// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
import { assert } from "../_util/asserts.ts";
const DEFAULT_BUFFER_SIZE = 32 * 1024;
/**
 * Copy N size at the most. If read size is lesser than N, then returns nread
 * @param r Reader
 * @param dest Writer
 * @param size Read size
 */ export async function copyN(r, dest, size) {
    let bytesRead = 0;
    let buf = new Uint8Array(DEFAULT_BUFFER_SIZE);
    while(bytesRead < size){
        if (size - bytesRead < DEFAULT_BUFFER_SIZE) {
            buf = new Uint8Array(size - bytesRead);
        }
        const result = await r.read(buf);
        const nread = result ?? 0;
        bytesRead += nread;
        if (nread > 0) {
            let n = 0;
            while(n < nread){
                n += await dest.write(buf.slice(n, nread));
            }
            assert(n === nread, "could not write");
        }
        if (result === null) {
            break;
        }
    }
    return bytesRead;
}
/**
 * Read big endian 16bit short from BufReader
 * @param buf
 */ export async function readShort(buf) {
    const high = await buf.readByte();
    if (high === null) return null;
    const low = await buf.readByte();
    if (low === null) throw new Deno.errors.UnexpectedEof();
    return high << 8 | low;
}
/**
 * Read big endian 32bit integer from BufReader
 * @param buf
 */ export async function readInt(buf) {
    const high = await readShort(buf);
    if (high === null) return null;
    const low = await readShort(buf);
    if (low === null) throw new Deno.errors.UnexpectedEof();
    return high << 16 | low;
}
const MAX_SAFE_INTEGER = BigInt(Number.MAX_SAFE_INTEGER);
/**
 * Read big endian 64bit long from BufReader
 * @param buf
 */ export async function readLong(buf) {
    const high = await readInt(buf);
    if (high === null) return null;
    const low = await readInt(buf);
    if (low === null) throw new Deno.errors.UnexpectedEof();
    const big = BigInt(high) << 32n | BigInt(low);
    // We probably should provide a similar API that returns BigInt values.
    if (big > MAX_SAFE_INTEGER) {
        throw new RangeError("Long value too big to be represented as a JavaScript number.");
    }
    return Number(big);
}
/**
 * Slice number into 64bit big endian byte array
 * @param d The number to be sliced
 * @param dest The sliced array
 */ export function sliceLongToBytes(d, dest = Array.from({
    length: 8
})) {
    let big = BigInt(d);
    for(let i = 0; i < 8; i++){
        dest[7 - i] = Number(big & 0xffn);
        big >>= 8n;
    }
    return dest;
}
//# sourceMappingURL=data:application/json;base64,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