// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
import { assert } from "../_util/asserts.ts";
import { copy } from "../bytes/mod.ts";
const MAX_SIZE = 2 ** 32 - 2;
const DEFAULT_CHUNK_SIZE = 16_640;
/** A variable-sized buffer of bytes with `read()` and `write()` methods.
 *
 * Buffer is almost always used with some I/O like files and sockets. It allows
 * one to buffer up a download from a socket. Buffer grows and shrinks as
 * necessary.
 *
 * Buffer is NOT the same thing as Node's Buffer. Node's Buffer was created in
 * 2009 before JavaScript had the concept of ArrayBuffers. It's simply a
 * non-standard ArrayBuffer.
 *
 * ArrayBuffer is a fixed memory allocation. Buffer is implemented on top of
 * ArrayBuffer.
 *
 * Based on [Go Buffer](https://golang.org/pkg/bytes/#Buffer). */ export class Buffer {
    #buf;
    #off = 0;
    #readable = new ReadableStream({
        type: "bytes",
        pull: (controller)=>{
            const view = new Uint8Array(controller.byobRequest.view.buffer);
            if (this.empty()) {
                // Buffer is empty, reset to recover space.
                this.reset();
                controller.close();
                controller.byobRequest.respond(0);
                return;
            }
            const nread = copy(this.#buf.subarray(this.#off), view);
            this.#off += nread;
            controller.byobRequest.respond(nread);
        },
        autoAllocateChunkSize: DEFAULT_CHUNK_SIZE
    });
    get readable() {
        return this.#readable;
    }
    #writable = new WritableStream({
        write: (chunk)=>{
            const m = this.#grow(chunk.byteLength);
            copy(chunk, this.#buf, m);
        }
    });
    get writable() {
        return this.#writable;
    }
    constructor(ab){
        this.#buf = ab === undefined ? new Uint8Array(0) : new Uint8Array(ab);
    }
    /** Returns a slice holding the unread portion of the buffer.
   *
   * The slice is valid for use only until the next buffer modification (that
   * is, only until the next call to a method like `read()`, `write()`,
   * `reset()`, or `truncate()`). If `options.copy` is false the slice aliases the buffer content at
   * least until the next buffer modification, so immediate changes to the
   * slice will affect the result of future reads.
   * @param options Defaults to `{ copy: true }`
   */ bytes(options = {
        copy: true
    }) {
        if (options.copy === false) return this.#buf.subarray(this.#off);
        return this.#buf.slice(this.#off);
    }
    /** Returns whether the unread portion of the buffer is empty. */ empty() {
        return this.#buf.byteLength <= this.#off;
    }
    /** A read only number of bytes of the unread portion of the buffer. */ get length() {
        return this.#buf.byteLength - this.#off;
    }
    /** The read only capacity of the buffer's underlying byte slice, that is,
   * the total space allocated for the buffer's data. */ get capacity() {
        return this.#buf.buffer.byteLength;
    }
    /** Discards all but the first `n` unread bytes from the buffer but
   * continues to use the same allocated storage. It throws if `n` is
   * negative or greater than the length of the buffer. */ truncate(n) {
        if (n === 0) {
            this.reset();
            return;
        }
        if (n < 0 || n > this.length) {
            throw Error("bytes.Buffer: truncation out of range");
        }
        this.#reslice(this.#off + n);
    }
    reset() {
        this.#reslice(0);
        this.#off = 0;
    }
    #tryGrowByReslice(n) {
        const l = this.#buf.byteLength;
        if (n <= this.capacity - l) {
            this.#reslice(l + n);
            return l;
        }
        return -1;
    }
    #reslice(len) {
        assert(len <= this.#buf.buffer.byteLength);
        this.#buf = new Uint8Array(this.#buf.buffer, 0, len);
    }
    #grow(n1) {
        const m = this.length;
        // If buffer is empty, reset to recover space.
        if (m === 0 && this.#off !== 0) {
            this.reset();
        }
        // Fast: Try to grow by means of a reslice.
        const i = this.#tryGrowByReslice(n1);
        if (i >= 0) {
            return i;
        }
        const c = this.capacity;
        if (n1 <= Math.floor(c / 2) - m) {
            // We can slide things down instead of allocating a new
            // ArrayBuffer. We only need m+n <= c to slide, but
            // we instead let capacity get twice as large so we
            // don't spend all our time copying.
            copy(this.#buf.subarray(this.#off), this.#buf);
        } else if (c + n1 > MAX_SIZE) {
            throw new Error("The buffer cannot be grown beyond the maximum size.");
        } else {
            // Not enough space anywhere, we need to allocate.
            const buf = new Uint8Array(Math.min(2 * c + n1, MAX_SIZE));
            copy(this.#buf.subarray(this.#off), buf);
            this.#buf = buf;
        }
        // Restore this.#off and len(this.#buf).
        this.#off = 0;
        this.#reslice(Math.min(m + n1, MAX_SIZE));
        return m;
    }
    /** Grows the buffer's capacity, if necessary, to guarantee space for
   * another `n` bytes. After `.grow(n)`, at least `n` bytes can be written to
   * the buffer without another allocation. If `n` is negative, `.grow()` will
   * throw. If the buffer can't grow it will throw an error.
   *
   * Based on Go Lang's
   * [Buffer.Grow](https://golang.org/pkg/bytes/#Buffer.Grow). */ grow(n) {
        if (n < 0) {
            throw Error("Buffer.grow: negative count");
        }
        const m = this.#grow(n);
        this.#reslice(m);
    }
}
/** A TransformStream that will only read & enqueue `size` amount of bytes.
 * This operation is chunk based and not BYOB based,
 * and as such will read more than needed.
 *
 * if options.error is set, then instead of terminating the stream,
 * an error will be thrown.
 *
 * ```ts
 * import { LimitedBytesTransformStream } from "https://deno.land/std@$STD_VERSION/streams/buffer.ts";
 * const res = await fetch("https://example.com");
 * const parts = res.body!
 *   .pipeThrough(new LimitedBytesTransformStream(512 * 1024));
 * ```
 */ export class LimitedBytesTransformStream extends TransformStream {
    #read = 0;
    constructor(size, options = {}){
        super({
            transform: (chunk, controller)=>{
                if (this.#read + chunk.byteLength > size) {
                    if (options.error) {
                        throw new RangeError(`Exceeded byte size limit of '${size}'`);
                    } else {
                        controller.terminate();
                    }
                } else {
                    this.#read += chunk.byteLength;
                    controller.enqueue(chunk);
                }
            }
        });
    }
}
/** A TransformStream that will only read & enqueue `size` amount of chunks.
 *
 * if options.error is set, then instead of terminating the stream,
 * an error will be thrown.
 *
 * ```ts
 * import { LimitedTransformStream } from "https://deno.land/std@$STD_VERSION/streams/buffer.ts";
 * const res = await fetch("https://example.com");
 * const parts = res.body!.pipeThrough(new LimitedTransformStream(50));
 * ```
 */ export class LimitedTransformStream extends TransformStream {
    #read = 0;
    constructor(size, options = {}){
        super({
            transform: (chunk, controller)=>{
                if (this.#read + 1 > size) {
                    if (options.error) {
                        throw new RangeError(`Exceeded chunk limit of '${size}'`);
                    } else {
                        controller.terminate();
                    }
                } else {
                    this.#read++;
                    controller.enqueue(chunk);
                }
            }
        });
    }
}
/**
 * A transform stream that only transforms from the zero-indexed `start` and `end` bytes (both inclusive).
 *
 * @example
 * ```ts
 * import { ByteSliceStream } from "https://deno.land/std@$STD_VERSION/streams/buffer.ts";
 * const response = await fetch("https://example.com");
 * const rangedStream = response.body!
 *   .pipeThrough(new ByteSliceStream(3, 8));
 * ```
 */ export class ByteSliceStream extends TransformStream {
    #offsetStart = 0;
    #offsetEnd = 0;
    constructor(start = 0, end = Infinity){
        super({
            start: ()=>{
                assert(start >= 0, "`start` must be greater than 0");
                end += 1;
            },
            transform: (chunk, controller)=>{
                this.#offsetStart = this.#offsetEnd;
                this.#offsetEnd += chunk.byteLength;
                if (this.#offsetEnd > start) {
                    if (this.#offsetStart < start) {
                        chunk = chunk.slice(start - this.#offsetStart);
                    }
                    if (this.#offsetEnd >= end) {
                        chunk = chunk.slice(0, chunk.byteLength - this.#offsetEnd + end);
                        controller.enqueue(chunk);
                        controller.terminate();
                    } else {
                        controller.enqueue(chunk);
                    }
                }
            }
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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