// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
// Copyright (c) Jason Campbell. MIT license
/**
 * {@linkcode createExtractor}, {@linkcode recognize} and {@linkcode test} functions
 * to handle many forms of front matter.
 *
 * Adapted from
 * [jxson/front-matter](https://github.com/jxson/front-matter/blob/36f139ef797bd9e5196a9ede03ef481d7fbca18e/index.js).
 *
 * @module
 */ export var Format;
(function(Format) {
    Format["YAML"] = "yaml";
    Format["TOML"] = "toml";
    Format["JSON"] = "json";
    Format["UNKNOWN"] = "unknown";
})(Format || (Format = {}));
const { isArray  } = Array;
const [RX_RECOGNIZE_YAML, RX_YAML] = createRegExp([
    "---yaml",
    "---"
], "= yaml =", "---");
const [RX_RECOGNIZE_TOML, RX_TOML] = createRegExp([
    "---toml",
    "---"
], "= toml =");
const [RX_RECOGNIZE_JSON, RX_JSON] = createRegExp([
    "---json",
    "---"
], "= json =");
const MAP_FORMAT_TO_RECOGNIZER_RX = {
    [Format.YAML]: RX_RECOGNIZE_YAML,
    [Format.TOML]: RX_RECOGNIZE_TOML,
    [Format.JSON]: RX_RECOGNIZE_JSON
};
const MAP_FORMAT_TO_EXTRACTOR_RX = {
    [Format.YAML]: RX_YAML,
    [Format.TOML]: RX_TOML,
    [Format.JSON]: RX_JSON
};
function getBeginToken(delimiter) {
    return isArray(delimiter) ? delimiter[0] : delimiter;
}
function getEndToken(delimiter) {
    return isArray(delimiter) ? delimiter[1] : delimiter;
}
function createRegExp(...dv) {
    const beginPattern = "(" + dv.map(getBeginToken).join("|") + ")";
    const pattern = "^(" + "\\ufeff?" + // Maybe byte order mark
    beginPattern + "$([\\s\\S]+?)" + "^(?:" + dv.map(getEndToken).join("|") + ")\\s*" + "$" + (Deno.build.os === "windows" ? "\\r?" : "") + "(?:\\n)?)";
    return [
        new RegExp("^" + beginPattern + "$", "im"),
        new RegExp(pattern, "im")
    ];
}
function _extract(str, rx, parse) {
    const match = rx.exec(str);
    if (!match || match.index !== 0) {
        throw new TypeError("Unexpected end of input");
    }
    const frontMatter = match.at(-1)?.replace(/^\s+|\s+$/g, "") || "";
    const attrs = parse(frontMatter);
    const body = str.replace(match[0], "");
    return {
        frontMatter,
        body,
        attrs
    };
}
/**
 * Factory that creates a function that extracts front matter from a string with the given parsers.
 * Supports YAML, TOML and JSON.
 *
 * @param formats A descriptor containing Format-parser pairs to use for each format.
 * @returns A function that extracts front matter from a string with the given parsers.
 *
 * ```ts
 * import { createExtractor, Format, Parser } from "https://deno.land/std@$STD_VERSION/encoding/front_matter/mod.ts";
 * import { assertEquals } from "https://deno.land/std@$STD_VERSION/testing/asserts.ts";
 * import { parse as parseYAML } from "https://deno.land/std@$STD_VERSION/encoding/yaml.ts";
 * import { parse as parseTOML } from "https://deno.land/std@$STD_VERSION/encoding/toml.ts";
 * const extractYAML = createExtractor({ [Format.YAML]: parseYAML as Parser });
 * const extractTOML = createExtractor({ [Format.TOML]: parseTOML as Parser });
 * const extractJSON = createExtractor({ [Format.JSON]: JSON.parse as Parser });
 * const extractYAMLOrJSON = createExtractor({
 *     [Format.YAML]: parseYAML as Parser,
 *     [Format.JSON]: JSON.parse as Parser,
 * });
 *
 * let { attrs, body, frontMatter } = extractYAML<{ title: string }>("---\ntitle: Three dashes marks the spot\n---\nferret");
 * assertEquals(attrs.title, "Three dashes marks the spot");
 * assertEquals(body, "ferret");
 * assertEquals(frontMatter, "title: Three dashes marks the spot");
 *
 * ({ attrs, body, frontMatter } = extractTOML<{ title: string }>("---toml\ntitle = 'Three dashes followed by format marks the spot'\n---\n"));
 * assertEquals(attrs.title, "Three dashes followed by format marks the spot");
 * assertEquals(body, "");
 * assertEquals(frontMatter, "title = 'Three dashes followed by format marks the spot'");
 *
 * ({ attrs, body, frontMatter } = extractJSON<{ title: string }>("---json\n{\"title\": \"Three dashes followed by format marks the spot\"}\n---\ngoat"));
 * assertEquals(attrs.title, "Three dashes followed by format marks the spot");
 * assertEquals(body, "goat");
 * assertEquals(frontMatter, "{\"title\": \"Three dashes followed by format marks the spot\"}");
 *
 * ({ attrs, body, frontMatter } = extractYAMLOrJSON<{ title: string }>("---\ntitle: Three dashes marks the spot\n---\nferret"));
 * assertEquals(attrs.title, "Three dashes marks the spot");
 * assertEquals(body, "ferret");
 * assertEquals(frontMatter, "title: Three dashes marks the spot");
 *
 * ({ attrs, body, frontMatter } = extractYAMLOrJSON<{ title: string }>("---json\n{\"title\": \"Three dashes followed by format marks the spot\"}\n---\ngoat"));
 * assertEquals(attrs.title, "Three dashes followed by format marks the spot");
 * assertEquals(body, "goat");
 * assertEquals(frontMatter, "{\"title\": \"Three dashes followed by format marks the spot\"}");
 * ```
 */ export function createExtractor(formats) {
    const formatKeys = Object.keys(formats);
    return function extract(str) {
        const format = recognize(str, formatKeys);
        const parser = formats[format];
        if (format === Format.UNKNOWN || !parser) {
            throw new TypeError(`Unsupported front matter format`);
        }
        return _extract(str, MAP_FORMAT_TO_EXTRACTOR_RX[format], parser);
    };
}
/**
 * Tests if a string has valid front matter. Supports YAML, TOML and JSON.
 *
 * @param str String to test.
 * @param formats A list of formats to test for. Defaults to all supported formats.
 *
 * ```ts
 * import { test, Format } from "https://deno.land/std@$STD_VERSION/encoding/front_matter/mod.ts";
 * import { assert } from "https://deno.land/std@$STD_VERSION/testing/asserts.ts";
 *
 * assert(test("---\ntitle: Three dashes marks the spot\n---\n"));
 * assert(test("---toml\ntitle = 'Three dashes followed by format marks the spot'\n---\n"));
 * assert(test("---json\n{\"title\": \"Three dashes followed by format marks the spot\"}\n---\n"));
 *
 * assert(!test("---json\n{\"title\": \"Three dashes followed by format marks the spot\"}\n---\n", [Format.YAML]));
 * ```
 */ export function test(str, formats) {
    if (!formats) {
        formats = Object.keys(MAP_FORMAT_TO_EXTRACTOR_RX);
    }
    for (const format of formats){
        if (format === Format.UNKNOWN) {
            throw new TypeError("Unable to test for unknown front matter format");
        }
        const match = MAP_FORMAT_TO_EXTRACTOR_RX[format].exec(str);
        if (match?.index === 0) {
            return true;
        }
    }
    return false;
}
/**
 * Recognizes the format of the front matter in a string. Supports YAML, TOML and JSON.
 *
 * @param str String to recognize.
 * @param formats A list of formats to recognize. Defaults to all supported formats.
 *
 * ```ts
 * import { recognize, Format } from "https://deno.land/std@$STD_VERSION/encoding/front_matter/mod.ts";
 * import { assertEquals } from "https://deno.land/std@$STD_VERSION/testing/asserts.ts";
 *
 * assertEquals(recognize("---\ntitle: Three dashes marks the spot\n---\n"), Format.YAML);
 * assertEquals(recognize("---toml\ntitle = 'Three dashes followed by format marks the spot'\n---\n"), Format.TOML);
 * assertEquals(recognize("---json\n{\"title\": \"Three dashes followed by format marks the spot\"}\n---\n"), Format.JSON);
 * assertEquals(recognize("---xml\n<title>Three dashes marks the spot</title>\n---\n"), Format.UNKNOWN);
 *
 * assertEquals(recognize("---json\n<title>Three dashes marks the spot</title>\n---\n", [Format.YAML]), Format.UNKNOWN);
 */ function recognize(str, formats) {
    if (!formats) {
        formats = Object.keys(MAP_FORMAT_TO_RECOGNIZER_RX);
    }
    const [firstLine] = str.split(/(\r?\n)/);
    for (const format of formats){
        if (format === Format.UNKNOWN) {
            continue;
        }
        if (MAP_FORMAT_TO_RECOGNIZER_RX[format].test(firstLine)) {
            return format;
        }
    }
    return Format.UNKNOWN;
}
//# sourceMappingURL=data:application/json;base64,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