// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
// Structured similarly to Go's cookie.go
// https://github.com/golang/go/blob/master/src/net/http/cookie.go
// This module is browser compatible.
import { assert } from "../_util/asserts.ts";
import { toIMF } from "../datetime/mod.ts";
const FIELD_CONTENT_REGEXP = /^(?=[\x20-\x7E]*$)[^()@<>,;:\\"\[\]?={}\s]+$/;
function toString(cookie) {
    if (!cookie.name) {
        return "";
    }
    const out = [];
    validateName(cookie.name);
    validateValue(cookie.name, cookie.value);
    out.push(`${cookie.name}=${cookie.value}`);
    // Fallback for invalid Set-Cookie
    // ref: https://tools.ietf.org/html/draft-ietf-httpbis-cookie-prefixes-00#section-3.1
    if (cookie.name.startsWith("__Secure")) {
        cookie.secure = true;
    }
    if (cookie.name.startsWith("__Host")) {
        cookie.path = "/";
        cookie.secure = true;
        delete cookie.domain;
    }
    if (cookie.secure) {
        out.push("Secure");
    }
    if (cookie.httpOnly) {
        out.push("HttpOnly");
    }
    if (typeof cookie.maxAge === "number" && Number.isInteger(cookie.maxAge)) {
        assert(cookie.maxAge >= 0, "Max-Age must be an integer superior or equal to 0");
        out.push(`Max-Age=${cookie.maxAge}`);
    }
    if (cookie.domain) {
        validateDomain(cookie.domain);
        out.push(`Domain=${cookie.domain}`);
    }
    if (cookie.sameSite) {
        out.push(`SameSite=${cookie.sameSite}`);
    }
    if (cookie.path) {
        validatePath(cookie.path);
        out.push(`Path=${cookie.path}`);
    }
    if (cookie.expires) {
        const { expires  } = cookie;
        const dateString = toIMF(typeof expires === "number" ? new Date(expires) : expires);
        out.push(`Expires=${dateString}`);
    }
    if (cookie.unparsed) {
        out.push(cookie.unparsed.join("; "));
    }
    return out.join("; ");
}
/**
 * Validate Cookie Name.
 * @param name Cookie name.
 */ function validateName(name) {
    if (name && !FIELD_CONTENT_REGEXP.test(name)) {
        throw new TypeError(`Invalid cookie name: "${name}".`);
    }
}
/**
 * Validate Path Value.
 * See {@link https://tools.ietf.org/html/rfc6265#section-4.1.2.4}.
 * @param path Path value.
 */ function validatePath(path) {
    if (path == null) {
        return;
    }
    for(let i = 0; i < path.length; i++){
        const c = path.charAt(i);
        if (c < String.fromCharCode(0x20) || c > String.fromCharCode(0x7E) || c == ";") {
            throw new Error(path + ": Invalid cookie path char '" + c + "'");
        }
    }
}
/**
 * Validate Cookie Value.
 * See {@link https://tools.ietf.org/html/rfc6265#section-4.1}.
 * @param value Cookie value.
 */ function validateValue(name, value) {
    if (value == null || name == null) return;
    for(let i = 0; i < value.length; i++){
        const c = value.charAt(i);
        if (c < String.fromCharCode(0x21) || c == String.fromCharCode(0x22) || c == String.fromCharCode(0x2c) || c == String.fromCharCode(0x3b) || c == String.fromCharCode(0x5c) || c == String.fromCharCode(0x7f)) {
            throw new Error("RFC2616 cookie '" + name + "' cannot contain character '" + c + "'");
        }
        if (c > String.fromCharCode(0x80)) {
            throw new Error("RFC2616 cookie '" + name + "' can only have US-ASCII chars as value" + c.charCodeAt(0).toString(16));
        }
    }
}
/**
 * Validate Cookie Domain.
 * See {@link https://datatracker.ietf.org/doc/html/rfc6265#section-4.1.2.3}.
 * @param domain Cookie domain.
 */ function validateDomain(domain) {
    if (domain == null) {
        return;
    }
    const char1 = domain.charAt(0);
    const charN = domain.charAt(domain.length - 1);
    if (char1 == "-" || charN == "." || charN == "-") {
        throw new Error("Invalid first/last char in cookie domain: " + domain);
    }
}
/**
 * Parse cookies of a header
 * @param headers The headers instance to get cookies from
 * @return Object with cookie names as keys
 */ export function getCookies(headers) {
    const cookie = headers.get("Cookie");
    if (cookie != null) {
        const out = {};
        const c = cookie.split(";");
        for (const kv of c){
            const [cookieKey, ...cookieVal] = kv.split("=");
            assert(cookieKey != null);
            const key = cookieKey.trim();
            out[key] = cookieVal.join("=");
        }
        return out;
    }
    return {};
}
/**
 * Set the cookie header properly in the headers
 * @param headers The headers instance to set the cookie to
 * @param cookie Cookie to set
 */ export function setCookie(headers, cookie) {
    // Parsing cookie headers to make consistent set-cookie header
    // ref: https://tools.ietf.org/html/rfc6265#section-4.1.1
    const v = toString(cookie);
    if (v) {
        headers.append("Set-Cookie", v);
    }
}
/**
 * Set the cookie header with empty value in the headers to delete it
 * @param headers The headers instance to delete the cookie from
 * @param name Name of cookie
 * @param attributes Additional cookie attributes
 */ export function deleteCookie(headers, name, attributes) {
    setCookie(headers, {
        name: name,
        value: "",
        expires: new Date(0),
        ...attributes
    });
}
function parseSetCookie(value) {
    const attrs = value.split(";").map((attr)=>attr.trim().split("=").map((keyOrValue)=>keyOrValue.trim()));
    const cookie = {
        name: attrs[0][0],
        value: attrs[0][1]
    };
    for (const [key, value1] of attrs.slice(1)){
        switch(key.toLocaleLowerCase()){
            case "expires":
                cookie.expires = new Date(value1);
                break;
            case "max-age":
                cookie.maxAge = Number(value1);
                if (cookie.maxAge < 0) {
                    console.warn("Max-Age must be an integer superior or equal to 0. Cookie ignored.");
                    return null;
                }
                break;
            case "domain":
                cookie.domain = value1;
                break;
            case "path":
                cookie.path = value1;
                break;
            case "secure":
                cookie.secure = true;
                break;
            case "httponly":
                cookie.httpOnly = true;
                break;
            case "samesite":
                cookie.sameSite = value1;
                break;
            default:
                if (!Array.isArray(cookie.unparsed)) {
                    cookie.unparsed = [];
                }
                cookie.unparsed.push([
                    key,
                    value1
                ].join("="));
        }
    }
    if (cookie.name.startsWith("__Secure-")) {
        /** This requirement is mentioned in https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Set-Cookie but not the RFC. */ if (!cookie.secure) {
            console.warn("Cookies with names starting with `__Secure-` must be set with the secure flag. Cookie ignored.");
            return null;
        }
    }
    if (cookie.name.startsWith("__Host-")) {
        if (!cookie.secure) {
            console.warn("Cookies with names starting with `__Host-` must be set with the secure flag. Cookie ignored.");
            return null;
        }
        if (cookie.domain !== undefined) {
            console.warn("Cookies with names starting with `__Host-` must not have a domain specified. Cookie ignored.");
            return null;
        }
        if (cookie.path !== "/") {
            console.warn("Cookies with names starting with `__Host-` must have path be `/`. Cookie has been ignored.");
            return null;
        }
    }
    return cookie;
}
/**
 * Parse set-cookies of a header
 * @param headers The headers instance to get set-cookies from
 * @return List of cookies
 */ export function getSetCookies(headers) {
    if (!headers.has("set-cookie")) {
        return [];
    }
    return [
        ...headers.entries()
    ].filter(([key])=>key === "set-cookie").map(([_, value])=>value)/** Parse each `set-cookie` header separately */ .map(parseSetCookie)/** Skip empty cookies */ .filter(Boolean);
}
//# sourceMappingURL=data:application/json;base64,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