// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/** This module is browser compatible. */ import { descend } from "./_comparators.ts";
export * from "./_comparators.ts";
/** Swaps the values at two indexes in an array. */ function swap(array, a, b) {
    const temp = array[a];
    array[a] = array[b];
    array[b] = temp;
}
/** Returns the parent index for a child index. */ function getParentIndex(index) {
    return Math.floor((index + 1) / 2) - 1;
}
/**
 * A priority queue implemented with a binary heap. The heap is in decending order by default,
 * using JavaScript's built in comparison operators to sort the values.
 */ export class BinaryHeap {
    #data;
    constructor(compare = descend){
        this.compare = compare;
        this.#data = [];
    }
    static from(collection, options) {
        let result;
        let unmappedValues = [];
        if (collection instanceof BinaryHeap) {
            result = new BinaryHeap(options?.compare ?? collection.compare);
            if (options?.compare || options?.map) {
                unmappedValues = collection.#data;
            } else {
                result.#data = Array.from(collection.#data);
            }
        } else {
            result = options?.compare ? new BinaryHeap(options.compare) : new BinaryHeap();
            unmappedValues = collection;
        }
        const values = options?.map ? Array.from(unmappedValues, options.map, options.thisArg) : unmappedValues;
        result.push(...values);
        return result;
    }
    /** The amount of values stored in the binary heap. */ get length() {
        return this.#data.length;
    }
    /** Returns the greatest value in the binary heap, or undefined if it is empty. */ peek() {
        return this.#data[0];
    }
    /** Removes the greatest value from the binary heap and returns it, or null if it is empty. */ pop() {
        const size = this.#data.length - 1;
        swap(this.#data, 0, size);
        let parent = 0;
        let right = 2 * (parent + 1);
        let left = right - 1;
        while(left < size){
            const greatestChild = right === size || this.compare(this.#data[left], this.#data[right]) <= 0 ? left : right;
            if (this.compare(this.#data[greatestChild], this.#data[parent]) < 0) {
                swap(this.#data, parent, greatestChild);
                parent = greatestChild;
            } else {
                break;
            }
            right = 2 * (parent + 1);
            left = right - 1;
        }
        return this.#data.pop();
    }
    /** Adds values to the binary heap. */ push(...values) {
        for (const value of values){
            let index = this.#data.length;
            let parent = getParentIndex(index);
            this.#data.push(value);
            while(index !== 0 && this.compare(this.#data[index], this.#data[parent]) < 0){
                swap(this.#data, parent, index);
                index = parent;
                parent = getParentIndex(index);
            }
        }
        return this.#data.length;
    }
    /** Removes all values from the binary heap. */ clear() {
        this.#data = [];
    }
    /** Checks if the binary heap is empty. */ isEmpty() {
        return this.#data.length === 0;
    }
    /** Returns an iterator for retrieving and removing values from the binary heap. */ *drain() {
        while(!this.isEmpty()){
            yield this.pop();
        }
    }
    *[Symbol.iterator]() {
        yield* this.drain();
    }
    compare;
}
//# sourceMappingURL=data:application/json;base64,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