// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
import { assert } from "../../_util/asserts.ts";
import { ERR_BARE_QUOTE, ERR_FIELD_COUNT, ERR_INVALID_DELIM, ERR_QUOTE, ParseError } from "./_io.ts";
export class Parser {
    #input = "";
    #cursor = 0;
    #options;
    constructor({ separator ="," , trimLeadingSpace =false , comment , lazyQuotes , fieldsPerRecord  } = {}){
        this.#options = {
            separator,
            trimLeadingSpace,
            comment,
            lazyQuotes,
            fieldsPerRecord
        };
    }
    #readLine() {
        if (this.#isEOF()) return null;
        if (!this.#input.startsWith("\r\n", this.#cursor) || !this.#input.startsWith("\n", this.#cursor)) {
            let buffer = "";
            let hadNewline = false;
            while(this.#cursor < this.#input.length){
                if (this.#input.startsWith("\r\n", this.#cursor)) {
                    hadNewline = true;
                    this.#cursor += 2;
                    break;
                }
                if (this.#input.startsWith("\n", this.#cursor)) {
                    hadNewline = true;
                    this.#cursor += 1;
                    break;
                }
                buffer += this.#input[this.#cursor];
                this.#cursor += 1;
            }
            if (!hadNewline && buffer.endsWith("\r")) {
                buffer = buffer.slice(0, -1);
            }
            return buffer;
        }
        return null;
    }
    #isEOF() {
        return this.#cursor >= this.#input.length;
    }
    #parseRecord(startLine) {
        let line = this.#readLine();
        if (line === null) return null;
        if (line.length === 0) {
            return [];
        }
        function runeCount(s) {
            // Array.from considers the surrogate pair.
            return Array.from(s).length;
        }
        let lineIndex = startLine + 1;
        // line starting with comment character is ignored
        if (this.#options.comment && line[0] === this.#options.comment) {
            return [];
        }
        assert(this.#options.separator != null);
        let fullLine = line;
        let quoteError = null;
        const quote = '"';
        const quoteLen = quote.length;
        const separatorLen = this.#options.separator.length;
        let recordBuffer = "";
        const fieldIndexes = [];
        parseField: for(;;){
            if (this.#options.trimLeadingSpace) {
                line = line.trimStart();
            }
            if (line.length === 0 || !line.startsWith(quote)) {
                // Non-quoted string field
                const i = line.indexOf(this.#options.separator);
                let field = line;
                if (i >= 0) {
                    field = field.substring(0, i);
                }
                // Check to make sure a quote does not appear in field.
                if (!this.#options.lazyQuotes) {
                    const j = field.indexOf(quote);
                    if (j >= 0) {
                        const col = runeCount(fullLine.slice(0, fullLine.length - line.slice(j).length));
                        quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_BARE_QUOTE);
                        break parseField;
                    }
                }
                recordBuffer += field;
                fieldIndexes.push(recordBuffer.length);
                if (i >= 0) {
                    line = line.substring(i + separatorLen);
                    continue parseField;
                }
                break parseField;
            } else {
                // Quoted string field
                line = line.substring(quoteLen);
                for(;;){
                    const i1 = line.indexOf(quote);
                    if (i1 >= 0) {
                        // Hit next quote.
                        recordBuffer += line.substring(0, i1);
                        line = line.substring(i1 + quoteLen);
                        if (line.startsWith(quote)) {
                            // `""` sequence (append quote).
                            recordBuffer += quote;
                            line = line.substring(quoteLen);
                        } else if (line.startsWith(this.#options.separator)) {
                            // `","` sequence (end of field).
                            line = line.substring(separatorLen);
                            fieldIndexes.push(recordBuffer.length);
                            continue parseField;
                        } else if (0 === line.length) {
                            // `"\n` sequence (end of line).
                            fieldIndexes.push(recordBuffer.length);
                            break parseField;
                        } else if (this.#options.lazyQuotes) {
                            // `"` sequence (bare quote).
                            recordBuffer += quote;
                        } else {
                            // `"*` sequence (invalid non-escaped quote).
                            const col1 = runeCount(fullLine.slice(0, fullLine.length - line.length - quoteLen));
                            quoteError = new ParseError(startLine + 1, lineIndex, col1, ERR_QUOTE);
                            break parseField;
                        }
                    } else if (line.length > 0 || !this.#isEOF()) {
                        // Hit end of line (copy all data so far).
                        recordBuffer += line;
                        const r = this.#readLine();
                        lineIndex++;
                        line = r ?? ""; // This is a workaround for making this module behave similarly to the encoding/csv/reader.go.
                        fullLine = line;
                        if (r === null) {
                            // Abrupt end of file (EOF or error).
                            if (!this.#options.lazyQuotes) {
                                const col2 = runeCount(fullLine);
                                quoteError = new ParseError(startLine + 1, lineIndex, col2, ERR_QUOTE);
                                break parseField;
                            }
                            fieldIndexes.push(recordBuffer.length);
                            break parseField;
                        }
                        recordBuffer += "\n"; // preserve line feed (This is because TextProtoReader removes it.)
                    } else {
                        // Abrupt end of file (EOF on error).
                        if (!this.#options.lazyQuotes) {
                            const col3 = runeCount(fullLine);
                            quoteError = new ParseError(startLine + 1, lineIndex, col3, ERR_QUOTE);
                            break parseField;
                        }
                        fieldIndexes.push(recordBuffer.length);
                        break parseField;
                    }
                }
            }
        }
        if (quoteError) {
            throw quoteError;
        }
        const result = [];
        let preIdx = 0;
        for (const i2 of fieldIndexes){
            result.push(recordBuffer.slice(preIdx, i2));
            preIdx = i2;
        }
        return result;
    }
    parse(input) {
        this.#input = input;
        this.#cursor = 0;
        const result = [];
        let _nbFields;
        let lineResult;
        let first = true;
        let lineIndex = 0;
        const INVALID_RUNE = [
            "\r",
            "\n",
            '"'
        ];
        const options = this.#options;
        if (INVALID_RUNE.includes(options.separator) || typeof options.comment === "string" && INVALID_RUNE.includes(options.comment) || options.separator === options.comment) {
            throw new Error(ERR_INVALID_DELIM);
        }
        for(;;){
            const r = this.#parseRecord(lineIndex);
            if (r === null) break;
            lineResult = r;
            lineIndex++;
            // If fieldsPerRecord is 0, Read sets it to
            // the number of fields in the first record
            if (first) {
                first = false;
                if (options.fieldsPerRecord !== undefined) {
                    if (options.fieldsPerRecord === 0) {
                        _nbFields = lineResult.length;
                    } else {
                        _nbFields = options.fieldsPerRecord;
                    }
                }
            }
            if (lineResult.length > 0) {
                if (_nbFields && _nbFields !== lineResult.length) {
                    throw new ParseError(lineIndex, lineIndex, null, ERR_FIELD_COUNT);
                }
                result.push(lineResult);
            }
        }
        return result;
    }
}
//# sourceMappingURL=data:application/json;base64,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