// Originally ported from Go:
// https://github.com/golang/go/blob/go1.12.5/src/encoding/csv/
// Copyright 2011 The Go Authors. All rights reserved. BSD license.
// https://github.com/golang/go/blob/master/LICENSE
// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
import { assert } from "../../_util/asserts.ts";
export const defaultReadOptions = {
    separator: ",",
    trimLeadingSpace: false
};
export async function readRecord(startLine, reader, opt = defaultReadOptions) {
    const line = await reader.readLine();
    if (line === null) return null;
    if (line.length === 0) {
        return [];
    }
    return parseRecord(line, reader, opt, startLine, startLine + 1);
}
export async function parseRecord(line, reader, opt = defaultReadOptions, startLine, lineIndex = startLine) {
    // line starting with comment character is ignored
    if (opt.comment && line[0] === opt.comment) {
        return [];
    }
    assert(opt.separator != null);
    let fullLine = line;
    let quoteError = null;
    const quote = '"';
    const quoteLen = quote.length;
    const separatorLen = opt.separator.length;
    let recordBuffer = "";
    const fieldIndexes = [];
    parseField: for(;;){
        if (opt.trimLeadingSpace) {
            line = line.trimStart();
        }
        if (line.length === 0 || !line.startsWith(quote)) {
            // Non-quoted string field
            const i = line.indexOf(opt.separator);
            let field = line;
            if (i >= 0) {
                field = field.substring(0, i);
            }
            // Check to make sure a quote does not appear in field.
            if (!opt.lazyQuotes) {
                const j = field.indexOf(quote);
                if (j >= 0) {
                    const col = runeCount(fullLine.slice(0, fullLine.length - line.slice(j).length));
                    quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_BARE_QUOTE);
                    break parseField;
                }
            }
            recordBuffer += field;
            fieldIndexes.push(recordBuffer.length);
            if (i >= 0) {
                line = line.substring(i + separatorLen);
                continue parseField;
            }
            break parseField;
        } else {
            // Quoted string field
            line = line.substring(quoteLen);
            for(;;){
                const i1 = line.indexOf(quote);
                if (i1 >= 0) {
                    // Hit next quote.
                    recordBuffer += line.substring(0, i1);
                    line = line.substring(i1 + quoteLen);
                    if (line.startsWith(quote)) {
                        // `""` sequence (append quote).
                        recordBuffer += quote;
                        line = line.substring(quoteLen);
                    } else if (line.startsWith(opt.separator)) {
                        // `","` sequence (end of field).
                        line = line.substring(separatorLen);
                        fieldIndexes.push(recordBuffer.length);
                        continue parseField;
                    } else if (0 === line.length) {
                        // `"\n` sequence (end of line).
                        fieldIndexes.push(recordBuffer.length);
                        break parseField;
                    } else if (opt.lazyQuotes) {
                        // `"` sequence (bare quote).
                        recordBuffer += quote;
                    } else {
                        // `"*` sequence (invalid non-escaped quote).
                        const col1 = runeCount(fullLine.slice(0, fullLine.length - line.length - quoteLen));
                        quoteError = new ParseError(startLine + 1, lineIndex, col1, ERR_QUOTE);
                        break parseField;
                    }
                } else if (line.length > 0 || !await reader.isEOF()) {
                    // Hit end of line (copy all data so far).
                    recordBuffer += line;
                    const r = await reader.readLine();
                    lineIndex++;
                    line = r ?? ""; // This is a workaround for making this module behave similarly to the encoding/csv/reader.go.
                    fullLine = line;
                    if (r === null) {
                        // Abrupt end of file (EOF or error).
                        if (!opt.lazyQuotes) {
                            const col2 = runeCount(fullLine);
                            quoteError = new ParseError(startLine + 1, lineIndex, col2, ERR_QUOTE);
                            break parseField;
                        }
                        fieldIndexes.push(recordBuffer.length);
                        break parseField;
                    }
                    recordBuffer += "\n"; // preserve line feed (This is because TextProtoReader removes it.)
                } else {
                    // Abrupt end of file (EOF on error).
                    if (!opt.lazyQuotes) {
                        const col3 = runeCount(fullLine);
                        quoteError = new ParseError(startLine + 1, lineIndex, col3, ERR_QUOTE);
                        break parseField;
                    }
                    fieldIndexes.push(recordBuffer.length);
                    break parseField;
                }
            }
        }
    }
    if (quoteError) {
        throw quoteError;
    }
    const result = [];
    let preIdx = 0;
    for (const i2 of fieldIndexes){
        result.push(recordBuffer.slice(preIdx, i2));
        preIdx = i2;
    }
    return result;
}
function runeCount(s) {
    // Array.from considers the surrogate pair.
    return Array.from(s).length;
}
/**
 * A ParseError is returned for parsing errors.
 * Line numbers are 1-indexed and columns are 0-indexed.
 */ export class ParseError extends SyntaxError {
    /** Line where the record starts*/ startLine;
    /** Line where the error occurred */ line;
    /** Column (rune index) where the error occurred */ column;
    constructor(start, line, column, message){
        super();
        this.startLine = start;
        this.column = column;
        this.line = line;
        if (message === ERR_FIELD_COUNT) {
            this.message = `record on line ${line}: ${message}`;
        } else if (start !== line) {
            this.message = `record on line ${start}; parse error on line ${line}, column ${column}: ${message}`;
        } else {
            this.message = `parse error on line ${line}, column ${column}: ${message}`;
        }
    }
}
export const ERR_BARE_QUOTE = 'bare " in non-quoted-field';
export const ERR_QUOTE = 'extraneous or missing " in quoted-field';
export const ERR_INVALID_DELIM = "Invalid Delimiter";
export const ERR_FIELD_COUNT = "wrong number of fields";
//# sourceMappingURL=data:application/json;base64,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