// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/** {@linkcode parse} function for parsing
 * [JSONC](https://code.visualstudio.com/docs/languages/json#_json-with-comments)
 * strings.
 *
 * This module is browser compatible.
 *
 * @module
 */ import { assert } from "../_util/asserts.ts";
/**
 * Converts a JSON with Comments (JSONC) string into an object.
 * If a syntax error is found, throw a SyntaxError.
 *
 * @param text A valid JSONC string.
 * @param options
 * @param options.allowTrailingComma Allow trailing commas at the end of arrays and objects. (default: `true`)
 *
 * ```ts
 * import * as JSONC from "https://deno.land/std@$STD_VERSION/encoding/jsonc.ts";
 *
 * JSONC.parse('{"foo": "bar", } // comment'); //=> { foo: "bar" }
 * JSONC.parse('{"foo": "bar" } // comment', { allowTrailingComma: false }); //=> { foo: "bar" }
 * ```
 */ export function parse(text, { allowTrailingComma =true  } = {}) {
    if (new.target) {
        throw new TypeError("parse is not a constructor");
    }
    return new JSONCParser(text, {
        allowTrailingComma
    }).parse();
}
var tokenType;
(function(tokenType) {
    tokenType[tokenType["beginObject"] = 0] = "beginObject";
    tokenType[tokenType["endObject"] = 1] = "endObject";
    tokenType[tokenType["beginArray"] = 2] = "beginArray";
    tokenType[tokenType["endArray"] = 3] = "endArray";
    tokenType[tokenType["nameSeparator"] = 4] = "nameSeparator";
    tokenType[tokenType["valueSeparator"] = 5] = "valueSeparator";
    tokenType[tokenType["nullOrTrueOrFalseOrNumber"] = 6] = "nullOrTrueOrFalseOrNumber";
    tokenType[tokenType["string"] = 7] = "string";
})(tokenType || (tokenType = {}));
const originalJSONParse = globalThis.JSON.parse;
// First tokenize and then parse the token.
class JSONCParser {
    #whitespace = new Set(" \t\r\n");
    #numberEndToken = new Set([
        ..."[]{}:,/",
        ...this.#whitespace
    ]);
    #text;
    #length;
    #tokenized;
    #options;
    constructor(text, options){
        this.#text = `${text}`;
        this.#length = this.#text.length;
        this.#tokenized = this.#tokenize();
        this.#options = options;
    }
    parse() {
        const token = this.#getNext();
        const res = this.#parseJSONValue(token);
        // make sure all characters have been read
        const { done , value  } = this.#tokenized.next();
        if (!done) {
            throw new SyntaxError(buildErrorMessage(value));
        }
        return res;
    }
    /** Read the next token. If the token is read to the end, it throws a SyntaxError. */ #getNext() {
        const { done , value  } = this.#tokenized.next();
        if (done) {
            throw new SyntaxError("Unexpected end of JSONC input");
        }
        return value;
    }
    /** Split the JSONC string into token units. Whitespace and comments are skipped. */ *#tokenize() {
        for(let i = 0; i < this.#length; i++){
            // skip whitespace
            if (this.#whitespace.has(this.#text[i])) {
                continue;
            }
            // skip multi line comment (`/*...*/`)
            if (this.#text[i] === "/" && this.#text[i + 1] === "*") {
                i += 2;
                let hasEndOfComment = false;
                for(; i < this.#length; i++){
                    if (this.#text[i] === "*" && this.#text[i + 1] === "/") {
                        hasEndOfComment = true;
                        break;
                    }
                }
                if (!hasEndOfComment) {
                    throw new SyntaxError("Unexpected end of JSONC input");
                }
                i++;
                continue;
            }
            // skip single line comment (`//...`)
            if (this.#text[i] === "/" && this.#text[i + 1] === "/") {
                i += 2;
                for(; i < this.#length; i++){
                    if (this.#text[i] === "\n" || this.#text[i] === "\r") {
                        break;
                    }
                }
                continue;
            }
            switch(this.#text[i]){
                case "{":
                    yield {
                        type: tokenType.beginObject,
                        position: i
                    };
                    break;
                case "}":
                    yield {
                        type: tokenType.endObject,
                        position: i
                    };
                    break;
                case "[":
                    yield {
                        type: tokenType.beginArray,
                        position: i
                    };
                    break;
                case "]":
                    yield {
                        type: tokenType.endArray,
                        position: i
                    };
                    break;
                case ":":
                    yield {
                        type: tokenType.nameSeparator,
                        position: i
                    };
                    break;
                case ",":
                    yield {
                        type: tokenType.valueSeparator,
                        position: i
                    };
                    break;
                case '"':
                    {
                        const startIndex = i;
                        // Need to handle consecutive backslashes correctly
                        // '"\\""' => '"'
                        // '"\\\\"' => '\\'
                        // '"\\\\\\""' => '\\"'
                        // '"\\\\\\\\"' => '\\\\'
                        let shouldEscapeNext = false;
                        i++;
                        for(; i < this.#length; i++){
                            if (this.#text[i] === '"' && !shouldEscapeNext) {
                                break;
                            }
                            shouldEscapeNext = this.#text[i] === "\\" && !shouldEscapeNext;
                        }
                        yield {
                            type: tokenType.string,
                            sourceText: this.#text.substring(startIndex, i + 1),
                            position: startIndex
                        };
                        break;
                    }
                default:
                    {
                        const startIndex1 = i;
                        for(; i < this.#length; i++){
                            if (this.#numberEndToken.has(this.#text[i])) {
                                break;
                            }
                        }
                        i--;
                        yield {
                            type: tokenType.nullOrTrueOrFalseOrNumber,
                            sourceText: this.#text.substring(startIndex1, i + 1),
                            position: startIndex1
                        };
                    }
            }
        }
    }
    #parseJSONValue(value1) {
        switch(value1.type){
            case tokenType.beginObject:
                return this.#parseObject();
            case tokenType.beginArray:
                return this.#parseArray();
            case tokenType.nullOrTrueOrFalseOrNumber:
                return this.#parseNullOrTrueOrFalseOrNumber(value1);
            case tokenType.string:
                return this.#parseString(value1);
            default:
                throw new SyntaxError(buildErrorMessage(value1));
        }
    }
    #parseObject() {
        const target = {};
        //   ┌─token1
        // { }
        //      ┌─────────────token1
        //      │   ┌─────────token2
        //      │   │   ┌─────token3
        //      │   │   │   ┌─token4
        //  { "key" : value }
        //      ┌───────────────token1
        //      │   ┌───────────token2
        //      │   │   ┌───────token3
        //      │   │   │   ┌───token4
        //      │   │   │   │ ┌─token1
        //  { "key" : value , }
        //      ┌─────────────────────────────token1
        //      │   ┌─────────────────────────token2
        //      │   │   ┌─────────────────────token3
        //      │   │   │   ┌─────────────────token4
        //      │   │   │   │   ┌─────────────token1
        //      │   │   │   │   │   ┌─────────token2
        //      │   │   │   │   │   │   ┌─────token3
        //      │   │   │   │   │   │   │   ┌─token4
        //  { "key" : value , "key" : value }
        for(let isFirst = true;; isFirst = false){
            const token1 = this.#getNext();
            if ((isFirst || this.#options.allowTrailingComma) && token1.type === tokenType.endObject) {
                return target;
            }
            if (token1.type !== tokenType.string) {
                throw new SyntaxError(buildErrorMessage(token1));
            }
            const key = this.#parseString(token1);
            const token2 = this.#getNext();
            if (token2.type !== tokenType.nameSeparator) {
                throw new SyntaxError(buildErrorMessage(token2));
            }
            const token3 = this.#getNext();
            target[key] = this.#parseJSONValue(token3);
            const token4 = this.#getNext();
            if (token4.type === tokenType.endObject) {
                return target;
            }
            if (token4.type !== tokenType.valueSeparator) {
                throw new SyntaxError(buildErrorMessage(token4));
            }
        }
    }
    #parseArray() {
        const target1 = [];
        //   ┌─token1
        // [ ]
        //      ┌─────────────token1
        //      │   ┌─────────token2
        //  [ value ]
        //      ┌───────token1
        //      │   ┌───token2
        //      │   │ ┌─token1
        //  [ value , ]
        //      ┌─────────────token1
        //      │   ┌─────────token2
        //      │   │   ┌─────token1
        //      │   │   │   ┌─token2
        //  [ value , value ]
        for(let isFirst1 = true;; isFirst1 = false){
            const token11 = this.#getNext();
            if ((isFirst1 || this.#options.allowTrailingComma) && token11.type === tokenType.endArray) {
                return target1;
            }
            target1.push(this.#parseJSONValue(token11));
            const token21 = this.#getNext();
            if (token21.type === tokenType.endArray) {
                return target1;
            }
            if (token21.type !== tokenType.valueSeparator) {
                throw new SyntaxError(buildErrorMessage(token21));
            }
        }
    }
    #parseString(value2) {
        let parsed;
        try {
            // Use JSON.parse to handle `\u0000` etc. correctly.
            parsed = originalJSONParse(value2.sourceText);
        } catch  {
            throw new SyntaxError(buildErrorMessage(value2));
        }
        assert(typeof parsed === "string");
        return parsed;
    }
    #parseNullOrTrueOrFalseOrNumber(value3) {
        if (value3.sourceText === "null") {
            return null;
        }
        if (value3.sourceText === "true") {
            return true;
        }
        if (value3.sourceText === "false") {
            return false;
        }
        let parsed1;
        try {
            // Use JSON.parse to handle `+100`, `Infinity` etc. correctly.
            parsed1 = originalJSONParse(value3.sourceText);
        } catch  {
            throw new SyntaxError(buildErrorMessage(value3));
        }
        assert(typeof parsed1 === "number");
        return parsed1;
    }
}
function buildErrorMessage({ type , sourceText , position  }) {
    let token = "";
    switch(type){
        case tokenType.beginObject:
            token = "{";
            break;
        case tokenType.endObject:
            token = "}";
            break;
        case tokenType.beginArray:
            token = "[";
            break;
        case tokenType.endArray:
            token = "]";
            break;
        case tokenType.nameSeparator:
            token = ":";
            break;
        case tokenType.valueSeparator:
            token = ",";
            break;
        case tokenType.nullOrTrueOrFalseOrNumber:
        case tokenType.string:
            // Truncate the string so that it is within 30 lengths.
            token = 30 < sourceText.length ? `${sourceText.slice(0, 30)}...` : sourceText;
            break;
        default:
            throw new Error("unreachable");
    }
    return `Unexpected token ${token} in JSONC at position ${position}`;
}
//# sourceMappingURL=data:application/json;base64,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