// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/**
 * Higher level API for dealing with OS signals.
 *
 * @module
 */ import { MuxAsyncIterator } from "../async/mux_async_iterator.ts";
import { deferred } from "../async/deferred.ts";
/**
 * Generates an AsyncIterable which can be awaited on for one or more signals.
 * `dispose()` can be called when you are finished waiting on the events.
 *
 * Example:
 *
 * ```ts
 *       import { signal } from "https://deno.land/std@$STD_VERSION/signal/mod.ts";
 *
 *       const sig = signal("SIGUSR1", "SIGINT");
 *       setTimeout(() => {}, 5000); // Prevents exiting immediately
 *
 *       for await (const _ of sig) {
 *         console.log("interrupt or usr1 signal received");
 *       }
 *
 *       // At some other point in your code when finished listening:
 *       sig.dispose();
 * ```
 *
 * @param signals - one or more signals to listen to
 */ export function signal(...signals) {
    const mux = new MuxAsyncIterator();
    if (signals.length < 1) {
        throw new Error("No signals are given. You need to specify at least one signal to create a signal stream.");
    }
    const streams = signals.map(createSignalStream);
    streams.forEach((stream)=>{
        mux.add(stream);
    });
    // Create dispose method for the muxer of signal streams.
    const dispose = ()=>{
        streams.forEach((stream)=>{
            stream.dispose();
        });
    };
    return Object.assign(mux, {
        dispose
    });
}
function createSignalStream(signal) {
    let streamContinues = deferred();
    const handler = ()=>{
        streamContinues.resolve(true);
    };
    Deno.addSignalListener(signal, handler);
    const gen = async function*() {
        while(await streamContinues){
            streamContinues = deferred();
            yield undefined;
        }
    };
    return Object.assign(gen(), {
        dispose () {
            streamContinues.resolve(false);
            Deno.removeSignalListener(signal, handler);
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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