// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/**
 * Load environment variables from `.env` files.
 *
 * @module
 */ import { difference, removeEmptyValues } from "./util.ts";
const RE_KeyValue = /^\s*(?:export\s+)?(?<key>[a-zA-Z_]+[a-zA-Z0-9_]*?)\s*=[\ \t]*('\n?(?<notInterpolated>(.|\n)*?)\n?'|"\n?(?<interpolated>(.|\n)*?)\n?"|(?<unquoted>[^\n#]*)) *#*.*$/gm;
const RE_ExpandValue = /(\${(?<inBrackets>.+?)(\:-(?<inBracketsDefault>.+))?}|(?<!\\)\$(?<notInBrackets>\w+)(\:-(?<notInBracketsDefault>.+))?)/g;
export function parse(rawDotenv, restrictEnvAccessTo = []) {
    const env = {};
    let match;
    const keysForExpandCheck = [];
    while((match = RE_KeyValue.exec(rawDotenv)) != null){
        const { key , interpolated , notInterpolated , unquoted  } = match?.groups;
        if (unquoted) {
            keysForExpandCheck.push(key);
        }
        env[key] = typeof notInterpolated === "string" ? notInterpolated : typeof interpolated === "string" ? expandCharacters(interpolated) : unquoted.trim();
    }
    //https://github.com/motdotla/dotenv-expand/blob/ed5fea5bf517a09fd743ce2c63150e88c8a5f6d1/lib/main.js#L23
    const variablesMap = {
        ...env,
        ...readEnv(restrictEnvAccessTo)
    };
    keysForExpandCheck.forEach((key)=>{
        env[key] = expand(env[key], variablesMap);
    });
    return env;
}
const defaultConfigOptions = {
    path: `.env`,
    export: false,
    safe: false,
    example: `.env.example`,
    allowEmptyValues: false,
    defaults: `.env.defaults`,
    restrictEnvAccessTo: []
};
export function configSync(options = {}) {
    const o = {
        ...defaultConfigOptions,
        ...options
    };
    const conf = parseFile(o.path, o.restrictEnvAccessTo);
    if (o.defaults) {
        const confDefaults = parseFile(o.defaults, o.restrictEnvAccessTo);
        for(const key in confDefaults){
            if (!(key in conf)) {
                conf[key] = confDefaults[key];
            }
        }
    }
    if (o.safe) {
        const confExample = parseFile(o.example, o.restrictEnvAccessTo);
        assertSafe(conf, confExample, o.allowEmptyValues, o.restrictEnvAccessTo);
    }
    if (o.export) {
        for(const key1 in conf){
            if (Deno.env.get(key1) !== undefined) continue;
            Deno.env.set(key1, conf[key1]);
        }
    }
    return conf;
}
export async function config(options = {}) {
    const o = {
        ...defaultConfigOptions,
        ...options
    };
    const conf = await parseFileAsync(o.path, o.restrictEnvAccessTo);
    if (o.defaults) {
        const confDefaults = await parseFileAsync(o.defaults, o.restrictEnvAccessTo);
        for(const key in confDefaults){
            if (!(key in conf)) {
                conf[key] = confDefaults[key];
            }
        }
    }
    if (o.safe) {
        const confExample = await parseFileAsync(o.example, o.restrictEnvAccessTo);
        assertSafe(conf, confExample, o.allowEmptyValues, o.restrictEnvAccessTo);
    }
    if (o.export) {
        for(const key1 in conf){
            if (Deno.env.get(key1) !== undefined) continue;
            Deno.env.set(key1, conf[key1]);
        }
    }
    return conf;
}
function parseFile(filepath, restrictEnvAccessTo = []) {
    try {
        return parse(new TextDecoder("utf-8").decode(Deno.readFileSync(filepath)), restrictEnvAccessTo);
    } catch (e) {
        if (e instanceof Deno.errors.NotFound) return {};
        throw e;
    }
}
async function parseFileAsync(filepath, restrictEnvAccessTo = []) {
    try {
        return parse(new TextDecoder("utf-8").decode(await Deno.readFile(filepath)), restrictEnvAccessTo);
    } catch (e) {
        if (e instanceof Deno.errors.NotFound) return {};
        throw e;
    }
}
function expandCharacters(str) {
    const charactersMap = {
        "\\n": "\n",
        "\\r": "\r",
        "\\t": "\t"
    };
    return str.replace(/\\([nrt])/g, ($1)=>charactersMap[$1]);
}
function assertSafe(conf, confExample, allowEmptyValues, restrictEnvAccessTo = []) {
    const currentEnv = readEnv(restrictEnvAccessTo);
    // Not all the variables have to be defined in .env, they can be supplied externally
    const confWithEnv = Object.assign({}, currentEnv, conf);
    const missing = difference(Object.keys(confExample), // If allowEmptyValues is false, filter out empty values from configuration
    Object.keys(allowEmptyValues ? confWithEnv : removeEmptyValues(confWithEnv)));
    if (missing.length > 0) {
        const errorMessages = [
            `The following variables were defined in the example file but are not present in the environment:\n  ${missing.join(", ")}`,
            `Make sure to add them to your env file.`,
            !allowEmptyValues && `If you expect any of these variables to be empty, you can set the allowEmptyValues option to true.`
        ];
        throw new MissingEnvVarsError(errorMessages.filter(Boolean).join("\n\n"), missing);
    }
}
// a guarded env access, that reads only a subset from the Deno.env object,
// if `restrictEnvAccessTo` property is passed.
function readEnv(restrictEnvAccessTo) {
    if (restrictEnvAccessTo && Array.isArray(restrictEnvAccessTo) && restrictEnvAccessTo.length > 0) {
        return restrictEnvAccessTo.reduce((accessedEnvVars, envVarName)=>{
            if (Deno.env.get(envVarName)) {
                accessedEnvVars[envVarName] = Deno.env.get(envVarName);
            }
            return accessedEnvVars;
        }, {});
    }
    return Deno.env.toObject();
}
export class MissingEnvVarsError extends Error {
    missing;
    constructor(message, missing){
        super(message);
        this.name = "MissingEnvVarsError";
        this.missing = missing;
        Object.setPrototypeOf(this, new.target.prototype);
    }
}
function expand(str, variablesMap) {
    if (RE_ExpandValue.test(str)) {
        return expand(str.replace(RE_ExpandValue, function(...params) {
            const { inBrackets , inBracketsDefault , notInBrackets , notInBracketsDefault  } = params[params.length - 1];
            const expandValue = inBrackets || notInBrackets;
            const defaultValue = inBracketsDefault || notInBracketsDefault;
            return variablesMap[expandValue] || expand(defaultValue, variablesMap);
        }), variablesMap);
    } else {
        return str;
    }
}
/**
 * @param object object to be stringified
 * @returns string of object
 * ```ts
 * import { stringify } from "https://deno.land/std@$STD_VERSION/dotenv/mod.ts";
 *
 * const object = { GREETING: "hello world" };
 * const string = stringify(object);
 * ```
 */ export function stringify(object) {
    const lines = [];
    for (const [key, value] of Object.entries(object)){
        let quote;
        let escapedValue = value ?? "";
        if (key.startsWith("#")) {
            console.warn(`key starts with a '#' indicates a comment and is ignored: '${key}'`);
            continue;
        } else if (escapedValue.includes("\n")) {
            // escape inner new lines
            escapedValue = escapedValue.replaceAll("\n", "\\n");
            quote = `"`;
        } else if (escapedValue.match(/\W/)) {
            quote = "'";
        }
        if (quote) {
            // escape inner quotes
            escapedValue = escapedValue.replaceAll(quote, `\\${quote}`);
            escapedValue = `${quote}${escapedValue}${quote}`;
        }
        const line = `${key}=${escapedValue}`;
        lines.push(line);
    }
    return lines.join("\n");
}
//# sourceMappingURL=data:application/json;base64,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