// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { bytesToUuid, uuidToBytes } from "./_common.ts";
import { concat } from "../bytes/mod.ts";
import { assert } from "../_util/asserts.ts";
const UUID_RE = /^[0-9a-f]{8}-[0-9a-f]{4}-[5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;
/**
 * Validate that the passed UUID is an RFC4122 v5 UUID.
 *
 * ```ts
 * import { generate as generateV5, validate } from "https://deno.land/std@$STD_VERSION/uuid/v5.ts";
 *
 * validate(await generateV5("6ba7b810-9dad-11d1-80b4-00c04fd430c8", new Uint8Array())); // true
 * validate(crypto.randomUUID()); // false
 * validate("this-is-not-a-uuid"); // false
 * ```
 */ export function validate(id) {
    return UUID_RE.test(id);
}
/**
 * Generate a RFC4122 v5 UUID (SHA-1 namespace).
 *
 * ```js
 * import { generate } from "https://deno.land/std@$STD_VERSION/uuid/v5.ts";
 *
 * const NAMESPACE_URL = "6ba7b810-9dad-11d1-80b4-00c04fd430c8";
 *
 * const uuid = await generate(NAMESPACE_URL, new TextEncoder().encode("python.org"));
 * uuid === "886313e1-3b8a-5372-9b90-0c9aee199e5d" // true
 * ```
 *
 * @param namespace The namespace to use, encoded as a UUID.
 * @param data The data to hash to calculate the SHA-1 digest for the UUID.
 */ export async function generate(namespace, data) {
    // TODO(lucacasonato): validate that `namespace` is a valid UUID.
    const space = uuidToBytes(namespace);
    assert(space.length === 16, "namespace must be a valid UUID");
    const toHash = concat(new Uint8Array(space), data);
    const buffer = await crypto.subtle.digest("sha-1", toHash);
    const bytes = new Uint8Array(buffer);
    bytes[6] = bytes[6] & 0x0f | 0x50;
    bytes[8] = bytes[8] & 0x3f | 0x80;
    return bytesToUuid(bytes);
}
//# sourceMappingURL=data:application/json;base64,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