// Copyright 2011 The Go Authors. All rights reserved. BSD license.
// https://github.com/golang/go/blob/master/LICENSE
// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/** Port of the Go
 * [encoding/csv](https://github.com/golang/go/blob/go1.12.5/src/encoding/csv/)
 * library.
 *
 * @module
 */ import { assert } from "../_util/asserts.ts";
import { Parser } from "./csv/_parser.ts";
export { ERR_BARE_QUOTE, ERR_FIELD_COUNT, ERR_INVALID_DELIM, ERR_QUOTE, ParseError } from "./csv/_io.ts";
const QUOTE = '"';
export const NEWLINE = "\r\n";
export class StringifyError extends Error {
    name = "StringifyError";
}
function getEscapedString(value, sep) {
    if (value === undefined || value === null) return "";
    let str = "";
    if (typeof value === "object") str = JSON.stringify(value);
    else str = String(value);
    // Is regex.test more performant here? If so, how to dynamically create?
    // https://stackoverflow.com/questions/3561493/
    if (str.includes(sep) || str.includes(NEWLINE) || str.includes(QUOTE)) {
        return `${QUOTE}${str.replaceAll(QUOTE, `${QUOTE}${QUOTE}`)}${QUOTE}`;
    }
    return str;
}
function normalizeColumn(column) {
    let header, prop;
    if (typeof column === "object") {
        if (Array.isArray(column)) {
            header = String(column[column.length - 1]);
            prop = column;
        } else {
            prop = Array.isArray(column.prop) ? column.prop : [
                column.prop
            ];
            header = typeof column.header === "string" ? column.header : String(prop[prop.length - 1]);
        }
    } else {
        header = String(column);
        prop = [
            column
        ];
    }
    return {
        header,
        prop
    };
}
/**
 * Returns an array of values from an object using the property accessors
 * (and optional transform function) in each column
 */ function getValuesFromItem(item, normalizedColumns) {
    const values = [];
    if (normalizedColumns.length) {
        for (const column of normalizedColumns){
            let value = item;
            for (const prop of column.prop){
                if (typeof value !== "object" || value === null) continue;
                if (Array.isArray(value)) {
                    if (typeof prop === "number") value = value[prop];
                    else {
                        throw new StringifyError('Property accessor is not of type "number"');
                    }
                } else value = value[prop];
            }
            values.push(value);
        }
    } else {
        if (Array.isArray(item)) {
            values.push(...item);
        } else if (typeof item === "object") {
            throw new StringifyError("No property accessor function was provided for object");
        } else {
            values.push(item);
        }
    }
    return values;
}
/**
 * @param data The array of objects to encode
 * @param options Output formatting options
 */ export function stringify(data, { headers =true , separator: sep = "," , columns =[]  } = {}) {
    if (sep.includes(QUOTE) || sep.includes(NEWLINE)) {
        const message = [
            "Separator cannot include the following strings:",
            '  - U+0022: Quotation mark (")',
            "  - U+000D U+000A: Carriage Return + Line Feed (\\r\\n)"
        ].join("\n");
        throw new StringifyError(message);
    }
    const normalizedColumns = columns.map(normalizeColumn);
    let output = "";
    if (headers) {
        output += normalizedColumns.map((column)=>getEscapedString(column.header, sep)).join(sep);
        output += NEWLINE;
    }
    for (const item of data){
        const values = getValuesFromItem(item, normalizedColumns);
        output += values.map((value)=>getEscapedString(value, sep)).join(sep);
        output += NEWLINE;
    }
    return output;
}
export function parse(input, opt = {
    skipFirstRow: false
}) {
    const parser = new Parser(opt);
    const r = parser.parse(input);
    if (opt.skipFirstRow || opt.columns) {
        let headers = [];
        let i = 0;
        if (opt.skipFirstRow) {
            const head = r.shift();
            assert(head != null);
            headers = head;
            i++;
        }
        if (opt.columns) {
            headers = opt.columns;
        }
        return r.map((e)=>{
            if (e.length !== headers.length) {
                throw new Error(`Error number of fields line: ${i}\nNumber of fields found: ${headers.length}\nExpected number of fields: ${e.length}`);
            }
            i++;
            const out = {};
            for(let j = 0; j < e.length; j++){
                out[headers[j]] = e[j];
            }
            return out;
        });
    }
    return r;
}
//# sourceMappingURL=data:application/json;base64,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