// deno-fmt-ignore-file
// deno-lint-ignore-file
// This code was bundled using `deno bundle` and it's not recommended to edit it manually

const LogLevels = {
    NOTSET: 0,
    DEBUG: 10,
    INFO: 20,
    WARN: 30,
    ERROR: 40,
    CRITICAL: 50
};
Object.keys(LogLevels).filter((key)=>isNaN(Number(key)));
const byLevel = {
    [LogLevels.NOTSET]: "NOTSET",
    [LogLevels.DEBUG]: "DEBUG",
    [LogLevels.INFO]: "INFO",
    [LogLevels.WARN]: "WARN",
    [LogLevels.ERROR]: "ERROR",
    [LogLevels.CRITICAL]: "CRITICAL"
};
function getLevelByName(name) {
    const level = LogLevels[name];
    if (level !== undefined) {
        return level;
    }
    throw new Error(`Cannot get log level: no level named ${name}`);
}
function getLevelName(level) {
    const levelName = byLevel[level];
    if (levelName) {
        return levelName;
    }
    throw new Error(`Cannot get log level: no name for level: ${level}`);
}
const DEFAULT_FORMATTER = ({ levelName, msg })=>`${levelName} ${msg}`;
class BaseHandler {
    #levelName;
    #level;
    formatter;
    constructor(levelName, options){
        const { formatter = DEFAULT_FORMATTER } = options ?? {};
        this.#levelName = levelName;
        this.#level = getLevelByName(levelName);
        this.formatter = formatter;
    }
    get level() {
        return this.#level;
    }
    set level(level) {
        this.#level = level;
        this.#levelName = getLevelName(level);
    }
    get levelName() {
        return this.#levelName;
    }
    set levelName(levelName) {
        this.#levelName = levelName;
        this.#level = getLevelByName(levelName);
    }
    handle(logRecord) {
        if (this.level > logRecord.level) return;
        const msg = this.format(logRecord);
        this.log(msg);
    }
    format(logRecord) {
        return this.formatter(logRecord);
    }
    setup() {}
    destroy() {}
    [Symbol.dispose]() {
        this.destroy();
    }
}
const { Deno: Deno1 } = globalThis;
const noColor = typeof Deno1?.noColor === "boolean" ? Deno1.noColor : false;
let enabled = !noColor;
function code(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run(str, code) {
    return enabled ? `${code.open}${str.replace(code.regexp, code.open)}${code.close}` : str;
}
function bold(str) {
    return run(str, code([
        1
    ], 22));
}
function red(str) {
    return run(str, code([
        31
    ], 39));
}
function yellow(str) {
    return run(str, code([
        33
    ], 39));
}
function blue(str) {
    return run(str, code([
        34
    ], 39));
}
new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]+)*|[a-zA-Z\\d]+(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TXZcf-nq-uy=><~]))"
].join("|"), "g");
function applyColors(msg, level) {
    switch(level){
        case LogLevels.INFO:
            msg = blue(msg);
            break;
        case LogLevels.WARN:
            msg = yellow(msg);
            break;
        case LogLevels.ERROR:
            msg = red(msg);
            break;
        case LogLevels.CRITICAL:
            msg = bold(red(msg));
            break;
        default:
            break;
    }
    return msg;
}
class ConsoleHandler extends BaseHandler {
    #useColors;
    constructor(levelName, options = {}){
        super(levelName, options);
        this.#useColors = options.useColors ?? true;
    }
    format(logRecord) {
        let msg = super.format(logRecord);
        if (this.#useColors) {
            msg = applyColors(msg, logRecord.level);
        }
        return msg;
    }
    log(msg) {
        console.log(msg);
    }
}
function writeAllSync(writer, data) {
    let nwritten = 0;
    while(nwritten < data.length){
        nwritten += writer.writeSync(data.subarray(nwritten));
    }
}
const fileSymbol = Symbol("file");
const bufSymbol = Symbol("buf");
const pointerSymbol = Symbol("pointer");
const filenameSymbol = Symbol("filename");
const modeSymbol = Symbol("mode");
const openOptionsSymbol = Symbol("openOptions");
const encoderSymbol = Symbol("encoder");
class FileHandler extends BaseHandler {
    [fileSymbol];
    [bufSymbol];
    [pointerSymbol] = 0;
    [filenameSymbol];
    [modeSymbol];
    [openOptionsSymbol];
    [encoderSymbol] = new TextEncoder();
    #unloadCallback = (()=>{
        this.destroy();
    }).bind(this);
    constructor(levelName, options){
        super(levelName, options);
        this[filenameSymbol] = options.filename;
        this[modeSymbol] = options.mode ?? "a";
        this[openOptionsSymbol] = {
            createNew: this[modeSymbol] === "x",
            create: this[modeSymbol] !== "x",
            append: this[modeSymbol] === "a",
            truncate: this[modeSymbol] !== "a",
            write: true
        };
        this[bufSymbol] = new Uint8Array(options.bufferSize ?? 4096);
    }
    setup() {
        this[fileSymbol] = Deno.openSync(this[filenameSymbol], this[openOptionsSymbol]);
        this.#resetBuffer();
        addEventListener("unload", this.#unloadCallback);
    }
    handle(logRecord) {
        super.handle(logRecord);
        if (logRecord.level > LogLevels.ERROR) {
            this.flush();
        }
    }
    log(msg) {
        const bytes = this[encoderSymbol].encode(msg + "\n");
        if (bytes.byteLength > this[bufSymbol].byteLength - this[pointerSymbol]) {
            this.flush();
        }
        if (bytes.byteLength > this[bufSymbol].byteLength) {
            writeAllSync(this[fileSymbol], bytes);
        } else {
            this[bufSymbol].set(bytes, this[pointerSymbol]);
            this[pointerSymbol] += bytes.byteLength;
        }
    }
    flush() {
        if (this[pointerSymbol] > 0 && this[fileSymbol]) {
            let written = 0;
            while(written < this[pointerSymbol]){
                written += this[fileSymbol].writeSync(this[bufSymbol].subarray(written, this[pointerSymbol]));
            }
            this.#resetBuffer();
        }
    }
    #resetBuffer() {
        this[pointerSymbol] = 0;
    }
    destroy() {
        this.flush();
        this[fileSymbol]?.close();
        this[fileSymbol] = undefined;
        removeEventListener("unload", this.#unloadCallback);
    }
}
class LogRecord {
    msg;
    #args;
    #datetime;
    level;
    levelName;
    loggerName;
    constructor(options){
        this.msg = options.msg;
        this.#args = [
            ...options.args
        ];
        this.level = options.level;
        this.loggerName = options.loggerName;
        this.#datetime = new Date();
        this.levelName = getLevelName(options.level);
    }
    get args() {
        return [
            ...this.#args
        ];
    }
    get datetime() {
        return new Date(this.#datetime.getTime());
    }
}
function asString(data, isProperty = false) {
    if (typeof data === "string") {
        if (isProperty) return `"${data}"`;
        return data;
    } else if (data === null || typeof data === "number" || typeof data === "bigint" || typeof data === "boolean" || typeof data === "undefined" || typeof data === "symbol") {
        return String(data);
    } else if (data instanceof Error) {
        return data.stack;
    } else if (typeof data === "object") {
        return `{${Object.entries(data).map(([k, v])=>`"${k}":${asString(v, true)}`).join(",")}}`;
    }
    return "undefined";
}
class Logger {
    #level;
    handlers;
    #loggerName;
    constructor(loggerName, levelName, options = {}){
        this.#loggerName = loggerName;
        this.#level = getLevelByName(levelName);
        this.handlers = options.handlers ?? [];
    }
    get level() {
        return this.#level;
    }
    set level(level) {
        try {
            this.#level = getLevelByName(getLevelName(level));
        } catch (_) {
            throw new TypeError(`Invalid log level: ${level}`);
        }
    }
    get levelName() {
        return getLevelName(this.#level);
    }
    set levelName(levelName) {
        this.#level = getLevelByName(levelName);
    }
    get loggerName() {
        return this.#loggerName;
    }
    #log(level, msg, ...args) {
        if (this.level > level) {
            return msg instanceof Function ? undefined : msg;
        }
        let fnResult;
        let logMessage;
        if (msg instanceof Function) {
            fnResult = msg();
            logMessage = asString(fnResult);
        } else {
            logMessage = asString(msg);
        }
        const record = new LogRecord({
            msg: logMessage,
            args: args,
            level: level,
            loggerName: this.loggerName
        });
        this.handlers.forEach((handler)=>{
            handler.handle(record);
        });
        return msg instanceof Function ? fnResult : msg;
    }
    debug(msg, ...args) {
        return this.#log(LogLevels.DEBUG, msg, ...args);
    }
    info(msg, ...args) {
        return this.#log(LogLevels.INFO, msg, ...args);
    }
    warn(msg, ...args) {
        return this.#log(LogLevels.WARN, msg, ...args);
    }
    error(msg, ...args) {
        return this.#log(LogLevels.ERROR, msg, ...args);
    }
    critical(msg, ...args) {
        return this.#log(LogLevels.CRITICAL, msg, ...args);
    }
}
const DEFAULT_LEVEL = "INFO";
const DEFAULT_CONFIG = {
    handlers: {
        default: new ConsoleHandler(DEFAULT_LEVEL)
    },
    loggers: {
        default: {
            level: DEFAULT_LEVEL,
            handlers: [
                "default"
            ]
        }
    }
};
const state = {
    handlers: new Map(),
    loggers: new Map(),
    config: DEFAULT_CONFIG
};
function getLogger(name) {
    if (!name) {
        const d = state.loggers.get("default");
        if (d === undefined) {
            throw new Error(`"default" logger must be set for getting logger without name`);
        }
        return d;
    }
    const result1 = state.loggers.get(name);
    if (!result1) {
        const logger = new Logger(name, "NOTSET", {
            handlers: []
        });
        state.loggers.set(name, logger);
        return logger;
    }
    return result1;
}
function setup(config) {
    state.config = {
        handlers: {
            ...DEFAULT_CONFIG.handlers,
            ...config.handlers
        },
        loggers: {
            ...DEFAULT_CONFIG.loggers,
            ...config.loggers
        }
    };
    state.handlers.forEach((handler)=>{
        handler.destroy();
    });
    state.handlers.clear();
    const handlers = state.config.handlers ?? {};
    for (const [handlerName, handler] of Object.entries(handlers)){
        handler.setup();
        state.handlers.set(handlerName, handler);
    }
    state.loggers.clear();
    const loggers = state.config.loggers ?? {};
    for (const [loggerName, loggerConfig] of Object.entries(loggers)){
        const handlerNames = loggerConfig.handlers ?? [];
        const handlers = [];
        handlerNames.forEach((handlerName)=>{
            const handler = state.handlers.get(handlerName);
            if (handler) {
                handlers.push(handler);
            }
        });
        const levelName = loggerConfig.level ?? DEFAULT_LEVEL;
        const logger = new Logger(loggerName, levelName, {
            handlers: handlers
        });
        state.loggers.set(loggerName, logger);
    }
}
setup(DEFAULT_CONFIG);
function debug(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").debug(msg, ...args);
    }
    return getLogger("default").debug(msg, ...args);
}
function error(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").error(msg, ...args);
    }
    return getLogger("default").error(msg, ...args);
}
function info(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").info(msg, ...args);
    }
    return getLogger("default").info(msg, ...args);
}
function warn(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").warn(msg, ...args);
    }
    return getLogger("default").warn(msg, ...args);
}
function assertPath(path) {
    if (typeof path !== "string") {
        throw new TypeError(`Path must be a string. Received ${JSON.stringify(path)}`);
    }
}
function stripSuffix(name, suffix) {
    if (suffix.length >= name.length) {
        return name;
    }
    const lenDiff = name.length - suffix.length;
    for(let i1 = suffix.length - 1; i1 >= 0; --i1){
        if (name.charCodeAt(lenDiff + i1) !== suffix.charCodeAt(i1)) {
            return name;
        }
    }
    return name.slice(0, -suffix.length);
}
function lastPathSegment(path, isSep, start = 0) {
    let matchedNonSeparator = false;
    let end = path.length;
    for(let i1 = path.length - 1; i1 >= start; --i1){
        if (isSep(path.charCodeAt(i1))) {
            if (matchedNonSeparator) {
                start = i1 + 1;
                break;
            }
        } else if (!matchedNonSeparator) {
            matchedNonSeparator = true;
            end = i1 + 1;
        }
    }
    return path.slice(start, end);
}
function assertArgs(path, suffix) {
    assertPath(path);
    if (path.length === 0) return path;
    if (typeof suffix !== "string") {
        throw new TypeError(`Suffix must be a string. Received ${JSON.stringify(suffix)}`);
    }
}
const CHAR_FORWARD_SLASH = 47;
function stripTrailingSeparators(segment, isSep) {
    if (segment.length <= 1) {
        return segment;
    }
    let end = segment.length;
    for(let i1 = segment.length - 1; i1 > 0; i1--){
        if (isSep(segment.charCodeAt(i1))) {
            end = i1;
        } else {
            break;
        }
    }
    return segment.slice(0, end);
}
function isPosixPathSeparator(code) {
    return code === 47;
}
function isPathSeparator(code) {
    return code === 47 || code === 92;
}
function isWindowsDeviceRoot(code) {
    return code >= 97 && code <= 122 || code >= 65 && code <= 90;
}
function basename(path, suffix = "") {
    assertArgs(path, suffix);
    let start = 0;
    if (path.length >= 2) {
        const drive = path.charCodeAt(0);
        if (isWindowsDeviceRoot(drive)) {
            if (path.charCodeAt(1) === 58) start = 2;
        }
    }
    const lastSegment = lastPathSegment(path, isPathSeparator, start);
    const strippedSegment = stripTrailingSeparators(lastSegment, isPathSeparator);
    return suffix ? stripSuffix(strippedSegment, suffix) : strippedSegment;
}
const SEPARATOR = "\\";
const SEPARATOR_PATTERN = /[\\/]+/;
function assertArg(path) {
    assertPath(path);
    if (path.length === 0) return ".";
}
function dirname(path) {
    assertArg(path);
    const len = path.length;
    let rootEnd = -1;
    let end = -1;
    let matchedSlash = true;
    let offset = 0;
    const code = path.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code)) {
            rootEnd = offset = 1;
            if (isPathSeparator(path.charCodeAt(1))) {
                let j = 2;
                let last = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path.charCodeAt(j))) break;
                }
                if (j < len && j !== last) {
                    last = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        last = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            return path;
                        }
                        if (j !== last) {
                            rootEnd = offset = j + 1;
                        }
                    }
                }
            }
        } else if (isWindowsDeviceRoot(code)) {
            if (path.charCodeAt(1) === 58) {
                rootEnd = offset = 2;
                if (len > 2) {
                    if (isPathSeparator(path.charCodeAt(2))) rootEnd = offset = 3;
                }
            }
        }
    } else if (isPathSeparator(code)) {
        return path;
    }
    for(let i1 = len - 1; i1 >= offset; --i1){
        if (isPathSeparator(path.charCodeAt(i1))) {
            if (!matchedSlash) {
                end = i1;
                break;
            }
        } else {
            matchedSlash = false;
        }
    }
    if (end === -1) {
        if (rootEnd === -1) return ".";
        else end = rootEnd;
    }
    return stripTrailingSeparators(path.slice(0, end), isPosixPathSeparator);
}
function extname(path) {
    assertPath(path);
    let start = 0;
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let preDotState = 0;
    if (path.length >= 2 && path.charCodeAt(1) === 58 && isWindowsDeviceRoot(path.charCodeAt(0))) {
        start = startPart = 2;
    }
    for(let i1 = path.length - 1; i1 >= start; --i1){
        const code = path.charCodeAt(i1);
        if (isPathSeparator(code)) {
            if (!matchedSlash) {
                startPart = i1 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i1 + 1;
        }
        if (code === 46) {
            if (startDot === -1) startDot = i1;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        return "";
    }
    return path.slice(startDot, end);
}
function _format(sep, pathObject) {
    const dir = pathObject.dir || pathObject.root;
    const base = pathObject.base || (pathObject.name || "") + (pathObject.ext || "");
    if (!dir) return base;
    if (base === sep) return dir;
    if (dir === pathObject.root) return dir + base;
    return dir + sep + base;
}
function assertArg1(pathObject) {
    if (pathObject === null || typeof pathObject !== "object") {
        throw new TypeError(`The "pathObject" argument must be of type Object. Received type ${typeof pathObject}`);
    }
}
function assertArg2(url) {
    url = url instanceof URL ? url : new URL(url);
    if (url.protocol !== "file:") {
        throw new TypeError("Must be a file URL.");
    }
    return url;
}
function fromFileUrl(url) {
    url = assertArg2(url);
    let path = decodeURIComponent(url.pathname.replace(/\//g, "\\").replace(/%(?![0-9A-Fa-f]{2})/g, "%25")).replace(/^\\*([A-Za-z]:)(\\|$)/, "$1\\");
    if (url.hostname !== "") {
        path = `\\\\${url.hostname}${path}`;
    }
    return path;
}
function isAbsolute(path) {
    assertPath(path);
    const len = path.length;
    if (len === 0) return false;
    const code = path.charCodeAt(0);
    if (isPathSeparator(code)) {
        return true;
    } else if (isWindowsDeviceRoot(code)) {
        if (len > 2 && path.charCodeAt(1) === 58) {
            if (isPathSeparator(path.charCodeAt(2))) return true;
        }
    }
    return false;
}
class AssertionError extends Error {
    constructor(message){
        super(message);
        this.name = "AssertionError";
    }
}
function assert(expr, msg = "") {
    if (!expr) {
        throw new AssertionError(msg);
    }
}
function assertArg3(path) {
    assertPath(path);
    if (path.length === 0) return ".";
}
function normalizeString(path, allowAboveRoot, separator, isPathSeparator) {
    let res = "";
    let lastSegmentLength = 0;
    let lastSlash = -1;
    let dots = 0;
    let code;
    for(let i1 = 0; i1 <= path.length; ++i1){
        if (i1 < path.length) code = path.charCodeAt(i1);
        else if (isPathSeparator(code)) break;
        else code = CHAR_FORWARD_SLASH;
        if (isPathSeparator(code)) {
            if (lastSlash === i1 - 1 || dots === 1) {} else if (lastSlash !== i1 - 1 && dots === 2) {
                if (res.length < 2 || lastSegmentLength !== 2 || res.charCodeAt(res.length - 1) !== 46 || res.charCodeAt(res.length - 2) !== 46) {
                    if (res.length > 2) {
                        const lastSlashIndex = res.lastIndexOf(separator);
                        if (lastSlashIndex === -1) {
                            res = "";
                            lastSegmentLength = 0;
                        } else {
                            res = res.slice(0, lastSlashIndex);
                            lastSegmentLength = res.length - 1 - res.lastIndexOf(separator);
                        }
                        lastSlash = i1;
                        dots = 0;
                        continue;
                    } else if (res.length === 2 || res.length === 1) {
                        res = "";
                        lastSegmentLength = 0;
                        lastSlash = i1;
                        dots = 0;
                        continue;
                    }
                }
                if (allowAboveRoot) {
                    if (res.length > 0) res += `${separator}..`;
                    else res = "..";
                    lastSegmentLength = 2;
                }
            } else {
                if (res.length > 0) res += separator + path.slice(lastSlash + 1, i1);
                else res = path.slice(lastSlash + 1, i1);
                lastSegmentLength = i1 - lastSlash - 1;
            }
            lastSlash = i1;
            dots = 0;
        } else if (code === 46 && dots !== -1) {
            ++dots;
        } else {
            dots = -1;
        }
    }
    return res;
}
function normalize(path) {
    assertArg3(path);
    const len = path.length;
    let rootEnd = 0;
    let device;
    let isAbsolute = false;
    const code = path.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code)) {
            isAbsolute = true;
            if (isPathSeparator(path.charCodeAt(1))) {
                let j = 2;
                let last = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path.charCodeAt(j))) break;
                }
                if (j < len && j !== last) {
                    const firstPart = path.slice(last, j);
                    last = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        last = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            return `\\\\${firstPart}\\${path.slice(last)}\\`;
                        } else if (j !== last) {
                            device = `\\\\${firstPart}\\${path.slice(last, j)}`;
                            rootEnd = j;
                        }
                    }
                }
            } else {
                rootEnd = 1;
            }
        } else if (isWindowsDeviceRoot(code)) {
            if (path.charCodeAt(1) === 58) {
                device = path.slice(0, 2);
                rootEnd = 2;
                if (len > 2) {
                    if (isPathSeparator(path.charCodeAt(2))) {
                        isAbsolute = true;
                        rootEnd = 3;
                    }
                }
            }
        }
    } else if (isPathSeparator(code)) {
        return "\\";
    }
    let tail;
    if (rootEnd < len) {
        tail = normalizeString(path.slice(rootEnd), !isAbsolute, "\\", isPathSeparator);
    } else {
        tail = "";
    }
    if (tail.length === 0 && !isAbsolute) tail = ".";
    if (tail.length > 0 && isPathSeparator(path.charCodeAt(len - 1))) {
        tail += "\\";
    }
    if (device === undefined) {
        if (isAbsolute) {
            if (tail.length > 0) return `\\${tail}`;
            else return "\\";
        } else if (tail.length > 0) {
            return tail;
        } else {
            return "";
        }
    } else if (isAbsolute) {
        if (tail.length > 0) return `${device}\\${tail}`;
        else return `${device}\\`;
    } else if (tail.length > 0) {
        return device + tail;
    } else {
        return device;
    }
}
function join(...paths) {
    if (paths.length === 0) return ".";
    let joined;
    let firstPart = null;
    for(let i1 = 0; i1 < paths.length; ++i1){
        const path = paths[i1];
        assertPath(path);
        if (path.length > 0) {
            if (joined === undefined) joined = firstPart = path;
            else joined += `\\${path}`;
        }
    }
    if (joined === undefined) return ".";
    let needsReplace = true;
    let slashCount = 0;
    assert(firstPart !== null);
    if (isPathSeparator(firstPart.charCodeAt(0))) {
        ++slashCount;
        const firstLen = firstPart.length;
        if (firstLen > 1) {
            if (isPathSeparator(firstPart.charCodeAt(1))) {
                ++slashCount;
                if (firstLen > 2) {
                    if (isPathSeparator(firstPart.charCodeAt(2))) ++slashCount;
                    else {
                        needsReplace = false;
                    }
                }
            }
        }
    }
    if (needsReplace) {
        for(; slashCount < joined.length; ++slashCount){
            if (!isPathSeparator(joined.charCodeAt(slashCount))) break;
        }
        if (slashCount >= 2) joined = `\\${joined.slice(slashCount)}`;
    }
    return normalize(joined);
}
function resolve(...pathSegments) {
    let resolvedDevice = "";
    let resolvedTail = "";
    let resolvedAbsolute = false;
    for(let i1 = pathSegments.length - 1; i1 >= -1; i1--){
        let path;
        const { Deno: Deno1 } = globalThis;
        if (i1 >= 0) {
            path = pathSegments[i1];
        } else if (!resolvedDevice) {
            if (typeof Deno1?.cwd !== "function") {
                throw new TypeError("Resolved a drive-letter-less path without a CWD.");
            }
            path = Deno1.cwd();
        } else {
            if (typeof Deno1?.env?.get !== "function" || typeof Deno1?.cwd !== "function") {
                throw new TypeError("Resolved a relative path without a CWD.");
            }
            path = Deno1.cwd();
            if (path === undefined || path.slice(0, 3).toLowerCase() !== `${resolvedDevice.toLowerCase()}\\`) {
                path = `${resolvedDevice}\\`;
            }
        }
        assertPath(path);
        const len = path.length;
        if (len === 0) continue;
        let rootEnd = 0;
        let device = "";
        let isAbsolute = false;
        const code = path.charCodeAt(0);
        if (len > 1) {
            if (isPathSeparator(code)) {
                isAbsolute = true;
                if (isPathSeparator(path.charCodeAt(1))) {
                    let j = 2;
                    let last = j;
                    for(; j < len; ++j){
                        if (isPathSeparator(path.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        const firstPart = path.slice(last, j);
                        last = j;
                        for(; j < len; ++j){
                            if (!isPathSeparator(path.charCodeAt(j))) break;
                        }
                        if (j < len && j !== last) {
                            last = j;
                            for(; j < len; ++j){
                                if (isPathSeparator(path.charCodeAt(j))) break;
                            }
                            if (j === len) {
                                device = `\\\\${firstPart}\\${path.slice(last)}`;
                                rootEnd = j;
                            } else if (j !== last) {
                                device = `\\\\${firstPart}\\${path.slice(last, j)}`;
                                rootEnd = j;
                            }
                        }
                    }
                } else {
                    rootEnd = 1;
                }
            } else if (isWindowsDeviceRoot(code)) {
                if (path.charCodeAt(1) === 58) {
                    device = path.slice(0, 2);
                    rootEnd = 2;
                    if (len > 2) {
                        if (isPathSeparator(path.charCodeAt(2))) {
                            isAbsolute = true;
                            rootEnd = 3;
                        }
                    }
                }
            }
        } else if (isPathSeparator(code)) {
            rootEnd = 1;
            isAbsolute = true;
        }
        if (device.length > 0 && resolvedDevice.length > 0 && device.toLowerCase() !== resolvedDevice.toLowerCase()) {
            continue;
        }
        if (resolvedDevice.length === 0 && device.length > 0) {
            resolvedDevice = device;
        }
        if (!resolvedAbsolute) {
            resolvedTail = `${path.slice(rootEnd)}\\${resolvedTail}`;
            resolvedAbsolute = isAbsolute;
        }
        if (resolvedAbsolute && resolvedDevice.length > 0) break;
    }
    resolvedTail = normalizeString(resolvedTail, !resolvedAbsolute, "\\", isPathSeparator);
    return resolvedDevice + (resolvedAbsolute ? "\\" : "") + resolvedTail || ".";
}
function assertArgs1(from, to) {
    assertPath(from);
    assertPath(to);
    if (from === to) return "";
}
function relative(from, to) {
    assertArgs1(from, to);
    const fromOrig = resolve(from);
    const toOrig = resolve(to);
    if (fromOrig === toOrig) return "";
    from = fromOrig.toLowerCase();
    to = toOrig.toLowerCase();
    if (from === to) return "";
    let fromStart = 0;
    let fromEnd = from.length;
    for(; fromStart < fromEnd; ++fromStart){
        if (from.charCodeAt(fromStart) !== 92) break;
    }
    for(; fromEnd - 1 > fromStart; --fromEnd){
        if (from.charCodeAt(fromEnd - 1) !== 92) break;
    }
    const fromLen = fromEnd - fromStart;
    let toStart = 0;
    let toEnd = to.length;
    for(; toStart < toEnd; ++toStart){
        if (to.charCodeAt(toStart) !== 92) break;
    }
    for(; toEnd - 1 > toStart; --toEnd){
        if (to.charCodeAt(toEnd - 1) !== 92) break;
    }
    const toLen = toEnd - toStart;
    const length = fromLen < toLen ? fromLen : toLen;
    let lastCommonSep = -1;
    let i1 = 0;
    for(; i1 <= length; ++i1){
        if (i1 === length) {
            if (toLen > length) {
                if (to.charCodeAt(toStart + i1) === 92) {
                    return toOrig.slice(toStart + i1 + 1);
                } else if (i1 === 2) {
                    return toOrig.slice(toStart + i1);
                }
            }
            if (fromLen > length) {
                if (from.charCodeAt(fromStart + i1) === 92) {
                    lastCommonSep = i1;
                } else if (i1 === 2) {
                    lastCommonSep = 3;
                }
            }
            break;
        }
        const fromCode = from.charCodeAt(fromStart + i1);
        const toCode = to.charCodeAt(toStart + i1);
        if (fromCode !== toCode) break;
        else if (fromCode === 92) lastCommonSep = i1;
    }
    if (i1 !== length && lastCommonSep === -1) {
        return toOrig;
    }
    let out = "";
    if (lastCommonSep === -1) lastCommonSep = 0;
    for(i1 = fromStart + lastCommonSep + 1; i1 <= fromEnd; ++i1){
        if (i1 === fromEnd || from.charCodeAt(i1) === 92) {
            if (out.length === 0) out += "..";
            else out += "\\..";
        }
    }
    if (out.length > 0) {
        return out + toOrig.slice(toStart + lastCommonSep, toEnd);
    } else {
        toStart += lastCommonSep;
        if (toOrig.charCodeAt(toStart) === 92) ++toStart;
        return toOrig.slice(toStart, toEnd);
    }
}
const WHITESPACE_ENCODINGS = {
    "\u0009": "%09",
    "\u000A": "%0A",
    "\u000B": "%0B",
    "\u000C": "%0C",
    "\u000D": "%0D",
    "\u0020": "%20"
};
function encodeWhitespace(string) {
    return string.replaceAll(/[\s]/g, (c)=>{
        return WHITESPACE_ENCODINGS[c] ?? c;
    });
}
function toFileUrl(path) {
    if (!isAbsolute(path)) {
        throw new TypeError("Must be an absolute path.");
    }
    const [, hostname, pathname] = path.match(/^(?:[/\\]{2}([^/\\]+)(?=[/\\](?:[^/\\]|$)))?(.*)/);
    const url = new URL("file:///");
    url.pathname = encodeWhitespace(pathname.replace(/%/g, "%25"));
    if (hostname !== undefined && hostname !== "localhost") {
        url.hostname = hostname;
        if (!url.hostname) {
            throw new TypeError("Invalid hostname.");
        }
    }
    return url;
}
function _common(paths, sep) {
    const [first = "", ...remaining] = paths;
    const parts = first.split(sep);
    let endOfPrefix = parts.length;
    let append = "";
    for (const path of remaining){
        const compare = path.split(sep);
        if (compare.length <= endOfPrefix) {
            endOfPrefix = compare.length;
            append = "";
        }
        for(let i1 = 0; i1 < endOfPrefix; i1++){
            if (compare[i1] !== parts[i1]) {
                endOfPrefix = i1;
                append = i1 === 0 ? "" : sep;
                break;
            }
        }
    }
    return parts.slice(0, endOfPrefix).join(sep) + append;
}
const regExpEscapeChars = [
    "!",
    "$",
    "(",
    ")",
    "*",
    "+",
    ".",
    "=",
    "?",
    "[",
    "\\",
    "^",
    "{",
    "|"
];
const rangeEscapeChars = [
    "-",
    "\\",
    "]"
];
function _globToRegExp(c, glob, { extended = true, globstar: globstarOption = true, caseInsensitive = false } = {}) {
    if (glob === "") {
        return /(?!)/;
    }
    let newLength = glob.length;
    for(; newLength > 1 && c.seps.includes(glob[newLength - 1]); newLength--);
    glob = glob.slice(0, newLength);
    let regExpString = "";
    for(let j = 0; j < glob.length;){
        let segment = "";
        const groupStack = [];
        let inRange = false;
        let inEscape = false;
        let endsWithSep = false;
        let i1 = j;
        for(; i1 < glob.length && !c.seps.includes(glob[i1]); i1++){
            if (inEscape) {
                inEscape = false;
                const escapeChars = inRange ? rangeEscapeChars : regExpEscapeChars;
                segment += escapeChars.includes(glob[i1]) ? `\\${glob[i1]}` : glob[i1];
                continue;
            }
            if (glob[i1] === c.escapePrefix) {
                inEscape = true;
                continue;
            }
            if (glob[i1] === "[") {
                if (!inRange) {
                    inRange = true;
                    segment += "[";
                    if (glob[i1 + 1] === "!") {
                        i1++;
                        segment += "^";
                    } else if (glob[i1 + 1] === "^") {
                        i1++;
                        segment += "\\^";
                    }
                    continue;
                } else if (glob[i1 + 1] === ":") {
                    let k = i1 + 1;
                    let value = "";
                    while(glob[k + 1] !== undefined && glob[k + 1] !== ":"){
                        value += glob[k + 1];
                        k++;
                    }
                    if (glob[k + 1] === ":" && glob[k + 2] === "]") {
                        i1 = k + 2;
                        if (value === "alnum") segment += "\\dA-Za-z";
                        else if (value === "alpha") segment += "A-Za-z";
                        else if (value === "ascii") segment += "\x00-\x7F";
                        else if (value === "blank") segment += "\t ";
                        else if (value === "cntrl") segment += "\x00-\x1F\x7F";
                        else if (value === "digit") segment += "\\d";
                        else if (value === "graph") segment += "\x21-\x7E";
                        else if (value === "lower") segment += "a-z";
                        else if (value === "print") segment += "\x20-\x7E";
                        else if (value === "punct") {
                            segment += "!\"#$%&'()*+,\\-./:;<=>?@[\\\\\\]^_‘{|}~";
                        } else if (value === "space") segment += "\\s\v";
                        else if (value === "upper") segment += "A-Z";
                        else if (value === "word") segment += "\\w";
                        else if (value === "xdigit") segment += "\\dA-Fa-f";
                        continue;
                    }
                }
            }
            if (glob[i1] === "]" && inRange) {
                inRange = false;
                segment += "]";
                continue;
            }
            if (inRange) {
                if (glob[i1] === "\\") {
                    segment += `\\\\`;
                } else {
                    segment += glob[i1];
                }
                continue;
            }
            if (glob[i1] === ")" && groupStack.length > 0 && groupStack[groupStack.length - 1] !== "BRACE") {
                segment += ")";
                const type = groupStack.pop();
                if (type === "!") {
                    segment += c.wildcard;
                } else if (type !== "@") {
                    segment += type;
                }
                continue;
            }
            if (glob[i1] === "|" && groupStack.length > 0 && groupStack[groupStack.length - 1] !== "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i1] === "+" && extended && glob[i1 + 1] === "(") {
                i1++;
                groupStack.push("+");
                segment += "(?:";
                continue;
            }
            if (glob[i1] === "@" && extended && glob[i1 + 1] === "(") {
                i1++;
                groupStack.push("@");
                segment += "(?:";
                continue;
            }
            if (glob[i1] === "?") {
                if (extended && glob[i1 + 1] === "(") {
                    i1++;
                    groupStack.push("?");
                    segment += "(?:";
                } else {
                    segment += ".";
                }
                continue;
            }
            if (glob[i1] === "!" && extended && glob[i1 + 1] === "(") {
                i1++;
                groupStack.push("!");
                segment += "(?!";
                continue;
            }
            if (glob[i1] === "{") {
                groupStack.push("BRACE");
                segment += "(?:";
                continue;
            }
            if (glob[i1] === "}" && groupStack[groupStack.length - 1] === "BRACE") {
                groupStack.pop();
                segment += ")";
                continue;
            }
            if (glob[i1] === "," && groupStack[groupStack.length - 1] === "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i1] === "*") {
                if (extended && glob[i1 + 1] === "(") {
                    i1++;
                    groupStack.push("*");
                    segment += "(?:";
                } else {
                    const prevChar = glob[i1 - 1];
                    let numStars = 1;
                    while(glob[i1 + 1] === "*"){
                        i1++;
                        numStars++;
                    }
                    const nextChar = glob[i1 + 1];
                    if (globstarOption && numStars === 2 && [
                        ...c.seps,
                        undefined
                    ].includes(prevChar) && [
                        ...c.seps,
                        undefined
                    ].includes(nextChar)) {
                        segment += c.globstar;
                        endsWithSep = true;
                    } else {
                        segment += c.wildcard;
                    }
                }
                continue;
            }
            segment += regExpEscapeChars.includes(glob[i1]) ? `\\${glob[i1]}` : glob[i1];
        }
        if (groupStack.length > 0 || inRange || inEscape) {
            segment = "";
            for (const c of glob.slice(j, i1)){
                segment += regExpEscapeChars.includes(c) ? `\\${c}` : c;
                endsWithSep = false;
            }
        }
        regExpString += segment;
        if (!endsWithSep) {
            regExpString += i1 < glob.length ? c.sep : c.sepMaybe;
            endsWithSep = true;
        }
        while(c.seps.includes(glob[i1]))i1++;
        if (!(i1 > j)) {
            throw new Error("Assertion failure: i > j (potential infinite loop)");
        }
        j = i1;
    }
    regExpString = `^${regExpString}$`;
    return new RegExp(regExpString, caseInsensitive ? "i" : "");
}
const constants = {
    sep: "(?:\\\\|/)+",
    sepMaybe: "(?:\\\\|/)*",
    seps: [
        "\\",
        "/"
    ],
    globstar: "(?:[^\\\\/]*(?:\\\\|/|$)+)*",
    wildcard: "[^\\\\/]*",
    escapePrefix: "`"
};
function globToRegExp(glob, options = {}) {
    return _globToRegExp(constants, glob, options);
}
function isGlob(str) {
    const chars = {
        "{": "}",
        "(": ")",
        "[": "]"
    };
    const regex = /\\(.)|(^!|\*|\?|[\].+)]\?|\[[^\\\]]+\]|\{[^\\}]+\}|\(\?[:!=][^\\)]+\)|\([^|]+\|[^\\)]+\))/;
    if (str === "") {
        return false;
    }
    let match;
    while(match = regex.exec(str)){
        if (match[2]) return true;
        let idx = match.index + match[0].length;
        const open = match[1];
        const close = open ? chars[open] : null;
        if (open && close) {
            const n = str.indexOf(close, idx);
            if (n !== -1) {
                idx = n + 1;
            }
        }
        str = str.slice(idx);
    }
    return false;
}
function normalizeGlob(glob, { globstar = false } = {}) {
    if (glob.match(/\0/g)) {
        throw new Error(`Glob contains invalid characters: "${glob}"`);
    }
    if (!globstar) {
        return normalize(glob);
    }
    const s = SEPARATOR_PATTERN.source;
    const badParentPattern = new RegExp(`(?<=(${s}|^)\\*\\*${s})\\.\\.(?=${s}|$)`, "g");
    return normalize(glob.replace(badParentPattern, "\0")).replace(/\0/g, "..");
}
function joinGlobs(globs, { extended = true, globstar = false } = {}) {
    if (!globstar || globs.length === 0) {
        return join(...globs);
    }
    if (globs.length === 0) return ".";
    let joined;
    for (const glob of globs){
        const path = glob;
        if (path.length > 0) {
            if (!joined) joined = path;
            else joined += `${SEPARATOR}${path}`;
        }
    }
    if (!joined) return ".";
    return normalizeGlob(joined, {
        extended,
        globstar
    });
}
function isPosixPathSeparator1(code) {
    return code === 47;
}
function basename1(path, suffix = "") {
    assertArgs(path, suffix);
    const lastSegment = lastPathSegment(path, isPosixPathSeparator1);
    const strippedSegment = stripTrailingSeparators(lastSegment, isPosixPathSeparator1);
    return suffix ? stripSuffix(strippedSegment, suffix) : strippedSegment;
}
const DELIMITER = ":";
const SEPARATOR1 = "/";
const SEPARATOR_PATTERN1 = /\/+/;
function dirname1(path) {
    assertArg(path);
    let end = -1;
    let matchedNonSeparator = false;
    for(let i1 = path.length - 1; i1 >= 1; --i1){
        if (isPosixPathSeparator1(path.charCodeAt(i1))) {
            if (matchedNonSeparator) {
                end = i1;
                break;
            }
        } else {
            matchedNonSeparator = true;
        }
    }
    if (end === -1) {
        return isPosixPathSeparator1(path.charCodeAt(0)) ? "/" : ".";
    }
    return stripTrailingSeparators(path.slice(0, end), isPosixPathSeparator1);
}
function extname1(path) {
    assertPath(path);
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let preDotState = 0;
    for(let i1 = path.length - 1; i1 >= 0; --i1){
        const code = path.charCodeAt(i1);
        if (isPosixPathSeparator1(code)) {
            if (!matchedSlash) {
                startPart = i1 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i1 + 1;
        }
        if (code === 46) {
            if (startDot === -1) startDot = i1;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        return "";
    }
    return path.slice(startDot, end);
}
function format(pathObject) {
    assertArg1(pathObject);
    return _format("/", pathObject);
}
function fromFileUrl1(url) {
    url = assertArg2(url);
    return decodeURIComponent(url.pathname.replace(/%(?![0-9A-Fa-f]{2})/g, "%25"));
}
function isAbsolute1(path) {
    assertPath(path);
    return path.length > 0 && isPosixPathSeparator1(path.charCodeAt(0));
}
function normalize1(path) {
    assertArg3(path);
    const isAbsolute = isPosixPathSeparator1(path.charCodeAt(0));
    const trailingSeparator = isPosixPathSeparator1(path.charCodeAt(path.length - 1));
    path = normalizeString(path, !isAbsolute, "/", isPosixPathSeparator1);
    if (path.length === 0 && !isAbsolute) path = ".";
    if (path.length > 0 && trailingSeparator) path += "/";
    if (isAbsolute) return `/${path}`;
    return path;
}
function join1(...paths) {
    if (paths.length === 0) return ".";
    let joined;
    for(let i1 = 0; i1 < paths.length; ++i1){
        const path = paths[i1];
        assertPath(path);
        if (path.length > 0) {
            if (!joined) joined = path;
            else joined += `/${path}`;
        }
    }
    if (!joined) return ".";
    return normalize1(joined);
}
function parse(path) {
    assertPath(path);
    const ret = {
        root: "",
        dir: "",
        base: "",
        ext: "",
        name: ""
    };
    if (path.length === 0) return ret;
    const isAbsolute = isPosixPathSeparator1(path.charCodeAt(0));
    let start;
    if (isAbsolute) {
        ret.root = "/";
        start = 1;
    } else {
        start = 0;
    }
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let i1 = path.length - 1;
    let preDotState = 0;
    for(; i1 >= start; --i1){
        const code = path.charCodeAt(i1);
        if (isPosixPathSeparator1(code)) {
            if (!matchedSlash) {
                startPart = i1 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i1 + 1;
        }
        if (code === 46) {
            if (startDot === -1) startDot = i1;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        if (end !== -1) {
            if (startPart === 0 && isAbsolute) {
                ret.base = ret.name = path.slice(1, end);
            } else {
                ret.base = ret.name = path.slice(startPart, end);
            }
        }
        ret.base = ret.base || "/";
    } else {
        if (startPart === 0 && isAbsolute) {
            ret.name = path.slice(1, startDot);
            ret.base = path.slice(1, end);
        } else {
            ret.name = path.slice(startPart, startDot);
            ret.base = path.slice(startPart, end);
        }
        ret.ext = path.slice(startDot, end);
    }
    if (startPart > 0) {
        ret.dir = stripTrailingSeparators(path.slice(0, startPart - 1), isPosixPathSeparator1);
    } else if (isAbsolute) ret.dir = "/";
    return ret;
}
function resolve1(...pathSegments) {
    let resolvedPath = "";
    let resolvedAbsolute = false;
    for(let i1 = pathSegments.length - 1; i1 >= -1 && !resolvedAbsolute; i1--){
        let path;
        if (i1 >= 0) path = pathSegments[i1];
        else {
            const { Deno: Deno1 } = globalThis;
            if (typeof Deno1?.cwd !== "function") {
                throw new TypeError("Resolved a relative path without a CWD.");
            }
            path = Deno1.cwd();
        }
        assertPath(path);
        if (path.length === 0) {
            continue;
        }
        resolvedPath = `${path}/${resolvedPath}`;
        resolvedAbsolute = isPosixPathSeparator1(path.charCodeAt(0));
    }
    resolvedPath = normalizeString(resolvedPath, !resolvedAbsolute, "/", isPosixPathSeparator1);
    if (resolvedAbsolute) {
        if (resolvedPath.length > 0) return `/${resolvedPath}`;
        else return "/";
    } else if (resolvedPath.length > 0) return resolvedPath;
    else return ".";
}
function relative1(from, to) {
    assertArgs1(from, to);
    from = resolve1(from);
    to = resolve1(to);
    if (from === to) return "";
    let fromStart = 1;
    const fromEnd = from.length;
    for(; fromStart < fromEnd; ++fromStart){
        if (!isPosixPathSeparator1(from.charCodeAt(fromStart))) break;
    }
    const fromLen = fromEnd - fromStart;
    let toStart = 1;
    const toEnd = to.length;
    for(; toStart < toEnd; ++toStart){
        if (!isPosixPathSeparator1(to.charCodeAt(toStart))) break;
    }
    const toLen = toEnd - toStart;
    const length = fromLen < toLen ? fromLen : toLen;
    let lastCommonSep = -1;
    let i1 = 0;
    for(; i1 <= length; ++i1){
        if (i1 === length) {
            if (toLen > length) {
                if (isPosixPathSeparator1(to.charCodeAt(toStart + i1))) {
                    return to.slice(toStart + i1 + 1);
                } else if (i1 === 0) {
                    return to.slice(toStart + i1);
                }
            } else if (fromLen > length) {
                if (isPosixPathSeparator1(from.charCodeAt(fromStart + i1))) {
                    lastCommonSep = i1;
                } else if (i1 === 0) {
                    lastCommonSep = 0;
                }
            }
            break;
        }
        const fromCode = from.charCodeAt(fromStart + i1);
        const toCode = to.charCodeAt(toStart + i1);
        if (fromCode !== toCode) break;
        else if (isPosixPathSeparator1(fromCode)) lastCommonSep = i1;
    }
    let out = "";
    for(i1 = fromStart + lastCommonSep + 1; i1 <= fromEnd; ++i1){
        if (i1 === fromEnd || isPosixPathSeparator1(from.charCodeAt(i1))) {
            if (out.length === 0) out += "..";
            else out += "/..";
        }
    }
    if (out.length > 0) return out + to.slice(toStart + lastCommonSep);
    else {
        toStart += lastCommonSep;
        if (isPosixPathSeparator1(to.charCodeAt(toStart))) ++toStart;
        return to.slice(toStart);
    }
}
function toFileUrl1(path) {
    if (!isAbsolute1(path)) {
        throw new TypeError("Must be an absolute path.");
    }
    const url = new URL("file:///");
    url.pathname = encodeWhitespace(path.replace(/%/g, "%25").replace(/\\/g, "%5C"));
    return url;
}
function toNamespacedPath(path) {
    return path;
}
function common(paths, sep = SEPARATOR1) {
    return _common(paths, sep);
}
const constants1 = {
    sep: "/+",
    sepMaybe: "/*",
    seps: [
        "/"
    ],
    globstar: "(?:[^/]*(?:/|$)+)*",
    wildcard: "[^/]*",
    escapePrefix: "\\"
};
function globToRegExp1(glob, options = {}) {
    return _globToRegExp(constants1, glob, options);
}
function normalizeGlob1(glob, { globstar = false } = {}) {
    if (glob.match(/\0/g)) {
        throw new Error(`Glob contains invalid characters: "${glob}"`);
    }
    if (!globstar) {
        return normalize1(glob);
    }
    const s = SEPARATOR_PATTERN1.source;
    const badParentPattern = new RegExp(`(?<=(${s}|^)\\*\\*${s})\\.\\.(?=${s}|$)`, "g");
    return normalize1(glob.replace(badParentPattern, "\0")).replace(/\0/g, "..");
}
function joinGlobs1(globs, { extended = true, globstar = false } = {}) {
    if (!globstar || globs.length === 0) {
        return join1(...globs);
    }
    if (globs.length === 0) return ".";
    let joined;
    for (const glob of globs){
        const path = glob;
        if (path.length > 0) {
            if (!joined) joined = path;
            else joined += `${SEPARATOR1}${path}`;
        }
    }
    if (!joined) return ".";
    return normalizeGlob1(joined, {
        extended,
        globstar
    });
}
const mod = {
    basename: basename1,
    DELIMITER,
    SEPARATOR: SEPARATOR1,
    SEPARATOR_PATTERN: SEPARATOR_PATTERN1,
    dirname: dirname1,
    extname: extname1,
    format,
    fromFileUrl: fromFileUrl1,
    isAbsolute: isAbsolute1,
    join: join1,
    normalize: normalize1,
    parse,
    relative: relative1,
    resolve: resolve1,
    toFileUrl: toFileUrl1,
    toNamespacedPath,
    common,
    globToRegExp: globToRegExp1,
    isGlob,
    joinGlobs: joinGlobs1,
    normalizeGlob: normalizeGlob1
};
const osType = (()=>{
    const { Deno: Deno1 } = globalThis;
    if (typeof Deno1?.build?.os === "string") {
        return Deno1.build.os;
    }
    const { navigator } = globalThis;
    if (navigator?.appVersion?.includes?.("Win")) {
        return "windows";
    }
    return "linux";
})();
const isWindows = osType === "windows";
function basename2(path, suffix = "") {
    return isWindows ? basename(path, suffix) : basename1(path, suffix);
}
const SEPARATOR2 = isWindows ? "\\" : "/";
const SEPARATOR_PATTERN2 = isWindows ? /[\\/]+/ : /\/+/;
function dirname2(path) {
    return isWindows ? dirname(path) : dirname1(path);
}
function extname2(path) {
    return isWindows ? extname(path) : extname1(path);
}
function fromFileUrl2(url) {
    return isWindows ? fromFileUrl(url) : fromFileUrl1(url);
}
function isAbsolute2(path) {
    return isWindows ? isAbsolute(path) : isAbsolute1(path);
}
function join2(...paths) {
    return isWindows ? join(...paths) : join1(...paths);
}
function normalize2(path) {
    return isWindows ? normalize(path) : normalize1(path);
}
function relative2(from, to) {
    return isWindows ? relative(from, to) : relative1(from, to);
}
function resolve2(...pathSegments) {
    return isWindows ? resolve(...pathSegments) : resolve1(...pathSegments);
}
function toFileUrl2(path) {
    return isWindows ? toFileUrl(path) : toFileUrl1(path);
}
function globToRegExp2(glob, options = {}) {
    return options.os === "windows" || !options.os && isWindows ? globToRegExp(glob, options) : globToRegExp1(glob, options);
}
function joinGlobs2(globs, options = {}) {
    return isWindows ? joinGlobs(globs, options) : joinGlobs1(globs, options);
}
const basename3 = basename2;
const extname3 = extname2;
const dirname3 = dirname2;
const isAbsolute3 = isAbsolute2;
const join3 = join2;
const relative3 = relative2;
const resolve3 = resolve2;
const normalize3 = normalize2;
const posix = {
    normalize: mod.normalize
};
function getFileInfoType(fileInfo) {
    return fileInfo.isFile ? "file" : fileInfo.isDirectory ? "dir" : fileInfo.isSymlink ? "symlink" : undefined;
}
async function ensureDir(dir) {
    try {
        const fileInfo = await Deno.stat(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
        return;
    } catch (err) {
        if (!(err instanceof Deno.errors.NotFound)) {
            throw err;
        }
    }
    try {
        await Deno.mkdir(dir, {
            recursive: true
        });
    } catch (err) {
        if (!(err instanceof Deno.errors.AlreadyExists)) {
            throw err;
        }
        const fileInfo = await Deno.stat(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
    }
}
function ensureDirSync(dir) {
    try {
        const fileInfo = Deno.statSync(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
        return;
    } catch (err) {
        if (!(err instanceof Deno.errors.NotFound)) {
            throw err;
        }
    }
    try {
        Deno.mkdirSync(dir, {
            recursive: true
        });
    } catch (err) {
        if (!(err instanceof Deno.errors.AlreadyExists)) {
            throw err;
        }
        const fileInfo = Deno.statSync(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
    }
}
function toPathString(pathUrl) {
    return pathUrl instanceof URL ? fromFileUrl2(pathUrl) : pathUrl;
}
function isSubdir(src, dest, sep = SEPARATOR2) {
    if (src === dest) {
        return false;
    }
    src = toPathString(src);
    const srcArray = src.split(sep);
    dest = toPathString(dest);
    const destArray = dest.split(sep);
    return srcArray.every((current, i1)=>destArray[i1] === current);
}
const isWindows1 = Deno.build.os === "windows";
async function ensureValidCopy(src, dest, options) {
    let destStat;
    try {
        destStat = await Deno.lstat(dest);
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return;
        }
        throw err;
    }
    if (options.isFolder && !destStat.isDirectory) {
        throw new Error(`Cannot overwrite non-directory '${dest}' with directory '${src}'.`);
    }
    if (!options.overwrite) {
        throw new Deno.errors.AlreadyExists(`'${dest}' already exists.`);
    }
    return destStat;
}
function ensureValidCopySync(src, dest, options) {
    let destStat;
    try {
        destStat = Deno.lstatSync(dest);
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return;
        }
        throw err;
    }
    if (options.isFolder && !destStat.isDirectory) {
        throw new Error(`Cannot overwrite non-directory '${dest}' with directory '${src}'.`);
    }
    if (!options.overwrite) {
        throw new Deno.errors.AlreadyExists(`'${dest}' already exists.`);
    }
    return destStat;
}
async function copyFile(src, dest, options) {
    await ensureValidCopy(src, dest, options);
    await Deno.copyFile(src, dest);
    if (options.preserveTimestamps) {
        const statInfo = await Deno.stat(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, statInfo.atime, statInfo.mtime);
    }
}
function copyFileSync(src, dest, options) {
    ensureValidCopySync(src, dest, options);
    Deno.copyFileSync(src, dest);
    if (options.preserveTimestamps) {
        const statInfo = Deno.statSync(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, statInfo.atime, statInfo.mtime);
    }
}
async function copySymLink(src, dest, options) {
    await ensureValidCopy(src, dest, options);
    const originSrcFilePath = await Deno.readLink(src);
    const type = getFileInfoType(await Deno.lstat(src));
    if (isWindows1) {
        await Deno.symlink(originSrcFilePath, dest, {
            type: type === "dir" ? "dir" : "file"
        });
    } else {
        await Deno.symlink(originSrcFilePath, dest);
    }
    if (options.preserveTimestamps) {
        const statInfo = await Deno.lstat(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, statInfo.atime, statInfo.mtime);
    }
}
function copySymlinkSync(src, dest, options) {
    ensureValidCopySync(src, dest, options);
    const originSrcFilePath = Deno.readLinkSync(src);
    const type = getFileInfoType(Deno.lstatSync(src));
    if (isWindows1) {
        Deno.symlinkSync(originSrcFilePath, dest, {
            type: type === "dir" ? "dir" : "file"
        });
    } else {
        Deno.symlinkSync(originSrcFilePath, dest);
    }
    if (options.preserveTimestamps) {
        const statInfo = Deno.lstatSync(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, statInfo.atime, statInfo.mtime);
    }
}
async function copyDir(src, dest, options) {
    const destStat = await ensureValidCopy(src, dest, {
        ...options,
        isFolder: true
    });
    if (!destStat) {
        await ensureDir(dest);
    }
    if (options.preserveTimestamps) {
        const srcStatInfo = await Deno.stat(src);
        assert(srcStatInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(srcStatInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, srcStatInfo.atime, srcStatInfo.mtime);
    }
    src = toPathString(src);
    dest = toPathString(dest);
    const promises = [];
    for await (const entry of Deno.readDir(src)){
        const srcPath = join2(src, entry.name);
        const destPath = join2(dest, basename2(srcPath));
        if (entry.isSymlink) {
            promises.push(copySymLink(srcPath, destPath, options));
        } else if (entry.isDirectory) {
            promises.push(copyDir(srcPath, destPath, options));
        } else if (entry.isFile) {
            promises.push(copyFile(srcPath, destPath, options));
        }
    }
    await Promise.all(promises);
}
function copyDirSync(src, dest, options) {
    const destStat = ensureValidCopySync(src, dest, {
        ...options,
        isFolder: true
    });
    if (!destStat) {
        ensureDirSync(dest);
    }
    if (options.preserveTimestamps) {
        const srcStatInfo = Deno.statSync(src);
        assert(srcStatInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(srcStatInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, srcStatInfo.atime, srcStatInfo.mtime);
    }
    src = toPathString(src);
    dest = toPathString(dest);
    for (const entry of Deno.readDirSync(src)){
        const srcPath = join2(src, entry.name);
        const destPath = join2(dest, basename2(srcPath));
        if (entry.isSymlink) {
            copySymlinkSync(srcPath, destPath, options);
        } else if (entry.isDirectory) {
            copyDirSync(srcPath, destPath, options);
        } else if (entry.isFile) {
            copyFileSync(srcPath, destPath, options);
        }
    }
}
async function copy(src, dest, options = {}) {
    src = resolve2(toPathString(src));
    dest = resolve2(toPathString(dest));
    if (src === dest) {
        throw new Error("Source and destination cannot be the same.");
    }
    const srcStat = await Deno.lstat(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot copy '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (srcStat.isSymlink) {
        await copySymLink(src, dest, options);
    } else if (srcStat.isDirectory) {
        await copyDir(src, dest, options);
    } else if (srcStat.isFile) {
        await copyFile(src, dest, options);
    }
}
function copySync(src, dest, options = {}) {
    src = resolve2(toPathString(src));
    dest = resolve2(toPathString(dest));
    if (src === dest) {
        throw new Error("Source and destination cannot be the same.");
    }
    const srcStat = Deno.lstatSync(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot copy '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (srcStat.isSymlink) {
        copySymlinkSync(src, dest, options);
    } else if (srcStat.isDirectory) {
        copyDirSync(src, dest, options);
    } else if (srcStat.isFile) {
        copyFileSync(src, dest, options);
    }
}
function existsSync(path, options) {
    try {
        const stat = Deno.statSync(path);
        if (options && (options.isReadable || options.isDirectory || options.isFile)) {
            if (options.isDirectory && options.isFile) {
                throw new TypeError("ExistsOptions.options.isDirectory and ExistsOptions.options.isFile must not be true together.");
            }
            if (options.isDirectory && !stat.isDirectory || options.isFile && !stat.isFile) {
                return false;
            }
            if (options.isReadable) {
                if (stat.mode === null) {
                    return true;
                }
                if (Deno.uid() === stat.uid) {
                    return (stat.mode & 0o400) === 0o400;
                } else if (Deno.gid() === stat.gid) {
                    return (stat.mode & 0o040) === 0o040;
                }
                return (stat.mode & 0o004) === 0o004;
            }
        }
        return true;
    } catch (error) {
        if (error instanceof Deno.errors.NotFound) {
            return false;
        }
        if (error instanceof Deno.errors.PermissionDenied) {
            if (Deno.permissions.querySync({
                name: "read",
                path
            }).state === "granted") {
                return !options?.isReadable;
            }
        }
        throw error;
    }
}
function createWalkEntrySync(path) {
    path = toPathString(path);
    path = normalize2(path);
    const name = basename2(path);
    const info = Deno.statSync(path);
    return {
        path,
        name,
        isFile: info.isFile,
        isDirectory: info.isDirectory,
        isSymlink: info.isSymlink
    };
}
class WalkError extends Error {
    root;
    constructor(cause, root){
        super(`${cause instanceof Error ? cause.message : cause} for path "${root}"`);
        this.cause = cause;
        this.name = this.constructor.name;
        this.root = root;
    }
}
function include(path, exts, match, skip) {
    if (exts && !exts.some((ext)=>path.endsWith(ext))) {
        return false;
    }
    if (match && !match.some((pattern)=>!!path.match(pattern))) {
        return false;
    }
    if (skip && skip.some((pattern)=>!!path.match(pattern))) {
        return false;
    }
    return true;
}
function wrapErrorWithPath(err, root) {
    if (err instanceof WalkError) return err;
    return new WalkError(err, root);
}
function* walkSync(root, { maxDepth = Infinity, includeFiles = true, includeDirs = true, includeSymlinks = true, followSymlinks = false, canonicalize = true, exts = undefined, match = undefined, skip = undefined } = {}) {
    root = toPathString(root);
    if (maxDepth < 0) {
        return;
    }
    if (includeDirs && include(root, exts, match, skip)) {
        yield createWalkEntrySync(root);
    }
    if (maxDepth < 1 || !include(root, undefined, undefined, skip)) {
        return;
    }
    let entries;
    try {
        entries = Deno.readDirSync(root);
    } catch (err) {
        throw wrapErrorWithPath(err, normalize2(root));
    }
    for (const entry of entries){
        let path = join2(root, entry.name);
        let { isSymlink, isDirectory } = entry;
        if (isSymlink) {
            if (!followSymlinks) {
                if (includeSymlinks && include(path, exts, match, skip)) {
                    yield {
                        path,
                        ...entry
                    };
                }
                continue;
            }
            const realPath = Deno.realPathSync(path);
            if (canonicalize) {
                path = realPath;
            }
            ({ isSymlink, isDirectory } = Deno.lstatSync(realPath));
        }
        if (isSymlink || isDirectory) {
            yield* walkSync(path, {
                maxDepth: maxDepth - 1,
                includeFiles,
                includeDirs,
                includeSymlinks,
                followSymlinks,
                exts,
                match,
                skip
            });
        } else if (includeFiles && include(path, exts, match, skip)) {
            yield {
                path,
                ...entry
            };
        }
    }
}
const isWindows2 = Deno.build.os === "windows";
function split(path) {
    const s = SEPARATOR_PATTERN2.source;
    const segments = path.replace(new RegExp(`^${s}|${s}$`, "g"), "").split(SEPARATOR_PATTERN2);
    const isAbsolute_ = isAbsolute2(path);
    return {
        segments,
        isAbsolute: isAbsolute_,
        hasTrailingSep: !!path.match(new RegExp(`${s}$`)),
        winRoot: isWindows2 && isAbsolute_ ? segments.shift() : undefined
    };
}
function throwUnlessNotFound(error) {
    if (!(error instanceof Deno.errors.NotFound)) {
        throw error;
    }
}
function comparePath(a, b) {
    if (a.path < b.path) return -1;
    if (a.path > b.path) return 1;
    return 0;
}
function* expandGlobSync(glob, { root, exclude = [], includeDirs = true, extended = true, globstar = true, caseInsensitive, followSymlinks, canonicalize } = {}) {
    const { segments, isAbsolute: isGlobAbsolute, hasTrailingSep, winRoot } = split(toPathString(glob));
    root ??= isGlobAbsolute ? winRoot ?? "/" : Deno.cwd();
    const globOptions = {
        extended,
        globstar,
        caseInsensitive
    };
    const absRoot = isGlobAbsolute ? root : resolve2(root);
    const resolveFromRoot = (path)=>resolve2(absRoot, path);
    const excludePatterns = exclude.map(resolveFromRoot).map((s)=>globToRegExp2(s, globOptions));
    const shouldInclude = (path)=>!excludePatterns.some((p)=>!!path.match(p));
    let fixedRoot = isGlobAbsolute ? winRoot !== undefined ? winRoot : "/" : absRoot;
    while(segments.length > 0 && !isGlob(segments[0])){
        const seg = segments.shift();
        assert(seg !== undefined);
        fixedRoot = joinGlobs2([
            fixedRoot,
            seg
        ], globOptions);
    }
    let fixedRootInfo;
    try {
        fixedRootInfo = createWalkEntrySync(fixedRoot);
    } catch (error) {
        return throwUnlessNotFound(error);
    }
    function* advanceMatch(walkInfo, globSegment) {
        if (!walkInfo.isDirectory) {
            return;
        } else if (globSegment === "..") {
            const parentPath = joinGlobs2([
                walkInfo.path,
                ".."
            ], globOptions);
            try {
                if (shouldInclude(parentPath)) {
                    return yield createWalkEntrySync(parentPath);
                }
            } catch (error) {
                throwUnlessNotFound(error);
            }
            return;
        } else if (globSegment === "**") {
            return yield* walkSync(walkInfo.path, {
                skip: excludePatterns,
                maxDepth: globstar ? Infinity : 1,
                followSymlinks,
                canonicalize
            });
        }
        const globPattern = globToRegExp2(globSegment, globOptions);
        for (const walkEntry of walkSync(walkInfo.path, {
            maxDepth: 1,
            skip: excludePatterns,
            followSymlinks
        })){
            if (walkEntry.path !== walkInfo.path && walkEntry.name.match(globPattern)) {
                yield walkEntry;
            }
        }
    }
    let currentMatches = [
        fixedRootInfo
    ];
    for (const segment of segments){
        const nextMatchMap = new Map();
        for (const currentMatch of currentMatches){
            for (const nextMatch of advanceMatch(currentMatch, segment)){
                nextMatchMap.set(nextMatch.path, nextMatch);
            }
        }
        currentMatches = [
            ...nextMatchMap.values()
        ].sort(comparePath);
    }
    if (hasTrailingSep) {
        currentMatches = currentMatches.filter((entry)=>entry.isDirectory);
    }
    if (!includeDirs) {
        currentMatches = currentMatches.filter((entry)=>!entry.isDirectory);
    }
    yield* currentMatches;
}
const LF = "\n";
const CRLF = "\r\n";
const EOL = Deno?.build.os === "windows" ? CRLF : LF;
const regDetect = /(?:\r?\n)/g;
function format1(content, eol) {
    return content.replace(regDetect, eol);
}
function isSamePath(src, dest) {
    src = toPathString(src);
    dest = toPathString(dest);
    return resolve2(src) === resolve2(dest);
}
const EXISTS_ERROR = new Deno.errors.AlreadyExists("dest already exists.");
class SubdirectoryMoveError extends Error {
    constructor(src, dest){
        super(`Cannot move '${src}' to a subdirectory of itself, '${dest}'.`);
        this.name = this.constructor.name;
    }
}
function moveSync(src, dest, { overwrite = false } = {}) {
    const srcStat = Deno.statSync(src);
    if (srcStat.isDirectory && (isSubdir(src, dest) || isSamePath(src, dest))) {
        throw new SubdirectoryMoveError(src, dest);
    }
    if (overwrite) {
        if (isSamePath(src, dest)) return;
        try {
            Deno.removeSync(dest, {
                recursive: true
            });
        } catch (error) {
            if (!(error instanceof Deno.errors.NotFound)) {
                throw error;
            }
        }
    } else {
        try {
            Deno.lstatSync(dest);
            throw EXISTS_ERROR;
        } catch (error) {
            if (error === EXISTS_ERROR) {
                throw error;
            }
        }
    }
    Deno.renameSync(src, dest);
}
function getFileInfoType1(fileInfo) {
    return fileInfo.isFile ? "file" : fileInfo.isDirectory ? "dir" : fileInfo.isSymlink ? "symlink" : undefined;
}
function isSubdir1(src, dest, sep = SEPARATOR2) {
    src = toPathString1(src);
    dest = toPathString1(dest);
    if (resolve2(src) === resolve2(dest)) {
        return false;
    }
    const srcArray = src.split(sep);
    const destArray = dest.split(sep);
    return srcArray.every((current, i1)=>destArray[i1] === current);
}
function toPathString1(pathUrl) {
    return pathUrl instanceof URL ? fromFileUrl2(pathUrl) : pathUrl;
}
function safeMoveSync(src, dest) {
    try {
        Deno.renameSync(src, dest);
    } catch (err) {
        if (err.code !== "EXDEV") {
            throw err;
        }
        copySync(src, dest, {
            overwrite: true
        });
        safeRemoveSync(src, {
            recursive: true
        });
    }
}
function safeRemoveSync(file, options = {}) {
    try {
        Deno.removeSync(file, options);
    } catch (e) {
        if (existsSync(file)) {
            throw e;
        }
    }
}
class UnsafeRemovalError extends Error {
    constructor(msg){
        super(msg);
    }
}
function safeRemoveDirSync(path, boundary) {
    if (path === boundary || isSubdir1(path, boundary)) {
        throw new UnsafeRemovalError(`Refusing to remove directory ${path} that isn't a subdirectory of ${boundary}`);
    }
    return safeRemoveSync(path, {
        recursive: true
    });
}
function safeModeFromFile(path) {
    if (Deno.build.os !== "windows") {
        const stat = Deno.statSync(path);
        if (stat.mode !== null) {
            return stat.mode;
        }
    }
}
function listCacheClear() {
    this.__data__ = [];
    this.size = 0;
}
var __VIRTUAL_FILE = listCacheClear;
function eq(value, other) {
    return value === other || value !== value && other !== other;
}
var __VIRTUAL_FILE1 = eq;
function assocIndexOf(array, key) {
    var length = array.length;
    while(length--){
        if (__VIRTUAL_FILE1(array[length][0], key)) {
            return length;
        }
    }
    return -1;
}
var __VIRTUAL_FILE2 = assocIndexOf;
var arrayProto = Array.prototype;
var splice = arrayProto.splice;
function listCacheDelete(key) {
    var data = this.__data__, index = __VIRTUAL_FILE2(data, key);
    if (index < 0) {
        return false;
    }
    var lastIndex = data.length - 1;
    if (index == lastIndex) {
        data.pop();
    } else {
        splice.call(data, index, 1);
    }
    --this.size;
    return true;
}
var __VIRTUAL_FILE3 = listCacheDelete;
function listCacheGet(key) {
    var data = this.__data__, index = __VIRTUAL_FILE2(data, key);
    return index < 0 ? void 0 : data[index][1];
}
var __VIRTUAL_FILE4 = listCacheGet;
function listCacheHas(key) {
    return __VIRTUAL_FILE2(this.__data__, key) > -1;
}
var __VIRTUAL_FILE5 = listCacheHas;
function listCacheSet(key, value) {
    var data = this.__data__, index = __VIRTUAL_FILE2(data, key);
    if (index < 0) {
        ++this.size;
        data.push([
            key,
            value
        ]);
    } else {
        data[index][1] = value;
    }
    return this;
}
var __VIRTUAL_FILE6 = listCacheSet;
function ListCache(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
ListCache.prototype.clear = __VIRTUAL_FILE;
ListCache.prototype["delete"] = __VIRTUAL_FILE3;
ListCache.prototype.get = __VIRTUAL_FILE4;
ListCache.prototype.has = __VIRTUAL_FILE5;
ListCache.prototype.set = __VIRTUAL_FILE6;
var __VIRTUAL_FILE7 = ListCache;
function stackClear() {
    this.__data__ = new __VIRTUAL_FILE7();
    this.size = 0;
}
var __VIRTUAL_FILE8 = stackClear;
function stackDelete(key) {
    var data = this.__data__, result1 = data["delete"](key);
    this.size = data.size;
    return result1;
}
var __VIRTUAL_FILE9 = stackDelete;
function stackGet(key) {
    return this.__data__.get(key);
}
var __VIRTUAL_FILE10 = stackGet;
function stackHas(key) {
    return this.__data__.has(key);
}
var __VIRTUAL_FILE11 = stackHas;
var commonjsGlobal = typeof globalThis !== "undefined" ? globalThis : typeof window !== "undefined" ? window : typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : {};
var freeGlobal = typeof commonjsGlobal == "object" && commonjsGlobal && commonjsGlobal.Object === Object && commonjsGlobal;
var __VIRTUAL_FILE12 = freeGlobal;
var freeSelf = typeof self == "object" && self && self.Object === Object && self;
var root = __VIRTUAL_FILE12 || freeSelf || Function("return this")();
var __VIRTUAL_FILE13 = root;
var Symbol1 = __VIRTUAL_FILE13.Symbol;
var __VIRTUAL_FILE14 = Symbol1;
var objectProto = Object.prototype;
var hasOwnProperty = objectProto.hasOwnProperty;
var nativeObjectToString = objectProto.toString;
var symToStringTag = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.toStringTag : void 0;
function getRawTag(value) {
    var isOwn = hasOwnProperty.call(value, symToStringTag), tag = value[symToStringTag];
    try {
        value[symToStringTag] = void 0;
        var unmasked = true;
    } catch (e) {}
    var result1 = nativeObjectToString.call(value);
    if (unmasked) {
        if (isOwn) {
            value[symToStringTag] = tag;
        } else {
            delete value[symToStringTag];
        }
    }
    return result1;
}
var __VIRTUAL_FILE15 = getRawTag;
var objectProto1 = Object.prototype;
var nativeObjectToString1 = objectProto1.toString;
function objectToString(value) {
    return nativeObjectToString1.call(value);
}
var __VIRTUAL_FILE16 = objectToString;
var nullTag = "[object Null]", undefinedTag = "[object Undefined]";
var symToStringTag1 = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.toStringTag : void 0;
function baseGetTag(value) {
    if (value == null) {
        return value === void 0 ? undefinedTag : nullTag;
    }
    return symToStringTag1 && symToStringTag1 in Object(value) ? __VIRTUAL_FILE15(value) : __VIRTUAL_FILE16(value);
}
var __VIRTUAL_FILE17 = baseGetTag;
function isObject(value) {
    var type = typeof value;
    return value != null && (type == "object" || type == "function");
}
var __VIRTUAL_FILE18 = isObject;
var asyncTag = "[object AsyncFunction]", funcTag = "[object Function]", genTag = "[object GeneratorFunction]", proxyTag = "[object Proxy]";
function isFunction(value) {
    if (!__VIRTUAL_FILE18(value)) {
        return false;
    }
    var tag = __VIRTUAL_FILE17(value);
    return tag == funcTag || tag == genTag || tag == asyncTag || tag == proxyTag;
}
var __VIRTUAL_FILE19 = isFunction;
var coreJsData = __VIRTUAL_FILE13["__core-js_shared__"];
var __VIRTUAL_FILE20 = coreJsData;
var maskSrcKey = function() {
    var uid = /[^.]+$/.exec(__VIRTUAL_FILE20 && __VIRTUAL_FILE20.keys && __VIRTUAL_FILE20.keys.IE_PROTO || "");
    return uid ? "Symbol(src)_1." + uid : "";
}();
function isMasked(func) {
    return !!maskSrcKey && maskSrcKey in func;
}
var __VIRTUAL_FILE21 = isMasked;
var funcProto = Function.prototype;
var funcToString = funcProto.toString;
function toSource(func) {
    if (func != null) {
        try {
            return funcToString.call(func);
        } catch (e) {}
        try {
            return func + "";
        } catch (e) {}
    }
    return "";
}
var __VIRTUAL_FILE22 = toSource;
var reRegExpChar = /[\\^$.*+?()[\]{}|]/g;
var reIsHostCtor = /^\[object .+?Constructor\]$/;
var funcProto1 = Function.prototype, objectProto2 = Object.prototype;
var funcToString1 = funcProto1.toString;
var hasOwnProperty1 = objectProto2.hasOwnProperty;
var reIsNative = RegExp("^" + funcToString1.call(hasOwnProperty1).replace(reRegExpChar, "\\$&").replace(/hasOwnProperty|(function).*?(?=\\\()| for .+?(?=\\\])/g, "$1.*?") + "$");
function baseIsNative(value) {
    if (!__VIRTUAL_FILE18(value) || __VIRTUAL_FILE21(value)) {
        return false;
    }
    var pattern = __VIRTUAL_FILE19(value) ? reIsNative : reIsHostCtor;
    return pattern.test(__VIRTUAL_FILE22(value));
}
var __VIRTUAL_FILE23 = baseIsNative;
function getValue(object, key) {
    return object == null ? void 0 : object[key];
}
var __VIRTUAL_FILE24 = getValue;
function getNative(object, key) {
    var value = __VIRTUAL_FILE24(object, key);
    return __VIRTUAL_FILE23(value) ? value : void 0;
}
var __VIRTUAL_FILE25 = getNative;
var Map1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "Map");
var __VIRTUAL_FILE26 = Map1;
var nativeCreate = __VIRTUAL_FILE25(Object, "create");
var __VIRTUAL_FILE27 = nativeCreate;
function hashClear() {
    this.__data__ = __VIRTUAL_FILE27 ? __VIRTUAL_FILE27(null) : {};
    this.size = 0;
}
var __VIRTUAL_FILE28 = hashClear;
function hashDelete(key) {
    var result1 = this.has(key) && delete this.__data__[key];
    this.size -= result1 ? 1 : 0;
    return result1;
}
var __VIRTUAL_FILE29 = hashDelete;
var HASH_UNDEFINED = "__lodash_hash_undefined__";
var objectProto3 = Object.prototype;
var hasOwnProperty2 = objectProto3.hasOwnProperty;
function hashGet(key) {
    var data = this.__data__;
    if (__VIRTUAL_FILE27) {
        var result1 = data[key];
        return result1 === HASH_UNDEFINED ? void 0 : result1;
    }
    return hasOwnProperty2.call(data, key) ? data[key] : void 0;
}
var __VIRTUAL_FILE30 = hashGet;
var objectProto4 = Object.prototype;
var hasOwnProperty3 = objectProto4.hasOwnProperty;
function hashHas(key) {
    var data = this.__data__;
    return __VIRTUAL_FILE27 ? data[key] !== void 0 : hasOwnProperty3.call(data, key);
}
var __VIRTUAL_FILE31 = hashHas;
var HASH_UNDEFINED1 = "__lodash_hash_undefined__";
function hashSet(key, value) {
    var data = this.__data__;
    this.size += this.has(key) ? 0 : 1;
    data[key] = __VIRTUAL_FILE27 && value === void 0 ? HASH_UNDEFINED1 : value;
    return this;
}
var __VIRTUAL_FILE32 = hashSet;
function Hash(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
Hash.prototype.clear = __VIRTUAL_FILE28;
Hash.prototype["delete"] = __VIRTUAL_FILE29;
Hash.prototype.get = __VIRTUAL_FILE30;
Hash.prototype.has = __VIRTUAL_FILE31;
Hash.prototype.set = __VIRTUAL_FILE32;
var __VIRTUAL_FILE33 = Hash;
function mapCacheClear() {
    this.size = 0;
    this.__data__ = {
        hash: new __VIRTUAL_FILE33(),
        map: new (__VIRTUAL_FILE26 || __VIRTUAL_FILE7)(),
        string: new __VIRTUAL_FILE33()
    };
}
var __VIRTUAL_FILE34 = mapCacheClear;
function isKeyable(value) {
    var type = typeof value;
    return type == "string" || type == "number" || type == "symbol" || type == "boolean" ? value !== "__proto__" : value === null;
}
var __VIRTUAL_FILE35 = isKeyable;
function getMapData(map, key) {
    var data = map.__data__;
    return __VIRTUAL_FILE35(key) ? data[typeof key == "string" ? "string" : "hash"] : data.map;
}
var __VIRTUAL_FILE36 = getMapData;
function mapCacheDelete(key) {
    var result1 = __VIRTUAL_FILE36(this, key)["delete"](key);
    this.size -= result1 ? 1 : 0;
    return result1;
}
var __VIRTUAL_FILE37 = mapCacheDelete;
function mapCacheGet(key) {
    return __VIRTUAL_FILE36(this, key).get(key);
}
var __VIRTUAL_FILE38 = mapCacheGet;
function mapCacheHas(key) {
    return __VIRTUAL_FILE36(this, key).has(key);
}
var __VIRTUAL_FILE39 = mapCacheHas;
function mapCacheSet(key, value) {
    var data = __VIRTUAL_FILE36(this, key), size = data.size;
    data.set(key, value);
    this.size += data.size == size ? 0 : 1;
    return this;
}
var __VIRTUAL_FILE40 = mapCacheSet;
function MapCache(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
MapCache.prototype.clear = __VIRTUAL_FILE34;
MapCache.prototype["delete"] = __VIRTUAL_FILE37;
MapCache.prototype.get = __VIRTUAL_FILE38;
MapCache.prototype.has = __VIRTUAL_FILE39;
MapCache.prototype.set = __VIRTUAL_FILE40;
var __VIRTUAL_FILE41 = MapCache;
var LARGE_ARRAY_SIZE = 200;
function stackSet(key, value) {
    var data = this.__data__;
    if (data instanceof __VIRTUAL_FILE7) {
        var pairs = data.__data__;
        if (!__VIRTUAL_FILE26 || pairs.length < LARGE_ARRAY_SIZE - 1) {
            pairs.push([
                key,
                value
            ]);
            this.size = ++data.size;
            return this;
        }
        data = this.__data__ = new __VIRTUAL_FILE41(pairs);
    }
    data.set(key, value);
    this.size = data.size;
    return this;
}
var __VIRTUAL_FILE42 = stackSet;
function Stack(entries) {
    var data = this.__data__ = new __VIRTUAL_FILE7(entries);
    this.size = data.size;
}
Stack.prototype.clear = __VIRTUAL_FILE8;
Stack.prototype["delete"] = __VIRTUAL_FILE9;
Stack.prototype.get = __VIRTUAL_FILE10;
Stack.prototype.has = __VIRTUAL_FILE11;
Stack.prototype.set = __VIRTUAL_FILE42;
var __VIRTUAL_FILE43 = Stack;
function arrayEach(array, iteratee) {
    var index = -1, length = array == null ? 0 : array.length;
    while(++index < length){
        if (iteratee(array[index], index, array) === false) {
            break;
        }
    }
    return array;
}
var __VIRTUAL_FILE44 = arrayEach;
var defineProperty = function() {
    try {
        var func = __VIRTUAL_FILE25(Object, "defineProperty");
        func({}, "", {});
        return func;
    } catch (e) {}
}();
var __VIRTUAL_FILE45 = defineProperty;
function baseAssignValue(object, key, value) {
    if (key == "__proto__" && __VIRTUAL_FILE45) {
        __VIRTUAL_FILE45(object, key, {
            configurable: true,
            enumerable: true,
            value,
            writable: true
        });
    } else {
        object[key] = value;
    }
}
var __VIRTUAL_FILE46 = baseAssignValue;
var objectProto5 = Object.prototype;
var hasOwnProperty4 = objectProto5.hasOwnProperty;
function assignValue(object, key, value) {
    var objValue = object[key];
    if (!(hasOwnProperty4.call(object, key) && __VIRTUAL_FILE1(objValue, value)) || value === void 0 && !(key in object)) {
        __VIRTUAL_FILE46(object, key, value);
    }
}
var __VIRTUAL_FILE47 = assignValue;
function copyObject(source, props, object, customizer) {
    var isNew = !object;
    object || (object = {});
    var index = -1, length = props.length;
    while(++index < length){
        var key = props[index];
        var newValue = customizer ? customizer(object[key], source[key], key, object, source) : void 0;
        if (newValue === void 0) {
            newValue = source[key];
        }
        if (isNew) {
            __VIRTUAL_FILE46(object, key, newValue);
        } else {
            __VIRTUAL_FILE47(object, key, newValue);
        }
    }
    return object;
}
var __VIRTUAL_FILE48 = copyObject;
function baseTimes(n, iteratee) {
    var index = -1, result1 = Array(n);
    while(++index < n){
        result1[index] = iteratee(index);
    }
    return result1;
}
var __VIRTUAL_FILE49 = baseTimes;
function isObjectLike(value) {
    return value != null && typeof value == "object";
}
var __VIRTUAL_FILE50 = isObjectLike;
var argsTag = "[object Arguments]";
function baseIsArguments(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE17(value) == argsTag;
}
var __VIRTUAL_FILE51 = baseIsArguments;
var objectProto6 = Object.prototype;
var hasOwnProperty5 = objectProto6.hasOwnProperty;
var propertyIsEnumerable = objectProto6.propertyIsEnumerable;
var isArguments = __VIRTUAL_FILE51(function() {
    return arguments;
}()) ? __VIRTUAL_FILE51 : function(value) {
    return __VIRTUAL_FILE50(value) && hasOwnProperty5.call(value, "callee") && !propertyIsEnumerable.call(value, "callee");
};
var __VIRTUAL_FILE52 = isArguments;
var isArray = Array.isArray;
var __VIRTUAL_FILE53 = isArray;
function stubFalse() {
    return false;
}
var __VIRTUAL_FILE54 = stubFalse;
function createCommonjsModule(fn) {
    var module1 = {
        exports: {}
    };
    return fn(module1, module1.exports), module1.exports;
}
var __VIRTUAL_FILE55 = createCommonjsModule(function(module1, exports) {
    var freeExports = exports && !exports.nodeType && exports;
    var freeModule = freeExports && true && module1 && !module1.nodeType && module1;
    var moduleExports = freeModule && freeModule.exports === freeExports;
    var Buffer1 = moduleExports ? __VIRTUAL_FILE13.Buffer : void 0;
    var nativeIsBuffer = Buffer1 ? Buffer1.isBuffer : void 0;
    var isBuffer = nativeIsBuffer || __VIRTUAL_FILE54;
    module1.exports = isBuffer;
});
var MAX_SAFE_INTEGER = 9007199254740991;
var reIsUint = /^(?:0|[1-9]\d*)$/;
function isIndex(value, length) {
    var type = typeof value;
    length = length == null ? MAX_SAFE_INTEGER : length;
    return !!length && (type == "number" || type != "symbol" && reIsUint.test(value)) && value > -1 && value % 1 == 0 && value < length;
}
var __VIRTUAL_FILE56 = isIndex;
var MAX_SAFE_INTEGER1 = 9007199254740991;
function isLength(value) {
    return typeof value == "number" && value > -1 && value % 1 == 0 && value <= MAX_SAFE_INTEGER1;
}
var __VIRTUAL_FILE57 = isLength;
var argsTag1 = "[object Arguments]", arrayTag = "[object Array]", boolTag = "[object Boolean]", dateTag = "[object Date]", errorTag = "[object Error]", funcTag1 = "[object Function]", mapTag = "[object Map]", numberTag = "[object Number]", objectTag = "[object Object]", regexpTag = "[object RegExp]", setTag = "[object Set]", stringTag = "[object String]", weakMapTag = "[object WeakMap]";
var arrayBufferTag = "[object ArrayBuffer]", dataViewTag = "[object DataView]", float32Tag = "[object Float32Array]", float64Tag = "[object Float64Array]", int8Tag = "[object Int8Array]", int16Tag = "[object Int16Array]", int32Tag = "[object Int32Array]", uint8Tag = "[object Uint8Array]", uint8ClampedTag = "[object Uint8ClampedArray]", uint16Tag = "[object Uint16Array]", uint32Tag = "[object Uint32Array]";
var typedArrayTags = {};
typedArrayTags[float32Tag] = typedArrayTags[float64Tag] = typedArrayTags[int8Tag] = typedArrayTags[int16Tag] = typedArrayTags[int32Tag] = typedArrayTags[uint8Tag] = typedArrayTags[uint8ClampedTag] = typedArrayTags[uint16Tag] = typedArrayTags[uint32Tag] = true;
typedArrayTags[argsTag1] = typedArrayTags[arrayTag] = typedArrayTags[arrayBufferTag] = typedArrayTags[boolTag] = typedArrayTags[dataViewTag] = typedArrayTags[dateTag] = typedArrayTags[errorTag] = typedArrayTags[funcTag1] = typedArrayTags[mapTag] = typedArrayTags[numberTag] = typedArrayTags[objectTag] = typedArrayTags[regexpTag] = typedArrayTags[setTag] = typedArrayTags[stringTag] = typedArrayTags[weakMapTag] = false;
function baseIsTypedArray(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE57(value.length) && !!typedArrayTags[__VIRTUAL_FILE17(value)];
}
var __VIRTUAL_FILE58 = baseIsTypedArray;
function baseUnary(func) {
    return function(value) {
        return func(value);
    };
}
var __VIRTUAL_FILE59 = baseUnary;
function createCommonjsModule1(fn) {
    var module1 = {
        exports: {}
    };
    return fn(module1, module1.exports), module1.exports;
}
var __VIRTUAL_FILE60 = createCommonjsModule1(function(module1, exports) {
    var freeExports = exports && !exports.nodeType && exports;
    var freeModule = freeExports && true && module1 && !module1.nodeType && module1;
    var moduleExports = freeModule && freeModule.exports === freeExports;
    var freeProcess = moduleExports && __VIRTUAL_FILE12.process;
    var nodeUtil = function() {
        try {
            var types = freeModule && freeModule.require && freeModule.require("util").types;
            if (types) {
                return types;
            }
            return freeProcess && freeProcess.binding && freeProcess.binding("util");
        } catch (e) {}
    }();
    module1.exports = nodeUtil;
});
var nodeIsTypedArray = __VIRTUAL_FILE60 && __VIRTUAL_FILE60.isTypedArray;
var isTypedArray = nodeIsTypedArray ? __VIRTUAL_FILE59(nodeIsTypedArray) : __VIRTUAL_FILE58;
var __VIRTUAL_FILE61 = isTypedArray;
var objectProto7 = Object.prototype;
var hasOwnProperty6 = objectProto7.hasOwnProperty;
function arrayLikeKeys(value, inherited) {
    var isArr = __VIRTUAL_FILE53(value), isArg = !isArr && __VIRTUAL_FILE52(value), isBuff = !isArr && !isArg && __VIRTUAL_FILE55(value), isType = !isArr && !isArg && !isBuff && __VIRTUAL_FILE61(value), skipIndexes = isArr || isArg || isBuff || isType, result1 = skipIndexes ? __VIRTUAL_FILE49(value.length, String) : [], length = result1.length;
    for(var key in value){
        if ((inherited || hasOwnProperty6.call(value, key)) && !(skipIndexes && (key == "length" || isBuff && (key == "offset" || key == "parent") || isType && (key == "buffer" || key == "byteLength" || key == "byteOffset") || __VIRTUAL_FILE56(key, length)))) {
            result1.push(key);
        }
    }
    return result1;
}
var __VIRTUAL_FILE62 = arrayLikeKeys;
var objectProto8 = Object.prototype;
function isPrototype(value) {
    var Ctor = value && value.constructor, proto = typeof Ctor == "function" && Ctor.prototype || objectProto8;
    return value === proto;
}
var __VIRTUAL_FILE63 = isPrototype;
function overArg(func, transform) {
    return function(arg) {
        return func(transform(arg));
    };
}
var __VIRTUAL_FILE64 = overArg;
var nativeKeys = __VIRTUAL_FILE64(Object.keys, Object);
var __VIRTUAL_FILE65 = nativeKeys;
var objectProto9 = Object.prototype;
var hasOwnProperty7 = objectProto9.hasOwnProperty;
function baseKeys(object) {
    if (!__VIRTUAL_FILE63(object)) {
        return __VIRTUAL_FILE65(object);
    }
    var result1 = [];
    for(var key in Object(object)){
        if (hasOwnProperty7.call(object, key) && key != "constructor") {
            result1.push(key);
        }
    }
    return result1;
}
var __VIRTUAL_FILE66 = baseKeys;
function isArrayLike(value) {
    return value != null && __VIRTUAL_FILE57(value.length) && !__VIRTUAL_FILE19(value);
}
var __VIRTUAL_FILE67 = isArrayLike;
function keys(object) {
    return __VIRTUAL_FILE67(object) ? __VIRTUAL_FILE62(object) : __VIRTUAL_FILE66(object);
}
var __VIRTUAL_FILE68 = keys;
function baseAssign(object, source) {
    return object && __VIRTUAL_FILE48(source, __VIRTUAL_FILE68(source), object);
}
var __VIRTUAL_FILE69 = baseAssign;
function nativeKeysIn(object) {
    var result1 = [];
    if (object != null) {
        for(var key in Object(object)){
            result1.push(key);
        }
    }
    return result1;
}
var __VIRTUAL_FILE70 = nativeKeysIn;
var objectProto10 = Object.prototype;
var hasOwnProperty8 = objectProto10.hasOwnProperty;
function baseKeysIn(object) {
    if (!__VIRTUAL_FILE18(object)) {
        return __VIRTUAL_FILE70(object);
    }
    var isProto = __VIRTUAL_FILE63(object), result1 = [];
    for(var key in object){
        if (!(key == "constructor" && (isProto || !hasOwnProperty8.call(object, key)))) {
            result1.push(key);
        }
    }
    return result1;
}
var __VIRTUAL_FILE71 = baseKeysIn;
function keysIn(object) {
    return __VIRTUAL_FILE67(object) ? __VIRTUAL_FILE62(object, true) : __VIRTUAL_FILE71(object);
}
var __VIRTUAL_FILE72 = keysIn;
function baseAssignIn(object, source) {
    return object && __VIRTUAL_FILE48(source, __VIRTUAL_FILE72(source), object);
}
var __VIRTUAL_FILE73 = baseAssignIn;
function createCommonjsModule2(fn) {
    var module1 = {
        exports: {}
    };
    return fn(module1, module1.exports), module1.exports;
}
var __VIRTUAL_FILE74 = createCommonjsModule2(function(module1, exports) {
    var freeExports = exports && !exports.nodeType && exports;
    var freeModule = freeExports && true && module1 && !module1.nodeType && module1;
    var moduleExports = freeModule && freeModule.exports === freeExports;
    var Buffer1 = moduleExports ? __VIRTUAL_FILE13.Buffer : void 0, allocUnsafe = Buffer1 ? Buffer1.allocUnsafe : void 0;
    function cloneBuffer(buffer, isDeep) {
        if (isDeep) {
            return buffer.slice();
        }
        var length = buffer.length, result1 = allocUnsafe ? allocUnsafe(length) : new buffer.constructor(length);
        buffer.copy(result1);
        return result1;
    }
    module1.exports = cloneBuffer;
});
function copyArray(source, array) {
    var index = -1, length = source.length;
    array || (array = Array(length));
    while(++index < length){
        array[index] = source[index];
    }
    return array;
}
var __VIRTUAL_FILE75 = copyArray;
function arrayFilter(array, predicate) {
    var index = -1, length = array == null ? 0 : array.length, resIndex = 0, result1 = [];
    while(++index < length){
        var value = array[index];
        if (predicate(value, index, array)) {
            result1[resIndex++] = value;
        }
    }
    return result1;
}
var __VIRTUAL_FILE76 = arrayFilter;
function stubArray() {
    return [];
}
var __VIRTUAL_FILE77 = stubArray;
var objectProto11 = Object.prototype;
var propertyIsEnumerable1 = objectProto11.propertyIsEnumerable;
var nativeGetSymbols = Object.getOwnPropertySymbols;
var getSymbols = !nativeGetSymbols ? __VIRTUAL_FILE77 : function(object) {
    if (object == null) {
        return [];
    }
    object = Object(object);
    return __VIRTUAL_FILE76(nativeGetSymbols(object), function(symbol) {
        return propertyIsEnumerable1.call(object, symbol);
    });
};
var __VIRTUAL_FILE78 = getSymbols;
function copySymbols(source, object) {
    return __VIRTUAL_FILE48(source, __VIRTUAL_FILE78(source), object);
}
var __VIRTUAL_FILE79 = copySymbols;
function arrayPush(array, values) {
    var index = -1, length = values.length, offset = array.length;
    while(++index < length){
        array[offset + index] = values[index];
    }
    return array;
}
var __VIRTUAL_FILE80 = arrayPush;
var getPrototype = __VIRTUAL_FILE64(Object.getPrototypeOf, Object);
var __VIRTUAL_FILE81 = getPrototype;
var nativeGetSymbols1 = Object.getOwnPropertySymbols;
var getSymbolsIn = !nativeGetSymbols1 ? __VIRTUAL_FILE77 : function(object) {
    var result1 = [];
    while(object){
        __VIRTUAL_FILE80(result1, __VIRTUAL_FILE78(object));
        object = __VIRTUAL_FILE81(object);
    }
    return result1;
};
var __VIRTUAL_FILE82 = getSymbolsIn;
function copySymbolsIn(source, object) {
    return __VIRTUAL_FILE48(source, __VIRTUAL_FILE82(source), object);
}
var __VIRTUAL_FILE83 = copySymbolsIn;
function baseGetAllKeys(object, keysFunc, symbolsFunc) {
    var result1 = keysFunc(object);
    return __VIRTUAL_FILE53(object) ? result1 : __VIRTUAL_FILE80(result1, symbolsFunc(object));
}
var __VIRTUAL_FILE84 = baseGetAllKeys;
function getAllKeys(object) {
    return __VIRTUAL_FILE84(object, __VIRTUAL_FILE68, __VIRTUAL_FILE78);
}
var __VIRTUAL_FILE85 = getAllKeys;
function getAllKeysIn(object) {
    return __VIRTUAL_FILE84(object, __VIRTUAL_FILE72, __VIRTUAL_FILE82);
}
var __VIRTUAL_FILE86 = getAllKeysIn;
var DataView1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "DataView");
var __VIRTUAL_FILE87 = DataView1;
var Promise1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "Promise");
var __VIRTUAL_FILE88 = Promise1;
var Set1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "Set");
var __VIRTUAL_FILE89 = Set1;
var WeakMap1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "WeakMap");
var __VIRTUAL_FILE90 = WeakMap1;
var mapTag1 = "[object Map]", objectTag1 = "[object Object]", promiseTag = "[object Promise]", setTag1 = "[object Set]", weakMapTag1 = "[object WeakMap]";
var dataViewTag1 = "[object DataView]";
var dataViewCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE87), mapCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE26), promiseCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE88), setCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE89), weakMapCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE90);
var getTag = __VIRTUAL_FILE17;
if (__VIRTUAL_FILE87 && getTag(new __VIRTUAL_FILE87(new ArrayBuffer(1))) != dataViewTag1 || __VIRTUAL_FILE26 && getTag(new __VIRTUAL_FILE26()) != mapTag1 || __VIRTUAL_FILE88 && getTag(__VIRTUAL_FILE88.resolve()) != promiseTag || __VIRTUAL_FILE89 && getTag(new __VIRTUAL_FILE89()) != setTag1 || __VIRTUAL_FILE90 && getTag(new __VIRTUAL_FILE90()) != weakMapTag1) {
    getTag = function(value) {
        var result1 = __VIRTUAL_FILE17(value), Ctor = result1 == objectTag1 ? value.constructor : void 0, ctorString = Ctor ? __VIRTUAL_FILE22(Ctor) : "";
        if (ctorString) {
            switch(ctorString){
                case dataViewCtorString:
                    return dataViewTag1;
                case mapCtorString:
                    return mapTag1;
                case promiseCtorString:
                    return promiseTag;
                case setCtorString:
                    return setTag1;
                case weakMapCtorString:
                    return weakMapTag1;
            }
        }
        return result1;
    };
}
var __VIRTUAL_FILE91 = getTag;
var objectProto12 = Object.prototype;
var hasOwnProperty9 = objectProto12.hasOwnProperty;
function initCloneArray(array) {
    var length = array.length, result1 = new array.constructor(length);
    if (length && typeof array[0] == "string" && hasOwnProperty9.call(array, "index")) {
        result1.index = array.index;
        result1.input = array.input;
    }
    return result1;
}
var __VIRTUAL_FILE92 = initCloneArray;
var Uint8Array1 = __VIRTUAL_FILE13.Uint8Array;
var __VIRTUAL_FILE93 = Uint8Array1;
function cloneArrayBuffer(arrayBuffer) {
    var result1 = new arrayBuffer.constructor(arrayBuffer.byteLength);
    new __VIRTUAL_FILE93(result1).set(new __VIRTUAL_FILE93(arrayBuffer));
    return result1;
}
var __VIRTUAL_FILE94 = cloneArrayBuffer;
function cloneDataView(dataView, isDeep) {
    var buffer = isDeep ? __VIRTUAL_FILE94(dataView.buffer) : dataView.buffer;
    return new dataView.constructor(buffer, dataView.byteOffset, dataView.byteLength);
}
var __VIRTUAL_FILE95 = cloneDataView;
var reFlags = /\w*$/;
function cloneRegExp(regexp) {
    var result1 = new regexp.constructor(regexp.source, reFlags.exec(regexp));
    result1.lastIndex = regexp.lastIndex;
    return result1;
}
var __VIRTUAL_FILE96 = cloneRegExp;
var symbolProto = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.prototype : void 0, symbolValueOf = symbolProto ? symbolProto.valueOf : void 0;
function cloneSymbol(symbol) {
    return symbolValueOf ? Object(symbolValueOf.call(symbol)) : {};
}
var __VIRTUAL_FILE97 = cloneSymbol;
function cloneTypedArray(typedArray, isDeep) {
    var buffer = isDeep ? __VIRTUAL_FILE94(typedArray.buffer) : typedArray.buffer;
    return new typedArray.constructor(buffer, typedArray.byteOffset, typedArray.length);
}
var __VIRTUAL_FILE98 = cloneTypedArray;
var boolTag1 = "[object Boolean]", dateTag1 = "[object Date]", mapTag2 = "[object Map]", numberTag1 = "[object Number]", regexpTag1 = "[object RegExp]", setTag2 = "[object Set]", stringTag1 = "[object String]", symbolTag = "[object Symbol]";
var arrayBufferTag1 = "[object ArrayBuffer]", dataViewTag2 = "[object DataView]", float32Tag1 = "[object Float32Array]", float64Tag1 = "[object Float64Array]", int8Tag1 = "[object Int8Array]", int16Tag1 = "[object Int16Array]", int32Tag1 = "[object Int32Array]", uint8Tag1 = "[object Uint8Array]", uint8ClampedTag1 = "[object Uint8ClampedArray]", uint16Tag1 = "[object Uint16Array]", uint32Tag1 = "[object Uint32Array]";
function initCloneByTag(object, tag, isDeep) {
    var Ctor = object.constructor;
    switch(tag){
        case arrayBufferTag1:
            return __VIRTUAL_FILE94(object);
        case boolTag1:
        case dateTag1:
            return new Ctor(+object);
        case dataViewTag2:
            return __VIRTUAL_FILE95(object, isDeep);
        case float32Tag1:
        case float64Tag1:
        case int8Tag1:
        case int16Tag1:
        case int32Tag1:
        case uint8Tag1:
        case uint8ClampedTag1:
        case uint16Tag1:
        case uint32Tag1:
            return __VIRTUAL_FILE98(object, isDeep);
        case mapTag2:
            return new Ctor();
        case numberTag1:
        case stringTag1:
            return new Ctor(object);
        case regexpTag1:
            return __VIRTUAL_FILE96(object);
        case setTag2:
            return new Ctor();
        case symbolTag:
            return __VIRTUAL_FILE97(object);
    }
}
var __VIRTUAL_FILE99 = initCloneByTag;
var objectCreate = Object.create;
var baseCreate = function() {
    function object() {}
    return function(proto) {
        if (!__VIRTUAL_FILE18(proto)) {
            return {};
        }
        if (objectCreate) {
            return objectCreate(proto);
        }
        object.prototype = proto;
        var result1 = new object();
        object.prototype = void 0;
        return result1;
    };
}();
var __VIRTUAL_FILE100 = baseCreate;
function initCloneObject(object) {
    return typeof object.constructor == "function" && !__VIRTUAL_FILE63(object) ? __VIRTUAL_FILE100(__VIRTUAL_FILE81(object)) : {};
}
var __VIRTUAL_FILE101 = initCloneObject;
var mapTag3 = "[object Map]";
function baseIsMap(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE91(value) == mapTag3;
}
var __VIRTUAL_FILE102 = baseIsMap;
var nodeIsMap = __VIRTUAL_FILE60 && __VIRTUAL_FILE60.isMap;
var isMap = nodeIsMap ? __VIRTUAL_FILE59(nodeIsMap) : __VIRTUAL_FILE102;
var __VIRTUAL_FILE103 = isMap;
var setTag3 = "[object Set]";
function baseIsSet(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE91(value) == setTag3;
}
var __VIRTUAL_FILE104 = baseIsSet;
var nodeIsSet = __VIRTUAL_FILE60 && __VIRTUAL_FILE60.isSet;
var isSet = nodeIsSet ? __VIRTUAL_FILE59(nodeIsSet) : __VIRTUAL_FILE104;
var __VIRTUAL_FILE105 = isSet;
var CLONE_DEEP_FLAG = 1, CLONE_FLAT_FLAG = 2, CLONE_SYMBOLS_FLAG = 4;
var argsTag2 = "[object Arguments]", arrayTag1 = "[object Array]", boolTag2 = "[object Boolean]", dateTag2 = "[object Date]", errorTag1 = "[object Error]", funcTag2 = "[object Function]", genTag1 = "[object GeneratorFunction]", mapTag4 = "[object Map]", numberTag2 = "[object Number]", objectTag2 = "[object Object]", regexpTag2 = "[object RegExp]", setTag4 = "[object Set]", stringTag2 = "[object String]", symbolTag1 = "[object Symbol]", weakMapTag2 = "[object WeakMap]";
var arrayBufferTag2 = "[object ArrayBuffer]", dataViewTag3 = "[object DataView]", float32Tag2 = "[object Float32Array]", float64Tag2 = "[object Float64Array]", int8Tag2 = "[object Int8Array]", int16Tag2 = "[object Int16Array]", int32Tag2 = "[object Int32Array]", uint8Tag2 = "[object Uint8Array]", uint8ClampedTag2 = "[object Uint8ClampedArray]", uint16Tag2 = "[object Uint16Array]", uint32Tag2 = "[object Uint32Array]";
var cloneableTags = {};
cloneableTags[argsTag2] = cloneableTags[arrayTag1] = cloneableTags[arrayBufferTag2] = cloneableTags[dataViewTag3] = cloneableTags[boolTag2] = cloneableTags[dateTag2] = cloneableTags[float32Tag2] = cloneableTags[float64Tag2] = cloneableTags[int8Tag2] = cloneableTags[int16Tag2] = cloneableTags[int32Tag2] = cloneableTags[mapTag4] = cloneableTags[numberTag2] = cloneableTags[objectTag2] = cloneableTags[regexpTag2] = cloneableTags[setTag4] = cloneableTags[stringTag2] = cloneableTags[symbolTag1] = cloneableTags[uint8Tag2] = cloneableTags[uint8ClampedTag2] = cloneableTags[uint16Tag2] = cloneableTags[uint32Tag2] = true;
cloneableTags[errorTag1] = cloneableTags[funcTag2] = cloneableTags[weakMapTag2] = false;
function baseClone(value, bitmask, customizer, key, object, stack) {
    var result1, isDeep = bitmask & CLONE_DEEP_FLAG, isFlat = bitmask & CLONE_FLAT_FLAG, isFull = bitmask & CLONE_SYMBOLS_FLAG;
    if (customizer) {
        result1 = object ? customizer(value, key, object, stack) : customizer(value);
    }
    if (result1 !== void 0) {
        return result1;
    }
    if (!__VIRTUAL_FILE18(value)) {
        return value;
    }
    var isArr = __VIRTUAL_FILE53(value);
    if (isArr) {
        result1 = __VIRTUAL_FILE92(value);
        if (!isDeep) {
            return __VIRTUAL_FILE75(value, result1);
        }
    } else {
        var tag = __VIRTUAL_FILE91(value), isFunc = tag == funcTag2 || tag == genTag1;
        if (__VIRTUAL_FILE55(value)) {
            return __VIRTUAL_FILE74(value, isDeep);
        }
        if (tag == objectTag2 || tag == argsTag2 || isFunc && !object) {
            result1 = isFlat || isFunc ? {} : __VIRTUAL_FILE101(value);
            if (!isDeep) {
                return isFlat ? __VIRTUAL_FILE83(value, __VIRTUAL_FILE73(result1, value)) : __VIRTUAL_FILE79(value, __VIRTUAL_FILE69(result1, value));
            }
        } else {
            if (!cloneableTags[tag]) {
                return object ? value : {};
            }
            result1 = __VIRTUAL_FILE99(value, tag, isDeep);
        }
    }
    stack || (stack = new __VIRTUAL_FILE43());
    var stacked = stack.get(value);
    if (stacked) {
        return stacked;
    }
    stack.set(value, result1);
    if (__VIRTUAL_FILE105(value)) {
        value.forEach(function(subValue) {
            result1.add(baseClone(subValue, bitmask, customizer, subValue, value, stack));
        });
    } else if (__VIRTUAL_FILE103(value)) {
        value.forEach(function(subValue, key2) {
            result1.set(key2, baseClone(subValue, bitmask, customizer, key2, value, stack));
        });
    }
    var keysFunc = isFull ? isFlat ? __VIRTUAL_FILE86 : __VIRTUAL_FILE85 : isFlat ? __VIRTUAL_FILE72 : __VIRTUAL_FILE68;
    var props = isArr ? void 0 : keysFunc(value);
    __VIRTUAL_FILE44(props || value, function(subValue, key2) {
        if (props) {
            key2 = subValue;
            subValue = value[key2];
        }
        __VIRTUAL_FILE47(result1, key2, baseClone(subValue, bitmask, customizer, key2, value, stack));
    });
    return result1;
}
var __VIRTUAL_FILE106 = baseClone;
var CLONE_DEEP_FLAG1 = 1, CLONE_SYMBOLS_FLAG1 = 4;
function cloneDeep(value) {
    return __VIRTUAL_FILE106(value, CLONE_DEEP_FLAG1 | CLONE_SYMBOLS_FLAG1);
}
var __VIRTUAL_FILE107 = cloneDeep;
var now = function() {
    return __VIRTUAL_FILE13.Date.now();
};
var __VIRTUAL_FILE108 = now;
var reWhitespace = /\s/;
function trimmedEndIndex(string) {
    var index = string.length;
    while(index-- && reWhitespace.test(string.charAt(index))){}
    return index;
}
var __VIRTUAL_FILE109 = trimmedEndIndex;
var reTrimStart = /^\s+/;
function baseTrim(string) {
    return string ? string.slice(0, __VIRTUAL_FILE109(string) + 1).replace(reTrimStart, "") : string;
}
var __VIRTUAL_FILE110 = baseTrim;
var symbolTag2 = "[object Symbol]";
function isSymbol(value) {
    return typeof value == "symbol" || __VIRTUAL_FILE50(value) && __VIRTUAL_FILE17(value) == symbolTag2;
}
var __VIRTUAL_FILE111 = isSymbol;
var NAN = 0 / 0;
var reIsBadHex = /^[-+]0x[0-9a-f]+$/i;
var reIsBinary = /^0b[01]+$/i;
var reIsOctal = /^0o[0-7]+$/i;
var freeParseInt = parseInt;
function toNumber(value) {
    if (typeof value == "number") {
        return value;
    }
    if (__VIRTUAL_FILE111(value)) {
        return NAN;
    }
    if (__VIRTUAL_FILE18(value)) {
        var other = typeof value.valueOf == "function" ? value.valueOf() : value;
        value = __VIRTUAL_FILE18(other) ? other + "" : other;
    }
    if (typeof value != "string") {
        return value === 0 ? value : +value;
    }
    value = __VIRTUAL_FILE110(value);
    var isBinary = reIsBinary.test(value);
    return isBinary || reIsOctal.test(value) ? freeParseInt(value.slice(2), isBinary ? 2 : 8) : reIsBadHex.test(value) ? NAN : +value;
}
var __VIRTUAL_FILE112 = toNumber;
var FUNC_ERROR_TEXT = "Expected a function";
var nativeMax = Math.max, nativeMin = Math.min;
function debounce(func, wait, options) {
    var lastArgs, lastThis, maxWait, result1, timerId, lastCallTime, lastInvokeTime = 0, leading = false, maxing = false, trailing = true;
    if (typeof func != "function") {
        throw new TypeError(FUNC_ERROR_TEXT);
    }
    wait = __VIRTUAL_FILE112(wait) || 0;
    if (__VIRTUAL_FILE18(options)) {
        leading = !!options.leading;
        maxing = "maxWait" in options;
        maxWait = maxing ? nativeMax(__VIRTUAL_FILE112(options.maxWait) || 0, wait) : maxWait;
        trailing = "trailing" in options ? !!options.trailing : trailing;
    }
    function invokeFunc(time) {
        var args = lastArgs, thisArg = lastThis;
        lastArgs = lastThis = void 0;
        lastInvokeTime = time;
        result1 = func.apply(thisArg, args);
        return result1;
    }
    function leadingEdge(time) {
        lastInvokeTime = time;
        timerId = setTimeout(timerExpired, wait);
        return leading ? invokeFunc(time) : result1;
    }
    function remainingWait(time) {
        var timeSinceLastCall = time - lastCallTime, timeSinceLastInvoke = time - lastInvokeTime, timeWaiting = wait - timeSinceLastCall;
        return maxing ? nativeMin(timeWaiting, maxWait - timeSinceLastInvoke) : timeWaiting;
    }
    function shouldInvoke(time) {
        var timeSinceLastCall = time - lastCallTime, timeSinceLastInvoke = time - lastInvokeTime;
        return lastCallTime === void 0 || timeSinceLastCall >= wait || timeSinceLastCall < 0 || maxing && timeSinceLastInvoke >= maxWait;
    }
    function timerExpired() {
        var time = __VIRTUAL_FILE108();
        if (shouldInvoke(time)) {
            return trailingEdge(time);
        }
        timerId = setTimeout(timerExpired, remainingWait(time));
    }
    function trailingEdge(time) {
        timerId = void 0;
        if (trailing && lastArgs) {
            return invokeFunc(time);
        }
        lastArgs = lastThis = void 0;
        return result1;
    }
    function cancel() {
        if (timerId !== void 0) {
            clearTimeout(timerId);
        }
        lastInvokeTime = 0;
        lastArgs = lastCallTime = lastThis = timerId = void 0;
    }
    function flush() {
        return timerId === void 0 ? result1 : trailingEdge(__VIRTUAL_FILE108());
    }
    function debounced() {
        var time = __VIRTUAL_FILE108(), isInvoking = shouldInvoke(time);
        lastArgs = arguments;
        lastThis = this;
        lastCallTime = time;
        if (isInvoking) {
            if (timerId === void 0) {
                return leadingEdge(lastCallTime);
            }
            if (maxing) {
                clearTimeout(timerId);
                timerId = setTimeout(timerExpired, wait);
                return invokeFunc(lastCallTime);
            }
        }
        if (timerId === void 0) {
            timerId = setTimeout(timerExpired, wait);
        }
        return result1;
    }
    debounced.cancel = cancel;
    debounced.flush = flush;
    return debounced;
}
var __VIRTUAL_FILE113 = debounce;
var HASH_UNDEFINED2 = "__lodash_hash_undefined__";
function setCacheAdd(value) {
    this.__data__.set(value, HASH_UNDEFINED2);
    return this;
}
var __VIRTUAL_FILE114 = setCacheAdd;
function setCacheHas(value) {
    return this.__data__.has(value);
}
var __VIRTUAL_FILE115 = setCacheHas;
function SetCache(values) {
    var index = -1, length = values == null ? 0 : values.length;
    this.__data__ = new __VIRTUAL_FILE41();
    while(++index < length){
        this.add(values[index]);
    }
}
SetCache.prototype.add = SetCache.prototype.push = __VIRTUAL_FILE114;
SetCache.prototype.has = __VIRTUAL_FILE115;
var __VIRTUAL_FILE116 = SetCache;
function baseFindIndex(array, predicate, fromIndex, fromRight) {
    var length = array.length, index = fromIndex + (fromRight ? 1 : -1);
    while(fromRight ? index-- : ++index < length){
        if (predicate(array[index], index, array)) {
            return index;
        }
    }
    return -1;
}
var __VIRTUAL_FILE117 = baseFindIndex;
function baseIsNaN(value) {
    return value !== value;
}
var __VIRTUAL_FILE118 = baseIsNaN;
function strictIndexOf(array, value, fromIndex) {
    var index = fromIndex - 1, length = array.length;
    while(++index < length){
        if (array[index] === value) {
            return index;
        }
    }
    return -1;
}
var __VIRTUAL_FILE119 = strictIndexOf;
function baseIndexOf(array, value, fromIndex) {
    return value === value ? __VIRTUAL_FILE119(array, value, fromIndex) : __VIRTUAL_FILE117(array, __VIRTUAL_FILE118, fromIndex);
}
var __VIRTUAL_FILE120 = baseIndexOf;
function arrayIncludes(array, value) {
    var length = array == null ? 0 : array.length;
    return !!length && __VIRTUAL_FILE120(array, value, 0) > -1;
}
var __VIRTUAL_FILE121 = arrayIncludes;
function arrayIncludesWith(array, value, comparator) {
    var index = -1, length = array == null ? 0 : array.length;
    while(++index < length){
        if (comparator(value, array[index])) {
            return true;
        }
    }
    return false;
}
var __VIRTUAL_FILE122 = arrayIncludesWith;
function arrayMap(array, iteratee) {
    var index = -1, length = array == null ? 0 : array.length, result1 = Array(length);
    while(++index < length){
        result1[index] = iteratee(array[index], index, array);
    }
    return result1;
}
var __VIRTUAL_FILE123 = arrayMap;
function cacheHas(cache, key) {
    return cache.has(key);
}
var __VIRTUAL_FILE124 = cacheHas;
var LARGE_ARRAY_SIZE1 = 200;
function baseDifference(array, values, iteratee, comparator) {
    var index = -1, includes = __VIRTUAL_FILE121, isCommon = true, length = array.length, result1 = [], valuesLength = values.length;
    if (!length) {
        return result1;
    }
    if (iteratee) {
        values = __VIRTUAL_FILE123(values, __VIRTUAL_FILE59(iteratee));
    }
    if (comparator) {
        includes = __VIRTUAL_FILE122;
        isCommon = false;
    } else if (values.length >= LARGE_ARRAY_SIZE1) {
        includes = __VIRTUAL_FILE124;
        isCommon = false;
        values = new __VIRTUAL_FILE116(values);
    }
    outer: while(++index < length){
        var value = array[index], computed = iteratee == null ? value : iteratee(value);
        value = comparator || value !== 0 ? value : 0;
        if (isCommon && computed === computed) {
            var valuesIndex = valuesLength;
            while(valuesIndex--){
                if (values[valuesIndex] === computed) {
                    continue outer;
                }
            }
            result1.push(value);
        } else if (!includes(values, computed, comparator)) {
            result1.push(value);
        }
    }
    return result1;
}
var __VIRTUAL_FILE125 = baseDifference;
var spreadableSymbol = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.isConcatSpreadable : void 0;
function isFlattenable(value) {
    return __VIRTUAL_FILE53(value) || __VIRTUAL_FILE52(value) || !!(spreadableSymbol && value && value[spreadableSymbol]);
}
var __VIRTUAL_FILE126 = isFlattenable;
function baseFlatten(array, depth, predicate, isStrict, result1) {
    var index = -1, length = array.length;
    predicate || (predicate = __VIRTUAL_FILE126);
    result1 || (result1 = []);
    while(++index < length){
        var value = array[index];
        if (depth > 0 && predicate(value)) {
            if (depth > 1) {
                baseFlatten(value, depth - 1, predicate, isStrict, result1);
            } else {
                __VIRTUAL_FILE80(result1, value);
            }
        } else if (!isStrict) {
            result1[result1.length] = value;
        }
    }
    return result1;
}
var __VIRTUAL_FILE127 = baseFlatten;
function identity(value) {
    return value;
}
var __VIRTUAL_FILE128 = identity;
function apply(func, thisArg, args) {
    switch(args.length){
        case 0:
            return func.call(thisArg);
        case 1:
            return func.call(thisArg, args[0]);
        case 2:
            return func.call(thisArg, args[0], args[1]);
        case 3:
            return func.call(thisArg, args[0], args[1], args[2]);
    }
    return func.apply(thisArg, args);
}
var __VIRTUAL_FILE129 = apply;
var nativeMax1 = Math.max;
function overRest(func, start, transform) {
    start = nativeMax1(start === void 0 ? func.length - 1 : start, 0);
    return function() {
        var args = arguments, index = -1, length = nativeMax1(args.length - start, 0), array = Array(length);
        while(++index < length){
            array[index] = args[start + index];
        }
        index = -1;
        var otherArgs = Array(start + 1);
        while(++index < start){
            otherArgs[index] = args[index];
        }
        otherArgs[start] = transform(array);
        return __VIRTUAL_FILE129(func, this, otherArgs);
    };
}
var __VIRTUAL_FILE130 = overRest;
function constant(value) {
    return function() {
        return value;
    };
}
var __VIRTUAL_FILE131 = constant;
var baseSetToString = !__VIRTUAL_FILE45 ? __VIRTUAL_FILE128 : function(func, string) {
    return __VIRTUAL_FILE45(func, "toString", {
        configurable: true,
        enumerable: false,
        value: __VIRTUAL_FILE131(string),
        writable: true
    });
};
var __VIRTUAL_FILE132 = baseSetToString;
var HOT_COUNT = 800, HOT_SPAN = 16;
var nativeNow = Date.now;
function shortOut(func) {
    var count = 0, lastCalled = 0;
    return function() {
        var stamp = nativeNow(), remaining = HOT_SPAN - (stamp - lastCalled);
        lastCalled = stamp;
        if (remaining > 0) {
            if (++count >= HOT_COUNT) {
                return arguments[0];
            }
        } else {
            count = 0;
        }
        return func.apply(void 0, arguments);
    };
}
var __VIRTUAL_FILE133 = shortOut;
var setToString = __VIRTUAL_FILE133(__VIRTUAL_FILE132);
var __VIRTUAL_FILE134 = setToString;
function baseRest(func, start) {
    return __VIRTUAL_FILE134(__VIRTUAL_FILE130(func, start, __VIRTUAL_FILE128), func + "");
}
var __VIRTUAL_FILE135 = baseRest;
function isArrayLikeObject(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE67(value);
}
var __VIRTUAL_FILE136 = isArrayLikeObject;
var difference = __VIRTUAL_FILE135(function(array, values) {
    return __VIRTUAL_FILE136(array) ? __VIRTUAL_FILE125(array, __VIRTUAL_FILE127(values, 1, __VIRTUAL_FILE136, true)) : [];
});
var __VIRTUAL_FILE137 = difference;
function createBaseFor(fromRight) {
    return function(object, iteratee, keysFunc) {
        var index = -1, iterable = Object(object), props = keysFunc(object), length = props.length;
        while(length--){
            var key = props[fromRight ? length : ++index];
            if (iteratee(iterable[key], key, iterable) === false) {
                break;
            }
        }
        return object;
    };
}
var __VIRTUAL_FILE138 = createBaseFor;
var baseFor = __VIRTUAL_FILE138();
var __VIRTUAL_FILE139 = baseFor;
function baseForOwn(object, iteratee) {
    return object && __VIRTUAL_FILE139(object, iteratee, __VIRTUAL_FILE68);
}
var __VIRTUAL_FILE140 = baseForOwn;
function createBaseEach(eachFunc, fromRight) {
    return function(collection, iteratee) {
        if (collection == null) {
            return collection;
        }
        if (!__VIRTUAL_FILE67(collection)) {
            return eachFunc(collection, iteratee);
        }
        var length = collection.length, index = fromRight ? length : -1, iterable = Object(collection);
        while(fromRight ? index-- : ++index < length){
            if (iteratee(iterable[index], index, iterable) === false) {
                break;
            }
        }
        return collection;
    };
}
var __VIRTUAL_FILE141 = createBaseEach;
var baseEach = __VIRTUAL_FILE141(__VIRTUAL_FILE140);
var __VIRTUAL_FILE142 = baseEach;
function castFunction(value) {
    return typeof value == "function" ? value : __VIRTUAL_FILE128;
}
var __VIRTUAL_FILE143 = castFunction;
function forEach(collection, iteratee) {
    var func = __VIRTUAL_FILE53(collection) ? __VIRTUAL_FILE44 : __VIRTUAL_FILE142;
    return func(collection, __VIRTUAL_FILE143(iteratee));
}
var __VIRTUAL_FILE144 = forEach;
var __VIRTUAL_FILE145 = __VIRTUAL_FILE144;
function assignMergeValue(object, key, value) {
    if (value !== void 0 && !__VIRTUAL_FILE1(object[key], value) || value === void 0 && !(key in object)) {
        __VIRTUAL_FILE46(object, key, value);
    }
}
var __VIRTUAL_FILE146 = assignMergeValue;
var objectTag3 = "[object Object]";
var funcProto2 = Function.prototype, objectProto13 = Object.prototype;
var funcToString2 = funcProto2.toString;
var hasOwnProperty10 = objectProto13.hasOwnProperty;
var objectCtorString = funcToString2.call(Object);
function isPlainObject(value) {
    if (!__VIRTUAL_FILE50(value) || __VIRTUAL_FILE17(value) != objectTag3) {
        return false;
    }
    var proto = __VIRTUAL_FILE81(value);
    if (proto === null) {
        return true;
    }
    var Ctor = hasOwnProperty10.call(proto, "constructor") && proto.constructor;
    return typeof Ctor == "function" && Ctor instanceof Ctor && funcToString2.call(Ctor) == objectCtorString;
}
var __VIRTUAL_FILE147 = isPlainObject;
function safeGet(object, key) {
    if (key === "constructor" && typeof object[key] === "function") {
        return;
    }
    if (key == "__proto__") {
        return;
    }
    return object[key];
}
var __VIRTUAL_FILE148 = safeGet;
function toPlainObject(value) {
    return __VIRTUAL_FILE48(value, __VIRTUAL_FILE72(value));
}
var __VIRTUAL_FILE149 = toPlainObject;
function baseMergeDeep(object, source, key, srcIndex, mergeFunc, customizer, stack) {
    var objValue = __VIRTUAL_FILE148(object, key), srcValue = __VIRTUAL_FILE148(source, key), stacked = stack.get(srcValue);
    if (stacked) {
        __VIRTUAL_FILE146(object, key, stacked);
        return;
    }
    var newValue = customizer ? customizer(objValue, srcValue, key + "", object, source, stack) : void 0;
    var isCommon = newValue === void 0;
    if (isCommon) {
        var isArr = __VIRTUAL_FILE53(srcValue), isBuff = !isArr && __VIRTUAL_FILE55(srcValue), isTyped = !isArr && !isBuff && __VIRTUAL_FILE61(srcValue);
        newValue = srcValue;
        if (isArr || isBuff || isTyped) {
            if (__VIRTUAL_FILE53(objValue)) {
                newValue = objValue;
            } else if (__VIRTUAL_FILE136(objValue)) {
                newValue = __VIRTUAL_FILE75(objValue);
            } else if (isBuff) {
                isCommon = false;
                newValue = __VIRTUAL_FILE74(srcValue, true);
            } else if (isTyped) {
                isCommon = false;
                newValue = __VIRTUAL_FILE98(srcValue, true);
            } else {
                newValue = [];
            }
        } else if (__VIRTUAL_FILE147(srcValue) || __VIRTUAL_FILE52(srcValue)) {
            newValue = objValue;
            if (__VIRTUAL_FILE52(objValue)) {
                newValue = __VIRTUAL_FILE149(objValue);
            } else if (!__VIRTUAL_FILE18(objValue) || __VIRTUAL_FILE19(objValue)) {
                newValue = __VIRTUAL_FILE101(srcValue);
            }
        } else {
            isCommon = false;
        }
    }
    if (isCommon) {
        stack.set(srcValue, newValue);
        mergeFunc(newValue, srcValue, srcIndex, customizer, stack);
        stack["delete"](srcValue);
    }
    __VIRTUAL_FILE146(object, key, newValue);
}
var __VIRTUAL_FILE150 = baseMergeDeep;
function baseMerge(object, source, srcIndex, customizer, stack) {
    if (object === source) {
        return;
    }
    __VIRTUAL_FILE139(source, function(srcValue, key) {
        stack || (stack = new __VIRTUAL_FILE43());
        if (__VIRTUAL_FILE18(srcValue)) {
            __VIRTUAL_FILE150(object, source, key, srcIndex, baseMerge, customizer, stack);
        } else {
            var newValue = customizer ? customizer(__VIRTUAL_FILE148(object, key), srcValue, key + "", object, source, stack) : void 0;
            if (newValue === void 0) {
                newValue = srcValue;
            }
            __VIRTUAL_FILE146(object, key, newValue);
        }
    }, __VIRTUAL_FILE72);
}
var __VIRTUAL_FILE151 = baseMerge;
function isIterateeCall(value, index, object) {
    if (!__VIRTUAL_FILE18(object)) {
        return false;
    }
    var type = typeof index;
    if (type == "number" ? __VIRTUAL_FILE67(object) && __VIRTUAL_FILE56(index, object.length) : type == "string" && index in object) {
        return __VIRTUAL_FILE1(object[index], value);
    }
    return false;
}
var __VIRTUAL_FILE152 = isIterateeCall;
function createAssigner(assigner) {
    return __VIRTUAL_FILE135(function(object, sources) {
        var index = -1, length = sources.length, customizer = length > 1 ? sources[length - 1] : void 0, guard = length > 2 ? sources[2] : void 0;
        customizer = assigner.length > 3 && typeof customizer == "function" ? (length--, customizer) : void 0;
        if (guard && __VIRTUAL_FILE152(sources[0], sources[1], guard)) {
            customizer = length < 3 ? void 0 : customizer;
            length = 1;
        }
        object = Object(object);
        while(++index < length){
            var source = sources[index];
            if (source) {
                assigner(object, source, index, customizer);
            }
        }
        return object;
    });
}
var __VIRTUAL_FILE153 = createAssigner;
var mergeWith = __VIRTUAL_FILE153(function(object, source, srcIndex, customizer) {
    __VIRTUAL_FILE151(object, source, srcIndex, customizer);
});
var __VIRTUAL_FILE154 = mergeWith;
var nativeFloor = Math.floor, nativeRandom = Math.random;
function baseRandom(lower, upper) {
    return lower + nativeFloor(nativeRandom() * (upper - lower + 1));
}
var __VIRTUAL_FILE155 = baseRandom;
function shuffleSelf(array, size) {
    var index = -1, length = array.length, lastIndex = length - 1;
    size = size === void 0 ? length : size;
    while(++index < size){
        var rand = __VIRTUAL_FILE155(index, lastIndex), value = array[rand];
        array[rand] = array[index];
        array[index] = value;
    }
    array.length = size;
    return array;
}
var __VIRTUAL_FILE156 = shuffleSelf;
function arrayShuffle(array) {
    return __VIRTUAL_FILE156(__VIRTUAL_FILE75(array));
}
var __VIRTUAL_FILE157 = arrayShuffle;
function baseValues(object, props) {
    return __VIRTUAL_FILE123(props, function(key) {
        return object[key];
    });
}
var __VIRTUAL_FILE158 = baseValues;
function values(object) {
    return object == null ? [] : __VIRTUAL_FILE158(object, __VIRTUAL_FILE68(object));
}
var __VIRTUAL_FILE159 = values;
function baseShuffle(collection) {
    return __VIRTUAL_FILE156(__VIRTUAL_FILE159(collection));
}
var __VIRTUAL_FILE160 = baseShuffle;
function shuffle(collection) {
    var func = __VIRTUAL_FILE53(collection) ? __VIRTUAL_FILE157 : __VIRTUAL_FILE160;
    return func(collection);
}
var __VIRTUAL_FILE161 = shuffle;
var assignInWith = __VIRTUAL_FILE153(function(object, source, srcIndex, customizer) {
    __VIRTUAL_FILE48(source, __VIRTUAL_FILE72(source), object, customizer);
});
var __VIRTUAL_FILE162 = assignInWith;
var domExcTag = "[object DOMException]", errorTag2 = "[object Error]";
function isError(value) {
    if (!__VIRTUAL_FILE50(value)) {
        return false;
    }
    var tag = __VIRTUAL_FILE17(value);
    return tag == errorTag2 || tag == domExcTag || typeof value.message == "string" && typeof value.name == "string" && !__VIRTUAL_FILE147(value);
}
var __VIRTUAL_FILE163 = isError;
var attempt = __VIRTUAL_FILE135(function(func, args) {
    try {
        return __VIRTUAL_FILE129(func, void 0, args);
    } catch (e) {
        return __VIRTUAL_FILE163(e) ? e : new Error(e);
    }
});
var __VIRTUAL_FILE164 = attempt;
var objectProto14 = Object.prototype;
var hasOwnProperty11 = objectProto14.hasOwnProperty;
function customDefaultsAssignIn(objValue, srcValue, key, object) {
    if (objValue === void 0 || __VIRTUAL_FILE1(objValue, objectProto14[key]) && !hasOwnProperty11.call(object, key)) {
        return srcValue;
    }
    return objValue;
}
var __VIRTUAL_FILE165 = customDefaultsAssignIn;
var stringEscapes = {
    "\\": "\\",
    "'": "'",
    "\n": "n",
    "\r": "r",
    "\u2028": "u2028",
    "\u2029": "u2029"
};
function escapeStringChar(chr) {
    return "\\" + stringEscapes[chr];
}
var __VIRTUAL_FILE166 = escapeStringChar;
var reInterpolate = /<%=([\s\S]+?)%>/g;
var __VIRTUAL_FILE167 = reInterpolate;
function basePropertyOf(object) {
    return function(key) {
        return object == null ? void 0 : object[key];
    };
}
var __VIRTUAL_FILE168 = basePropertyOf;
var htmlEscapes = {
    "&": "&amp;",
    "<": "&lt;",
    ">": "&gt;",
    '"': "&quot;",
    "'": "&#39;"
};
var escapeHtmlChar = __VIRTUAL_FILE168(htmlEscapes);
var __VIRTUAL_FILE169 = escapeHtmlChar;
var INFINITY = 1 / 0;
var symbolProto1 = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.prototype : void 0, symbolToString = symbolProto1 ? symbolProto1.toString : void 0;
function baseToString(value) {
    if (typeof value == "string") {
        return value;
    }
    if (__VIRTUAL_FILE53(value)) {
        return __VIRTUAL_FILE123(value, baseToString) + "";
    }
    if (__VIRTUAL_FILE111(value)) {
        return symbolToString ? symbolToString.call(value) : "";
    }
    var result1 = value + "";
    return result1 == "0" && 1 / value == -INFINITY ? "-0" : result1;
}
var __VIRTUAL_FILE170 = baseToString;
function toString(value) {
    return value == null ? "" : __VIRTUAL_FILE170(value);
}
var __VIRTUAL_FILE171 = toString;
var reUnescapedHtml = /[&<>"']/g, reHasUnescapedHtml = RegExp(reUnescapedHtml.source);
function escape(string) {
    string = __VIRTUAL_FILE171(string);
    return string && reHasUnescapedHtml.test(string) ? string.replace(reUnescapedHtml, __VIRTUAL_FILE169) : string;
}
var __VIRTUAL_FILE172 = escape;
var reEscape = /<%-([\s\S]+?)%>/g;
var __VIRTUAL_FILE173 = reEscape;
var reEvaluate = /<%([\s\S]+?)%>/g;
var __VIRTUAL_FILE174 = reEvaluate;
var templateSettings = {
    escape: __VIRTUAL_FILE173,
    evaluate: __VIRTUAL_FILE174,
    interpolate: __VIRTUAL_FILE167,
    variable: "",
    imports: {
        _: {
            escape: __VIRTUAL_FILE172
        }
    }
};
var __VIRTUAL_FILE175 = templateSettings;
var INVALID_TEMPL_VAR_ERROR_TEXT = "Invalid `variable` option passed into `_.template`";
var reEmptyStringLeading = /\b__p \+= '';/g, reEmptyStringMiddle = /\b(__p \+=) '' \+/g, reEmptyStringTrailing = /(__e\(.*?\)|\b__t\)) \+\n'';/g;
var reForbiddenIdentifierChars = /[()=,{}\[\]\/\s]/;
var reEsTemplate = /\$\{([^\\}]*(?:\\.[^\\}]*)*)\}/g;
var reNoMatch = /($^)/;
var reUnescapedString = /['\n\r\u2028\u2029\\]/g;
var objectProto15 = Object.prototype;
var hasOwnProperty12 = objectProto15.hasOwnProperty;
function template(string, options, guard) {
    var settings = __VIRTUAL_FILE175.imports._.templateSettings || __VIRTUAL_FILE175;
    if (guard && __VIRTUAL_FILE152(string, options, guard)) {
        options = void 0;
    }
    string = __VIRTUAL_FILE171(string);
    options = __VIRTUAL_FILE162({}, options, settings, __VIRTUAL_FILE165);
    var imports = __VIRTUAL_FILE162({}, options.imports, settings.imports, __VIRTUAL_FILE165), importsKeys = __VIRTUAL_FILE68(imports), importsValues = __VIRTUAL_FILE158(imports, importsKeys);
    var isEscaping, isEvaluating, index = 0, interpolate = options.interpolate || reNoMatch, source = "__p += '";
    var reDelimiters = RegExp((options.escape || reNoMatch).source + "|" + interpolate.source + "|" + (interpolate === __VIRTUAL_FILE167 ? reEsTemplate : reNoMatch).source + "|" + (options.evaluate || reNoMatch).source + "|$", "g");
    var sourceURL = hasOwnProperty12.call(options, "sourceURL") ? "//# sourceURL=" + (options.sourceURL + "").replace(/\s/g, " ") + "\n" : "";
    string.replace(reDelimiters, function(match, escapeValue, interpolateValue, esTemplateValue, evaluateValue, offset) {
        interpolateValue || (interpolateValue = esTemplateValue);
        source += string.slice(index, offset).replace(reUnescapedString, __VIRTUAL_FILE166);
        if (escapeValue) {
            isEscaping = true;
            source += "' +\n__e(" + escapeValue + ") +\n'";
        }
        if (evaluateValue) {
            isEvaluating = true;
            source += "';\n" + evaluateValue + ";\n__p += '";
        }
        if (interpolateValue) {
            source += "' +\n((__t = (" + interpolateValue + ")) == null ? '' : __t) +\n'";
        }
        index = offset + match.length;
        return match;
    });
    source += "';\n";
    var variable = hasOwnProperty12.call(options, "variable") && options.variable;
    if (!variable) {
        source = "with (obj) {\n" + source + "\n}\n";
    } else if (reForbiddenIdentifierChars.test(variable)) {
        throw new Error(INVALID_TEMPL_VAR_ERROR_TEXT);
    }
    source = (isEvaluating ? source.replace(reEmptyStringLeading, "") : source).replace(reEmptyStringMiddle, "$1").replace(reEmptyStringTrailing, "$1;");
    source = "function(" + (variable || "obj") + ") {\n" + (variable ? "" : "obj || (obj = {});\n") + "var __t, __p = ''" + (isEscaping ? ", __e = _.escape" : "") + (isEvaluating ? ", __j = Array.prototype.join;\nfunction print() { __p += __j.call(arguments, '') }\n" : ";\n") + source + "return __p\n}";
    var result1 = __VIRTUAL_FILE164(function() {
        return Function(importsKeys, sourceURL + "return " + source).apply(void 0, importsValues);
    });
    result1.source = source;
    if (__VIRTUAL_FILE163(result1)) {
        throw result1;
    }
    return result1;
}
var __VIRTUAL_FILE176 = template;
function noop() {}
var __VIRTUAL_FILE177 = noop;
function setToArray(set) {
    var index = -1, result1 = Array(set.size);
    set.forEach(function(value) {
        result1[++index] = value;
    });
    return result1;
}
var __VIRTUAL_FILE178 = setToArray;
var INFINITY1 = 1 / 0;
var createSet = !(__VIRTUAL_FILE89 && 1 / __VIRTUAL_FILE178(new __VIRTUAL_FILE89([
    ,
    -0
]))[1] == INFINITY1) ? __VIRTUAL_FILE177 : function(values) {
    return new __VIRTUAL_FILE89(values);
};
var __VIRTUAL_FILE179 = createSet;
var LARGE_ARRAY_SIZE2 = 200;
function baseUniq(array, iteratee, comparator) {
    var index = -1, includes = __VIRTUAL_FILE121, length = array.length, isCommon = true, result1 = [], seen = result1;
    if (comparator) {
        isCommon = false;
        includes = __VIRTUAL_FILE122;
    } else if (length >= LARGE_ARRAY_SIZE2) {
        var set = iteratee ? null : __VIRTUAL_FILE179(array);
        if (set) {
            return __VIRTUAL_FILE178(set);
        }
        isCommon = false;
        includes = __VIRTUAL_FILE124;
        seen = new __VIRTUAL_FILE116();
    } else {
        seen = iteratee ? [] : result1;
    }
    outer: while(++index < length){
        var value = array[index], computed = iteratee ? iteratee(value) : value;
        value = comparator || value !== 0 ? value : 0;
        if (isCommon && computed === computed) {
            var seenIndex = seen.length;
            while(seenIndex--){
                if (seen[seenIndex] === computed) {
                    continue outer;
                }
            }
            if (iteratee) {
                seen.push(computed);
            }
            result1.push(value);
        } else if (!includes(seen, computed, comparator)) {
            if (seen !== result1) {
                seen.push(computed);
            }
            result1.push(value);
        }
    }
    return result1;
}
var __VIRTUAL_FILE180 = baseUniq;
function uniq(array) {
    return array && array.length ? __VIRTUAL_FILE180(array) : [];
}
var __VIRTUAL_FILE181 = uniq;
function arraySome(array, predicate) {
    var index = -1, length = array == null ? 0 : array.length;
    while(++index < length){
        if (predicate(array[index], index, array)) {
            return true;
        }
    }
    return false;
}
var __VIRTUAL_FILE182 = arraySome;
var COMPARE_PARTIAL_FLAG = 1, COMPARE_UNORDERED_FLAG = 2;
function equalArrays(array, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = bitmask & COMPARE_PARTIAL_FLAG, arrLength = array.length, othLength = other.length;
    if (arrLength != othLength && !(isPartial && othLength > arrLength)) {
        return false;
    }
    var arrStacked = stack.get(array);
    var othStacked = stack.get(other);
    if (arrStacked && othStacked) {
        return arrStacked == other && othStacked == array;
    }
    var index = -1, result1 = true, seen = bitmask & COMPARE_UNORDERED_FLAG ? new __VIRTUAL_FILE116() : void 0;
    stack.set(array, other);
    stack.set(other, array);
    while(++index < arrLength){
        var arrValue = array[index], othValue = other[index];
        if (customizer) {
            var compared = isPartial ? customizer(othValue, arrValue, index, other, array, stack) : customizer(arrValue, othValue, index, array, other, stack);
        }
        if (compared !== void 0) {
            if (compared) {
                continue;
            }
            result1 = false;
            break;
        }
        if (seen) {
            if (!__VIRTUAL_FILE182(other, function(othValue2, othIndex) {
                if (!__VIRTUAL_FILE124(seen, othIndex) && (arrValue === othValue2 || equalFunc(arrValue, othValue2, bitmask, customizer, stack))) {
                    return seen.push(othIndex);
                }
            })) {
                result1 = false;
                break;
            }
        } else if (!(arrValue === othValue || equalFunc(arrValue, othValue, bitmask, customizer, stack))) {
            result1 = false;
            break;
        }
    }
    stack["delete"](array);
    stack["delete"](other);
    return result1;
}
var __VIRTUAL_FILE183 = equalArrays;
function mapToArray(map) {
    var index = -1, result1 = Array(map.size);
    map.forEach(function(value, key) {
        result1[++index] = [
            key,
            value
        ];
    });
    return result1;
}
var __VIRTUAL_FILE184 = mapToArray;
var COMPARE_PARTIAL_FLAG1 = 1, COMPARE_UNORDERED_FLAG1 = 2;
var boolTag3 = "[object Boolean]", dateTag3 = "[object Date]", errorTag3 = "[object Error]", mapTag5 = "[object Map]", numberTag3 = "[object Number]", regexpTag3 = "[object RegExp]", setTag5 = "[object Set]", stringTag3 = "[object String]", symbolTag3 = "[object Symbol]";
var arrayBufferTag3 = "[object ArrayBuffer]", dataViewTag4 = "[object DataView]";
var symbolProto2 = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.prototype : void 0, symbolValueOf1 = symbolProto2 ? symbolProto2.valueOf : void 0;
function equalByTag(object, other, tag, bitmask, customizer, equalFunc, stack) {
    switch(tag){
        case dataViewTag4:
            if (object.byteLength != other.byteLength || object.byteOffset != other.byteOffset) {
                return false;
            }
            object = object.buffer;
            other = other.buffer;
        case arrayBufferTag3:
            if (object.byteLength != other.byteLength || !equalFunc(new __VIRTUAL_FILE93(object), new __VIRTUAL_FILE93(other))) {
                return false;
            }
            return true;
        case boolTag3:
        case dateTag3:
        case numberTag3:
            return __VIRTUAL_FILE1(+object, +other);
        case errorTag3:
            return object.name == other.name && object.message == other.message;
        case regexpTag3:
        case stringTag3:
            return object == other + "";
        case mapTag5:
            var convert = __VIRTUAL_FILE184;
        case setTag5:
            var isPartial = bitmask & COMPARE_PARTIAL_FLAG1;
            convert || (convert = __VIRTUAL_FILE178);
            if (object.size != other.size && !isPartial) {
                return false;
            }
            var stacked = stack.get(object);
            if (stacked) {
                return stacked == other;
            }
            bitmask |= COMPARE_UNORDERED_FLAG1;
            stack.set(object, other);
            var result1 = __VIRTUAL_FILE183(convert(object), convert(other), bitmask, customizer, equalFunc, stack);
            stack["delete"](object);
            return result1;
        case symbolTag3:
            if (symbolValueOf1) {
                return symbolValueOf1.call(object) == symbolValueOf1.call(other);
            }
    }
    return false;
}
var __VIRTUAL_FILE185 = equalByTag;
var COMPARE_PARTIAL_FLAG2 = 1;
var objectProto16 = Object.prototype;
var hasOwnProperty13 = objectProto16.hasOwnProperty;
function equalObjects(object, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = bitmask & COMPARE_PARTIAL_FLAG2, objProps = __VIRTUAL_FILE85(object), objLength = objProps.length, othProps = __VIRTUAL_FILE85(other), othLength = othProps.length;
    if (objLength != othLength && !isPartial) {
        return false;
    }
    var index = objLength;
    while(index--){
        var key = objProps[index];
        if (!(isPartial ? key in other : hasOwnProperty13.call(other, key))) {
            return false;
        }
    }
    var objStacked = stack.get(object);
    var othStacked = stack.get(other);
    if (objStacked && othStacked) {
        return objStacked == other && othStacked == object;
    }
    var result1 = true;
    stack.set(object, other);
    stack.set(other, object);
    var skipCtor = isPartial;
    while(++index < objLength){
        key = objProps[index];
        var objValue = object[key], othValue = other[key];
        if (customizer) {
            var compared = isPartial ? customizer(othValue, objValue, key, other, object, stack) : customizer(objValue, othValue, key, object, other, stack);
        }
        if (!(compared === void 0 ? objValue === othValue || equalFunc(objValue, othValue, bitmask, customizer, stack) : compared)) {
            result1 = false;
            break;
        }
        skipCtor || (skipCtor = key == "constructor");
    }
    if (result1 && !skipCtor) {
        var objCtor = object.constructor, othCtor = other.constructor;
        if (objCtor != othCtor && "constructor" in object && "constructor" in other && !(typeof objCtor == "function" && objCtor instanceof objCtor && typeof othCtor == "function" && othCtor instanceof othCtor)) {
            result1 = false;
        }
    }
    stack["delete"](object);
    stack["delete"](other);
    return result1;
}
var __VIRTUAL_FILE186 = equalObjects;
var COMPARE_PARTIAL_FLAG3 = 1;
var argsTag3 = "[object Arguments]", arrayTag2 = "[object Array]", objectTag4 = "[object Object]";
var objectProto17 = Object.prototype;
var hasOwnProperty14 = objectProto17.hasOwnProperty;
function baseIsEqualDeep(object, other, bitmask, customizer, equalFunc, stack) {
    var objIsArr = __VIRTUAL_FILE53(object), othIsArr = __VIRTUAL_FILE53(other), objTag = objIsArr ? arrayTag2 : __VIRTUAL_FILE91(object), othTag = othIsArr ? arrayTag2 : __VIRTUAL_FILE91(other);
    objTag = objTag == argsTag3 ? objectTag4 : objTag;
    othTag = othTag == argsTag3 ? objectTag4 : othTag;
    var objIsObj = objTag == objectTag4, othIsObj = othTag == objectTag4, isSameTag = objTag == othTag;
    if (isSameTag && __VIRTUAL_FILE55(object)) {
        if (!__VIRTUAL_FILE55(other)) {
            return false;
        }
        objIsArr = true;
        objIsObj = false;
    }
    if (isSameTag && !objIsObj) {
        stack || (stack = new __VIRTUAL_FILE43());
        return objIsArr || __VIRTUAL_FILE61(object) ? __VIRTUAL_FILE183(object, other, bitmask, customizer, equalFunc, stack) : __VIRTUAL_FILE185(object, other, objTag, bitmask, customizer, equalFunc, stack);
    }
    if (!(bitmask & COMPARE_PARTIAL_FLAG3)) {
        var objIsWrapped = objIsObj && hasOwnProperty14.call(object, "__wrapped__"), othIsWrapped = othIsObj && hasOwnProperty14.call(other, "__wrapped__");
        if (objIsWrapped || othIsWrapped) {
            var objUnwrapped = objIsWrapped ? object.value() : object, othUnwrapped = othIsWrapped ? other.value() : other;
            stack || (stack = new __VIRTUAL_FILE43());
            return equalFunc(objUnwrapped, othUnwrapped, bitmask, customizer, stack);
        }
    }
    if (!isSameTag) {
        return false;
    }
    stack || (stack = new __VIRTUAL_FILE43());
    return __VIRTUAL_FILE186(object, other, bitmask, customizer, equalFunc, stack);
}
var __VIRTUAL_FILE187 = baseIsEqualDeep;
function baseIsEqual(value, other, bitmask, customizer, stack) {
    if (value === other) {
        return true;
    }
    if (value == null || other == null || !__VIRTUAL_FILE50(value) && !__VIRTUAL_FILE50(other)) {
        return value !== value && other !== other;
    }
    return __VIRTUAL_FILE187(value, other, bitmask, customizer, baseIsEqual, stack);
}
var __VIRTUAL_FILE188 = baseIsEqual;
var COMPARE_PARTIAL_FLAG4 = 1, COMPARE_UNORDERED_FLAG2 = 2;
function baseIsMatch(object, source, matchData, customizer) {
    var index = matchData.length, length = index, noCustomizer = !customizer;
    if (object == null) {
        return !length;
    }
    object = Object(object);
    while(index--){
        var data = matchData[index];
        if (noCustomizer && data[2] ? data[1] !== object[data[0]] : !(data[0] in object)) {
            return false;
        }
    }
    while(++index < length){
        data = matchData[index];
        var key = data[0], objValue = object[key], srcValue = data[1];
        if (noCustomizer && data[2]) {
            if (objValue === void 0 && !(key in object)) {
                return false;
            }
        } else {
            var stack = new __VIRTUAL_FILE43();
            if (customizer) {
                var result1 = customizer(objValue, srcValue, key, object, source, stack);
            }
            if (!(result1 === void 0 ? __VIRTUAL_FILE188(srcValue, objValue, COMPARE_PARTIAL_FLAG4 | COMPARE_UNORDERED_FLAG2, customizer, stack) : result1)) {
                return false;
            }
        }
    }
    return true;
}
var __VIRTUAL_FILE189 = baseIsMatch;
function isStrictComparable(value) {
    return value === value && !__VIRTUAL_FILE18(value);
}
var __VIRTUAL_FILE190 = isStrictComparable;
function getMatchData(object) {
    var result1 = __VIRTUAL_FILE68(object), length = result1.length;
    while(length--){
        var key = result1[length], value = object[key];
        result1[length] = [
            key,
            value,
            __VIRTUAL_FILE190(value)
        ];
    }
    return result1;
}
var __VIRTUAL_FILE191 = getMatchData;
function matchesStrictComparable(key, srcValue) {
    return function(object) {
        if (object == null) {
            return false;
        }
        return object[key] === srcValue && (srcValue !== void 0 || key in Object(object));
    };
}
var __VIRTUAL_FILE192 = matchesStrictComparable;
function baseMatches(source) {
    var matchData = __VIRTUAL_FILE191(source);
    if (matchData.length == 1 && matchData[0][2]) {
        return __VIRTUAL_FILE192(matchData[0][0], matchData[0][1]);
    }
    return function(object) {
        return object === source || __VIRTUAL_FILE189(object, source, matchData);
    };
}
var __VIRTUAL_FILE193 = baseMatches;
var reIsDeepProp = /\.|\[(?:[^[\]]*|(["'])(?:(?!\1)[^\\]|\\.)*?\1)\]/, reIsPlainProp = /^\w*$/;
function isKey(value, object) {
    if (__VIRTUAL_FILE53(value)) {
        return false;
    }
    var type = typeof value;
    if (type == "number" || type == "symbol" || type == "boolean" || value == null || __VIRTUAL_FILE111(value)) {
        return true;
    }
    return reIsPlainProp.test(value) || !reIsDeepProp.test(value) || object != null && value in Object(object);
}
var __VIRTUAL_FILE194 = isKey;
var FUNC_ERROR_TEXT1 = "Expected a function";
function memoize(func, resolver) {
    if (typeof func != "function" || resolver != null && typeof resolver != "function") {
        throw new TypeError(FUNC_ERROR_TEXT1);
    }
    var memoized = function() {
        var args = arguments, key = resolver ? resolver.apply(this, args) : args[0], cache = memoized.cache;
        if (cache.has(key)) {
            return cache.get(key);
        }
        var result1 = func.apply(this, args);
        memoized.cache = cache.set(key, result1) || cache;
        return result1;
    };
    memoized.cache = new (memoize.Cache || __VIRTUAL_FILE41)();
    return memoized;
}
memoize.Cache = __VIRTUAL_FILE41;
var __VIRTUAL_FILE195 = memoize;
var MAX_MEMOIZE_SIZE = 500;
function memoizeCapped(func) {
    var result1 = __VIRTUAL_FILE195(func, function(key) {
        if (cache.size === MAX_MEMOIZE_SIZE) {
            cache.clear();
        }
        return key;
    });
    var cache = result1.cache;
    return result1;
}
var __VIRTUAL_FILE196 = memoizeCapped;
var rePropName = /[^.[\]]+|\[(?:(-?\d+(?:\.\d+)?)|(["'])((?:(?!\2)[^\\]|\\.)*?)\2)\]|(?=(?:\.|\[\])(?:\.|\[\]|$))/g;
var reEscapeChar = /\\(\\)?/g;
var stringToPath = __VIRTUAL_FILE196(function(string) {
    var result1 = [];
    if (string.charCodeAt(0) === 46) {
        result1.push("");
    }
    string.replace(rePropName, function(match, number, quote, subString) {
        result1.push(quote ? subString.replace(reEscapeChar, "$1") : number || match);
    });
    return result1;
});
var __VIRTUAL_FILE197 = stringToPath;
function castPath(value, object) {
    if (__VIRTUAL_FILE53(value)) {
        return value;
    }
    return __VIRTUAL_FILE194(value, object) ? [
        value
    ] : __VIRTUAL_FILE197(__VIRTUAL_FILE171(value));
}
var __VIRTUAL_FILE198 = castPath;
var INFINITY2 = 1 / 0;
function toKey(value) {
    if (typeof value == "string" || __VIRTUAL_FILE111(value)) {
        return value;
    }
    var result1 = value + "";
    return result1 == "0" && 1 / value == -INFINITY2 ? "-0" : result1;
}
var __VIRTUAL_FILE199 = toKey;
function baseGet(object, path) {
    path = __VIRTUAL_FILE198(path, object);
    var index = 0, length = path.length;
    while(object != null && index < length){
        object = object[__VIRTUAL_FILE199(path[index++])];
    }
    return index && index == length ? object : void 0;
}
var __VIRTUAL_FILE200 = baseGet;
function get(object, path, defaultValue) {
    var result1 = object == null ? void 0 : __VIRTUAL_FILE200(object, path);
    return result1 === void 0 ? defaultValue : result1;
}
var __VIRTUAL_FILE201 = get;
function baseHasIn(object, key) {
    return object != null && key in Object(object);
}
var __VIRTUAL_FILE202 = baseHasIn;
function hasPath(object, path, hasFunc) {
    path = __VIRTUAL_FILE198(path, object);
    var index = -1, length = path.length, result1 = false;
    while(++index < length){
        var key = __VIRTUAL_FILE199(path[index]);
        if (!(result1 = object != null && hasFunc(object, key))) {
            break;
        }
        object = object[key];
    }
    if (result1 || ++index != length) {
        return result1;
    }
    length = object == null ? 0 : object.length;
    return !!length && __VIRTUAL_FILE57(length) && __VIRTUAL_FILE56(key, length) && (__VIRTUAL_FILE53(object) || __VIRTUAL_FILE52(object));
}
var __VIRTUAL_FILE203 = hasPath;
function hasIn(object, path) {
    return object != null && __VIRTUAL_FILE203(object, path, __VIRTUAL_FILE202);
}
var __VIRTUAL_FILE204 = hasIn;
var COMPARE_PARTIAL_FLAG5 = 1, COMPARE_UNORDERED_FLAG3 = 2;
function baseMatchesProperty(path, srcValue) {
    if (__VIRTUAL_FILE194(path) && __VIRTUAL_FILE190(srcValue)) {
        return __VIRTUAL_FILE192(__VIRTUAL_FILE199(path), srcValue);
    }
    return function(object) {
        var objValue = __VIRTUAL_FILE201(object, path);
        return objValue === void 0 && objValue === srcValue ? __VIRTUAL_FILE204(object, path) : __VIRTUAL_FILE188(srcValue, objValue, COMPARE_PARTIAL_FLAG5 | COMPARE_UNORDERED_FLAG3);
    };
}
var __VIRTUAL_FILE205 = baseMatchesProperty;
function baseProperty(key) {
    return function(object) {
        return object == null ? void 0 : object[key];
    };
}
var __VIRTUAL_FILE206 = baseProperty;
function basePropertyDeep(path) {
    return function(object) {
        return __VIRTUAL_FILE200(object, path);
    };
}
var __VIRTUAL_FILE207 = basePropertyDeep;
function property(path) {
    return __VIRTUAL_FILE194(path) ? __VIRTUAL_FILE206(__VIRTUAL_FILE199(path)) : __VIRTUAL_FILE207(path);
}
var __VIRTUAL_FILE208 = property;
function baseIteratee(value) {
    if (typeof value == "function") {
        return value;
    }
    if (value == null) {
        return __VIRTUAL_FILE128;
    }
    if (typeof value == "object") {
        return __VIRTUAL_FILE53(value) ? __VIRTUAL_FILE205(value[0], value[1]) : __VIRTUAL_FILE193(value);
    }
    return __VIRTUAL_FILE208(value);
}
var __VIRTUAL_FILE209 = baseIteratee;
function uniqBy(array, iteratee) {
    return array && array.length ? __VIRTUAL_FILE180(array, __VIRTUAL_FILE209(iteratee, 2)) : [];
}
var __VIRTUAL_FILE210 = uniqBy;
function isEqual(value, other) {
    return __VIRTUAL_FILE188(value, other);
}
var __VIRTUAL_FILE211 = isEqual;
function baseMap(collection, iteratee) {
    var index = -1, result1 = __VIRTUAL_FILE67(collection) ? Array(collection.length) : [];
    __VIRTUAL_FILE142(collection, function(value, key, collection2) {
        result1[++index] = iteratee(value, key, collection2);
    });
    return result1;
}
var __VIRTUAL_FILE212 = baseMap;
function baseSortBy(array, comparer) {
    var length = array.length;
    array.sort(comparer);
    while(length--){
        array[length] = array[length].value;
    }
    return array;
}
var __VIRTUAL_FILE213 = baseSortBy;
function compareAscending(value, other) {
    if (value !== other) {
        var valIsDefined = value !== void 0, valIsNull = value === null, valIsReflexive = value === value, valIsSymbol = __VIRTUAL_FILE111(value);
        var othIsDefined = other !== void 0, othIsNull = other === null, othIsReflexive = other === other, othIsSymbol = __VIRTUAL_FILE111(other);
        if (!othIsNull && !othIsSymbol && !valIsSymbol && value > other || valIsSymbol && othIsDefined && othIsReflexive && !othIsNull && !othIsSymbol || valIsNull && othIsDefined && othIsReflexive || !valIsDefined && othIsReflexive || !valIsReflexive) {
            return 1;
        }
        if (!valIsNull && !valIsSymbol && !othIsSymbol && value < other || othIsSymbol && valIsDefined && valIsReflexive && !valIsNull && !valIsSymbol || othIsNull && valIsDefined && valIsReflexive || !othIsDefined && valIsReflexive || !othIsReflexive) {
            return -1;
        }
    }
    return 0;
}
var __VIRTUAL_FILE214 = compareAscending;
function compareMultiple(object, other, orders) {
    var index = -1, objCriteria = object.criteria, othCriteria = other.criteria, length = objCriteria.length, ordersLength = orders.length;
    while(++index < length){
        var result1 = __VIRTUAL_FILE214(objCriteria[index], othCriteria[index]);
        if (result1) {
            if (index >= ordersLength) {
                return result1;
            }
            var order = orders[index];
            return result1 * (order == "desc" ? -1 : 1);
        }
    }
    return object.index - other.index;
}
var __VIRTUAL_FILE215 = compareMultiple;
function baseOrderBy(collection, iteratees, orders) {
    if (iteratees.length) {
        iteratees = __VIRTUAL_FILE123(iteratees, function(iteratee) {
            if (__VIRTUAL_FILE53(iteratee)) {
                return function(value) {
                    return __VIRTUAL_FILE200(value, iteratee.length === 1 ? iteratee[0] : iteratee);
                };
            }
            return iteratee;
        });
    } else {
        iteratees = [
            __VIRTUAL_FILE128
        ];
    }
    var index = -1;
    iteratees = __VIRTUAL_FILE123(iteratees, __VIRTUAL_FILE59(__VIRTUAL_FILE209));
    var result1 = __VIRTUAL_FILE212(collection, function(value, key, collection2) {
        var criteria = __VIRTUAL_FILE123(iteratees, function(iteratee) {
            return iteratee(value);
        });
        return {
            criteria,
            index: ++index,
            value
        };
    });
    return __VIRTUAL_FILE213(result1, function(object, other) {
        return __VIRTUAL_FILE215(object, other, orders);
    });
}
var __VIRTUAL_FILE216 = baseOrderBy;
function orderBy(collection, iteratees, orders, guard) {
    if (collection == null) {
        return [];
    }
    if (!__VIRTUAL_FILE53(iteratees)) {
        iteratees = iteratees == null ? [] : [
            iteratees
        ];
    }
    orders = guard ? void 0 : orders;
    if (!__VIRTUAL_FILE53(orders)) {
        orders = orders == null ? [] : [
            orders
        ];
    }
    return __VIRTUAL_FILE216(collection, iteratees, orders);
}
var __VIRTUAL_FILE217 = orderBy;
const cloneDeep1 = __VIRTUAL_FILE107;
const debounce1 = __VIRTUAL_FILE113;
const shuffle1 = __VIRTUAL_FILE161;
const toString1 = __VIRTUAL_FILE171;
const uniq1 = __VIRTUAL_FILE181;
const uniqBy1 = __VIRTUAL_FILE210;
const escape1 = __VIRTUAL_FILE172;
const isWindows3 = Deno.build.os === "windows";
const isMac = Deno.build.os === "darwin";
const isLinux = Deno.build.os === "linux";
const os = Deno.build.os;
function getenv(name, defaultValue) {
    const value = Deno.env.get(name);
    if (value === undefined) {
        if (defaultValue === undefined) {
            throw new Error(`Required environment variable ${name} not specified.`);
        } else {
            return defaultValue;
        }
    } else {
        return value;
    }
}
function withPath(paths) {
    const delimiter = isWindows3 ? ";" : ":";
    const currentPath = Deno.env.get("PATH") || "";
    if (paths.append !== undefined && paths.prepend !== undefined) {
        return currentPath;
    } else if (paths.append?.length === 0 && paths.prepend?.length === 0) {
        return currentPath;
    } else {
        const modifiedPaths = [
            currentPath
        ];
        if (paths.append) {
            modifiedPaths.unshift(...paths.append);
        }
        if (paths.prepend) {
            modifiedPaths.push(...paths.prepend);
        }
        return modifiedPaths.join(delimiter);
    }
}
class MuxAsyncIterator {
    #iteratorCount = 0;
    #yields = [];
    #throws = [];
    #signal = Promise.withResolvers();
    add(iterable) {
        ++this.#iteratorCount;
        this.#callIteratorNext(iterable[Symbol.asyncIterator]());
    }
    async #callIteratorNext(iterator) {
        try {
            const { value, done } = await iterator.next();
            if (done) {
                --this.#iteratorCount;
            } else {
                this.#yields.push({
                    iterator,
                    value
                });
            }
        } catch (e) {
            this.#throws.push(e);
        }
        this.#signal.resolve();
    }
    async *iterate() {
        while(this.#iteratorCount > 0){
            await this.#signal.promise;
            for (const { iterator, value } of this.#yields){
                yield value;
                this.#callIteratorNext(iterator);
            }
            if (this.#throws.length) {
                for (const e of this.#throws){
                    throw e;
                }
            }
            this.#yields.length = 0;
            this.#signal = Promise.withResolvers();
        }
    }
    [Symbol.asyncIterator]() {
        return this.iterate();
    }
}
const ERROR_WHILE_MAPPING_MESSAGE = "Threw while mapping.";
function pooledMap(poolLimit, array, iteratorFn) {
    const res = new TransformStream({
        async transform (p, controller) {
            try {
                const s = await p;
                controller.enqueue(s);
            } catch (e) {
                if (e instanceof AggregateError && e.message === ERROR_WHILE_MAPPING_MESSAGE) {
                    controller.error(e);
                }
            }
        }
    });
    (async ()=>{
        const writer = res.writable.getWriter();
        const executing = [];
        try {
            for await (const item of array){
                const p = Promise.resolve().then(()=>iteratorFn(item));
                writer.write(p);
                const e = p.then(()=>executing.splice(executing.indexOf(e), 1));
                executing.push(e);
                if (executing.length >= poolLimit) {
                    await Promise.race(executing);
                }
            }
            await Promise.all(executing);
            writer.close();
        } catch  {
            const errors = [];
            for (const result1 of (await Promise.allSettled(executing))){
                if (result1.status === "rejected") {
                    errors.push(result1.reason);
                }
            }
            writer.write(Promise.reject(new AggregateError(errors, ERROR_WHILE_MAPPING_MESSAGE))).catch(()=>{});
        }
    })();
    return Symbol.asyncIterator in res.readable && typeof res.readable[Symbol.asyncIterator] === "function" ? res.readable[Symbol.asyncIterator]() : async function*() {
        const reader = res.readable.getReader();
        while(true){
            const { done, value } = await reader.read();
            if (done) break;
            yield value;
        }
        reader.releaseLock();
    }();
}
const DEFAULT_BUFFER_SIZE = 32 * 1024;
async function* iterateReader(reader, options) {
    const bufSize = options?.bufSize ?? DEFAULT_BUFFER_SIZE;
    const b = new Uint8Array(bufSize);
    while(true){
        const result1 = await reader.read(b);
        if (result1 === null) {
            break;
        }
        yield b.slice(0, result1);
    }
}
const cleanupHandlers = [];
function onCleanup(handler) {
    cleanupHandlers.push(handler);
}
function exitWithCleanup(code) {
    for(let i1 = cleanupHandlers.length - 1; i1 >= 0; i1--){
        const handler = cleanupHandlers[i1];
        try {
            handler();
        } catch (error) {
            info("Error occurred during cleanup: " + error);
        }
    }
    Deno.exit(code);
}
const processList = new Map();
let processCount = 0;
let cleanupRegistered = false;
function registerForExitCleanup(process) {
    const thisProcessId = ++processCount;
    processList.set(thisProcessId, process);
    return thisProcessId;
}
function unregisterForExitCleanup(processId) {
    processList.delete(processId);
}
function ensureCleanup() {
    if (!cleanupRegistered) {
        cleanupRegistered = true;
        onCleanup(()=>{
            for (const process of processList.values()){
                try {
                    process.kill();
                    process.close();
                } catch (error) {
                    info("Error occurred during cleanup: " + error);
                }
            }
        });
    }
}
async function execProcess(options, stdin, mergeOutput, stderrFilter, respectStreams, timeout) {
    const withTimeout = (promise)=>{
        return timeout ? Promise.race([
            promise,
            new Promise((_, reject)=>setTimeout(()=>reject(new Error("Process timed out")), timeout))
        ]) : promise;
    };
    ensureCleanup();
    try {
        debug(`[execProcess] ${options.cmd.join(" ")}`);
        const process = Deno.run({
            ...options,
            stdin: stdin !== undefined ? "piped" : options.stdin,
            stdout: typeof options.stdout === "number" ? options.stdout : "piped",
            stderr: typeof options.stderr === "number" ? options.stderr : "piped"
        });
        const thisProcessId = registerForExitCleanup(process);
        if (stdin !== undefined) {
            if (!process.stdin) {
                unregisterForExitCleanup(thisProcessId);
                throw new Error("Process stdin not available");
            }
            const buffer = new TextEncoder().encode(stdin);
            let offset = 0;
            while(offset < buffer.length){
                const end = Math.min(offset + 4096, buffer.length);
                const window1 = buffer.subarray(offset, end);
                const written = await process.stdin.write(window1);
                offset += written;
            }
            process.stdin.close();
        }
        let stdoutText = "";
        let stderrText = "";
        if (mergeOutput) {
            const multiplexIterator = new MuxAsyncIterator();
            const addStream = (stream, filter)=>{
                if (stream !== null) {
                    const streamIter = filter ? filteredAsyncIterator(iterateReader(stream), filter) : iterateReader(stream);
                    multiplexIterator.add(streamIter);
                }
            };
            addStream(process.stdout);
            addStream(process.stderr, stderrFilter);
            const allOutput = await processOutput(multiplexIterator, mergeOutput === "stderr>stdout" ? options.stdout : options.stderr);
            if (mergeOutput === "stderr>stdout") {
                stdoutText = allOutput;
            } else {
                stderrText = allOutput;
            }
            const closeStream = (stream)=>{
                if (stream) {
                    stream.close();
                }
            };
            closeStream(process.stdout);
            closeStream(process.stderr);
        } else {
            const promises = [];
            if (process.stdout !== null) {
                promises.push(processOutput(iterateReader(process.stdout), options.stdout, respectStreams ? "stdout" : undefined).then((text)=>{
                    stdoutText = text;
                    process.stdout.close();
                }));
            }
            if (process.stderr != null) {
                const iterator = stderrFilter ? filteredAsyncIterator(iterateReader(process.stderr), stderrFilter) : iterateReader(process.stderr);
                promises.push(processOutput(iterator, options.stderr, respectStreams ? "stderr" : undefined).then((text)=>{
                    stderrText = text;
                    process.stderr.close();
                }));
            }
            await withTimeout(Promise.all(promises));
        }
        const status = await withTimeout(process.status());
        process.close();
        unregisterForExitCleanup(thisProcessId);
        debug(`[execProcess] Success: ${status.success}, code: ${status.code}`);
        return {
            success: status.success,
            code: status.code,
            stdout: stdoutText,
            stderr: stderrText
        };
    } catch (e) {
        throw new Error(`Error executing '${options.cmd[0]}': ${e.message}`);
    }
}
function processSuccessResult() {
    return {
        success: true,
        code: 0
    };
}
function filteredAsyncIterator(iterator, filter) {
    const encoder = new TextEncoder();
    const decoder = new TextDecoder();
    return pooledMap(1, iterator, (data)=>{
        return Promise.resolve(encoder.encode(filter(decoder.decode(data))));
    });
}
async function processOutput(iterator, output, which) {
    const decoder = new TextDecoder();
    let outputText = "";
    for await (const chunk of iterator){
        if (output === "inherit" || output === undefined) {
            if (which === "stdout") {
                Deno.stdout.writeSync(chunk);
            } else if (which === "stderr") {
                Deno.stderr.writeSync(chunk);
            } else {
                info(decoder.decode(chunk), {
                    newline: false
                });
            }
        }
        const text = decoder.decode(chunk);
        outputText += text;
    }
    return outputText;
}
const kSkipHidden = /[/\\][\.]/;
function removeIfExists(file) {
    if (existsSync(file)) {
        safeRemoveSync(file, {
            recursive: true
        });
    }
}
function safeRemoveIfExists(file) {
    try {
        removeIfExists(file);
    } catch (error) {
        warn(`Error removing file ${file}: ${error.message}`);
    }
}
function removeIfEmptyDir(dir) {
    if (existsSync(dir)) {
        let empty = true;
        for (const _entry of Deno.readDirSync(dir)){
            empty = false;
            break;
        }
        if (empty) {
            safeRemoveSync(dir, {
                recursive: true
            });
            return true;
        }
        return false;
    } else {
        return false;
    }
}
function isModifiedAfter(file, otherFile) {
    if (!existsSync(file)) {
        throw new Error(`${file} does not exist`);
    }
    if (!existsSync(otherFile)) {
        return true;
    }
    const fileInfo = Deno.statSync(file);
    const otherfileInfo = Deno.statSync(otherFile);
    if (fileInfo.mtime === null || otherfileInfo.mtime === null) {
        return true;
    } else {
        return fileInfo.mtime > otherfileInfo.mtime;
    }
}
function dirAndStem(file) {
    return [
        dirname2(file),
        basename2(file, extname2(file))
    ];
}
function expandPath(path) {
    if (path === "~") {
        return getenv("HOME", "~");
    } else {
        return path.replace(/^~\//, getenv("HOME", "~") + "/");
    }
}
function safeExistsSync(path) {
    try {
        return existsSync(path);
    } catch  {
        return false;
    }
}
async function which(cmd) {
    const args = isWindows3 ? [
        "CMD",
        "/C",
        "where",
        cmd
    ] : [
        "which",
        cmd
    ];
    const result1 = await execProcess({
        cmd: args,
        stderr: "piped",
        stdout: "piped"
    });
    if (result1.code === 0) {
        return isWindows3 ? result1.stdout?.split("\n")[0].trim() : result1.stdout?.trim();
    } else {
        return undefined;
    }
}
function filterPaths(root, paths, globs, options) {
    const expandGlobs = (targetGlobs)=>{
        const expanded = [];
        for (const glob of targetGlobs){
            const needSlash = !root.endsWith("/") && !glob.startsWith("/");
            const regex = globToRegExp2(`${root}${needSlash ? "/" : ""}${glob}`);
            const matchingFiles = paths.filter((path)=>{
                return regex.test(path);
            });
            expanded.push(...matchingFiles);
        }
        return __VIRTUAL_FILE181(expanded);
    };
    return resolveGlobs(root, globs, expandGlobs, options);
}
function resolvePathGlobs(root, globs, exclude, options) {
    const expandGlobs = (targetGlobs)=>{
        const expanded = [];
        for (const glob of targetGlobs){
            for (const file of expandGlobSync(glob, {
                root,
                exclude,
                includeDirs: true,
                extended: true,
                globstar: true
            })){
                expanded.push(file.path);
            }
        }
        return __VIRTUAL_FILE181(expanded);
    };
    return resolveGlobs(root, globs, expandGlobs, options);
}
function pathWithForwardSlashes(path) {
    return path.replace(/\\/g, "/");
}
function ensureTrailingSlash(path) {
    if (path && !path.endsWith("/")) {
        return path + "/";
    } else {
        return path;
    }
}
function resolveGlobs(root, globs, expandGlobs, options) {
    const includeGlobs = [];
    const excludeGlobs = [];
    const asFullGlob = (glob, preferSmart)=>{
        const useSmartGlobs = ()=>{
            if (options?.mode === "strict") {
                return false;
            } else if (options?.mode === "always") {
                return true;
            } else if (options?.mode === "auto") {
                if (preferSmart) {
                    return true;
                } else {
                    return isGlob(glob);
                }
            } else {
                return true;
            }
        };
        const smartGlob = useSmartGlobs();
        if (glob.startsWith("\\!")) {
            glob = glob.slice(1);
        }
        if (smartGlob) {
            glob = glob.replace(/^\.([\/\\])+/, "$1");
            if (glob.endsWith("/")) {
                glob = glob + "**/*";
            } else {
                const fullPath = join2(root, glob);
                try {
                    if (Deno.statSync(fullPath).isDirectory) {
                        glob = glob + "/**/*";
                    }
                } catch  {}
            }
        }
        if (!glob.startsWith("/")) {
            if (smartGlob && (!options || !options.explicitSubfolderSearch)) {
                return "**/" + glob;
            } else {
                return glob;
            }
        } else {
            return glob.slice(1);
        }
    };
    for (const glob of globs){
        if (glob.startsWith("!")) {
            excludeGlobs.push(asFullGlob(glob.slice(1), true));
        } else {
            includeGlobs.push(asFullGlob(glob));
        }
    }
    const includeFiles = expandGlobs(includeGlobs);
    const excludeFiles = expandGlobs(excludeGlobs);
    return {
        include: includeFiles,
        exclude: excludeFiles
    };
}
function normalizePath(path) {
    let file = path instanceof URL ? fromFileUrl2(path) : path;
    if (!isAbsolute2(file)) {
        file = join3(Deno.cwd(), file);
    }
    file = normalize3(file);
    return file.replace(/^\w:\\/, (m)=>m[0].toUpperCase() + ":\\");
}
function suggestUserBinPaths() {
    if (!isWindows3) {
        const possiblePaths = [
            "/usr/local/bin",
            "~/.local/bin",
            "~/bin"
        ];
        const pathRaw = Deno.env.get("PATH");
        const paths = pathRaw ? pathRaw.split(":") : [];
        return possiblePaths.filter((path)=>{
            return paths.includes(path) || paths.includes(expandPath(path));
        });
    } else {
        throw new Error("suggestUserBinPaths not currently supported on Windows");
    }
}
Deno.realPathSync = normalizePath;
function withAttempts(callable) {
    for(let i1 = 0; i1 < 5; i1++){
        try {
            return callable();
        } catch (err) {
            if (err.message) {
                debug("Error attempting to create temp file: " + err.message);
                if (i1 === 5 - 1) {
                    throw err;
                } else {
                    debug(`Retrying... ${i1 + 1} of ${5 - 1}`);
                }
            }
        }
    }
    throw new Error("Shouldn't get here");
}
function withAttemptsAsync(callable) {
    const inner = (attempt)=>{
        return callable().catch((err)=>{
            if (err.message) {
                debug("Error attempting to create temp file: " + err.message);
            }
            if (attempt === 5 - 1) {
                throw err;
            } else {
                debug(`Retrying... ${attempt + 1} of ${5 - 1}`);
                return inner(attempt + 1);
            }
        });
    };
    return inner(0);
}
const oldMakeTempFile = Deno.makeTempFile;
const oldMakeTempFileSync = Deno.makeTempFileSync;
const oldMakeTempDir = Deno.makeTempDir;
const oldMakeTempDirSync = Deno.makeTempDirSync;
function makeTempFileSync(options) {
    return withAttempts(()=>oldMakeTempFileSync(options));
}
function makeTempFile(options) {
    return withAttemptsAsync(()=>oldMakeTempFile(options));
}
function makeTempDirSync(options) {
    return withAttempts(()=>oldMakeTempDirSync(options));
}
function makeTempDir(options) {
    return withAttemptsAsync(()=>oldMakeTempDir(options));
}
Deno.makeTempFile = makeTempFile;
Deno.makeTempFileSync = makeTempFileSync;
Deno.makeTempDir = makeTempDir;
Deno.makeTempDirSync = makeTempDirSync;
const oldReadTextFile = Deno.readTextFile;
const oldReadTextFileSync = Deno.readTextFileSync;
Deno.readTextFile = async (path, options)=>{
    try {
        const result1 = await oldReadTextFile(path, options);
        return result1;
    } catch (err) {
        if (err.message) {
            err.message = err.message + "\n" + "Path: " + path;
        }
        throw err;
    }
};
Deno.readTextFileSync = (path)=>{
    try {
        const result1 = oldReadTextFileSync(path);
        return result1;
    } catch (err) {
        if (err.message) {
            err.message = err.message + "\n" + "Path: " + path;
        }
        throw err;
    }
};
const { Deno: Deno2 } = globalThis;
const noColor1 = typeof Deno2?.noColor === "boolean" ? Deno2.noColor : false;
let enabled1 = !noColor1;
function setColorEnabled(value) {
    if (Deno2?.noColor) {
        return;
    }
    enabled1 = value;
}
function getColorEnabled() {
    return enabled1;
}
function code1(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run1(str, code) {
    return enabled1 ? `${code.open}${str.replace(code.regexp, code.open)}${code.close}` : str;
}
function reset(str) {
    return run1(str, code1([
        0
    ], 0));
}
function bold1(str) {
    return run1(str, code1([
        1
    ], 22));
}
function dim(str) {
    return run1(str, code1([
        2
    ], 22));
}
function italic(str) {
    return run1(str, code1([
        3
    ], 23));
}
function underline(str) {
    return run1(str, code1([
        4
    ], 24));
}
function inverse(str) {
    return run1(str, code1([
        7
    ], 27));
}
function hidden(str) {
    return run1(str, code1([
        8
    ], 28));
}
function strikethrough(str) {
    return run1(str, code1([
        9
    ], 29));
}
function black(str) {
    return run1(str, code1([
        30
    ], 39));
}
function red1(str) {
    return run1(str, code1([
        31
    ], 39));
}
function green(str) {
    return run1(str, code1([
        32
    ], 39));
}
function yellow1(str) {
    return run1(str, code1([
        33
    ], 39));
}
function blue1(str) {
    return run1(str, code1([
        34
    ], 39));
}
function magenta(str) {
    return run1(str, code1([
        35
    ], 39));
}
function cyan(str) {
    return run1(str, code1([
        36
    ], 39));
}
function white(str) {
    return run1(str, code1([
        37
    ], 39));
}
function gray(str) {
    return brightBlack(str);
}
function brightBlack(str) {
    return run1(str, code1([
        90
    ], 39));
}
function brightRed(str) {
    return run1(str, code1([
        91
    ], 39));
}
function brightGreen(str) {
    return run1(str, code1([
        92
    ], 39));
}
function brightYellow(str) {
    return run1(str, code1([
        93
    ], 39));
}
function brightBlue(str) {
    return run1(str, code1([
        94
    ], 39));
}
function brightMagenta(str) {
    return run1(str, code1([
        95
    ], 39));
}
function brightCyan(str) {
    return run1(str, code1([
        96
    ], 39));
}
function brightWhite(str) {
    return run1(str, code1([
        97
    ], 39));
}
function bgBlack(str) {
    return run1(str, code1([
        40
    ], 49));
}
function bgRed(str) {
    return run1(str, code1([
        41
    ], 49));
}
function bgGreen(str) {
    return run1(str, code1([
        42
    ], 49));
}
function bgYellow(str) {
    return run1(str, code1([
        43
    ], 49));
}
function bgBlue(str) {
    return run1(str, code1([
        44
    ], 49));
}
function bgMagenta(str) {
    return run1(str, code1([
        45
    ], 49));
}
function bgCyan(str) {
    return run1(str, code1([
        46
    ], 49));
}
function bgWhite(str) {
    return run1(str, code1([
        47
    ], 49));
}
function bgBrightBlack(str) {
    return run1(str, code1([
        100
    ], 49));
}
function bgBrightRed(str) {
    return run1(str, code1([
        101
    ], 49));
}
function bgBrightGreen(str) {
    return run1(str, code1([
        102
    ], 49));
}
function bgBrightYellow(str) {
    return run1(str, code1([
        103
    ], 49));
}
function bgBrightBlue(str) {
    return run1(str, code1([
        104
    ], 49));
}
function bgBrightMagenta(str) {
    return run1(str, code1([
        105
    ], 49));
}
function bgBrightCyan(str) {
    return run1(str, code1([
        106
    ], 49));
}
function bgBrightWhite(str) {
    return run1(str, code1([
        107
    ], 49));
}
function clampAndTruncate(n, max = 255, min = 0) {
    return Math.trunc(Math.max(Math.min(n, max), min));
}
function rgb8(str, color) {
    return run1(str, code1([
        38,
        5,
        clampAndTruncate(color)
    ], 39));
}
function bgRgb8(str, color) {
    return run1(str, code1([
        48,
        5,
        clampAndTruncate(color)
    ], 49));
}
function rgb24(str, color) {
    if (typeof color === "number") {
        return run1(str, code1([
            38,
            2,
            color >> 16 & 0xff,
            color >> 8 & 0xff,
            color & 0xff
        ], 39));
    }
    return run1(str, code1([
        38,
        2,
        clampAndTruncate(color.r),
        clampAndTruncate(color.g),
        clampAndTruncate(color.b)
    ], 39));
}
function bgRgb24(str, color) {
    if (typeof color === "number") {
        return run1(str, code1([
            48,
            2,
            color >> 16 & 0xff,
            color >> 8 & 0xff,
            color & 0xff
        ], 49));
    }
    return run1(str, code1([
        48,
        2,
        clampAndTruncate(color.r),
        clampAndTruncate(color.g),
        clampAndTruncate(color.b)
    ], 49));
}
const ANSI_PATTERN = new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]+)*|[a-zA-Z\\d]+(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-nq-uy=><~]))"
].join("|"), "g");
function stripColor(string) {
    return string.replace(ANSI_PATTERN, "");
}
const mod1 = {
    setColorEnabled: setColorEnabled,
    getColorEnabled: getColorEnabled,
    reset: reset,
    bold: bold1,
    dim: dim,
    italic: italic,
    underline: underline,
    inverse: inverse,
    hidden: hidden,
    strikethrough: strikethrough,
    black: black,
    red: red1,
    green: green,
    yellow: yellow1,
    blue: blue1,
    magenta: magenta,
    cyan: cyan,
    white: white,
    gray: gray,
    brightBlack: brightBlack,
    brightRed: brightRed,
    brightGreen: brightGreen,
    brightYellow: brightYellow,
    brightBlue: brightBlue,
    brightMagenta: brightMagenta,
    brightCyan: brightCyan,
    brightWhite: brightWhite,
    bgBlack: bgBlack,
    bgRed: bgRed,
    bgGreen: bgGreen,
    bgYellow: bgYellow,
    bgBlue: bgBlue,
    bgMagenta: bgMagenta,
    bgCyan: bgCyan,
    bgWhite: bgWhite,
    bgBrightBlack: bgBrightBlack,
    bgBrightRed: bgBrightRed,
    bgBrightGreen: bgBrightGreen,
    bgBrightYellow: bgBrightYellow,
    bgBrightBlue: bgBrightBlue,
    bgBrightMagenta: bgBrightMagenta,
    bgBrightCyan: bgBrightCyan,
    bgBrightWhite: bgBrightWhite,
    rgb8: rgb8,
    bgRgb8: bgRgb8,
    rgb24: rgb24,
    bgRgb24: bgRgb24,
    stripColor: stripColor
};
function distance(a, b) {
    if (a.length == 0) {
        return b.length;
    }
    if (b.length == 0) {
        return a.length;
    }
    const matrix = [];
    for(let i1 = 0; i1 <= b.length; i1++){
        matrix[i1] = [
            i1
        ];
    }
    for(let j = 0; j <= a.length; j++){
        matrix[0][j] = j;
    }
    for(let i1 = 1; i1 <= b.length; i1++){
        for(let j = 1; j <= a.length; j++){
            if (b.charAt(i1 - 1) == a.charAt(j - 1)) {
                matrix[i1][j] = matrix[i1 - 1][j - 1];
            } else {
                matrix[i1][j] = Math.min(matrix[i1 - 1][j - 1] + 1, Math.min(matrix[i1][j - 1] + 1, matrix[i1 - 1][j] + 1));
            }
        }
    }
    return matrix[b.length][a.length];
}
function paramCaseToCamelCase(str) {
    return str.replace(/-([a-z])/g, (g)=>g[1].toUpperCase());
}
function underscoreToCamelCase(str) {
    return str.replace(/([a-z])([A-Z])/g, "$1_$2").toLowerCase().replace(/_([a-z])/g, (g)=>g[1].toUpperCase());
}
function getOption(flags, name) {
    while(name[0] === "-"){
        name = name.slice(1);
    }
    for (const flag of flags){
        if (isOption(flag, name)) {
            return flag;
        }
    }
    return;
}
function didYouMeanOption(option, options) {
    const optionNames = options.map((option)=>[
            option.name,
            ...option.aliases ?? []
        ]).flat().map((option)=>getFlag(option));
    return didYouMean(" Did you mean option", getFlag(option), optionNames);
}
function didYouMeanType(type, types) {
    return didYouMean(" Did you mean type", type, types);
}
function didYouMean(message, type, types) {
    const match = closest(type, types);
    return match ? `${message} "${match}"?` : "";
}
function getFlag(name) {
    if (name.startsWith("-")) {
        return name;
    }
    if (name.length > 1) {
        return `--${name}`;
    }
    return `-${name}`;
}
function isOption(option, name) {
    return option.name === name || option.aliases && option.aliases.indexOf(name) !== -1;
}
function matchWildCardOptions(name, flags) {
    for (const option of flags){
        if (option.name.indexOf("*") === -1) {
            continue;
        }
        let matched = matchWildCardOption(name, option);
        if (matched) {
            matched = {
                ...matched,
                name
            };
            flags.push(matched);
            return matched;
        }
    }
}
function matchWildCardOption(name, option) {
    const parts = option.name.split(".");
    const parts2 = name.split(".");
    if (parts.length !== parts2.length) {
        return false;
    }
    const count = Math.max(parts.length, parts2.length);
    for(let i1 = 0; i1 < count; i1++){
        if (parts[i1] !== parts2[i1] && parts[i1] !== "*") {
            return false;
        }
    }
    return option;
}
function closest(str, arr) {
    let minDistance = Infinity;
    let minIndex = 0;
    for(let i1 = 0; i1 < arr.length; i1++){
        const dist = distance(str, arr[i1]);
        if (dist < minDistance) {
            minDistance = dist;
            minIndex = i1;
        }
    }
    return arr[minIndex];
}
function getDefaultValue(option) {
    return typeof option.default === "function" ? option.default() : option.default;
}
class FlagsError extends Error {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, FlagsError.prototype);
    }
}
class UnknownRequiredOptionError extends FlagsError {
    constructor(option, options){
        super(`Unknown required option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownRequiredOptionError.prototype);
    }
}
class UnknownConflictingOptionError extends FlagsError {
    constructor(option, options){
        super(`Unknown conflicting option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownConflictingOptionError.prototype);
    }
}
class UnknownTypeError extends FlagsError {
    constructor(type, types){
        super(`Unknown type "${type}".${didYouMeanType(type, types)}`);
        Object.setPrototypeOf(this, UnknownTypeError.prototype);
    }
}
class ValidationError extends FlagsError {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, ValidationError.prototype);
    }
}
class DuplicateOptionError extends ValidationError {
    constructor(name){
        super(`Option "${getFlag(name).replace(/^--no-/, "--")}" can only occur once, but was found several times.`);
        Object.setPrototypeOf(this, DuplicateOptionError.prototype);
    }
}
class InvalidOptionError extends ValidationError {
    constructor(option, options){
        super(`Invalid option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, InvalidOptionError.prototype);
    }
}
class UnknownOptionError extends ValidationError {
    constructor(option, options){
        super(`Unknown option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownOptionError.prototype);
    }
}
class MissingOptionValueError extends ValidationError {
    constructor(option){
        super(`Missing value for option "${getFlag(option)}".`);
        Object.setPrototypeOf(this, MissingOptionValueError.prototype);
    }
}
class InvalidOptionValueError extends ValidationError {
    constructor(option, expected, value){
        super(`Option "${getFlag(option)}" must be of type "${expected}", but got "${value}".`);
        Object.setPrototypeOf(this, InvalidOptionValueError.prototype);
    }
}
class UnexpectedOptionValueError extends ValidationError {
    constructor(option, value){
        super(`Option "${getFlag(option)}" doesn't take a value, but got "${value}".`);
        Object.setPrototypeOf(this, InvalidOptionValueError.prototype);
    }
}
class OptionNotCombinableError extends ValidationError {
    constructor(option){
        super(`Option "${getFlag(option)}" cannot be combined with other options.`);
        Object.setPrototypeOf(this, OptionNotCombinableError.prototype);
    }
}
class ConflictingOptionError extends ValidationError {
    constructor(option, conflictingOption){
        super(`Option "${getFlag(option)}" conflicts with option "${getFlag(conflictingOption)}".`);
        Object.setPrototypeOf(this, ConflictingOptionError.prototype);
    }
}
class DependingOptionError extends ValidationError {
    constructor(option, dependingOption){
        super(`Option "${getFlag(option)}" depends on option "${getFlag(dependingOption)}".`);
        Object.setPrototypeOf(this, DependingOptionError.prototype);
    }
}
class MissingRequiredOptionError extends ValidationError {
    constructor(option){
        super(`Missing required option "${getFlag(option)}".`);
        Object.setPrototypeOf(this, MissingRequiredOptionError.prototype);
    }
}
class UnexpectedRequiredArgumentError extends ValidationError {
    constructor(arg){
        super(`An required argument cannot follow an optional argument, but "${arg}"  is defined as required.`);
        Object.setPrototypeOf(this, UnexpectedRequiredArgumentError.prototype);
    }
}
class UnexpectedArgumentAfterVariadicArgumentError extends ValidationError {
    constructor(arg){
        super(`An argument cannot follow an variadic argument, but got "${arg}".`);
        Object.setPrototypeOf(this, UnexpectedArgumentAfterVariadicArgumentError.prototype);
    }
}
class InvalidTypeError extends ValidationError {
    constructor({ label, name, value, type }, expected){
        super(`${label} "${name}" must be of type "${type}", but got "${value}".` + (expected ? ` Expected values: ${expected.map((value)=>`"${value}"`).join(", ")}` : ""));
        Object.setPrototypeOf(this, MissingOptionValueError.prototype);
    }
}
var OptionType;
(function(OptionType) {
    OptionType["STRING"] = "string";
    OptionType["NUMBER"] = "number";
    OptionType["INTEGER"] = "integer";
    OptionType["BOOLEAN"] = "boolean";
})(OptionType || (OptionType = {}));
const __boolean = (type)=>{
    if (~[
        "1",
        "true"
    ].indexOf(type.value)) {
        return true;
    }
    if (~[
        "0",
        "false"
    ].indexOf(type.value)) {
        return false;
    }
    throw new InvalidTypeError(type, [
        "true",
        "false",
        "1",
        "0"
    ]);
};
const number = (type)=>{
    const value = Number(type.value);
    if (Number.isFinite(value)) {
        return value;
    }
    throw new InvalidTypeError(type);
};
const string = ({ value })=>{
    return value;
};
function validateFlags(ctx, opts, options = new Map()) {
    if (!opts.flags) {
        return;
    }
    setDefaultValues(ctx, opts);
    const optionNames = Object.keys(ctx.flags);
    if (!optionNames.length && opts.allowEmpty) {
        return;
    }
    if (ctx.standalone) {
        validateStandaloneOption(ctx, options, optionNames);
        return;
    }
    for (const [name, option] of options){
        validateUnknownOption(option, opts);
        validateConflictingOptions(ctx, option);
        validateDependingOptions(ctx, option);
        validateRequiredValues(ctx, option, name);
    }
    validateRequiredOptions(ctx, options, opts);
}
function validateUnknownOption(option, opts) {
    if (!getOption(opts.flags ?? [], option.name)) {
        throw new UnknownOptionError(option.name, opts.flags ?? []);
    }
}
function setDefaultValues(ctx, opts) {
    if (!opts.flags?.length) {
        return;
    }
    for (const option of opts.flags){
        let name;
        let defaultValue = undefined;
        if (option.name.startsWith("no-")) {
            const propName = option.name.replace(/^no-/, "");
            if (typeof ctx.flags[propName] !== "undefined") {
                continue;
            }
            const positiveOption = getOption(opts.flags, propName);
            if (positiveOption) {
                continue;
            }
            name = paramCaseToCamelCase(propName);
            defaultValue = true;
        }
        if (!name) {
            name = paramCaseToCamelCase(option.name);
        }
        const hasDefaultValue = (!opts.ignoreDefaults || typeof opts.ignoreDefaults[name] === "undefined") && typeof ctx.flags[name] === "undefined" && (typeof option.default !== "undefined" || typeof defaultValue !== "undefined");
        if (hasDefaultValue) {
            ctx.flags[name] = getDefaultValue(option) ?? defaultValue;
            ctx.defaults[option.name] = true;
            if (typeof option.value === "function") {
                ctx.flags[name] = option.value(ctx.flags[name]);
            }
        }
    }
}
function validateStandaloneOption(ctx, options, optionNames) {
    if (!ctx.standalone || optionNames.length === 1) {
        return;
    }
    for (const [_, opt] of options){
        if (!ctx.defaults[opt.name] && opt !== ctx.standalone) {
            throw new OptionNotCombinableError(ctx.standalone.name);
        }
    }
}
function validateConflictingOptions(ctx, option) {
    if (!option.conflicts?.length) {
        return;
    }
    for (const flag of option.conflicts){
        if (isset(flag, ctx.flags)) {
            throw new ConflictingOptionError(option.name, flag);
        }
    }
}
function validateDependingOptions(ctx, option) {
    if (!option.depends) {
        return;
    }
    for (const flag of option.depends){
        if (!isset(flag, ctx.flags) && !ctx.defaults[option.name]) {
            throw new DependingOptionError(option.name, flag);
        }
    }
}
function validateRequiredValues(ctx, option, name) {
    if (!option.args) {
        return;
    }
    const isArray = option.args.length > 1;
    for(let i1 = 0; i1 < option.args.length; i1++){
        const arg = option.args[i1];
        if (arg.optional) {
            continue;
        }
        const hasValue = isArray ? typeof ctx.flags[name][i1] !== "undefined" : typeof ctx.flags[name] !== "undefined";
        if (!hasValue) {
            throw new MissingOptionValueError(option.name);
        }
    }
}
function validateRequiredOptions(ctx, options, opts) {
    if (!opts.flags?.length) {
        return;
    }
    const optionsValues = [
        ...options.values()
    ];
    for (const option of opts.flags){
        if (!option.required || paramCaseToCamelCase(option.name) in ctx.flags) {
            continue;
        }
        const conflicts = option.conflicts ?? [];
        const hasConflict = conflicts.find((flag)=>!!ctx.flags[flag]);
        const hasConflicts = hasConflict || optionsValues.find((opt)=>opt.conflicts?.find((flag)=>flag === option.name));
        if (hasConflicts) {
            continue;
        }
        throw new MissingRequiredOptionError(option.name);
    }
}
function isset(flagName, flags) {
    const name = paramCaseToCamelCase(flagName);
    return typeof flags[name] !== "undefined";
}
const integer = (type)=>{
    const value = Number(type.value);
    if (Number.isInteger(value)) {
        return value;
    }
    throw new InvalidTypeError(type);
};
const DefaultTypes = {
    string,
    number,
    integer,
    boolean: __boolean
};
function parseFlags(argsOrCtx, opts = {}) {
    let args;
    let ctx;
    if (Array.isArray(argsOrCtx)) {
        ctx = {};
        args = argsOrCtx;
    } else {
        ctx = argsOrCtx;
        args = argsOrCtx.unknown;
        argsOrCtx.unknown = [];
    }
    args = args.slice();
    ctx.flags ??= {};
    ctx.literal ??= [];
    ctx.unknown ??= [];
    ctx.stopEarly = false;
    ctx.stopOnUnknown = false;
    ctx.defaults ??= {};
    opts.dotted ??= true;
    validateOptions(opts);
    const options = parseArgs(ctx, args, opts);
    validateFlags(ctx, opts, options);
    if (opts.dotted) {
        parseDottedOptions(ctx);
    }
    return ctx;
}
function validateOptions(opts) {
    opts.flags?.forEach((opt)=>{
        opt.depends?.forEach((flag)=>{
            if (!opts.flags || !getOption(opts.flags, flag)) {
                throw new UnknownRequiredOptionError(flag, opts.flags ?? []);
            }
        });
        opt.conflicts?.forEach((flag)=>{
            if (!opts.flags || !getOption(opts.flags, flag)) {
                throw new UnknownConflictingOptionError(flag, opts.flags ?? []);
            }
        });
    });
}
function parseArgs(ctx, args, opts) {
    const optionsMap = new Map();
    let inLiteral = false;
    for(let argsIndex = 0; argsIndex < args.length; argsIndex++){
        let option;
        let current = args[argsIndex];
        let currentValue;
        let negate = false;
        if (inLiteral) {
            ctx.literal.push(current);
            continue;
        } else if (current === "--") {
            inLiteral = true;
            continue;
        } else if (ctx.stopEarly || ctx.stopOnUnknown) {
            ctx.unknown.push(current);
            continue;
        }
        const isFlag = current.length > 1 && current[0] === "-";
        if (!isFlag) {
            if (opts.stopEarly) {
                ctx.stopEarly = true;
            }
            ctx.unknown.push(current);
            continue;
        }
        const isShort = current[1] !== "-";
        const isLong = isShort ? false : current.length > 3 && current[2] !== "-";
        if (!isShort && !isLong) {
            throw new InvalidOptionError(current, opts.flags ?? []);
        }
        if (isShort && current.length > 2 && current[2] !== ".") {
            args.splice(argsIndex, 1, ...splitFlags(current));
            current = args[argsIndex];
        } else if (isLong && current.startsWith("--no-")) {
            negate = true;
        }
        const equalSignIndex = current.indexOf("=");
        if (equalSignIndex !== -1) {
            currentValue = current.slice(equalSignIndex + 1) || undefined;
            current = current.slice(0, equalSignIndex);
        }
        if (opts.flags) {
            option = getOption(opts.flags, current);
            if (!option) {
                const name = current.replace(/^-+/, "");
                option = matchWildCardOptions(name, opts.flags);
                if (!option) {
                    if (opts.stopOnUnknown) {
                        ctx.stopOnUnknown = true;
                        ctx.unknown.push(args[argsIndex]);
                        continue;
                    }
                    throw new UnknownOptionError(current, opts.flags);
                }
            }
        } else {
            option = {
                name: current.replace(/^-+/, ""),
                optionalValue: true,
                type: OptionType.STRING
            };
        }
        if (option.standalone) {
            ctx.standalone = option;
        }
        const positiveName = negate ? option.name.replace(/^no-?/, "") : option.name;
        const propName = paramCaseToCamelCase(positiveName);
        if (typeof ctx.flags[propName] !== "undefined") {
            if (!opts.flags?.length) {
                option.collect = true;
            } else if (!option.collect && !ctx.defaults[option.name]) {
                throw new DuplicateOptionError(current);
            }
        }
        if (option.type && !option.args?.length) {
            option.args = [
                {
                    type: option.type,
                    optional: option.optionalValue,
                    variadic: option.variadic,
                    list: option.list,
                    separator: option.separator
                }
            ];
        }
        if (opts.flags?.length && !option.args?.length && typeof currentValue !== "undefined") {
            throw new UnexpectedOptionValueError(option.name, currentValue);
        }
        let optionArgsIndex = 0;
        let inOptionalArg = false;
        const next = ()=>currentValue ?? args[argsIndex + 1];
        const previous = ctx.flags[propName];
        parseNext(option);
        if (typeof ctx.flags[propName] === "undefined") {
            if (option.args?.length && !option.args?.[optionArgsIndex].optional) {
                throw new MissingOptionValueError(option.name);
            } else if (typeof option.default !== "undefined" && (option.type || option.value || option.args?.length)) {
                ctx.flags[propName] = getDefaultValue(option);
            } else {
                setFlagValue(true);
            }
        }
        if (option.value) {
            const value = option.value(ctx.flags[propName], previous);
            setFlagValue(value);
        } else if (option.collect) {
            const value = typeof previous !== "undefined" ? Array.isArray(previous) ? previous : [
                previous
            ] : [];
            value.push(ctx.flags[propName]);
            setFlagValue(value);
        }
        optionsMap.set(propName, option);
        opts.option?.(option, ctx.flags[propName]);
        function parseNext(option) {
            if (negate) {
                setFlagValue(false);
                return;
            } else if (!option.args?.length) {
                setFlagValue(undefined);
                return;
            }
            const arg = option.args[optionArgsIndex];
            if (!arg) {
                const flag = next();
                throw new UnknownOptionError(flag, opts.flags ?? []);
            }
            if (!arg.type) {
                arg.type = OptionType.BOOLEAN;
            }
            if (!option.args?.length && arg.type === OptionType.BOOLEAN && arg.optional === undefined) {
                arg.optional = true;
            }
            if (arg.optional) {
                inOptionalArg = true;
            } else if (inOptionalArg) {
                throw new UnexpectedRequiredArgumentError(option.name);
            }
            let result1;
            let increase = false;
            if (arg.list && hasNext(arg)) {
                const parsed = next().split(arg.separator || ",").map((nextValue)=>{
                    const value = parseValue(option, arg, nextValue);
                    if (typeof value === "undefined") {
                        throw new InvalidOptionValueError(option.name, arg.type ?? "?", nextValue);
                    }
                    return value;
                });
                if (parsed?.length) {
                    result1 = parsed;
                }
            } else {
                if (hasNext(arg)) {
                    result1 = parseValue(option, arg, next());
                } else if (arg.optional && arg.type === OptionType.BOOLEAN) {
                    result1 = true;
                }
            }
            if (increase && typeof currentValue === "undefined") {
                argsIndex++;
                if (!arg.variadic) {
                    optionArgsIndex++;
                } else if (option.args[optionArgsIndex + 1]) {
                    throw new UnexpectedArgumentAfterVariadicArgumentError(next());
                }
            }
            if (typeof result1 !== "undefined" && (option.args.length > 1 || arg.variadic)) {
                if (!ctx.flags[propName]) {
                    setFlagValue([]);
                }
                ctx.flags[propName].push(result1);
                if (hasNext(arg)) {
                    parseNext(option);
                }
            } else {
                setFlagValue(result1);
            }
            function hasNext(arg) {
                if (!option.args?.length) {
                    return false;
                }
                const nextValue = currentValue ?? args[argsIndex + 1];
                if (!nextValue) {
                    return false;
                }
                if (option.args.length > 1 && optionArgsIndex >= option.args.length) {
                    return false;
                }
                if (!arg.optional) {
                    return true;
                }
                if (option.equalsSign && arg.optional && !arg.variadic && typeof currentValue === "undefined") {
                    return false;
                }
                if (arg.optional || arg.variadic) {
                    return nextValue[0] !== "-" || typeof currentValue !== "undefined" || arg.type === OptionType.NUMBER && !isNaN(Number(nextValue));
                }
                return false;
            }
            function parseValue(option, arg, value) {
                const result1 = opts.parse ? opts.parse({
                    label: "Option",
                    type: arg.type || OptionType.STRING,
                    name: `--${option.name}`,
                    value
                }) : parseDefaultType(option, arg, value);
                if (typeof result1 !== "undefined") {
                    increase = true;
                }
                return result1;
            }
        }
        function setFlagValue(value) {
            ctx.flags[propName] = value;
            if (ctx.defaults[propName]) {
                delete ctx.defaults[propName];
            }
        }
    }
    return optionsMap;
}
function parseDottedOptions(ctx) {
    ctx.flags = Object.keys(ctx.flags).reduce((result1, key)=>{
        if (~key.indexOf(".")) {
            key.split(".").reduce((result1, subKey, index, parts)=>{
                if (index === parts.length - 1) {
                    result1[subKey] = ctx.flags[key];
                } else {
                    result1[subKey] = result1[subKey] ?? {};
                }
                return result1[subKey];
            }, result1);
        } else {
            result1[key] = ctx.flags[key];
        }
        return result1;
    }, {});
}
function splitFlags(flag) {
    flag = flag.slice(1);
    const normalized = [];
    const index = flag.indexOf("=");
    const flags = (index !== -1 ? flag.slice(0, index) : flag).split("");
    if (isNaN(Number(flag[flag.length - 1]))) {
        flags.forEach((val)=>normalized.push(`-${val}`));
    } else {
        normalized.push(`-${flags.shift()}`);
        if (flags.length) {
            normalized.push(flags.join(""));
        }
    }
    if (index !== -1) {
        normalized[normalized.length - 1] += flag.slice(index);
    }
    return normalized;
}
function parseDefaultType(option, arg, value) {
    const type = arg.type || OptionType.STRING;
    const parseType = DefaultTypes[type];
    if (!parseType) {
        throw new UnknownTypeError(type, Object.keys(DefaultTypes));
    }
    return parseType({
        label: "Option",
        type,
        name: `--${option.name}`,
        value
    });
}
function didYouMeanCommand(command, commands, excludes = []) {
    const commandNames = commands.map((command)=>command.getName()).filter((command)=>!excludes.includes(command));
    return didYouMean(" Did you mean command", command, commandNames);
}
const ARGUMENT_REGEX = /^[<\[].+[\]>]$/;
const ARGUMENT_DETAILS_REGEX = /[<\[:>\]]/;
function splitArguments(args) {
    const parts = args.trim().split(/[, =] */g);
    const typeParts = [];
    while(parts[parts.length - 1] && ARGUMENT_REGEX.test(parts[parts.length - 1])){
        typeParts.unshift(parts.pop());
    }
    const typeDefinition = typeParts.join(" ");
    return {
        flags: parts,
        typeDefinition,
        equalsSign: args.includes("=")
    };
}
function parseArgumentsDefinition(argsDefinition, validate = true, all) {
    const argumentDetails = [];
    let hasOptional = false;
    let hasVariadic = false;
    const parts = argsDefinition.split(/ +/);
    for (const arg of parts){
        if (validate && hasVariadic) {
            throw new UnexpectedArgumentAfterVariadicArgumentError(arg);
        }
        const parts = arg.split(ARGUMENT_DETAILS_REGEX);
        if (!parts[1]) {
            if (all) {
                argumentDetails.push(parts[0]);
            }
            continue;
        }
        const type = parts[2] || OptionType.STRING;
        const details = {
            optional: arg[0] === "[",
            name: parts[1],
            action: parts[3] || type,
            variadic: false,
            list: type ? arg.indexOf(type + "[]") !== -1 : false,
            type
        };
        if (validate && !details.optional && hasOptional) {
            throw new UnexpectedRequiredArgumentError(details.name);
        }
        if (arg[0] === "[") {
            hasOptional = true;
        }
        if (details.name.length > 3) {
            const istVariadicLeft = details.name.slice(0, 3) === "...";
            const istVariadicRight = details.name.slice(-3) === "...";
            hasVariadic = details.variadic = istVariadicLeft || istVariadicRight;
            if (istVariadicLeft) {
                details.name = details.name.slice(3);
            } else if (istVariadicRight) {
                details.name = details.name.slice(0, -3);
            }
        }
        argumentDetails.push(details);
    }
    return argumentDetails;
}
function dedent(str) {
    const lines = str.split(/\r?\n|\r/g);
    let text = "";
    let indent = 0;
    for (const line of lines){
        if (text || line.trim()) {
            if (!text) {
                text = line.trimStart();
                indent = line.length - text.length;
            } else {
                text += line.slice(indent);
            }
            text += "\n";
        }
    }
    return text.trimEnd();
}
function getDescription(description, __short) {
    return __short ? description.trim().split("\n", 1)[0].trim() : dedent(description);
}
class CommandError extends Error {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, CommandError.prototype);
    }
}
class ValidationError1 extends CommandError {
    exitCode;
    cmd;
    constructor(message, { exitCode } = {}){
        super(message);
        Object.setPrototypeOf(this, ValidationError1.prototype);
        this.exitCode = exitCode ?? 2;
    }
}
class DuplicateOptionNameError extends CommandError {
    constructor(optionName, commandName){
        super(`An option with name '${bold1(getFlag(optionName))}' is already registered on command '${bold1(commandName)}'. If it is intended to override the option, set the '${bold1("override")}' option of the '${bold1("option")}' method to true.`);
        Object.setPrototypeOf(this, DuplicateOptionNameError.prototype);
    }
}
class MissingCommandNameCompletionsError extends CommandError {
    constructor(shell){
        super(`Failed to generate shell completions. Missing main command name. Use '${bold1('cmd.name("<comand-name>")')}' to set the name of the main command or use the '${bold1("--name")}' option from the '${bold1("completions")}' command to set the cli name: '${bold1(`<command> completions ${shell} --name <cli-name>`)}'.`);
        Object.setPrototypeOf(this, MissingCommandNameCompletionsError.prototype);
    }
}
class MissingCommandNameError extends CommandError {
    constructor(){
        super("Missing command name.");
        Object.setPrototypeOf(this, MissingCommandNameError.prototype);
    }
}
class DuplicateCommandNameError extends CommandError {
    constructor(name){
        super(`Duplicate command name "${name}".`);
        Object.setPrototypeOf(this, DuplicateCommandNameError.prototype);
    }
}
class DuplicateCommandAliasError extends CommandError {
    constructor(alias){
        super(`Duplicate command alias "${alias}".`);
        Object.setPrototypeOf(this, DuplicateCommandAliasError.prototype);
    }
}
class CommandNotFoundError extends CommandError {
    constructor(name, commands, excluded){
        super(`Unknown command "${name}".${didYouMeanCommand(name, commands, excluded)}`);
        Object.setPrototypeOf(this, CommandNotFoundError.prototype);
    }
}
class DuplicateTypeError extends CommandError {
    constructor(name){
        super(`Type with name "${name}" already exists.`);
        Object.setPrototypeOf(this, DuplicateTypeError.prototype);
    }
}
class DuplicateCompletionError extends CommandError {
    constructor(name){
        super(`Completion with name "${name}" already exists.`);
        Object.setPrototypeOf(this, DuplicateCompletionError.prototype);
    }
}
class DuplicateExampleError extends CommandError {
    constructor(name){
        super(`Example with name "${name}" already exists.`);
        Object.setPrototypeOf(this, DuplicateExampleError.prototype);
    }
}
class DuplicateEnvVarError extends CommandError {
    constructor(name){
        super(`Environment variable with name "${name}" already exists.`);
        Object.setPrototypeOf(this, DuplicateEnvVarError.prototype);
    }
}
class MissingRequiredEnvVarError extends ValidationError1 {
    constructor(envVar){
        super(`Missing required environment variable "${envVar.names[0]}".`);
        Object.setPrototypeOf(this, MissingRequiredEnvVarError.prototype);
    }
}
class TooManyEnvVarValuesError extends CommandError {
    constructor(name){
        super(`An environment variable can only have one value, but "${name}" has more than one.`);
        Object.setPrototypeOf(this, TooManyEnvVarValuesError.prototype);
    }
}
class UnexpectedOptionalEnvVarValueError extends CommandError {
    constructor(name){
        super(`An environment variable cannot have an optional value, but "${name}" is defined as optional.`);
        Object.setPrototypeOf(this, UnexpectedOptionalEnvVarValueError.prototype);
    }
}
class UnexpectedVariadicEnvVarValueError extends CommandError {
    constructor(name){
        super(`An environment variable cannot have an variadic value, but "${name}" is defined as variadic.`);
        Object.setPrototypeOf(this, UnexpectedVariadicEnvVarValueError.prototype);
    }
}
class DefaultCommandNotFoundError extends CommandError {
    constructor(name, commands){
        super(`Default command "${name}" not found.${didYouMeanCommand(name, commands)}`);
        Object.setPrototypeOf(this, DefaultCommandNotFoundError.prototype);
    }
}
class CommandExecutableNotFoundError extends CommandError {
    constructor(name){
        super(`Command executable not found: ${name}`);
        Object.setPrototypeOf(this, CommandExecutableNotFoundError.prototype);
    }
}
class UnknownCompletionCommandError extends CommandError {
    constructor(name, commands){
        super(`Auto-completion failed. Unknown command "${name}".${didYouMeanCommand(name, commands)}`);
        Object.setPrototypeOf(this, UnknownCompletionCommandError.prototype);
    }
}
class UnknownCommandError extends ValidationError1 {
    constructor(name, commands, excluded){
        super(`Unknown command "${name}".${didYouMeanCommand(name, commands, excluded)}`);
        Object.setPrototypeOf(this, UnknownCommandError.prototype);
    }
}
class NoArgumentsAllowedError extends ValidationError1 {
    constructor(name){
        super(`No arguments allowed for command "${name}".`);
        Object.setPrototypeOf(this, NoArgumentsAllowedError.prototype);
    }
}
class MissingArgumentsError extends ValidationError1 {
    constructor(names){
        super(`Missing argument(s): ${names.join(", ")}`);
        Object.setPrototypeOf(this, MissingArgumentsError.prototype);
    }
}
class MissingArgumentError extends ValidationError1 {
    constructor(name){
        super(`Missing argument: ${name}`);
        Object.setPrototypeOf(this, MissingArgumentError.prototype);
    }
}
class TooManyArgumentsError extends ValidationError1 {
    constructor(args){
        super(`Too many arguments: ${args.join(" ")}`);
        Object.setPrototypeOf(this, TooManyArgumentsError.prototype);
    }
}
class Type {
}
class BooleanType extends Type {
    parse(type) {
        return __boolean(type);
    }
    complete() {
        return [
            "true",
            "false"
        ];
    }
}
class StringType extends Type {
    parse(type) {
        return string(type);
    }
}
class FileType extends StringType {
    constructor(){
        super();
    }
}
class IntegerType extends Type {
    parse(type) {
        return integer(type);
    }
}
class NumberType extends Type {
    parse(type) {
        return number(type);
    }
}
const border = {
    top: "─",
    topMid: "┬",
    topLeft: "┌",
    topRight: "┐",
    bottom: "─",
    bottomMid: "┴",
    bottomLeft: "└",
    bottomRight: "┘",
    left: "│",
    leftMid: "├",
    mid: "─",
    midMid: "┼",
    right: "│",
    rightMid: "┤",
    middle: "│"
};
class Cell {
    value;
    options;
    get length() {
        return this.toString().length;
    }
    static from(value) {
        let cell;
        if (value instanceof Cell) {
            cell = new this(value.getValue());
            cell.options = {
                ...value.options
            };
        } else {
            cell = new this(value);
        }
        return cell;
    }
    constructor(value){
        this.value = value;
        this.options = {};
    }
    toString() {
        return this.value.toString();
    }
    getValue() {
        return this.value;
    }
    setValue(value) {
        this.value = value;
        return this;
    }
    clone(value) {
        return Cell.from(value ?? this);
    }
    border(enable = true, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    colSpan(span, override = true) {
        if (override || typeof this.options.colSpan === "undefined") {
            this.options.colSpan = span;
        }
        return this;
    }
    rowSpan(span, override = true) {
        if (override || typeof this.options.rowSpan === "undefined") {
            this.options.rowSpan = span;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    getBorder() {
        return this.options.border === true;
    }
    getColSpan() {
        return typeof this.options.colSpan === "number" && this.options.colSpan > 0 ? this.options.colSpan : 1;
    }
    getRowSpan() {
        return typeof this.options.rowSpan === "number" && this.options.rowSpan > 0 ? this.options.rowSpan : 1;
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
class Column {
    static from(options) {
        const opts = options instanceof Column ? options.opts : options;
        return new Column().options(opts);
    }
    opts = {};
    options(options) {
        Object.assign(this.opts, options);
        return this;
    }
    minWidth(width) {
        this.opts.minWidth = width;
        return this;
    }
    maxWidth(width) {
        this.opts.maxWidth = width;
        return this;
    }
    border(border = true) {
        this.opts.border = border;
        return this;
    }
    padding(padding) {
        this.opts.padding = padding;
        return this;
    }
    align(direction) {
        this.opts.align = direction;
        return this;
    }
    getMinWidth() {
        return this.opts.minWidth;
    }
    getMaxWidth() {
        return this.opts.maxWidth;
    }
    getBorder() {
        return this.opts.border;
    }
    getPadding() {
        return this.opts.padding;
    }
    getAlign() {
        return this.opts.align;
    }
}
const __default = JSON.parse("{\n  \"UNICODE_VERSION\": \"15.0.0\",\n  \"tables\": [\n    {\n      \"d\": \"AAECAwQFBgcICQoLDA0OAw8DDwkQCRESERIA\",\n      \"r\": \"AQEBAgEBAQEBAQEBAQEBBwEHAVABBwcBBwF4\"\n    },\n    {\n      \"d\": \"AAECAwQFBgcGCAYJCgsMDQ4PEAYREhMUBhUWFxgZGhscHR4fICEiIyIkJSYnKCkqJSssLS4vMDEyMzQ1Njc4OToGOzwKBj0GPj9AQUIGQwZEBkVGR0hJSktMTQZOBgoGT1BRUlNUVVZXWFkGWgZbBlxdXl1fYGFiY2RlZmdoBmlqBmsGAQZsBm1uO29wcXI7czt0dXZ3OwY7eHkGent8Bn0Gfn+AgYKDhIWGBoc7iAZdO4kGiosGAXGMBo0GjgaPBpAGkQaSBpMGlJUGlpcGmJmam5ydnp+gLgahLKIGo6SlpganqKmqqwasBq0Grq8GsLGyswa0BrUGtre4Brm6uwZHvAa9vga/wME7wjvDxAbFO8bHO8gGyQbKywbMzQbOBs/Q0QbSBr8GvgbT1AbUBtUG1gbXBtjZ2tsG3N0G3t/g4eLjO+Tl5ufoO+k76gbrBuztOwbu7/AGO+XxCgYKCwZd8g==\",\n      \"r\": \"AQEBAQEBAQEBAQEBAQEBAQEBAQMBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQECBQEOAQEBAQEBAQEBAwEBAQEBAQEBAQIBAwEIAQEBAQEBAQEBAQEBAQIBAQEBAQEBAQEBAQEBAQEBDQEBBQEBAQEBAgEBAwEBAQEBAQEBAQEBbQHaAQEFAQEBBAECAQEBAQEBAQEBAwGuASFkCAELAQEBAQEBAQEHAQMBAQEaAQIBCAEFAQEBAQEBAQEBAQEBAQEBAQEBAQECAQEBAQIBAQEBAQEBAwEDAQEBAQEBAQUBAQEBAQEBBAEBAVIBAdkBARABAQFfARMBAYoBBAEBBQEmAUkBAQcBAQIBHgEBARUBAQEBAQUBAQcBDwEBARoBAgEBAQEBAQECAQEBAQEBAQEBAQEBAQEBAQMBBAEBAgEBAQEUfwEBAQIDAXj/AQ==\"\n    },\n    {\n      \"d\": \"AFUVAF3Xd3X/93//VXVVV9VX9V91f1/31X93XVXdVdVV9dVV/VVX1X9X/131VfXVVXV3V1VdVV1V1/1dV1X/3VUAVf3/3/9fVf3/3/9fVV1V/11VFQBQVQEAEEEQVQBQVQBAVFUVAFVUVQUAEAAUBFBVFVFVAEBVBQBUVRUAVVFVBRAAAVBVAVVQVQBVBQBAVUVUAQBUUQEAVQVVUVVUAVRVUVUFVUVBVVRBFRRQUVVQUVUBEFRRVQVVBQBRVRQBVFVRVUFVBVVFVVRVUVVUVQRUBQRQVUFVBVVFVVBVBVVQVRVUAVRVUVUFVVFVRVUFRFVRAEBVFQBAVVEAVFUAQFVQVRFRVQEAQAAEVQEAAQBUVUVVAQQAQVVQBVRVAVRVRUFVUVVRVaoAVQFVBVRVBVUFVQVVEABQVUUBAFVRVRUAVUFVUVVAFVRVRVUBVRUUVUUAQEQBAFQVABRVAEBVAFUEQFRFVRUAVVBVBVAQUFVFUBFQVQAFVUAABABUUVVUUFUVANd/X3//BUD3XdV1VQAEAFVXVdX9V1VXVQBUVdVdVdV1VX111VXVV9V//1X/X1VdVf9fVV9VdVdV1VX31dfVXXX9193/d1X/VV9VV3VVX//1VfVVXVVdVdVVdVWlVWlVqVaWVf/f/1X/Vf/1X1Xf/19V9VVf9df1X1X1X1XVVWlVfV31VVpVd1V3VapV33/fVZVVlVX1WVWlVelV+v/v//7/31Xv/6/77/tVWaVVVlVdVWaVmlX1/1WpVVZVlVWVVlVW+V9VFVBVAKqaqlWqWlWqVaoKoKpqqapqgapVqaqpqmqqVapqqv+qVqpqVRVAAFBVBVVQVUUVVUFVVFVQVQBQVRVVBQBQVRUAUFWqVkBVFQVQVVFVAUBBVRVVVFVUVQQUVAVRVVBVRVVRVFFVqlVFVQCqWlUAqmqqaqpVqlZVqmpVAV1VUVVUVQVAVQFBVQBVQBVVQVUAVRVUVQFVBQBUVQVQVVFVAEBVFFRVFVBVFUBBUUVVUVVAVRUAAQBUVRVVUFUFAEBVARRVFVAEVUVVFQBAVVRVBQBUAFRVAAVEVUVVFQBEFQRVBVBVEFRVUFUVAEARVFUVUQAQVQEFEABVFQBBVRVEFVUABVVUVQEAQFUVABRAVRVVAUABVQUAQFBVAEAAEFUFAAUABEFVAUBFEAAQVVARVRVUVVBVBUBVRFVUFQBQVQBUVQBAVRVVFUBVqlRVWlWqVapaVapWVaqpqmmqalVlVWpZVapVqlVBAFUAUABAVRVQVRUAQAEAVQVQVQVUVQBAFQBUVVFVVFUVAAEAVQBAABQAEARAVUVVAFUAQFUAQFVWVZVV/39V/1//X1X/76uq6v9XVWpVqlWqVlVaVapaVapWVamqmqqmqlWqapWqVapWqmqmqpaqWlWVaqpVZVVpVVZVlapVqlpVVmqpVapVlVZVqlZVqlVWVapqqpqqVapWqlZVqpqqWlWlqlWqVlWqVlVRVQD/Xw==\",\n      \"r\": \"CBcBCAEBAQEBAQEBAQECAQEBAQEBAQEBAQEBAQMBAQECAQEBAQEBAQEBAQEBBAEBGAEDAQwBAwEIAQEBAQEBAQgcCAEDAQEBAQEDAQEBDQEDEAELAQEBEQEKAQEBDgEBAgIBAQoBBQQBCAEBAQEBAQEHAQEHBgEWAQIBDQECAgEFAQECAgEKAQ0BAQIKAQ0BDQEBAQEBAQEBAgEHAQ4BAQEBAQQBBgEBDgEBAQEBAQcBAQIBAQEBBAEFAQEBDgEBAQEBAQECAQcBDwECAQwCDQEBAQEBAQECAQgBAQEEAQcBDQEBAQEBAQQBBwERAQEBARYBAQECAQEBGAECAQIBARIBBgEBDQECAQEBAQECAQgBAQEZAQEBAgYBAQEDAQECAQEBAQMBCBgIBwEMAQEGAQcBBwEQAQEBAQEBAgIBCgEBDQEIAQ0BAQEBAQEBBgEBDgEBAQEBAQEBAgEMBwEMAQwBAQEBCQECAwEHAQEBAQ0BAQEBDgIBBgEDAQEBAQEBAQMBAQEBAgEBAQEBAQEBCAEBAgEBAQEBAQkBCAgBAwECAQEBAgEBAQkBAQEBAwECAQMBAQIBBwEFAQEDAQYBAQEBAgEBAQEBAQEBAQECAgEDAQECBAIDAgIBBQEEAQEBAwEPAQEBCyIBCAEJAwQBAQIBAQEBAgECAQEBAQMBAQEBAwEBAQEBAQEBAQgBAQMDAgEBAwEEAQIBAQEBBAEBAQEBAQECAQEBAQEBAQEBAQEHAQQBAwEBAQcBAgUBBgECAQYBAQwBAQEUAQELCAYBFgMFAQYDAQoBAQMBARQBAQkBAQoBBgEVAwsBCgIPAQ0BGQEBAgEHARQBAwIBBgEBAQUBBgQBAgEJAQEBBQECAQMHAQELAQECCQEQAQECAgECAQsBDAEBAQEBCgEBAQsBAQEECQ4BCAQCAQEECAEEAQEFCAEPAQEEAQEPAQgBFAEBAQEBAQEKAQEJAQ8BEAEBEwEBAQIBCwEBDgENAwEKAQEBAQELAQEBAQECAQwBCAEBAQEBDgEDAQwBAQECAQEXAQEBAQEHAgEBBQEIAQEBAQEQAgEBBQEUAQEBAQEbAQEBAQEGARQBAQEBARkBAQEBCQEBAQEQAQIBDwEBARQBAQEBBwEBAQkBAQEBAQECAQEBCwECAQEVAQEBAQQBBQEBAQEOAQEBAQEBEgEBFgEBAgEMAQEBAQ8BAQMBFgEBDgEBBQEPAQETAQECAQMOAgUBCgIBGQEBAQEIAQMBBwEBAwECEwgBAQcLAQUBFwEBAQEDAQEBBwEBBAEBDg0BAQwBAQEDAQQBAQEDBAEBBAEBAQEBEAEPAQgBAQsBAQ4BEQEMAgEBBwEOAQEHAQEBAQQBBAEDCwECAQEBAwEBBggBAgEBAREBBQMKAQEBAwQCEQEBHgEPAQIBAQYEAQYBAwEUAQUMAQEBAQEBAQECAQEBAgEIAwEBBgsBAgEODAMBAgEBCwEBAQEBAwECAQECAQEBBwgPAQ==\"\n    }\n  ]\n}");
function runLengthDecode({ d, r }) {
    const data = atob(d);
    const runLengths = atob(r);
    let out = "";
    for (const [i1, ch] of [
        ...runLengths
    ].entries()){
        out += data[i1].repeat(ch.codePointAt(0));
    }
    return Uint8Array.from([
        ...out
    ].map((x)=>x.codePointAt(0)));
}
let tables = null;
function lookupWidth(cp) {
    if (!tables) tables = __default.tables.map(runLengthDecode);
    const t1Offset = tables[0][cp >> 13 & 0xff];
    const t2Offset = tables[1][128 * t1Offset + (cp >> 6 & 0x7f)];
    const packedWidths = tables[2][16 * t2Offset + (cp >> 2 & 0xf)];
    const width = packedWidths >> 2 * (cp & 0b11) & 0b11;
    return width === 3 ? 1 : width;
}
const cache = new Map();
function charWidth(ch) {
    if (cache.has(ch)) return cache.get(ch);
    const cp = ch.codePointAt(0);
    let v = null;
    if (cp < 0x7f) {
        v = cp >= 0x20 ? 1 : cp === 0 ? 0 : null;
    } else if (cp >= 0xa0) {
        v = lookupWidth(cp);
    } else {
        v = null;
    }
    cache.set(ch, v);
    return v;
}
function unicodeWidth(str) {
    return [
        ...str
    ].map((ch)=>charWidth(ch) ?? 0).reduce((a, b)=>a + b, 0);
}
function unicodeWidth1(str) {
    return unicodeWidth(str);
}
const strLength = (str)=>{
    return unicodeWidth1(stripColor(str));
};
function consumeWords(length, content) {
    let consumed = "";
    const words = content.split("\n")[0]?.split(/ /g);
    for(let i1 = 0; i1 < words.length; i1++){
        const word = words[i1];
        if (consumed) {
            const nextLength = strLength(word);
            const consumedLength = strLength(consumed);
            if (consumedLength + nextLength >= length) {
                break;
            }
        }
        consumed += (i1 > 0 ? " " : "") + word;
    }
    return consumed;
}
function longest(index, rows, maxWidth) {
    const cellLengths = rows.map((row)=>{
        const cell = row[index];
        const cellValue = cell instanceof Cell && cell.getColSpan() > 1 ? "" : cell?.toString() || "";
        return cellValue.split("\n").map((line)=>{
            const str = typeof maxWidth === "undefined" ? line : consumeWords(maxWidth, line);
            return strLength(str) || 0;
        });
    }).flat();
    return Math.max(...cellLengths);
}
class Row extends Array {
    options = {};
    static from(cells) {
        const row = new this(...cells);
        if (cells instanceof Row) {
            row.options = {
                ...cells.options
            };
        }
        return row;
    }
    clone() {
        const row = new Row(...this.map((cell)=>cell instanceof Cell ? cell.clone() : cell));
        row.options = {
            ...this.options
        };
        return row;
    }
    border(enable = true, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    getBorder() {
        return this.options.border === true;
    }
    hasBorder() {
        return this.getBorder() || this.some((cell)=>cell instanceof Cell && cell.getBorder());
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
class TableLayout {
    table;
    options;
    constructor(table, options){
        this.table = table;
        this.options = options;
    }
    toString() {
        const opts = this.createLayout();
        return opts.rows.length ? this.renderRows(opts) : "";
    }
    createLayout() {
        Object.keys(this.options.chars).forEach((key)=>{
            if (typeof this.options.chars[key] !== "string") {
                this.options.chars[key] = "";
            }
        });
        const hasBodyBorder = this.table.getBorder() || this.table.hasBodyBorder();
        const hasHeaderBorder = this.table.hasHeaderBorder();
        const hasBorder = hasHeaderBorder || hasBodyBorder;
        const rows = this.#getRows();
        const columns = Math.max(...rows.map((row)=>row.length));
        for(let rowIndex = 0; rowIndex < rows.length; rowIndex++){
            const row = rows[rowIndex];
            const length = row.length;
            if (length < columns) {
                const diff = columns - length;
                for(let i1 = 0; i1 < diff; i1++){
                    row.push(this.createCell(null, row, rowIndex, length + i1));
                }
            }
        }
        const padding = [];
        const width = [];
        for(let colIndex = 0; colIndex < columns; colIndex++){
            const column = this.options.columns.at(colIndex);
            const minColWidth = column?.getMinWidth() ?? (Array.isArray(this.options.minColWidth) ? this.options.minColWidth[colIndex] : this.options.minColWidth);
            const maxColWidth = column?.getMaxWidth() ?? (Array.isArray(this.options.maxColWidth) ? this.options.maxColWidth[colIndex] : this.options.maxColWidth);
            const colWidth = longest(colIndex, rows, maxColWidth);
            width[colIndex] = Math.min(maxColWidth, Math.max(minColWidth, colWidth));
            padding[colIndex] = column?.getPadding() ?? (Array.isArray(this.options.padding) ? this.options.padding[colIndex] : this.options.padding);
        }
        return {
            padding,
            width,
            rows,
            columns,
            hasBorder,
            hasBodyBorder,
            hasHeaderBorder
        };
    }
    #getRows() {
        const header = this.table.getHeader();
        const rows = header ? [
            header,
            ...this.table
        ] : this.table.slice();
        const hasSpan = rows.some((row)=>row.some((cell)=>cell instanceof Cell && (cell.getColSpan() > 1 || cell.getRowSpan() > 1)));
        if (hasSpan) {
            return this.spanRows(rows);
        }
        return rows.map((row, rowIndex)=>{
            const newRow = this.createRow(row);
            for(let colIndex = 0; colIndex < row.length; colIndex++){
                newRow[colIndex] = this.createCell(row[colIndex], newRow, rowIndex, colIndex);
            }
            return newRow;
        });
    }
    spanRows(rows) {
        const rowSpan = [];
        let colSpan = 1;
        let rowIndex = -1;
        while(true){
            rowIndex++;
            if (rowIndex === rows.length && rowSpan.every((span)=>span === 1)) {
                break;
            }
            const row = rows[rowIndex] = this.createRow(rows[rowIndex] || []);
            let colIndex = -1;
            while(true){
                colIndex++;
                if (colIndex === row.length && colIndex === rowSpan.length && colSpan === 1) {
                    break;
                }
                if (colSpan > 1) {
                    colSpan--;
                    rowSpan[colIndex] = rowSpan[colIndex - 1];
                    row.splice(colIndex, this.getDeleteCount(rows, rowIndex, colIndex), row[colIndex - 1]);
                    continue;
                }
                if (rowSpan[colIndex] > 1) {
                    rowSpan[colIndex]--;
                    rows[rowIndex].splice(colIndex, this.getDeleteCount(rows, rowIndex, colIndex), rows[rowIndex - 1][colIndex]);
                    continue;
                }
                const cell = row[colIndex] = this.createCell(row[colIndex] || null, row, rowIndex, colIndex);
                colSpan = cell.getColSpan();
                rowSpan[colIndex] = cell.getRowSpan();
            }
        }
        return rows;
    }
    getDeleteCount(rows, rowIndex, colIndex) {
        return colIndex <= rows[rowIndex].length - 1 && typeof rows[rowIndex][colIndex] === "undefined" ? 1 : 0;
    }
    createRow(row) {
        return Row.from(row).border(this.table.getBorder(), false).align(this.table.getAlign(), false);
    }
    createCell(cell, row, rowIndex, colIndex) {
        const column = this.options.columns.at(colIndex);
        const isHeaderRow = this.isHeaderRow(rowIndex);
        return Cell.from(cell ?? "").border((isHeaderRow ? null : column?.getBorder()) ?? row.getBorder(), false).align((isHeaderRow ? null : column?.getAlign()) ?? row.getAlign(), false);
    }
    isHeaderRow(rowIndex) {
        return rowIndex === 0 && this.table.getHeader() !== undefined;
    }
    renderRows(opts) {
        let result1 = "";
        const rowSpan = new Array(opts.columns).fill(1);
        for(let rowIndex = 0; rowIndex < opts.rows.length; rowIndex++){
            result1 += this.renderRow(rowSpan, rowIndex, opts);
        }
        return result1.slice(0, -1);
    }
    renderRow(rowSpan, rowIndex, opts, isMultiline) {
        const row = opts.rows[rowIndex];
        const prevRow = opts.rows[rowIndex - 1];
        const nextRow = opts.rows[rowIndex + 1];
        let result1 = "";
        let colSpan = 1;
        if (!isMultiline && rowIndex === 0 && row.hasBorder()) {
            result1 += this.renderBorderRow(undefined, row, rowSpan, opts);
        }
        let isMultilineRow = false;
        result1 += " ".repeat(this.options.indent || 0);
        for(let colIndex = 0; colIndex < opts.columns; colIndex++){
            if (colSpan > 1) {
                colSpan--;
                rowSpan[colIndex] = rowSpan[colIndex - 1];
                continue;
            }
            result1 += this.renderCell(colIndex, row, opts);
            if (rowSpan[colIndex] > 1) {
                if (!isMultiline) {
                    rowSpan[colIndex]--;
                }
            } else if (!prevRow || prevRow[colIndex] !== row[colIndex]) {
                rowSpan[colIndex] = row[colIndex].getRowSpan();
            }
            colSpan = row[colIndex].getColSpan();
            if (rowSpan[colIndex] === 1 && row[colIndex].length) {
                isMultilineRow = true;
            }
        }
        if (opts.columns > 0) {
            if (row[opts.columns - 1].getBorder()) {
                result1 += this.options.chars.right;
            } else if (opts.hasBorder) {
                result1 += " ";
            }
        }
        result1 += "\n";
        if (isMultilineRow) {
            return result1 + this.renderRow(rowSpan, rowIndex, opts, isMultilineRow);
        }
        if (opts.rows.length > 1 && (rowIndex === 0 && opts.hasHeaderBorder || rowIndex < opts.rows.length - 1 && opts.hasBodyBorder)) {
            result1 += this.renderBorderRow(row, nextRow, rowSpan, opts);
        }
        if (rowIndex === opts.rows.length - 1 && row.hasBorder()) {
            result1 += this.renderBorderRow(row, undefined, rowSpan, opts);
        }
        return result1;
    }
    renderCell(colIndex, row, opts, noBorder) {
        let result1 = "";
        const prevCell = row[colIndex - 1];
        const cell = row[colIndex];
        if (!noBorder) {
            if (colIndex === 0) {
                if (cell.getBorder()) {
                    result1 += this.options.chars.left;
                } else if (opts.hasBorder) {
                    result1 += " ";
                }
            } else {
                if (cell.getBorder() || prevCell?.getBorder()) {
                    result1 += this.options.chars.middle;
                } else if (opts.hasBorder) {
                    result1 += " ";
                }
            }
        }
        let maxLength = opts.width[colIndex];
        const colSpan = cell.getColSpan();
        if (colSpan > 1) {
            for(let o = 1; o < colSpan; o++){
                maxLength += opts.width[colIndex + o] + opts.padding[colIndex + o];
                if (opts.hasBorder) {
                    maxLength += opts.padding[colIndex + o] + 1;
                }
            }
        }
        const { current, next } = this.renderCellValue(cell, maxLength);
        row[colIndex].setValue(next.getValue());
        if (opts.hasBorder) {
            result1 += " ".repeat(opts.padding[colIndex]);
        }
        result1 += current;
        if (opts.hasBorder || colIndex < opts.columns - 1) {
            result1 += " ".repeat(opts.padding[colIndex]);
        }
        return result1;
    }
    renderCellValue(cell, maxLength) {
        const length = Math.min(maxLength, strLength(cell.toString()));
        let words = consumeWords(length, cell.toString());
        const breakWord = strLength(words) > length;
        if (breakWord) {
            words = words.slice(0, length);
        }
        const next = cell.toString().slice(words.length + (breakWord ? 0 : 1));
        const fillLength = maxLength - strLength(words);
        const align = cell.getAlign();
        let current;
        if (fillLength === 0) {
            current = words;
        } else if (align === "left") {
            current = words + " ".repeat(fillLength);
        } else if (align === "center") {
            current = " ".repeat(Math.floor(fillLength / 2)) + words + " ".repeat(Math.ceil(fillLength / 2));
        } else if (align === "right") {
            current = " ".repeat(fillLength) + words;
        } else {
            throw new Error("Unknown direction: " + align);
        }
        return {
            current,
            next: cell.clone(next)
        };
    }
    renderBorderRow(prevRow, nextRow, rowSpan, opts) {
        let result1 = "";
        let colSpan = 1;
        for(let colIndex = 0; colIndex < opts.columns; colIndex++){
            if (rowSpan[colIndex] > 1) {
                if (!nextRow) {
                    throw new Error("invalid layout");
                }
                if (colSpan > 1) {
                    colSpan--;
                    continue;
                }
            }
            result1 += this.renderBorderCell(colIndex, prevRow, nextRow, rowSpan, opts);
            colSpan = nextRow?.[colIndex].getColSpan() ?? 1;
        }
        return result1.length ? " ".repeat(this.options.indent) + result1 + "\n" : "";
    }
    renderBorderCell(colIndex, prevRow, nextRow, rowSpan, opts) {
        const a1 = prevRow?.[colIndex - 1];
        const a2 = nextRow?.[colIndex - 1];
        const b1 = prevRow?.[colIndex];
        const b2 = nextRow?.[colIndex];
        const a1Border = !!a1?.getBorder();
        const a2Border = !!a2?.getBorder();
        const b1Border = !!b1?.getBorder();
        const b2Border = !!b2?.getBorder();
        const hasColSpan = (cell)=>(cell?.getColSpan() ?? 1) > 1;
        const hasRowSpan = (cell)=>(cell?.getRowSpan() ?? 1) > 1;
        let result1 = "";
        if (colIndex === 0) {
            if (rowSpan[colIndex] > 1) {
                if (b1Border) {
                    result1 += this.options.chars.left;
                } else {
                    result1 += " ";
                }
            } else if (b1Border && b2Border) {
                result1 += this.options.chars.leftMid;
            } else if (b1Border) {
                result1 += this.options.chars.bottomLeft;
            } else if (b2Border) {
                result1 += this.options.chars.topLeft;
            } else {
                result1 += " ";
            }
        } else if (colIndex < opts.columns) {
            if (a1Border && b2Border || b1Border && a2Border) {
                const a1ColSpan = hasColSpan(a1);
                const a2ColSpan = hasColSpan(a2);
                const b1ColSpan = hasColSpan(b1);
                const b2ColSpan = hasColSpan(b2);
                const a1RowSpan = hasRowSpan(a1);
                const a2RowSpan = hasRowSpan(a2);
                const b1RowSpan = hasRowSpan(b1);
                const b2RowSpan = hasRowSpan(b2);
                const hasAllBorder = a1Border && b2Border && b1Border && a2Border;
                const hasAllRowSpan = a1RowSpan && b1RowSpan && a2RowSpan && b2RowSpan;
                const hasAllColSpan = a1ColSpan && b1ColSpan && a2ColSpan && b2ColSpan;
                if (hasAllRowSpan && hasAllBorder) {
                    result1 += this.options.chars.middle;
                } else if (hasAllColSpan && hasAllBorder && a1 === b1 && a2 === b2) {
                    result1 += this.options.chars.mid;
                } else if (a1ColSpan && b1ColSpan && a1 === b1) {
                    result1 += this.options.chars.topMid;
                } else if (a2ColSpan && b2ColSpan && a2 === b2) {
                    result1 += this.options.chars.bottomMid;
                } else if (a1RowSpan && a2RowSpan && a1 === a2) {
                    result1 += this.options.chars.leftMid;
                } else if (b1RowSpan && b2RowSpan && b1 === b2) {
                    result1 += this.options.chars.rightMid;
                } else {
                    result1 += this.options.chars.midMid;
                }
            } else if (a1Border && b1Border) {
                if (hasColSpan(a1) && hasColSpan(b1) && a1 === b1) {
                    result1 += this.options.chars.bottom;
                } else {
                    result1 += this.options.chars.bottomMid;
                }
            } else if (b1Border && b2Border) {
                if (rowSpan[colIndex] > 1) {
                    result1 += this.options.chars.left;
                } else {
                    result1 += this.options.chars.leftMid;
                }
            } else if (b2Border && a2Border) {
                if (hasColSpan(a2) && hasColSpan(b2) && a2 === b2) {
                    result1 += this.options.chars.top;
                } else {
                    result1 += this.options.chars.topMid;
                }
            } else if (a1Border && a2Border) {
                if (hasRowSpan(a1) && a1 === a2) {
                    result1 += this.options.chars.right;
                } else {
                    result1 += this.options.chars.rightMid;
                }
            } else if (a1Border) {
                result1 += this.options.chars.bottomRight;
            } else if (b1Border) {
                result1 += this.options.chars.bottomLeft;
            } else if (a2Border) {
                result1 += this.options.chars.topRight;
            } else if (b2Border) {
                result1 += this.options.chars.topLeft;
            } else {
                result1 += " ";
            }
        }
        const length = opts.padding[colIndex] + opts.width[colIndex] + opts.padding[colIndex];
        if (rowSpan[colIndex] > 1 && nextRow) {
            result1 += this.renderCell(colIndex, nextRow, opts, true);
            if (nextRow[colIndex] === nextRow[nextRow.length - 1]) {
                if (b1Border) {
                    result1 += this.options.chars.right;
                } else {
                    result1 += " ";
                }
                return result1;
            }
        } else if (b1Border && b2Border) {
            result1 += this.options.chars.mid.repeat(length);
        } else if (b1Border) {
            result1 += this.options.chars.bottom.repeat(length);
        } else if (b2Border) {
            result1 += this.options.chars.top.repeat(length);
        } else {
            result1 += " ".repeat(length);
        }
        if (colIndex === opts.columns - 1) {
            if (b1Border && b2Border) {
                result1 += this.options.chars.rightMid;
            } else if (b1Border) {
                result1 += this.options.chars.bottomRight;
            } else if (b2Border) {
                result1 += this.options.chars.topRight;
            } else {
                result1 += " ";
            }
        }
        return result1;
    }
}
class Table extends Array {
    static _chars = {
        ...border
    };
    options = {
        indent: 0,
        border: false,
        maxColWidth: Infinity,
        minColWidth: 0,
        padding: 1,
        chars: {
            ...Table._chars
        },
        columns: []
    };
    headerRow;
    static from(rows) {
        const table = new this(...rows);
        if (rows instanceof Table) {
            table.options = {
                ...rows.options
            };
            table.headerRow = rows.headerRow ? Row.from(rows.headerRow) : undefined;
        }
        return table;
    }
    static fromJson(rows) {
        return new this().fromJson(rows);
    }
    static chars(chars) {
        Object.assign(this._chars, chars);
        return this;
    }
    static render(rows) {
        Table.from(rows).render();
    }
    fromJson(rows) {
        this.header(Object.keys(rows[0]));
        this.body(rows.map((row)=>Object.values(row)));
        return this;
    }
    columns(columns) {
        this.options.columns = columns.map((column)=>column instanceof Column ? column : Column.from(column));
        return this;
    }
    column(index, column) {
        if (column instanceof Column) {
            this.options.columns[index] = column;
        } else if (this.options.columns[index]) {
            this.options.columns[index].options(column);
        } else {
            this.options.columns[index] = Column.from(column);
        }
        return this;
    }
    header(header) {
        this.headerRow = header instanceof Row ? header : Row.from(header);
        return this;
    }
    body(rows) {
        this.length = 0;
        this.push(...rows);
        return this;
    }
    clone() {
        const table = new Table(...this.map((row)=>row instanceof Row ? row.clone() : Row.from(row).clone()));
        table.options = {
            ...this.options
        };
        table.headerRow = this.headerRow?.clone();
        return table;
    }
    toString() {
        return new TableLayout(this, this.options).toString();
    }
    render() {
        console.log(this.toString());
        return this;
    }
    maxColWidth(width, override = true) {
        if (override || typeof this.options.maxColWidth === "undefined") {
            this.options.maxColWidth = width;
        }
        return this;
    }
    minColWidth(width, override = true) {
        if (override || typeof this.options.minColWidth === "undefined") {
            this.options.minColWidth = width;
        }
        return this;
    }
    indent(width, override = true) {
        if (override || typeof this.options.indent === "undefined") {
            this.options.indent = width;
        }
        return this;
    }
    padding(padding, override = true) {
        if (override || typeof this.options.padding === "undefined") {
            this.options.padding = padding;
        }
        return this;
    }
    border(enable = true, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    chars(chars) {
        Object.assign(this.options.chars, chars);
        return this;
    }
    getHeader() {
        return this.headerRow;
    }
    getBody() {
        return [
            ...this
        ];
    }
    getMaxColWidth() {
        return this.options.maxColWidth;
    }
    getMinColWidth() {
        return this.options.minColWidth;
    }
    getIndent() {
        return this.options.indent;
    }
    getPadding() {
        return this.options.padding;
    }
    getBorder() {
        return this.options.border === true;
    }
    hasHeaderBorder() {
        const hasBorder = this.headerRow?.hasBorder();
        return hasBorder === true || this.getBorder() && hasBorder !== false;
    }
    hasBodyBorder() {
        return this.getBorder() || this.options.columns.some((column)=>column.getBorder()) || this.some((row)=>row instanceof Row ? row.hasBorder() : row.some((cell)=>cell instanceof Cell ? cell.getBorder() : false));
    }
    hasBorder() {
        return this.hasHeaderBorder() || this.hasBodyBorder();
    }
    getAlign() {
        return this.options.align ?? "left";
    }
    getColumns() {
        return this.options.columns;
    }
    getColumn(index) {
        return this.options.columns[index] ??= new Column();
    }
}
class HelpGenerator {
    cmd;
    indent;
    options;
    static generate(cmd, options) {
        return new HelpGenerator(cmd, options).generate();
    }
    constructor(cmd, options = {}){
        this.cmd = cmd;
        this.indent = 2;
        this.options = {
            types: false,
            hints: true,
            colors: true,
            long: false,
            ...options
        };
    }
    generate() {
        const areColorsEnabled = getColorEnabled();
        setColorEnabled(this.options.colors);
        const result1 = this.generateHeader() + this.generateMeta() + this.generateDescription() + this.generateOptions() + this.generateCommands() + this.generateEnvironmentVariables() + this.generateExamples();
        setColorEnabled(areColorsEnabled);
        return result1;
    }
    generateHeader() {
        const usage = this.cmd.getUsage();
        const rows = [
            [
                bold1("Usage:"),
                brightMagenta(this.cmd.getPath() + (usage ? " " + highlightArguments(usage, this.options.types) : ""))
            ]
        ];
        const version = this.cmd.getVersion();
        if (version) {
            rows.push([
                bold1("Version:"),
                yellow1(`${this.cmd.getVersion()}`)
            ]);
        }
        return "\n" + Table.from(rows).padding(1).toString() + "\n";
    }
    generateMeta() {
        const meta = Object.entries(this.cmd.getMeta());
        if (!meta.length) {
            return "";
        }
        const rows = [];
        for (const [name, value] of meta){
            rows.push([
                bold1(`${name}: `) + value
            ]);
        }
        return "\n" + Table.from(rows).padding(1).toString() + "\n";
    }
    generateDescription() {
        if (!this.cmd.getDescription()) {
            return "";
        }
        return this.label("Description") + Table.from([
            [
                dedent(this.cmd.getDescription())
            ]
        ]).indent(this.indent).maxColWidth(140).padding(1).toString() + "\n";
    }
    generateOptions() {
        const options = this.cmd.getOptions(false);
        if (!options.length) {
            return "";
        }
        let groups = [];
        const hasGroups = options.some((option)=>option.groupName);
        if (hasGroups) {
            for (const option of options){
                let group = groups.find((group)=>group.name === option.groupName);
                if (!group) {
                    group = {
                        name: option.groupName,
                        options: []
                    };
                    groups.push(group);
                }
                group.options.push(option);
            }
        } else {
            groups = [
                {
                    name: "Options",
                    options
                }
            ];
        }
        let result1 = "";
        for (const group of groups){
            result1 += this.generateOptionGroup(group);
        }
        return result1;
    }
    generateOptionGroup(group) {
        if (!group.options.length) {
            return "";
        }
        const hasTypeDefinitions = !!group.options.find((option)=>!!option.typeDefinition);
        if (hasTypeDefinitions) {
            return this.label(group.name ?? "Options") + Table.from([
                ...group.options.map((option)=>[
                        option.flags.map((flag)=>brightBlue(flag)).join(", "),
                        highlightArguments(option.typeDefinition || "", this.options.types),
                        red1(bold1("-")),
                        getDescription(option.description, !this.options.long),
                        this.generateHints(option)
                    ])
            ]).padding([
                2,
                2,
                1,
                2
            ]).indent(this.indent).maxColWidth([
                60,
                60,
                1,
                80,
                60
            ]).toString() + "\n";
        }
        return this.label(group.name ?? "Options") + Table.from([
            ...group.options.map((option)=>[
                    option.flags.map((flag)=>brightBlue(flag)).join(", "),
                    red1(bold1("-")),
                    getDescription(option.description, !this.options.long),
                    this.generateHints(option)
                ])
        ]).indent(this.indent).maxColWidth([
            60,
            1,
            80,
            60
        ]).padding([
            2,
            1,
            2
        ]).toString() + "\n";
    }
    generateCommands() {
        const commands = this.cmd.getCommands(false);
        if (!commands.length) {
            return "";
        }
        const hasTypeDefinitions = !!commands.find((command)=>!!command.getArgsDefinition());
        if (hasTypeDefinitions) {
            return this.label("Commands") + Table.from([
                ...commands.map((command)=>[
                        [
                            command.getName(),
                            ...command.getAliases()
                        ].map((name)=>brightBlue(name)).join(", "),
                        highlightArguments(command.getArgsDefinition() || "", this.options.types),
                        red1(bold1("-")),
                        command.getShortDescription()
                    ])
            ]).indent(this.indent).maxColWidth([
                60,
                60,
                1,
                80
            ]).padding([
                2,
                2,
                1,
                2
            ]).toString() + "\n";
        }
        return this.label("Commands") + Table.from([
            ...commands.map((command)=>[
                    [
                        command.getName(),
                        ...command.getAliases()
                    ].map((name)=>brightBlue(name)).join(", "),
                    red1(bold1("-")),
                    command.getShortDescription()
                ])
        ]).maxColWidth([
            60,
            1,
            80
        ]).padding([
            2,
            1,
            2
        ]).indent(this.indent).toString() + "\n";
    }
    generateEnvironmentVariables() {
        const envVars = this.cmd.getEnvVars(false);
        if (!envVars.length) {
            return "";
        }
        return this.label("Environment variables") + Table.from([
            ...envVars.map((envVar)=>[
                    envVar.names.map((name)=>brightBlue(name)).join(", "),
                    highlightArgumentDetails(envVar.details, this.options.types),
                    red1(bold1("-")),
                    this.options.long ? dedent(envVar.description) : envVar.description.trim().split("\n", 1)[0],
                    envVar.required ? `(${yellow1(`required`)})` : ""
                ])
        ]).padding([
            2,
            2,
            1,
            2
        ]).indent(this.indent).maxColWidth([
            60,
            60,
            1,
            80,
            10
        ]).toString() + "\n";
    }
    generateExamples() {
        const examples = this.cmd.getExamples();
        if (!examples.length) {
            return "";
        }
        return this.label("Examples") + Table.from(examples.map((example)=>[
                dim(bold1(`${capitalize(example.name)}:`)),
                dedent(example.description)
            ])).padding(1).indent(this.indent).maxColWidth(150).toString() + "\n";
    }
    generateHints(option) {
        if (!this.options.hints) {
            return "";
        }
        const hints = [];
        option.required && hints.push(yellow1(`required`));
        if (typeof option.default !== "undefined") {
            const defaultValue = getDefaultValue(option);
            if (typeof defaultValue !== "undefined") {
                hints.push(bold1(`Default: `) + inspect(defaultValue, this.options.colors));
            }
        }
        option.depends?.length && hints.push(yellow1(bold1(`Depends: `)) + italic(option.depends.map(getFlag).join(", ")));
        option.conflicts?.length && hints.push(red1(bold1(`Conflicts: `)) + italic(option.conflicts.map(getFlag).join(", ")));
        const type = this.cmd.getType(option.args[0]?.type)?.handler;
        if (type instanceof Type) {
            const possibleValues = type.values?.(this.cmd, this.cmd.getParent());
            if (possibleValues?.length) {
                hints.push(bold1(`Values: `) + possibleValues.map((value)=>inspect(value, this.options.colors)).join(", "));
            }
        }
        if (hints.length) {
            return `(${hints.join(", ")})`;
        }
        return "";
    }
    label(label) {
        return "\n" + bold1(`${label}:`) + "\n\n";
    }
}
function capitalize(string) {
    return (string?.charAt(0).toUpperCase() + string.slice(1)) ?? "";
}
function inspect(value, colors) {
    return Deno.inspect(value, {
        depth: 1,
        colors,
        trailingComma: false
    });
}
function highlightArguments(argsDefinition, types = true) {
    if (!argsDefinition) {
        return "";
    }
    return parseArgumentsDefinition(argsDefinition, false, true).map((arg)=>typeof arg === "string" ? arg : highlightArgumentDetails(arg, types)).join(" ");
}
function highlightArgumentDetails(arg, types = true) {
    let str = "";
    str += yellow1(arg.optional ? "[" : "<");
    let name = "";
    name += arg.name;
    if (arg.variadic) {
        name += "...";
    }
    name = brightMagenta(name);
    str += name;
    if (types) {
        str += yellow1(":");
        str += red1(arg.type);
        if (arg.list) {
            str += green("[]");
        }
    }
    str += yellow1(arg.optional ? "]" : ">");
    return str;
}
class Command {
    types = new Map();
    rawArgs = [];
    literalArgs = [];
    _name = "COMMAND";
    _parent;
    _globalParent;
    ver;
    desc = "";
    _usage;
    actionHandler;
    globalActionHandler;
    options = [];
    commands = new Map();
    examples = [];
    envVars = [];
    aliases = [];
    completions = new Map();
    cmd = this;
    argsDefinition;
    isExecutable = false;
    throwOnError = false;
    _allowEmpty = false;
    _stopEarly = false;
    defaultCommand;
    _useRawArgs = false;
    args = [];
    isHidden = false;
    isGlobal = false;
    hasDefaults = false;
    _versionOptions;
    _helpOptions;
    _versionOption;
    _helpOption;
    _help;
    _shouldExit;
    _meta = {};
    _groupName = null;
    _noGlobals = false;
    errorHandler;
    versionOption(flags, desc, opts) {
        this._versionOptions = flags === false ? flags : {
            flags,
            desc,
            opts: typeof opts === "function" ? {
                action: opts
            } : opts
        };
        return this;
    }
    helpOption(flags, desc, opts) {
        this._helpOptions = flags === false ? flags : {
            flags,
            desc,
            opts: typeof opts === "function" ? {
                action: opts
            } : opts
        };
        return this;
    }
    command(nameAndArguments, cmdOrDescription, override) {
        this.reset();
        const result1 = splitArguments(nameAndArguments);
        const name = result1.flags.shift();
        const aliases = result1.flags;
        if (!name) {
            throw new MissingCommandNameError();
        }
        if (this.getBaseCommand(name, true)) {
            if (!override) {
                throw new DuplicateCommandNameError(name);
            }
            this.removeCommand(name);
        }
        let description;
        let cmd;
        if (typeof cmdOrDescription === "string") {
            description = cmdOrDescription;
        }
        if (cmdOrDescription instanceof Command) {
            cmd = cmdOrDescription.reset();
        } else {
            cmd = new Command();
        }
        cmd._name = name;
        cmd._parent = this;
        if (description) {
            cmd.description(description);
        }
        if (result1.typeDefinition) {
            cmd.arguments(result1.typeDefinition);
        }
        aliases.forEach((alias)=>cmd.alias(alias));
        this.commands.set(name, cmd);
        this.select(name);
        return this;
    }
    alias(alias) {
        if (this.cmd._name === alias || this.cmd.aliases.includes(alias)) {
            throw new DuplicateCommandAliasError(alias);
        }
        this.cmd.aliases.push(alias);
        return this;
    }
    reset() {
        this._groupName = null;
        this.cmd = this;
        return this;
    }
    select(name) {
        const cmd = this.getBaseCommand(name, true);
        if (!cmd) {
            throw new CommandNotFoundError(name, this.getBaseCommands(true));
        }
        this.cmd = cmd;
        return this;
    }
    name(name) {
        this.cmd._name = name;
        return this;
    }
    version(version) {
        if (typeof version === "string") {
            this.cmd.ver = ()=>version;
        } else if (typeof version === "function") {
            this.cmd.ver = version;
        }
        return this;
    }
    meta(name, value) {
        this.cmd._meta[name] = value;
        return this;
    }
    getMeta(name) {
        return typeof name === "undefined" ? this._meta : this._meta[name];
    }
    help(help) {
        if (typeof help === "string") {
            this.cmd._help = ()=>help;
        } else if (typeof help === "function") {
            this.cmd._help = help;
        } else {
            this.cmd._help = (cmd, options)=>HelpGenerator.generate(cmd, {
                    ...help,
                    ...options
                });
        }
        return this;
    }
    description(description) {
        this.cmd.desc = description;
        return this;
    }
    usage(usage) {
        this.cmd._usage = usage;
        return this;
    }
    hidden() {
        this.cmd.isHidden = true;
        return this;
    }
    global() {
        this.cmd.isGlobal = true;
        return this;
    }
    executable() {
        this.cmd.isExecutable = true;
        return this;
    }
    arguments(args) {
        this.cmd.argsDefinition = args;
        return this;
    }
    action(fn) {
        this.cmd.actionHandler = fn;
        return this;
    }
    globalAction(fn) {
        this.cmd.globalActionHandler = fn;
        return this;
    }
    allowEmpty(allowEmpty) {
        this.cmd._allowEmpty = allowEmpty !== false;
        return this;
    }
    stopEarly(stopEarly = true) {
        this.cmd._stopEarly = stopEarly;
        return this;
    }
    useRawArgs(useRawArgs = true) {
        this.cmd._useRawArgs = useRawArgs;
        return this;
    }
    default(name) {
        this.cmd.defaultCommand = name;
        return this;
    }
    globalType(name, handler, options) {
        return this.type(name, handler, {
            ...options,
            global: true
        });
    }
    type(name, handler, options) {
        if (this.cmd.types.get(name) && !options?.override) {
            throw new DuplicateTypeError(name);
        }
        this.cmd.types.set(name, {
            ...options,
            name,
            handler: handler
        });
        if (handler instanceof Type && (typeof handler.complete !== "undefined" || typeof handler.values !== "undefined")) {
            const completeHandler = (cmd, parent)=>handler.complete?.(cmd, parent) || [];
            this.complete(name, completeHandler, options);
        }
        return this;
    }
    globalComplete(name, complete, options) {
        return this.complete(name, complete, {
            ...options,
            global: true
        });
    }
    complete(name, complete, options) {
        if (this.cmd.completions.has(name) && !options?.override) {
            throw new DuplicateCompletionError(name);
        }
        this.cmd.completions.set(name, {
            name,
            complete,
            ...options
        });
        return this;
    }
    throwErrors() {
        this.cmd.throwOnError = true;
        return this;
    }
    error(handler) {
        this.cmd.errorHandler = handler;
        return this;
    }
    getErrorHandler() {
        return this.errorHandler ?? this._parent?.errorHandler;
    }
    noExit() {
        this.cmd._shouldExit = false;
        this.throwErrors();
        return this;
    }
    noGlobals() {
        this.cmd._noGlobals = true;
        return this;
    }
    shouldThrowErrors() {
        return this.throwOnError || !!this._parent?.shouldThrowErrors();
    }
    shouldExit() {
        return this._shouldExit ?? this._parent?.shouldExit() ?? true;
    }
    group(name) {
        this.cmd._groupName = name;
        return this;
    }
    globalOption(flags, desc, opts) {
        if (typeof opts === "function") {
            return this.option(flags, desc, {
                value: opts,
                global: true
            });
        }
        return this.option(flags, desc, {
            ...opts,
            global: true
        });
    }
    option(flags, desc, opts) {
        if (typeof opts === "function") {
            opts = {
                value: opts
            };
        }
        const result1 = splitArguments(flags);
        const args = result1.typeDefinition ? parseArgumentsDefinition(result1.typeDefinition) : [];
        const option = {
            ...opts,
            name: "",
            description: desc,
            args,
            flags: result1.flags,
            equalsSign: result1.equalsSign,
            typeDefinition: result1.typeDefinition,
            groupName: this._groupName ?? undefined
        };
        if (option.separator) {
            for (const arg of args){
                if (arg.list) {
                    arg.separator = option.separator;
                }
            }
        }
        for (const part of option.flags){
            const arg = part.trim();
            const isLong = /^--/.test(arg);
            const name = isLong ? arg.slice(2) : arg.slice(1);
            if (this.cmd.getBaseOption(name, true)) {
                if (opts?.override) {
                    this.removeOption(name);
                } else {
                    throw new DuplicateOptionNameError(name, this.getPath());
                }
            }
            if (!option.name && isLong) {
                option.name = name;
            } else if (!option.aliases) {
                option.aliases = [
                    name
                ];
            } else {
                option.aliases.push(name);
            }
        }
        if (option.prepend) {
            this.cmd.options.unshift(option);
        } else {
            this.cmd.options.push(option);
        }
        return this;
    }
    example(name, description) {
        if (this.cmd.hasExample(name)) {
            throw new DuplicateExampleError(name);
        }
        this.cmd.examples.push({
            name,
            description
        });
        return this;
    }
    globalEnv(name, description, options) {
        return this.env(name, description, {
            ...options,
            global: true
        });
    }
    env(name, description, options) {
        const result1 = splitArguments(name);
        if (!result1.typeDefinition) {
            result1.typeDefinition = "<value:boolean>";
        }
        if (result1.flags.some((envName)=>this.cmd.getBaseEnvVar(envName, true))) {
            throw new DuplicateEnvVarError(name);
        }
        const details = parseArgumentsDefinition(result1.typeDefinition);
        if (details.length > 1) {
            throw new TooManyEnvVarValuesError(name);
        } else if (details.length && details[0].optional) {
            throw new UnexpectedOptionalEnvVarValueError(name);
        } else if (details.length && details[0].variadic) {
            throw new UnexpectedVariadicEnvVarValueError(name);
        }
        this.cmd.envVars.push({
            name: result1.flags[0],
            names: result1.flags,
            description,
            type: details[0].type,
            details: details.shift(),
            ...options
        });
        return this;
    }
    parse(args = Deno.args) {
        const ctx = {
            unknown: args.slice(),
            flags: {},
            env: {},
            literal: [],
            stopEarly: false,
            stopOnUnknown: false,
            defaults: {},
            actions: []
        };
        return this.parseCommand(ctx);
    }
    async parseCommand(ctx) {
        try {
            this.reset();
            this.registerDefaults();
            this.rawArgs = ctx.unknown.slice();
            if (this.isExecutable) {
                await this.executeExecutable(ctx.unknown);
                return {
                    options: {},
                    args: [],
                    cmd: this,
                    literal: []
                };
            } else if (this._useRawArgs) {
                await this.parseEnvVars(ctx, this.envVars);
                return await this.execute(ctx.env, ctx.unknown);
            }
            let preParseGlobals = false;
            let subCommand;
            if (ctx.unknown.length > 0) {
                subCommand = this.getSubCommand(ctx);
                if (!subCommand) {
                    const optionName = ctx.unknown[0].replace(/^-+/, "");
                    const option = this.getOption(optionName, true);
                    if (option?.global) {
                        preParseGlobals = true;
                        await this.parseGlobalOptionsAndEnvVars(ctx);
                    }
                }
            }
            if (subCommand || ctx.unknown.length > 0) {
                subCommand ??= this.getSubCommand(ctx);
                if (subCommand) {
                    subCommand._globalParent = this;
                    return subCommand.parseCommand(ctx);
                }
            }
            await this.parseOptionsAndEnvVars(ctx, preParseGlobals);
            const options = {
                ...ctx.env,
                ...ctx.flags
            };
            const args = this.parseArguments(ctx, options);
            this.literalArgs = ctx.literal;
            if (ctx.actions.length) {
                await Promise.all(ctx.actions.map((action)=>action.call(this, options, ...args)));
            }
            if (ctx.standalone) {
                return {
                    options,
                    args,
                    cmd: this,
                    literal: this.literalArgs
                };
            }
            return await this.execute(options, args);
        } catch (error) {
            this.handleError(error);
        }
    }
    getSubCommand(ctx) {
        const subCommand = this.getCommand(ctx.unknown[0], true);
        if (subCommand) {
            ctx.unknown.shift();
        }
        return subCommand;
    }
    async parseGlobalOptionsAndEnvVars(ctx) {
        const isHelpOption = this.getHelpOption()?.flags.includes(ctx.unknown[0]);
        const envVars = [
            ...this.envVars.filter((envVar)=>envVar.global),
            ...this.getGlobalEnvVars(true)
        ];
        await this.parseEnvVars(ctx, envVars, !isHelpOption);
        const options = [
            ...this.options.filter((option)=>option.global),
            ...this.getGlobalOptions(true)
        ];
        this.parseOptions(ctx, options, {
            stopEarly: true,
            stopOnUnknown: true,
            dotted: false
        });
    }
    async parseOptionsAndEnvVars(ctx, preParseGlobals) {
        const helpOption = this.getHelpOption();
        const isVersionOption = this._versionOption?.flags.includes(ctx.unknown[0]);
        const isHelpOption = helpOption && ctx.flags?.[helpOption.name] === true;
        const envVars = preParseGlobals ? this.envVars.filter((envVar)=>!envVar.global) : this.getEnvVars(true);
        await this.parseEnvVars(ctx, envVars, !isHelpOption && !isVersionOption);
        const options = this.getOptions(true);
        this.parseOptions(ctx, options);
    }
    registerDefaults() {
        if (this.hasDefaults || this.getParent()) {
            return this;
        }
        this.hasDefaults = true;
        this.reset();
        !this.types.has("string") && this.type("string", new StringType(), {
            global: true
        });
        !this.types.has("number") && this.type("number", new NumberType(), {
            global: true
        });
        !this.types.has("integer") && this.type("integer", new IntegerType(), {
            global: true
        });
        !this.types.has("boolean") && this.type("boolean", new BooleanType(), {
            global: true
        });
        !this.types.has("file") && this.type("file", new FileType(), {
            global: true
        });
        if (!this._help) {
            this.help({});
        }
        if (this._versionOptions !== false && (this._versionOptions || this.ver)) {
            this.option(this._versionOptions?.flags || "-V, --version", this._versionOptions?.desc || "Show the version number for this program.", {
                standalone: true,
                prepend: true,
                action: async function() {
                    const __long = this.getRawArgs().includes(`--${this._versionOption?.name}`);
                    if (__long) {
                        await checkVersion(this);
                        this.showLongVersion();
                    } else {
                        this.showVersion();
                    }
                    this.exit();
                },
                ...this._versionOptions?.opts ?? {}
            });
            this._versionOption = this.options[0];
        }
        if (this._helpOptions !== false) {
            this.option(this._helpOptions?.flags || "-h, --help", this._helpOptions?.desc || "Show this help.", {
                standalone: true,
                global: true,
                prepend: true,
                action: async function() {
                    const __long = this.getRawArgs().includes(`--${this.getHelpOption()?.name}`);
                    await checkVersion(this);
                    this.showHelp({
                        long: __long
                    });
                    this.exit();
                },
                ...this._helpOptions?.opts ?? {}
            });
            this._helpOption = this.options[0];
        }
        return this;
    }
    async execute(options, args) {
        if (this.defaultCommand) {
            const cmd = this.getCommand(this.defaultCommand, true);
            if (!cmd) {
                throw new DefaultCommandNotFoundError(this.defaultCommand, this.getCommands());
            }
            cmd._globalParent = this;
            return cmd.execute(options, args);
        }
        await this.executeGlobalAction(options, args);
        if (this.actionHandler) {
            await this.actionHandler(options, ...args);
        }
        return {
            options,
            args,
            cmd: this,
            literal: this.literalArgs
        };
    }
    async executeGlobalAction(options, args) {
        if (!this._noGlobals) {
            await this._parent?.executeGlobalAction(options, args);
        }
        await this.globalActionHandler?.(options, ...args);
    }
    async executeExecutable(args) {
        const command = this.getPath().replace(/\s+/g, "-");
        await Deno.permissions.request({
            name: "run",
            command
        });
        try {
            const cmd = new Deno.Command(command, {
                args
            });
            const output = await cmd.output();
            if (!output.success) {
                Deno.exit(output.code);
            }
        } catch (error) {
            if (error instanceof Deno.errors.NotFound) {
                throw new CommandExecutableNotFoundError(command);
            }
            throw error;
        }
    }
    parseOptions(ctx, options, { stopEarly = this._stopEarly, stopOnUnknown = false, dotted = true } = {}) {
        parseFlags(ctx, {
            stopEarly,
            stopOnUnknown,
            dotted,
            allowEmpty: this._allowEmpty,
            flags: options,
            ignoreDefaults: ctx.env,
            parse: (type)=>this.parseType(type),
            option: (option)=>{
                if (option.action) {
                    ctx.actions.push(option.action);
                }
            }
        });
    }
    parseType(type) {
        const typeSettings = this.getType(type.type);
        if (!typeSettings) {
            throw new UnknownTypeError(type.type, this.getTypes().map((type)=>type.name));
        }
        return typeSettings.handler instanceof Type ? typeSettings.handler.parse(type) : typeSettings.handler(type);
    }
    async parseEnvVars(ctx, envVars, validate = true) {
        for (const envVar of envVars){
            const env = await this.findEnvVar(envVar.names);
            if (env) {
                const parseType = (value)=>{
                    return this.parseType({
                        label: "Environment variable",
                        type: envVar.type,
                        name: env.name,
                        value
                    });
                };
                const propertyName = underscoreToCamelCase(envVar.prefix ? envVar.names[0].replace(new RegExp(`^${envVar.prefix}`), "") : envVar.names[0]);
                if (envVar.details.list) {
                    ctx.env[propertyName] = env.value.split(envVar.details.separator ?? ",").map(parseType);
                } else {
                    ctx.env[propertyName] = parseType(env.value);
                }
                if (envVar.value && typeof ctx.env[propertyName] !== "undefined") {
                    ctx.env[propertyName] = envVar.value(ctx.env[propertyName]);
                }
            } else if (envVar.required && validate) {
                throw new MissingRequiredEnvVarError(envVar);
            }
        }
    }
    async findEnvVar(names) {
        for (const name of names){
            const status = await Deno.permissions.query({
                name: "env",
                variable: name
            });
            if (status.state === "granted") {
                const value = Deno.env.get(name);
                if (value) {
                    return {
                        name,
                        value
                    };
                }
            }
        }
        return undefined;
    }
    parseArguments(ctx, options) {
        const params = [];
        const args = ctx.unknown.slice();
        if (!this.hasArguments()) {
            if (args.length) {
                if (this.hasCommands(true)) {
                    if (this.hasCommand(args[0], true)) {
                        throw new TooManyArgumentsError(args);
                    } else {
                        throw new UnknownCommandError(args[0], this.getCommands());
                    }
                } else {
                    throw new NoArgumentsAllowedError(this.getPath());
                }
            }
        } else {
            if (!args.length) {
                const required = this.getArguments().filter((expectedArg)=>!expectedArg.optional).map((expectedArg)=>expectedArg.name);
                if (required.length) {
                    const optionNames = Object.keys(options);
                    const hasStandaloneOption = !!optionNames.find((name)=>this.getOption(name, true)?.standalone);
                    if (!hasStandaloneOption) {
                        throw new MissingArgumentsError(required);
                    }
                }
            } else {
                for (const expectedArg of this.getArguments()){
                    if (!args.length) {
                        if (expectedArg.optional) {
                            break;
                        }
                        throw new MissingArgumentError(expectedArg.name);
                    }
                    let arg;
                    const parseArgValue = (value)=>{
                        return expectedArg.list ? value.split(",").map((value)=>parseArgType(value)) : parseArgType(value);
                    };
                    const parseArgType = (value)=>{
                        return this.parseType({
                            label: "Argument",
                            type: expectedArg.type,
                            name: expectedArg.name,
                            value
                        });
                    };
                    if (expectedArg.variadic) {
                        arg = args.splice(0, args.length).map((value)=>parseArgValue(value));
                    } else {
                        arg = parseArgValue(args.shift());
                    }
                    if (expectedArg.variadic && Array.isArray(arg)) {
                        params.push(...arg);
                    } else if (typeof arg !== "undefined") {
                        params.push(arg);
                    }
                }
                if (args.length) {
                    throw new TooManyArgumentsError(args);
                }
            }
        }
        return params;
    }
    handleError(error) {
        this.throw(error instanceof ValidationError ? new ValidationError1(error.message) : error instanceof Error ? error : new Error(`[non-error-thrown] ${error}`));
    }
    throw(error) {
        if (error instanceof ValidationError1) {
            error.cmd = this;
        }
        this.getErrorHandler()?.(error, this);
        if (this.shouldThrowErrors() || !(error instanceof ValidationError1)) {
            throw error;
        }
        this.showHelp();
        console.error(red1(`  ${bold1("error")}: ${error.message}\n`));
        Deno.exit(error instanceof ValidationError1 ? error.exitCode : 1);
    }
    getName() {
        return this._name;
    }
    getParent() {
        return this._parent;
    }
    getGlobalParent() {
        return this._globalParent;
    }
    getMainCommand() {
        return this._parent?.getMainCommand() ?? this;
    }
    getAliases() {
        return this.aliases;
    }
    getPath(name) {
        return this._parent ? this._parent.getPath(name) + " " + this._name : name || this._name;
    }
    getArgsDefinition() {
        return this.argsDefinition;
    }
    getArgument(name) {
        return this.getArguments().find((arg)=>arg.name === name);
    }
    getArguments() {
        if (!this.args.length && this.argsDefinition) {
            this.args = parseArgumentsDefinition(this.argsDefinition);
        }
        return this.args;
    }
    hasArguments() {
        return !!this.argsDefinition;
    }
    getVersion() {
        return this.getVersionHandler()?.call(this, this);
    }
    getVersionHandler() {
        return this.ver ?? this._parent?.getVersionHandler();
    }
    getDescription() {
        return typeof this.desc === "function" ? this.desc = this.desc() : this.desc;
    }
    getUsage() {
        return this._usage ?? [
            this.getArgsDefinition(),
            this.getRequiredOptionsDefinition()
        ].join(" ").trim();
    }
    getRequiredOptionsDefinition() {
        return this.getOptions().filter((option)=>option.required).map((option)=>[
                findFlag(option.flags),
                option.typeDefinition
            ].filter((v)=>v).join(" ").trim()).join(" ");
    }
    getShortDescription() {
        return getDescription(this.getDescription(), true);
    }
    getRawArgs() {
        return this.rawArgs;
    }
    getLiteralArgs() {
        return this.literalArgs;
    }
    showVersion() {
        console.log(this.getVersion());
    }
    getLongVersion() {
        return `${bold1(this.getMainCommand().getName())} ${brightBlue(this.getVersion() ?? "")}` + Object.entries(this.getMeta()).map(([k, v])=>`\n${bold1(k)} ${brightBlue(v)}`).join("");
    }
    showLongVersion() {
        console.log(this.getLongVersion());
    }
    showHelp(options) {
        console.log(this.getHelp(options));
    }
    getHelp(options) {
        this.registerDefaults();
        return this.getHelpHandler().call(this, this, options ?? {});
    }
    getHelpHandler() {
        return this._help ?? this._parent?.getHelpHandler();
    }
    exit(code = 0) {
        if (this.shouldExit()) {
            Deno.exit(code);
        }
    }
    hasOptions(hidden) {
        return this.getOptions(hidden).length > 0;
    }
    getOptions(hidden) {
        return this.getGlobalOptions(hidden).concat(this.getBaseOptions(hidden));
    }
    getBaseOptions(hidden) {
        if (!this.options.length) {
            return [];
        }
        return hidden ? this.options.slice(0) : this.options.filter((opt)=>!opt.hidden);
    }
    getGlobalOptions(hidden) {
        const helpOption = this.getHelpOption();
        const getGlobals = (cmd, noGlobals, options = [], names = [])=>{
            if (cmd.options.length) {
                for (const option of cmd.options){
                    if (option.global && !this.options.find((opt)=>opt.name === option.name) && names.indexOf(option.name) === -1 && (hidden || !option.hidden)) {
                        if (noGlobals && option !== helpOption) {
                            continue;
                        }
                        names.push(option.name);
                        options.push(option);
                    }
                }
            }
            return cmd._parent ? getGlobals(cmd._parent, noGlobals || cmd._noGlobals, options, names) : options;
        };
        return this._parent ? getGlobals(this._parent, this._noGlobals) : [];
    }
    hasOption(name, hidden) {
        return !!this.getOption(name, hidden);
    }
    getOption(name, hidden) {
        return this.getBaseOption(name, hidden) ?? this.getGlobalOption(name, hidden);
    }
    getBaseOption(name, hidden) {
        const option = this.options.find((option)=>option.name === name || option.aliases?.includes(name));
        return option && (hidden || !option.hidden) ? option : undefined;
    }
    getGlobalOption(name, hidden) {
        const helpOption = this.getHelpOption();
        const getGlobalOption = (parent, noGlobals)=>{
            const option = parent.getBaseOption(name, hidden);
            if (!option?.global) {
                return parent._parent && getGlobalOption(parent._parent, noGlobals || parent._noGlobals);
            }
            if (noGlobals && option !== helpOption) {
                return;
            }
            return option;
        };
        return this._parent && getGlobalOption(this._parent, this._noGlobals);
    }
    removeOption(name) {
        const index = this.options.findIndex((option)=>option.name === name);
        if (index === -1) {
            return;
        }
        return this.options.splice(index, 1)[0];
    }
    hasCommands(hidden) {
        return this.getCommands(hidden).length > 0;
    }
    getCommands(hidden) {
        return this.getGlobalCommands(hidden).concat(this.getBaseCommands(hidden));
    }
    getBaseCommands(hidden) {
        const commands = Array.from(this.commands.values());
        return hidden ? commands : commands.filter((cmd)=>!cmd.isHidden);
    }
    getGlobalCommands(hidden) {
        const getCommands = (command, noGlobals, commands = [], names = [])=>{
            if (command.commands.size) {
                for (const [_, cmd] of command.commands){
                    if (cmd.isGlobal && this !== cmd && !this.commands.has(cmd._name) && names.indexOf(cmd._name) === -1 && (hidden || !cmd.isHidden)) {
                        if (noGlobals && cmd?.getName() !== "help") {
                            continue;
                        }
                        names.push(cmd._name);
                        commands.push(cmd);
                    }
                }
            }
            return command._parent ? getCommands(command._parent, noGlobals || command._noGlobals, commands, names) : commands;
        };
        return this._parent ? getCommands(this._parent, this._noGlobals) : [];
    }
    hasCommand(name, hidden) {
        return !!this.getCommand(name, hidden);
    }
    getCommand(name, hidden) {
        return this.getBaseCommand(name, hidden) ?? this.getGlobalCommand(name, hidden);
    }
    getBaseCommand(name, hidden) {
        for (const cmd of this.commands.values()){
            if (cmd._name === name || cmd.aliases.includes(name)) {
                return cmd && (hidden || !cmd.isHidden) ? cmd : undefined;
            }
        }
    }
    getGlobalCommand(name, hidden) {
        const getGlobalCommand = (parent, noGlobals)=>{
            const cmd = parent.getBaseCommand(name, hidden);
            if (!cmd?.isGlobal) {
                return parent._parent && getGlobalCommand(parent._parent, noGlobals || parent._noGlobals);
            }
            if (noGlobals && cmd.getName() !== "help") {
                return;
            }
            return cmd;
        };
        return this._parent && getGlobalCommand(this._parent, this._noGlobals);
    }
    removeCommand(name) {
        const command = this.getBaseCommand(name, true);
        if (command) {
            this.commands.delete(command._name);
        }
        return command;
    }
    getTypes() {
        return this.getGlobalTypes().concat(this.getBaseTypes());
    }
    getBaseTypes() {
        return Array.from(this.types.values());
    }
    getGlobalTypes() {
        const getTypes = (cmd, types = [], names = [])=>{
            if (cmd) {
                if (cmd.types.size) {
                    cmd.types.forEach((type)=>{
                        if (type.global && !this.types.has(type.name) && names.indexOf(type.name) === -1) {
                            names.push(type.name);
                            types.push(type);
                        }
                    });
                }
                return getTypes(cmd._parent, types, names);
            }
            return types;
        };
        return getTypes(this._parent);
    }
    getType(name) {
        return this.getBaseType(name) ?? this.getGlobalType(name);
    }
    getBaseType(name) {
        return this.types.get(name);
    }
    getGlobalType(name) {
        if (!this._parent) {
            return;
        }
        const cmd = this._parent.getBaseType(name);
        if (!cmd?.global) {
            return this._parent.getGlobalType(name);
        }
        return cmd;
    }
    getCompletions() {
        return this.getGlobalCompletions().concat(this.getBaseCompletions());
    }
    getBaseCompletions() {
        return Array.from(this.completions.values());
    }
    getGlobalCompletions() {
        const getCompletions = (cmd, completions = [], names = [])=>{
            if (cmd) {
                if (cmd.completions.size) {
                    cmd.completions.forEach((completion)=>{
                        if (completion.global && !this.completions.has(completion.name) && names.indexOf(completion.name) === -1) {
                            names.push(completion.name);
                            completions.push(completion);
                        }
                    });
                }
                return getCompletions(cmd._parent, completions, names);
            }
            return completions;
        };
        return getCompletions(this._parent);
    }
    getCompletion(name) {
        return this.getBaseCompletion(name) ?? this.getGlobalCompletion(name);
    }
    getBaseCompletion(name) {
        return this.completions.get(name);
    }
    getGlobalCompletion(name) {
        if (!this._parent) {
            return;
        }
        const completion = this._parent.getBaseCompletion(name);
        if (!completion?.global) {
            return this._parent.getGlobalCompletion(name);
        }
        return completion;
    }
    hasEnvVars(hidden) {
        return this.getEnvVars(hidden).length > 0;
    }
    getEnvVars(hidden) {
        return this.getGlobalEnvVars(hidden).concat(this.getBaseEnvVars(hidden));
    }
    getBaseEnvVars(hidden) {
        if (!this.envVars.length) {
            return [];
        }
        return hidden ? this.envVars.slice(0) : this.envVars.filter((env)=>!env.hidden);
    }
    getGlobalEnvVars(hidden) {
        if (this._noGlobals) {
            return [];
        }
        const getEnvVars = (cmd, envVars = [], names = [])=>{
            if (cmd) {
                if (cmd.envVars.length) {
                    cmd.envVars.forEach((envVar)=>{
                        if (envVar.global && !this.envVars.find((env)=>env.names[0] === envVar.names[0]) && names.indexOf(envVar.names[0]) === -1 && (hidden || !envVar.hidden)) {
                            names.push(envVar.names[0]);
                            envVars.push(envVar);
                        }
                    });
                }
                return getEnvVars(cmd._parent, envVars, names);
            }
            return envVars;
        };
        return getEnvVars(this._parent);
    }
    hasEnvVar(name, hidden) {
        return !!this.getEnvVar(name, hidden);
    }
    getEnvVar(name, hidden) {
        return this.getBaseEnvVar(name, hidden) ?? this.getGlobalEnvVar(name, hidden);
    }
    getBaseEnvVar(name, hidden) {
        const envVar = this.envVars.find((env)=>env.names.indexOf(name) !== -1);
        return envVar && (hidden || !envVar.hidden) ? envVar : undefined;
    }
    getGlobalEnvVar(name, hidden) {
        if (!this._parent || this._noGlobals) {
            return;
        }
        const envVar = this._parent.getBaseEnvVar(name, hidden);
        if (!envVar?.global) {
            return this._parent.getGlobalEnvVar(name, hidden);
        }
        return envVar;
    }
    hasExamples() {
        return this.examples.length > 0;
    }
    getExamples() {
        return this.examples;
    }
    hasExample(name) {
        return !!this.getExample(name);
    }
    getExample(name) {
        return this.examples.find((example)=>example.name === name);
    }
    getHelpOption() {
        return this._helpOption ?? this._parent?.getHelpOption();
    }
}
async function checkVersion(cmd) {
    const mainCommand = cmd.getMainCommand();
    const upgradeCommand = mainCommand.getCommand("upgrade");
    if (!isUpgradeCommand(upgradeCommand)) {
        return;
    }
    const latestVersion = await upgradeCommand.getLatestVersion();
    const currentVersion = mainCommand.getVersion();
    if (currentVersion === latestVersion) {
        return;
    }
    const versionHelpText = `(New version available: ${latestVersion}. Run '${mainCommand.getName()} upgrade' to upgrade to the latest version!)`;
    mainCommand.version(`${currentVersion}  ${bold1(yellow1(versionHelpText))}`);
}
function findFlag(flags) {
    for (const flag of flags){
        if (flag.startsWith("--")) {
            return flag;
        }
    }
    return flags[0];
}
function isUpgradeCommand(command) {
    return command instanceof Command && "getLatestVersion" in command;
}
class CommandType extends StringType {
    complete(_cmd, parent) {
        return parent?.getCommands(false).map((cmd)=>cmd.getName()) || [];
    }
}
class HelpCommand extends Command {
    constructor(cmd){
        super();
        return this.type("command", new CommandType()).arguments("[command:command]").description("Show this help or the help of a sub-command.").noGlobals().action(async (_, name)=>{
            if (!cmd) {
                cmd = name ? this.getGlobalParent()?.getBaseCommand(name) : this.getGlobalParent();
            }
            if (!cmd) {
                const cmds = this.getGlobalParent()?.getCommands();
                throw new UnknownCommandError(name ?? "", cmds ?? [], [
                    this.getName(),
                    ...this.getAliases()
                ]);
            }
            await checkVersion(cmd);
            cmd.showHelp();
            if (this.shouldExit()) {
                Deno.exit(0);
            }
        });
    }
}
class BashCompletionsGenerator {
    name;
    cmd;
    static generate(name, cmd) {
        if (!name || name === "COMMAND") {
            throw new MissingCommandNameCompletionsError("bash");
        }
        return new BashCompletionsGenerator(name, cmd).generate();
    }
    constructor(name, cmd){
        this.name = name;
        this.cmd = cmd;
    }
    generate() {
        const path = this.cmd.getPath(this.name);
        const version = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env bash
# bash completion support for ${path}${version}

_${replaceSpecialChars(path)}() {
  local word cur prev listFiles
  local -a opts
  COMPREPLY=()
  cur="\${COMP_WORDS[COMP_CWORD]}"
  prev="\${COMP_WORDS[COMP_CWORD-1]}"
  cmd="_"
  opts=()
  listFiles=0

  _${replaceSpecialChars(this.name)}_complete() {
    local action="$1"; shift
    mapfile -t values < <( ${this.name} completions complete "\${action}" "\${@}" )
    for i in "\${values[@]}"; do
      opts+=("$i")
    done
  }

  _${replaceSpecialChars(this.name)}_expand() {
    [ "$cur" != "\${cur%\\\\}" ] && cur="$cur\\\\"
  
    # expand ~username type directory specifications
    if [[ "$cur" == \\~*/* ]]; then
      # shellcheck disable=SC2086
      eval cur=$cur
      
    elif [[ "$cur" == \\~* ]]; then
      cur=\${cur#\\~}
      # shellcheck disable=SC2086,SC2207
      COMPREPLY=( $( compgen -P '~' -u $cur ) )
      return \${#COMPREPLY[@]}
    fi
  }

  # shellcheck disable=SC2120
  _${replaceSpecialChars(this.name)}_file_dir() {
    listFiles=1
    local IFS=$'\\t\\n' xspec #glob
    _${replaceSpecialChars(this.name)}_expand || return 0
  
    if [ "\${1:-}" = -d ]; then
      # shellcheck disable=SC2206,SC2207,SC2086
      COMPREPLY=( \${COMPREPLY[@]:-} $( compgen -d -- $cur ) )
      #eval "$glob"    # restore glob setting.
      return 0
    fi
  
    xspec=\${1:+"!*.$1"}	# set only if glob passed in as $1
    # shellcheck disable=SC2206,SC2207
    COMPREPLY=( \${COMPREPLY[@]:-} $( compgen -f -X "$xspec" -- "$cur" ) \
          $( compgen -d -- "$cur" ) )
  }

  ${this.generateCompletions(this.name, this.cmd).trim()}

  for word in "\${COMP_WORDS[@]}"; do
    case "\${word}" in
      -*) ;;
      *)
        cmd_tmp="\${cmd}_\${word//[^[:alnum:]]/_}"
        if type "\${cmd_tmp}" &>/dev/null; then
          cmd="\${cmd_tmp}"
        fi
    esac
  done

  \${cmd}

  if [[ listFiles -eq 1 ]]; then
    return 0
  fi

  if [[ \${#opts[@]} -eq 0 ]]; then
    # shellcheck disable=SC2207
    COMPREPLY=($(compgen -f "\${cur}"))
    return 0
  fi

  local values
  values="$( printf "\\n%s" "\${opts[@]}" )"
  local IFS=$'\\n'
  # shellcheck disable=SC2207
  local result=($(compgen -W "\${values[@]}" -- "\${cur}"))
  if [[ \${#result[@]} -eq 0 ]]; then
    # shellcheck disable=SC2207
    COMPREPLY=($(compgen -f "\${cur}"))
  else
    # shellcheck disable=SC2207
    COMPREPLY=($(printf '%q\\n' "\${result[@]}"))
  fi

  return 0
}

complete -F _${replaceSpecialChars(path)} -o bashdefault -o default ${path}`;
    }
    generateCompletions(name, command, path = "", index = 1) {
        path = (path ? path + " " : "") + name;
        const commandCompletions = this.generateCommandCompletions(command, path, index);
        const childCommandCompletions = command.getCommands(false).filter((subCommand)=>subCommand !== command).map((subCommand)=>this.generateCompletions(subCommand.getName(), subCommand, path, index + 1)).join("");
        return `${commandCompletions}

${childCommandCompletions}`;
    }
    generateCommandCompletions(command, path, index) {
        const flags = this.getFlags(command);
        const childCommandNames = command.getCommands(false).map((childCommand)=>childCommand.getName());
        const completionsPath = ~path.indexOf(" ") ? " " + path.split(" ").slice(1).join(" ") : "";
        const optionArguments = this.generateOptionArguments(command, completionsPath);
        const completionsCmd = this.generateCommandCompletionsCommand(command, completionsPath);
        return `  __${replaceSpecialChars(path)}() {
    opts=(${[
            ...flags,
            ...childCommandNames
        ].join(" ")})
    ${completionsCmd}
    if [[ \${cur} == -* || \${COMP_CWORD} -eq ${index} ]] ; then
      return 0
    fi
    ${optionArguments}
  }`;
    }
    getFlags(command) {
        return command.getOptions(false).map((option)=>option.flags).flat();
    }
    generateOptionArguments(command, completionsPath) {
        let opts = "";
        const options = command.getOptions(false);
        if (options.length) {
            opts += 'case "${prev}" in';
            for (const option of options){
                const flags = option.flags.map((flag)=>flag.trim()).join("|");
                const completionsCmd = this.generateOptionCompletionsCommand(command, option.args, completionsPath, {
                    standalone: option.standalone
                });
                opts += `\n      ${flags}) ${completionsCmd} ;;`;
            }
            opts += "\n    esac";
        }
        return opts;
    }
    generateCommandCompletionsCommand(command, path) {
        const args = command.getArguments();
        if (args.length) {
            const type = command.getType(args[0].type);
            if (type && type.handler instanceof FileType) {
                return `_${replaceSpecialChars(this.name)}_file_dir`;
            }
            return `_${replaceSpecialChars(this.name)}_complete ${args[0].action}${path}`;
        }
        return "";
    }
    generateOptionCompletionsCommand(command, args, path, opts) {
        if (args.length) {
            const type = command.getType(args[0].type);
            if (type && type.handler instanceof FileType) {
                return `opts=(); _${replaceSpecialChars(this.name)}_file_dir`;
            }
            return `opts=(); _${replaceSpecialChars(this.name)}_complete ${args[0].action}${path}`;
        }
        if (opts?.standalone) {
            return "opts=()";
        }
        return "";
    }
}
function replaceSpecialChars(str) {
    return str.replace(/[^a-zA-Z0-9]/g, "_");
}
class BashCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        return this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for bash.

To enable bash completions for this program add following line to your ${dim(italic("~/.bashrc"))}:

    ${dim(italic(`source <(${baseCmd.getPath()} completions bash)`))}`;
        }).noGlobals().option("-n, --name <command-name>", "The name of the main command.").action(({ name = this.getMainCommand().getName() })=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            console.log(BashCompletionsGenerator.generate(name, baseCmd));
        });
    }
}
class FishCompletionsGenerator {
    name;
    cmd;
    static generate(name, cmd) {
        if (!name || name === "COMMAND") {
            throw new MissingCommandNameCompletionsError("fish");
        }
        return new FishCompletionsGenerator(name, cmd).generate();
    }
    constructor(name, cmd){
        this.name = name;
        this.cmd = cmd;
    }
    generate() {
        const path = this.cmd.getPath(this.name);
        const version = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env fish
# fish completion support for ${path}${version}

function __fish_${replaceSpecialChars1(this.name)}_using_command
  set -l cmds ${getCommandFnNames(this.name, this.cmd).join(" ")}
  set -l words (commandline -opc)
  set -l cmd "_"
  for word in $words
    switch $word
      case '-*'
        continue
      case '*'
        set word (string replace -r -a '\\W' '_' $word)
        set -l cmd_tmp $cmd"_$word"
        if contains $cmd_tmp $cmds
          set cmd $cmd_tmp
        end
    end
  end
  if test "$cmd" = "$argv[1]"
    return 0
  end
  return 1
end

${this.generateCompletions(this.name, this.cmd).trim()}`;
    }
    generateCompletions(name, command) {
        const parent = command.getParent();
        let result1 = ``;
        if (parent) {
            result1 += "\n" + this.complete(parent, {
                description: command.getShortDescription(),
                arguments: name
            });
        }
        const commandArgs = command.getArguments();
        if (commandArgs.length) {
            result1 += "\n" + this.complete(command, {
                arguments: commandArgs.length ? this.getCompletionCommand(command, commandArgs[0]) : undefined
            });
        }
        for (const option of command.getOptions(false)){
            result1 += "\n" + this.completeOption(command, option);
        }
        for (const subCommand of command.getCommands(false)){
            result1 += this.generateCompletions(subCommand.getName(), subCommand);
        }
        return result1;
    }
    completeOption(command, option) {
        const shortOption = option.flags.find((flag)=>flag.length === 2)?.replace(/^(-)+/, "");
        const longOption = option.flags.find((flag)=>flag.length > 2)?.replace(/^(-)+/, "");
        return this.complete(command, {
            description: getDescription(option.description),
            shortOption: shortOption,
            longOption: longOption,
            required: true,
            standalone: option.standalone,
            arguments: option.args.length ? this.getCompletionCommand(command, option.args[0]) : undefined
        });
    }
    complete(command, options) {
        const cmd = [
            "complete"
        ];
        cmd.push("-c", this.name);
        cmd.push("-n", `'__fish_${replaceSpecialChars1(this.name)}_using_command __${replaceSpecialChars1(command.getPath(this.name))}'`);
        options.shortOption && cmd.push("-s", options.shortOption);
        options.longOption && cmd.push("-l", options.longOption);
        options.standalone && cmd.push("-x");
        cmd.push("-k");
        cmd.push("-f");
        if (options.arguments) {
            options.required && cmd.push("-r");
            cmd.push("-a", options.arguments);
        }
        if (options.description) {
            const description = getDescription(options.description, true).replace(/'/g, "\\'");
            cmd.push("-d", `'${description}'`);
        }
        return cmd.join(" ");
    }
    getCompletionCommand(cmd, arg) {
        const type = cmd.getType(arg.type);
        if (type && type.handler instanceof FileType) {
            return `'(__fish_complete_path)'`;
        }
        return `'(${this.name} completions complete ${arg.action + " " + getCompletionsPath(cmd)})'`;
    }
}
function getCommandFnNames(name, cmd, cmds = []) {
    cmds.push(`__${replaceSpecialChars1(cmd.getPath(name))}`);
    cmd.getCommands(false).forEach((command)=>{
        getCommandFnNames(name, command, cmds);
    });
    return cmds;
}
function getCompletionsPath(command) {
    return command.getPath().split(" ").slice(1).join(" ");
}
function replaceSpecialChars1(str) {
    return str.replace(/[^a-zA-Z0-9]/g, "_");
}
class FishCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        return this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for fish.

To enable fish completions for this program add following line to your ${dim(italic("~/.config/fish/config.fish"))}:

    ${dim(italic(`source (${baseCmd.getPath()} completions fish | psub)`))}`;
        }).noGlobals().option("-n, --name <command-name>", "The name of the main command.").action(({ name = this.getMainCommand().getName() })=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            console.log(FishCompletionsGenerator.generate(name, baseCmd));
        });
    }
}
class ZshCompletionsGenerator {
    name;
    cmd;
    actions;
    static generate(name, cmd) {
        if (!name || name === "COMMAND") {
            throw new MissingCommandNameCompletionsError("zsh");
        }
        return new ZshCompletionsGenerator(name, cmd).generate();
    }
    constructor(name, cmd){
        this.name = name;
        this.cmd = cmd;
        this.actions = new Map();
    }
    generate() {
        const path = this.cmd.getPath(this.name);
        const version = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env zsh
# zsh completion support for ${path}${version}

autoload -U is-at-least

# shellcheck disable=SC2154
(( $+functions[__${replaceSpecialChars2(this.name)}_complete] )) ||
function __${replaceSpecialChars2(this.name)}_complete {
  local name="$1"; shift
  local action="$1"; shift
  integer ret=1
  local -a values
  local expl lines
  _tags "$name"
  while _tags; do
    if _requested "$name"; then
      # shellcheck disable=SC2034
      lines="$(${this.name} completions complete "\${action}" "\${@}")"
      values=("\${(ps:\\n:)lines}")
      if (( \${#values[@]} )); then
        while _next_label "$name" expl "$action"; do
          compadd -S '' "\${expl[@]}" "\${values[@]}"
        done
      fi
    fi
  done
}

${this.generateCompletions(this.name, this.cmd).trim()}

# _${replaceSpecialChars2(path)} "\${@}"

compdef _${replaceSpecialChars2(path)} ${path}`;
    }
    generateCompletions(name, command, path = "") {
        if (!command.hasCommands(false) && !command.hasOptions(false) && !command.hasArguments()) {
            return "";
        }
        path = (path ? path + " " : "") + name;
        return `# shellcheck disable=SC2154
(( $+functions[_${replaceSpecialChars2(path)}] )) ||
function _${replaceSpecialChars2(path)}() {` + (!command.getParent() ? `
  local state` : "") + this.generateCommandCompletions(command, path) + this.generateSubCommandCompletions(command, path) + this.generateArgumentCompletions(command, path) + this.generateActions(command) + `\n}\n\n` + command.getCommands(false).filter((subCommand)=>subCommand !== command).map((subCommand)=>this.generateCompletions(subCommand.getName(), subCommand, path)).join("");
    }
    generateCommandCompletions(command, path) {
        const commands = command.getCommands(false);
        let completions = commands.map((subCommand)=>`'${subCommand.getName()}:${subCommand.getShortDescription().replace(/'/g, "'\"'\"'")}'`).join("\n      ");
        if (completions) {
            completions = `
    local -a commands
    # shellcheck disable=SC2034
    commands=(
      ${completions}
    )
    _describe 'command' commands`;
        }
        if (command.hasArguments()) {
            const completionsPath = path.split(" ").slice(1).join(" ");
            const arg = command.getArguments()[0];
            const action = this.addAction(arg, completionsPath);
            if (action && command.getCompletion(arg.action)) {
                completions += `\n    __${replaceSpecialChars2(this.name)}_complete ${action.arg.name} ${action.arg.action} ${action.cmd}`;
            }
        }
        if (completions) {
            completions = `\n\n  function _commands() {${completions}\n  }`;
        }
        return completions;
    }
    generateSubCommandCompletions(command, path) {
        if (command.hasCommands(false)) {
            const actions = command.getCommands(false).map((command)=>`${command.getName()}) _${replaceSpecialChars2(path + " " + command.getName())} ;;`).join("\n      ");
            return `\n
  function _command_args() {
    case "\${words[1]}" in\n      ${actions}\n    esac
  }`;
        }
        return "";
    }
    generateArgumentCompletions(command, path) {
        this.actions.clear();
        const options = this.generateOptions(command, path);
        let argIndex = 0;
        let argsCommand = "\n\n  _arguments -w -s -S -C";
        if (command.hasOptions()) {
            argsCommand += ` \\\n    ${options.join(" \\\n    ")}`;
        }
        if (command.hasCommands(false) || command.getArguments().filter((arg)=>command.getCompletion(arg.action)).length) {
            argsCommand += ` \\\n    '${++argIndex}:command:_commands'`;
        }
        if (command.hasArguments() || command.hasCommands(false)) {
            const args = [];
            for (const arg of command.getArguments().slice(1)){
                const type = command.getType(arg.type);
                if (type && type.handler instanceof FileType) {
                    const fileCompletions = this.getFileCompletions(type);
                    if (arg.variadic) {
                        argIndex++;
                        for(let i1 = 0; i1 < 5; i1++){
                            args.push(`${argIndex + i1}${arg.optional ? "::" : ":"}${arg.name}:${fileCompletions}`);
                        }
                    } else {
                        args.push(`${++argIndex}${arg.optional ? "::" : ":"}${arg.name}:${fileCompletions}`);
                    }
                } else {
                    const completionsPath = path.split(" ").slice(1).join(" ");
                    const action = this.addAction(arg, completionsPath);
                    args.push(`${++argIndex}${arg.optional ? "::" : ":"}${arg.name}:->${action.name}`);
                }
            }
            argsCommand += args.map((arg)=>`\\\n    '${arg}'`).join("");
            if (command.hasCommands(false)) {
                argsCommand += ` \\\n    '*::sub command:->command_args'`;
            }
        }
        return argsCommand;
    }
    generateOptions(command, path) {
        const options = [];
        const cmdArgs = path.split(" ");
        cmdArgs.shift();
        const completionsPath = cmdArgs.join(" ");
        const excludedFlags = command.getOptions(false).map((option)=>option.standalone ? option.flags : false).flat().filter((flag)=>typeof flag === "string");
        for (const option of command.getOptions(false)){
            options.push(this.generateOption(command, option, completionsPath, excludedFlags));
        }
        return options;
    }
    generateOption(command, option, completionsPath, excludedOptions) {
        let args = "";
        for (const arg of option.args){
            const type = command.getType(arg.type);
            const optionalValue = arg.optional ? "::" : ":";
            if (type && type.handler instanceof FileType) {
                const fileCompletions = this.getFileCompletions(type);
                args += `${optionalValue}${arg.name}:${fileCompletions}`;
            } else {
                const action = this.addAction(arg, completionsPath);
                args += `${optionalValue}${arg.name}:->${action.name}`;
            }
        }
        const description = getDescription(option.description, true).replace(/\[/g, "\\[").replace(/]/g, "\\]").replace(/"/g, '\\"').replace(/'/g, "'\"'\"'");
        const collect = option.collect ? "*" : "";
        const equalsSign = option.equalsSign ? "=" : "";
        const flags = option.flags.map((flag)=>`${flag}${equalsSign}`);
        let result1 = "";
        if (option.standalone) {
            result1 += "'(- *)'";
        } else {
            const excludedFlags = [
                ...excludedOptions
            ];
            if (option.conflicts?.length) {
                excludedFlags.push(...option.conflicts.map((opt)=>"--" + opt.replace(/^--/, "")));
            }
            if (!option.collect) {
                excludedFlags.push(...option.flags);
            }
            if (excludedFlags.length) {
                result1 += `'(${excludedFlags.join(" ")})'`;
            }
        }
        if (collect || flags.length > 1) {
            result1 += `{${collect}${flags.join(",")}}`;
        } else {
            result1 += `${flags.join(",")}`;
        }
        return `${result1}'[${description}]${args}'`;
    }
    getFileCompletions(type) {
        if (!(type.handler instanceof FileType)) {
            return "";
        }
        return "_files";
    }
    addAction(arg, cmd) {
        const action = `${arg.name}-${arg.action}`;
        if (!this.actions.has(action)) {
            this.actions.set(action, {
                arg: arg,
                label: `${arg.name}: ${arg.action}`,
                name: action,
                cmd
            });
        }
        return this.actions.get(action);
    }
    generateActions(command) {
        let actions = [];
        if (this.actions.size) {
            actions = Array.from(this.actions).map(([name, action])=>`${name}) __${replaceSpecialChars2(this.name)}_complete ${action.arg.name} ${action.arg.action} ${action.cmd} ;;`);
        }
        if (command.hasCommands(false)) {
            actions.unshift(`command_args) _command_args ;;`);
        }
        if (actions.length) {
            return `\n\n  case "$state" in\n    ${actions.join("\n    ")}\n  esac`;
        }
        return "";
    }
}
function replaceSpecialChars2(str) {
    return str.replace(/[^a-zA-Z0-9]/g, "_");
}
class ZshCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        return this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for zsh.

To enable zsh completions for this program add following line to your ${dim(italic("~/.zshrc"))}:

    ${dim(italic(`source <(${baseCmd.getPath()} completions zsh)`))}`;
        }).noGlobals().option("-n, --name <command-name>", "The name of the main command.", {
            default: ()=>this.getMainCommand().getName()
        }).action(({ name })=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            console.log(ZshCompletionsGenerator.generate(name, baseCmd));
        });
    }
}
class CompleteCommand extends Command {
    constructor(cmd){
        super();
        return this.description("Get completions for given action from given command.").noGlobals().arguments("<action:string> [command...:string]").action(async (_, action, ...commandNames)=>{
            let parent;
            const completeCommand = commandNames?.reduce((cmd, name)=>{
                parent = cmd;
                const childCmd = cmd.getCommand(name, false);
                if (!childCmd) {
                    throw new UnknownCompletionCommandError(name, cmd.getCommands());
                }
                return childCmd;
            }, cmd || this.getMainCommand()) ?? (cmd || this.getMainCommand());
            const completion = completeCommand.getCompletion(action);
            const result1 = await completion?.complete(completeCommand, parent) ?? [];
            if (result1?.length) {
                Deno.stdout.writeSync(new TextEncoder().encode(result1.join("\n")));
            }
        }).reset();
    }
}
class CompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        return this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions.

To enable shell completions for this program add the following line to your ${dim(italic("~/.bashrc"))} or similar:

    ${dim(italic(`source <(${baseCmd.getPath()} completions [shell])`))}

    For more information run ${dim(italic(`${baseCmd.getPath()} completions [shell] --help`))}
`;
        }).noGlobals().action(()=>this.showHelp()).command("bash", new BashCompletionsCommand(this.#cmd)).command("fish", new FishCompletionsCommand(this.#cmd)).command("zsh", new ZshCompletionsCommand(this.#cmd)).command("complete", new CompleteCommand(this.#cmd)).hidden().reset();
    }
}
const State = {
    PASSTHROUGH: 0,
    PERCENT: 1,
    POSITIONAL: 2,
    PRECISION: 3,
    WIDTH: 4
};
const WorP = {
    WIDTH: 0,
    PRECISION: 1
};
const F = {
    sign: 1,
    mantissa: 2,
    fractional: 3,
    esign: 4,
    exponent: 5
};
class Flags {
    plus;
    dash;
    sharp;
    space;
    zero;
    lessthan;
    width = -1;
    precision = -1;
}
const min = Math.min;
const UNICODE_REPLACEMENT_CHARACTER = "\ufffd";
const FLOAT_REGEXP = /(-?)(\d)\.?(\d*)e([+-])(\d+)/;
class Printf {
    format;
    args;
    i;
    state = State.PASSTHROUGH;
    verb = "";
    buf = "";
    argNum = 0;
    flags = new Flags();
    haveSeen;
    tmpError;
    constructor(format, ...args){
        this.format = format;
        this.args = args;
        this.haveSeen = Array.from({
            length: args.length
        });
        this.i = 0;
    }
    doPrintf() {
        for(; this.i < this.format.length; ++this.i){
            const c = this.format[this.i];
            switch(this.state){
                case State.PASSTHROUGH:
                    if (c === "%") {
                        this.state = State.PERCENT;
                    } else {
                        this.buf += c;
                    }
                    break;
                case State.PERCENT:
                    if (c === "%") {
                        this.buf += c;
                        this.state = State.PASSTHROUGH;
                    } else {
                        this.handleFormat();
                    }
                    break;
                default:
                    throw Error("Should be unreachable, certainly a bug in the lib.");
            }
        }
        let extras = false;
        let err = "%!(EXTRA";
        for(let i1 = 0; i1 !== this.haveSeen.length; ++i1){
            if (!this.haveSeen[i1]) {
                extras = true;
                err += ` '${Deno.inspect(this.args[i1])}'`;
            }
        }
        err += ")";
        if (extras) {
            this.buf += err;
        }
        return this.buf;
    }
    handleFormat() {
        this.flags = new Flags();
        const flags = this.flags;
        for(; this.i < this.format.length; ++this.i){
            const c = this.format[this.i];
            switch(this.state){
                case State.PERCENT:
                    switch(c){
                        case "[":
                            this.handlePositional();
                            this.state = State.POSITIONAL;
                            break;
                        case "+":
                            flags.plus = true;
                            break;
                        case "<":
                            flags.lessthan = true;
                            break;
                        case "-":
                            flags.dash = true;
                            flags.zero = false;
                            break;
                        case "#":
                            flags.sharp = true;
                            break;
                        case " ":
                            flags.space = true;
                            break;
                        case "0":
                            flags.zero = !flags.dash;
                            break;
                        default:
                            if ("1" <= c && c <= "9" || c === "." || c === "*") {
                                if (c === ".") {
                                    this.flags.precision = 0;
                                    this.state = State.PRECISION;
                                    this.i++;
                                } else {
                                    this.state = State.WIDTH;
                                }
                                this.handleWidthAndPrecision(flags);
                            } else {
                                this.handleVerb();
                                return;
                            }
                    }
                    break;
                case State.POSITIONAL:
                    if (c === "*") {
                        const worp = this.flags.precision === -1 ? WorP.WIDTH : WorP.PRECISION;
                        this.handleWidthOrPrecisionRef(worp);
                        this.state = State.PERCENT;
                        break;
                    } else {
                        this.handleVerb();
                        return;
                    }
                default:
                    throw new Error(`Should not be here ${this.state}, library bug!`);
            }
        }
    }
    handleWidthOrPrecisionRef(wOrP) {
        if (this.argNum >= this.args.length) {
            return;
        }
        const arg = this.args[this.argNum];
        this.haveSeen[this.argNum] = true;
        if (typeof arg === "number") {
            switch(wOrP){
                case WorP.WIDTH:
                    this.flags.width = arg;
                    break;
                default:
                    this.flags.precision = arg;
            }
        } else {
            const tmp = wOrP === WorP.WIDTH ? "WIDTH" : "PREC";
            this.tmpError = `%!(BAD ${tmp} '${this.args[this.argNum]}')`;
        }
        this.argNum++;
    }
    handleWidthAndPrecision(flags) {
        const fmt = this.format;
        for(; this.i !== this.format.length; ++this.i){
            const c = fmt[this.i];
            switch(this.state){
                case State.WIDTH:
                    switch(c){
                        case ".":
                            this.flags.precision = 0;
                            this.state = State.PRECISION;
                            break;
                        case "*":
                            this.handleWidthOrPrecisionRef(WorP.WIDTH);
                            break;
                        default:
                            {
                                const val = parseInt(c);
                                if (isNaN(val)) {
                                    this.i--;
                                    this.state = State.PERCENT;
                                    return;
                                }
                                flags.width = flags.width === -1 ? 0 : flags.width;
                                flags.width *= 10;
                                flags.width += val;
                            }
                    }
                    break;
                case State.PRECISION:
                    {
                        if (c === "*") {
                            this.handleWidthOrPrecisionRef(WorP.PRECISION);
                            break;
                        }
                        const val = parseInt(c);
                        if (isNaN(val)) {
                            this.i--;
                            this.state = State.PERCENT;
                            return;
                        }
                        flags.precision *= 10;
                        flags.precision += val;
                        break;
                    }
                default:
                    throw new Error("can't be here. bug.");
            }
        }
    }
    handlePositional() {
        if (this.format[this.i] !== "[") {
            throw new Error("Can't happen? Bug.");
        }
        let positional = 0;
        const format = this.format;
        this.i++;
        let err = false;
        for(; this.i !== this.format.length; ++this.i){
            if (format[this.i] === "]") {
                break;
            }
            positional *= 10;
            const val = parseInt(format[this.i], 10);
            if (isNaN(val)) {
                this.tmpError = "%!(BAD INDEX)";
                err = true;
            }
            positional += val;
        }
        if (positional - 1 >= this.args.length) {
            this.tmpError = "%!(BAD INDEX)";
            err = true;
        }
        this.argNum = err ? this.argNum : positional - 1;
    }
    handleLessThan() {
        const arg = this.args[this.argNum];
        if ((arg || {}).constructor.name !== "Array") {
            throw new Error(`arg ${arg} is not an array. Todo better error handling`);
        }
        let str = "[ ";
        for(let i1 = 0; i1 !== arg.length; ++i1){
            if (i1 !== 0) str += ", ";
            str += this._handleVerb(arg[i1]);
        }
        return str + " ]";
    }
    handleVerb() {
        const verb = this.format[this.i];
        this.verb = verb || this.verb;
        if (this.tmpError) {
            this.buf += this.tmpError;
            this.tmpError = undefined;
            if (this.argNum < this.haveSeen.length) {
                this.haveSeen[this.argNum] = true;
            }
        } else if (this.args.length <= this.argNum) {
            this.buf += `%!(MISSING '${verb}')`;
        } else {
            const arg = this.args[this.argNum];
            this.haveSeen[this.argNum] = true;
            if (this.flags.lessthan) {
                this.buf += this.handleLessThan();
            } else {
                this.buf += this._handleVerb(arg);
            }
        }
        this.argNum++;
        this.state = State.PASSTHROUGH;
    }
    _handleVerb(arg) {
        switch(this.verb){
            case "t":
                return this.pad(arg.toString());
            case "b":
                return this.fmtNumber(arg, 2);
            case "c":
                return this.fmtNumberCodePoint(arg);
            case "d":
                return this.fmtNumber(arg, 10);
            case "o":
                return this.fmtNumber(arg, 8);
            case "x":
                return this.fmtHex(arg);
            case "X":
                return this.fmtHex(arg, true);
            case "e":
                return this.fmtFloatE(arg);
            case "E":
                return this.fmtFloatE(arg, true);
            case "f":
            case "F":
                return this.fmtFloatF(arg);
            case "g":
                return this.fmtFloatG(arg);
            case "G":
                return this.fmtFloatG(arg, true);
            case "s":
                return this.fmtString(arg);
            case "T":
                return this.fmtString(typeof arg);
            case "v":
                return this.fmtV(arg);
            case "j":
                return this.fmtJ(arg);
            case "i":
                return this.fmtI(arg, false);
            case "I":
                return this.fmtI(arg, true);
            default:
                return `%!(BAD VERB '${this.verb}')`;
        }
    }
    pad(s) {
        const padding = this.flags.zero ? "0" : " ";
        if (this.flags.dash) {
            return s.padEnd(this.flags.width, padding);
        }
        return s.padStart(this.flags.width, padding);
    }
    padNum(nStr, neg) {
        let sign;
        if (neg) {
            sign = "-";
        } else if (this.flags.plus || this.flags.space) {
            sign = this.flags.plus ? "+" : " ";
        } else {
            sign = "";
        }
        const zero = this.flags.zero;
        if (!zero) {
            nStr = sign + nStr;
        }
        const pad = zero ? "0" : " ";
        const len = zero ? this.flags.width - sign.length : this.flags.width;
        if (this.flags.dash) {
            nStr = nStr.padEnd(len, pad);
        } else {
            nStr = nStr.padStart(len, pad);
        }
        if (zero) {
            nStr = sign + nStr;
        }
        return nStr;
    }
    fmtNumber(n, radix, upcase = false) {
        let num = Math.abs(n).toString(radix);
        const prec = this.flags.precision;
        if (prec !== -1) {
            this.flags.zero = false;
            num = n === 0 && prec === 0 ? "" : num;
            while(num.length < prec){
                num = "0" + num;
            }
        }
        let prefix = "";
        if (this.flags.sharp) {
            switch(radix){
                case 2:
                    prefix += "0b";
                    break;
                case 8:
                    prefix += num.startsWith("0") ? "" : "0";
                    break;
                case 16:
                    prefix += "0x";
                    break;
                default:
                    throw new Error("cannot handle base: " + radix);
            }
        }
        num = num.length === 0 ? num : prefix + num;
        if (upcase) {
            num = num.toUpperCase();
        }
        return this.padNum(num, n < 0);
    }
    fmtNumberCodePoint(n) {
        let s = "";
        try {
            s = String.fromCodePoint(n);
        } catch  {
            s = UNICODE_REPLACEMENT_CHARACTER;
        }
        return this.pad(s);
    }
    fmtFloatSpecial(n) {
        if (isNaN(n)) {
            this.flags.zero = false;
            return this.padNum("NaN", false);
        }
        if (n === Number.POSITIVE_INFINITY) {
            this.flags.zero = false;
            this.flags.plus = true;
            return this.padNum("Inf", false);
        }
        if (n === Number.NEGATIVE_INFINITY) {
            this.flags.zero = false;
            return this.padNum("Inf", true);
        }
        return "";
    }
    roundFractionToPrecision(fractional, precision) {
        let round = false;
        if (fractional.length > precision) {
            fractional = "1" + fractional;
            let tmp = parseInt(fractional.slice(0, precision + 2)) / 10;
            tmp = Math.round(tmp);
            fractional = Math.floor(tmp).toString();
            round = fractional[0] === "2";
            fractional = fractional.slice(1);
        } else {
            while(fractional.length < precision){
                fractional += "0";
            }
        }
        return [
            fractional,
            round
        ];
    }
    fmtFloatE(n, upcase = false) {
        const special = this.fmtFloatSpecial(n);
        if (special !== "") {
            return special;
        }
        const m = n.toExponential().match(FLOAT_REGEXP);
        if (!m) {
            throw Error("can't happen, bug");
        }
        const precision = this.flags.precision !== -1 ? this.flags.precision : 6;
        const [fractional, rounding] = this.roundFractionToPrecision(m[F.fractional] || "", precision);
        let e = m[F.exponent];
        let esign = m[F.esign];
        let mantissa = parseInt(m[F.mantissa]);
        if (rounding) {
            mantissa += 1;
            if (10 <= mantissa) {
                mantissa = 1;
                const r = parseInt(esign + e) + 1;
                e = r.toString();
                esign = r < 0 ? "-" : "+";
            }
        }
        e = e.length === 1 ? "0" + e : e;
        const val = `${mantissa}.${fractional}${upcase ? "E" : "e"}${esign}${e}`;
        return this.padNum(val, n < 0);
    }
    fmtFloatF(n) {
        const special = this.fmtFloatSpecial(n);
        if (special !== "") {
            return special;
        }
        function expandNumber(n) {
            if (Number.isSafeInteger(n)) {
                return n.toString() + ".";
            }
            const t = n.toExponential().split("e");
            let m = t[0].replace(".", "");
            const e = parseInt(t[1]);
            if (e < 0) {
                let nStr = "0.";
                for(let i1 = 0; i1 !== Math.abs(e) - 1; ++i1){
                    nStr += "0";
                }
                return nStr += m;
            } else {
                const splIdx = e + 1;
                while(m.length < splIdx){
                    m += "0";
                }
                return m.slice(0, splIdx) + "." + m.slice(splIdx);
            }
        }
        const val = expandNumber(Math.abs(n));
        let [dig, fractional] = val.split(".");
        const precision = this.flags.precision !== -1 ? this.flags.precision : 6;
        let round = false;
        [fractional, round] = this.roundFractionToPrecision(fractional, precision);
        if (round) {
            dig = (parseInt(dig) + 1).toString();
        }
        return this.padNum(`${dig}.${fractional}`, n < 0);
    }
    fmtFloatG(n, upcase = false) {
        const special = this.fmtFloatSpecial(n);
        if (special !== "") {
            return special;
        }
        let P = this.flags.precision !== -1 ? this.flags.precision : 6;
        P = P === 0 ? 1 : P;
        const m = n.toExponential().match(FLOAT_REGEXP);
        if (!m) {
            throw Error("can't happen");
        }
        const X = parseInt(m[F.exponent]) * (m[F.esign] === "-" ? -1 : 1);
        let nStr = "";
        if (P > X && X >= -4) {
            this.flags.precision = P - (X + 1);
            nStr = this.fmtFloatF(n);
            if (!this.flags.sharp) {
                nStr = nStr.replace(/\.?0*$/, "");
            }
        } else {
            this.flags.precision = P - 1;
            nStr = this.fmtFloatE(n);
            if (!this.flags.sharp) {
                nStr = nStr.replace(/\.?0*e/, upcase ? "E" : "e");
            }
        }
        return nStr;
    }
    fmtString(s) {
        if (this.flags.precision !== -1) {
            s = s.slice(0, this.flags.precision);
        }
        return this.pad(s);
    }
    fmtHex(val, upper = false) {
        switch(typeof val){
            case "number":
                return this.fmtNumber(val, 16, upper);
            case "string":
                {
                    const sharp = this.flags.sharp && val.length !== 0;
                    let hex = sharp ? "0x" : "";
                    const prec = this.flags.precision;
                    const end = prec !== -1 ? min(prec, val.length) : val.length;
                    for(let i1 = 0; i1 !== end; ++i1){
                        if (i1 !== 0 && this.flags.space) {
                            hex += sharp ? " 0x" : " ";
                        }
                        const c = (val.charCodeAt(i1) & 0xff).toString(16);
                        hex += c.length === 1 ? `0${c}` : c;
                    }
                    if (upper) {
                        hex = hex.toUpperCase();
                    }
                    return this.pad(hex);
                }
            default:
                throw new Error("currently only number and string are implemented for hex");
        }
    }
    fmtV(val) {
        if (this.flags.sharp) {
            const options = this.flags.precision !== -1 ? {
                depth: this.flags.precision
            } : {};
            return this.pad(Deno.inspect(val, options));
        } else {
            const p = this.flags.precision;
            return p === -1 ? val.toString() : val.toString().slice(0, p);
        }
    }
    fmtJ(val) {
        return JSON.stringify(val);
    }
    fmtI(val, compact) {
        return Deno.inspect(val, {
            colors: !Deno?.noColor,
            compact,
            depth: Infinity,
            iterableLimit: Infinity
        });
    }
}
const { Deno: Deno3 } = globalThis;
const noColor2 = typeof Deno3?.noColor === "boolean" ? Deno3.noColor : false;
let enabled2 = !noColor2;
function code2(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run2(str, code) {
    return enabled2 ? `${code.open}${str.replace(code.regexp, code.open)}${code.close}` : str;
}
function bold2(str) {
    return run2(str, code2([
        1
    ], 22));
}
function dim1(str) {
    return run2(str, code2([
        2
    ], 22));
}
function underline1(str) {
    return run2(str, code2([
        4
    ], 24));
}
function red2(str) {
    return run2(str, code2([
        31
    ], 39));
}
function green1(str) {
    return run2(str, code2([
        32
    ], 39));
}
function yellow2(str) {
    return run2(str, code2([
        33
    ], 39));
}
function blue2(str) {
    return run2(str, code2([
        34
    ], 39));
}
function brightRed1(str) {
    return run2(str, code2([
        91
    ], 39));
}
const ANSI_PATTERN1 = new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]+)*|[a-zA-Z\\d]+(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TXZcf-nq-uy=><~]))"
].join("|"), "g");
function stripColor1(string) {
    return stripAnsiCode(string);
}
function stripAnsiCode(string) {
    return string.replace(ANSI_PATTERN1, "");
}
function glb(array, value, compare) {
    compare = compare || ((a, b)=>a - b);
    if (array.length === 0) {
        return -1;
    }
    if (array.length === 1) {
        if (compare(value, array[0]) < 0) {
            return -1;
        } else {
            return 0;
        }
    }
    let left = 0;
    let right = array.length - 1;
    const vLeft = array[left], vRight = array[right];
    if (compare(value, vRight) >= 0) {
        return right;
    }
    if (compare(value, vLeft) < 0) {
        return -1;
    }
    while(right - left > 1){
        const center = left + (right - left >> 1);
        const vCenter = array[center];
        const cmp = compare(value, vCenter);
        if (cmp < 0) {
            right = center;
        } else if (cmp === 0) {
            left = center;
        } else {
            left = center;
        }
    }
    return left;
}
class InternalError extends Error {
    constructor(message, printName = true, printStack = true){
        super(message);
        this.name = "Internal Error";
        this.printName = printName;
        this.printStack = printStack;
    }
    printName;
    printStack;
}
class UnreachableError extends InternalError {
    constructor(){
        super("Unreachable code was reached.", true, true);
    }
}
class ErrorEx extends Error {
    constructor(name, message, printName = true, printStack = true){
        super(message);
        this.name = name;
        this.printName = printName;
        this.printStack = printStack;
    }
    printName;
    printStack;
}
function asErrorEx(e, defaultShowStack = true) {
    if (e instanceof ErrorEx) {
        return e;
    } else if (e instanceof Error) {
        e.printName = e.name !== "Error";
        e.printStack = defaultShowStack && !!e.message;
        return e;
    } else {
        return new ErrorEx("Error", String(e), false, true);
    }
}
let Deno4;
try {
    Deno4 = globalThis.Deno;
} catch (_e) {}
const noColor3 = typeof (Deno4 && Deno4.noColor) === "boolean" ? Deno4.noColor : true;
let enabled3 = !noColor3;
function code3(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run3(str, code) {
    return enabled3 ? `${code.open}${str.replace(code.regexp, code.open)}${code.close}` : str;
}
function red3(str) {
    return run3(str, code3([
        31
    ], 39));
}
function blue3(str) {
    return run3(str, code3([
        34
    ], 39));
}
const ANSI_PATTERN2 = new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:[a-zA-Z\\d]*(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-ntqry=><~]))"
].join("|"), "g");
function stripColor2(string) {
    return string.replace(ANSI_PATTERN2, "");
}
function platformHasNonAsciiCharacters() {
    try {
        return Deno.build.os !== "windows";
    } catch (_e) {
        return true;
    }
}
function tidyverseInfo(msg) {
    if (platformHasNonAsciiCharacters()) {
        return `${blue3("ℹ")} ${msg}`;
    } else {
        return `${blue3("i")} ${msg}`;
    }
}
function tidyverseError(msg) {
    if (platformHasNonAsciiCharacters()) {
        return `${red3("✖")} ${msg}`;
    } else {
        return `${red3("x")} ${msg}`;
    }
}
function tidyverseFormatError(msg) {
    let { heading, error, info } = msg;
    if (msg.location) {
        heading = `${locationString(msg.location)} ${heading}`;
    }
    if (msg.fileName) {
        heading = `In file ${msg.fileName}\n${heading}`;
    }
    const strings = [
        heading,
        msg.sourceContext,
        ...error.map(tidyverseError),
        ...Object.values(info).map(tidyverseInfo),
        ""
    ];
    return strings.join("\n");
}
function quotedStringColor(msg) {
    return blue3(msg);
}
function addFileInfo(msg, src) {
    if (src.fileName !== undefined) {
        msg.fileName = src.fileName;
    }
}
function addInstancePathInfo(msg, instancePath) {
    if (instancePath.length) {
        const niceInstancePath = instancePath.map((s)=>blue3(String(s))).join(":");
        msg.info["instance-path-location"] = `The error happened in location ${niceInstancePath}.`;
    }
}
function locationString(loc) {
    const { start, end } = loc;
    if (start.line === end.line) {
        if (start.column === end.column) {
            return `(line ${start.line + 1}, column ${start.column + 1})`;
        } else {
            return `(line ${start.line + 1}, columns ${start.column + 1}--${end.column + 1})`;
        }
    } else {
        return `(line ${start.line + 1}, column ${start.column + 1} through line ${end.line + 1}, column ${end.column + 1})`;
    }
}
function errorKey(err) {
    const positionKey = (pos)=>`${pos.line}-${pos.column}`;
    return `${err.fileName || ""}-${positionKey(err.location.start)}-${positionKey(err.location.end)}`;
}
function reportOnce(reporter, reportSet) {
    const errorsReported = reportSet || new Set();
    return (err)=>{
        const key = errorKey(err);
        if (errorsReported.has(key)) {
            return;
        }
        errorsReported.add(key);
        reporter(err);
    };
}
function lines(text) {
    return text.split(/\r?\n/);
}
function normalizeNewlines(text) {
    return lines(text).join("\n");
}
function trimEmptyLines(lines, trim = "all") {
    if (trim === "all" || trim === "leading") {
        const firstNonEmpty = lines.findIndex((line)=>line.trim().length > 0);
        if (firstNonEmpty === -1) {
            return [];
        }
        lines = lines.slice(firstNonEmpty);
    }
    if (trim === "all" || trim === "trailing") {
        let lastNonEmpty = -1;
        for(let i1 = lines.length - 1; i1 >= 0; i1--){
            if (lines[i1].trim().length > 0) {
                lastNonEmpty = i1;
                break;
            }
        }
        if (lastNonEmpty > -1) {
            lines = lines.slice(0, lastNonEmpty + 1);
        }
    }
    return lines;
}
function* matchAll(text, regexp) {
    if (!regexp.global) {
        throw new Error("matchAll requires global regexps");
    }
    let match;
    while((match = regexp.exec(text)) !== null){
        yield match;
    }
}
function* lineOffsets(text) {
    yield 0;
    for (const match of matchAll(text, /\r?\n/g)){
        yield match.index + match[0].length;
    }
}
function indexToLineCol(text) {
    const offsets = Array.from(lineOffsets(text));
    return function(offset) {
        if (offset === 0) {
            return {
                line: 0,
                column: 0
            };
        }
        const startIndex = glb(offsets, offset);
        return {
            line: startIndex,
            column: offset - offsets[startIndex]
        };
    };
}
function lineColToIndex(text) {
    const offsets = Array.from(lineOffsets(text));
    return function(position) {
        return offsets[position.line] + position.column;
    };
}
function formatLineRange(text, firstLine, lastLine) {
    const lineWidth = Math.max(String(firstLine + 1).length, String(lastLine + 1).length);
    const pad = " ".repeat(lineWidth);
    const ls = lines(text);
    const result1 = [];
    for(let i1 = firstLine; i1 <= lastLine; ++i1){
        const numberStr = `${pad}${i1 + 1}: `.slice(-(lineWidth + 2));
        const lineStr = ls[i1];
        result1.push({
            lineNumber: i1,
            content: numberStr + quotedStringColor(lineStr),
            rawLine: ls[i1]
        });
    }
    return {
        prefixWidth: lineWidth + 2,
        lines: result1
    };
}
function editDistance(w1, w2) {
    const cost = (c)=>{
        if ("_-".indexOf(c) !== -1) {
            return 1;
        }
        return 10;
    };
    const cost2 = (c1, c2)=>{
        if (c1 === c2) {
            return 0;
        }
        if ("_-".indexOf(c1) !== -1 && "_-".indexOf(c2) !== -1) {
            return 1;
        }
        if (c1.toLocaleLowerCase() === c2.toLocaleLowerCase()) {
            return 1;
        }
        const cc1 = c1.charCodeAt(0);
        const cc2 = c2.charCodeAt(0);
        if (cc1 >= 48 && cc1 <= 57 && cc2 >= 48 && cc2 <= 57) {
            return 1;
        }
        return 10;
    };
    const s1 = w1.length + 1;
    const s2 = w2.length + 1;
    const v = new Int32Array(s1 * s2);
    for(let i1 = 0; i1 < s1; ++i1){
        for(let j = 0; j < s2; ++j){
            if (i1 === 0 && j === 0) {
                continue;
            } else if (i1 === 0) {
                v[i1 * s2 + j] = v[i1 * s2 + (j - 1)] + cost(w2[j - 1]);
            } else if (j === 0) {
                v[i1 * s2 + j] = v[(i1 - 1) * s2 + j] + cost(w1[i1 - 1]);
            } else {
                v[i1 * s2 + j] = Math.min(v[(i1 - 1) * s2 + (j - 1)] + cost2(w1[i1 - 1], w2[j - 1]), v[i1 * s2 + (j - 1)] + cost(w2[j - 1]), v[(i1 - 1) * s2 + j] + cost(w1[i1 - 1]));
            }
        }
    }
    return v[(w1.length + 1) * (w2.length + 1) - 1];
}
function detectCaseConvention(key) {
    if (key.toLocaleLowerCase() !== key) {
        return "capitalizationCase";
    }
    const underscoreIndex = key.indexOf("_");
    if (underscoreIndex !== -1 && underscoreIndex !== 0 && underscoreIndex !== key.length - 1) {
        return "underscore_case";
    }
    const dashIndex = key.indexOf("-");
    if (dashIndex !== -1 && dashIndex !== 0 && dashIndex !== key.length - 1) {
        return "dash-case";
    }
    return undefined;
}
function resolveCaseConventionRegex(keys, conventions) {
    if (conventions !== undefined) {
        if (conventions.length === 0) {
            throw new InternalError("resolveCaseConventionRegex requires nonempty `conventions`");
        }
        return {
            pattern: conventions.map((c)=>`(${c})`).join("|"),
            list: conventions
        };
    }
    const disallowedNearMisses = [];
    const keySet = new Set(keys);
    const addNearMiss = (value)=>{
        if (!keySet.has(value)) {
            disallowedNearMisses.push(value);
        }
    };
    const foundConventions = new Set();
    for (const key of keys){
        const found = detectCaseConvention(key);
        if (found) {
            foundConventions.add(found);
        }
        switch(found){
            case "capitalizationCase":
                addNearMiss(toUnderscoreCase(key));
                addNearMiss(toDashCase(key));
                break;
            case "dash-case":
                addNearMiss(toUnderscoreCase(key));
                addNearMiss(toCapitalizationCase(key));
                break;
            case "underscore_case":
                addNearMiss(toDashCase(key));
                addNearMiss(toCapitalizationCase(key));
                break;
        }
    }
    if (foundConventions.size === 0) {
        return {
            pattern: undefined,
            list: []
        };
    }
    return {
        pattern: `(?!(${disallowedNearMisses.map((c)=>`^${c}$`).join("|")}))`,
        list: Array.from(foundConventions)
    };
}
function toDashCase(str) {
    return toUnderscoreCase(str).replace(/_/g, "-");
}
function toUnderscoreCase(str) {
    return str.replace(/([A-Z]+)/g, (_match, p1)=>`-${p1}`).replace(/-/g, "_").split("_").filter((x)=>x.length).join("_").toLocaleLowerCase();
}
function toCapitalizationCase(str) {
    return toUnderscoreCase(str).replace(/_(.)/g, (_match, p1)=>p1.toLocaleUpperCase());
}
function capitalizeWord(str) {
    return str.slice(0, 1).toUpperCase() + str.slice(1);
}
function capitalizeTitle(str) {
    return str.split(/\s+/).map((str, index, arr)=>{
        if (index === 0 || index === arr.length - 1 || !isNotCapitalized(str)) {
            return capitalizeWord(str);
        } else {
            return str;
        }
    }).join(" ");
}
function isNotCapitalized(str) {
    return [
        "a",
        "an",
        "the",
        "for",
        "and",
        "nor",
        "but",
        "or",
        "yet",
        "so",
        "with",
        "at",
        "by",
        "to",
        "in",
        "for",
        "from",
        "of",
        "on"
    ].includes(str);
}
const kLastPunctuationRegex = /([\S\s]*)[\.\?\!]/;
function trimSentence(text) {
    const match = text.match(kLastPunctuationRegex);
    if (match) {
        return {
            text: match[0],
            trimmed: true
        };
    } else {
        return {
            text,
            trimmed: false
        };
    }
}
function trimLength(text, length) {
    if (text.length < length) {
        return {
            text,
            trimmed: false
        };
    } else {
        return {
            text: text.substring(0, length),
            trimmed: true
        };
    }
}
function trimSpace(text) {
    const lastSpace = text.lastIndexOf(" ");
    if (lastSpace > 0) {
        return {
            text: text.substring(0, lastSpace),
            trimmed: true
        };
    } else {
        return {
            text,
            trimmed: false
        };
    }
}
function truncateText(text, length, breakAt) {
    const trimEnd = (text)=>{
        if ([
            ",",
            "/",
            ":"
        ].includes(text.charAt(text.length - 1))) {
            return text.substring(0, text.length - 1);
        } else {
            return text;
        }
    };
    const trimAtSpace = (text)=>{
        const spaceResult = trimSpace(text.substring(0, text.length - 1));
        return trimEnd(spaceResult.text) + "…";
    };
    const trimPunc = (text)=>{
        const puncResult = trimSentence(text);
        if (puncResult.trimmed) {
            return puncResult.text;
        } else {
            return trimAtSpace(puncResult.text);
        }
    };
    const lengthResult = trimLength(text, length);
    if (lengthResult.trimmed) {
        if (breakAt === "punctuation") {
            return trimPunc(lengthResult.text);
        } else {
            return trimAtSpace(lengthResult.text);
        }
    } else {
        return lengthResult.text;
    }
}
const booleanSchema = {
    "type": "boolean",
    "description": "be `true` or `false`",
    "completions": [
        "true",
        "false"
    ],
    "exhaustiveCompletions": true
};
const numberSchema = {
    "type": "number",
    "description": "be a number"
};
const stringSchema = {
    "type": "string",
    "description": "be a string"
};
const nullSchema = {
    "type": "null",
    "description": "be the null value",
    "completions": [
        "null"
    ],
    "exhaustiveCompletions": true
};
function schemaType(schema) {
    if (schema === false) {
        return "false";
    }
    if (schema === true) {
        return "true";
    }
    return schema.type;
}
function schemaDispatch(s, d) {
    const st = schemaType(s);
    if (d[st]) {
        d[st](s);
    }
}
function schemaCall(s, d, other) {
    const st = schemaType(s);
    if (d[st]) {
        return d[st](s);
    }
    if (other) {
        return other(s);
    }
    throw new Error(`Internal Error: Dispatch failed for type ${st}`);
}
function schemaDocString(d) {
    if (typeof d === "string") {
        return d;
    }
    if (d.short) {
        return d.short;
    }
    return "";
}
function schemaDescription(schema) {
    if (schema === true) {
        return `be anything`;
    } else if (schema === false) {
        return `be no possible value`;
    } else {
        return schema.description || `be ${schemaType(schema)}`;
    }
}
let globalInternalIdCounter = 0;
function internalId() {
    return {
        _internalId: ++globalInternalIdCounter
    };
}
function tagSchema(schema, tags) {
    return {
        ...schema,
        tags: {
            ...schema.tags || {},
            ...tags
        }
    };
}
function anySchema(description) {
    return {
        ...internalId(),
        description,
        "type": "any"
    };
}
function enumSchema(...args) {
    if (args.length === 0) {
        throw new InternalError("Empty enum schema not supported.");
    }
    return {
        ...internalId(),
        "type": "enum",
        "enum": args,
        "description": args.length > 1 ? `be one of: ${args.map((x)=>"`" + x + "`").join(", ")}` : `be '${args[0]}'`,
        "completions": args.map(String),
        "exhaustiveCompletions": true
    };
}
function regexSchema(arg, description) {
    const result1 = {
        ...internalId(),
        "type": "string",
        "pattern": arg
    };
    if (description) {
        result1.description = description;
    } else {
        result1.description = `be a string that satisfies regex "${arg}"`;
    }
    return result1;
}
function anyOfSchema(...args) {
    return {
        ...internalId(),
        "type": "anyOf",
        "anyOf": args,
        "description": `be at least one of: ${args.map((x)=>schemaDescription(x).slice(3)).join(", ")}`
    };
}
function allOfSchema(...args) {
    return {
        ...internalId(),
        "type": "allOf",
        "allOf": args,
        "description": `be all of: ${args.map((x)=>schemaDescription(x).slice(3)).join(", ")}`
    };
}
function objectSchema(params = {}) {
    let { properties, patternProperties, required, additionalProperties, description, baseSchema, exhaustive, completions: completionsParam, namingConvention, propertyNames: propertyNamesSchema, closed } = params;
    required = required || [];
    properties = properties || {};
    patternProperties = patternProperties || {};
    let tags = {};
    let tagsAreSet = false;
    let propertyNames = propertyNamesSchema;
    if (completionsParam) {
        tags["completions"] = completionsParam;
        tagsAreSet = true;
    }
    const createCaseConventionSchema = (props)=>{
        if (namingConvention === "ignore") {
            return undefined;
        }
        const objectKeys = Object.getOwnPropertyNames(props);
        const { pattern, list } = resolveCaseConventionRegex(objectKeys, namingConvention);
        if (pattern === undefined) {
            return undefined;
        }
        if (propertyNames !== undefined) {
            console.error("Warning: propertyNames and case convention detection are mutually exclusive.");
            console.error("Add `namingConvention: 'ignore'` to your schema definition to remove this warning.");
            return undefined;
        }
        const tags = {
            "case-convention": list,
            "error-importance": -5,
            "case-detection": true
        };
        return {
            errorMessage: "property ${value} does not match case convention " + `${objectKeys.join(",")}`,
            "type": "string",
            pattern,
            tags
        };
    };
    const hasDescription = description !== undefined;
    description = description || "be an object";
    let result1 = undefined;
    if (baseSchema) {
        if (!Array.isArray(baseSchema)) {
            baseSchema = [
                baseSchema
            ];
        }
        if (baseSchema.some((s)=>s.type !== "object")) {
            throw new InternalError("Attempted to extend a non-object schema");
        }
        if (baseSchema.length <= 0) {
            throw new InternalError("base schema cannot be empty list");
        }
        let temp = {
            ...internalId()
        };
        for (const base of baseSchema){
            temp = Object.assign(temp, base);
        }
        result1 = temp;
        if (result1 === undefined) {
            throw new UnreachableError();
        }
        if (result1.$id) {
            delete result1.$id;
        }
        for (const base of baseSchema){
            if (base.exhaustiveCompletions) {
                result1.exhaustiveCompletions = true;
            }
        }
        if (hasDescription) {
            result1.description = description;
        }
        const m = new Map();
        for (const base of baseSchema){
            for (const [k, v] of Object.entries(base.properties || {})){
                if (!m.has(k)) {
                    m.set(k, []);
                }
                m.get(k).push([
                    v,
                    base.$id
                ]);
            }
        }
        const errorMsgs = new Set();
        for (const [k, l1] of m){
            if (l1.length > 1) {
                errorMsgs.add(`Internal Error: base schemas ${l1.map((x)=>x[1]).join(", ")} share property ${k}.`);
            }
        }
        if (errorMsgs.size > 0) {
            console.error([
                ...errorMsgs
            ].toSorted((a, b)=>a.localeCompare(b)).join("\n"));
            console.error("This is a bug in quarto's schemas.");
            console.error("Note that we don't throw in order to allow build-js to finish, but the generated schemas will be invalid.");
        }
        result1.properties = Object.assign({}, ...baseSchema.map((s)=>s.properties), properties);
        result1.patternProperties = Object.assign({}, ...baseSchema.map((s)=>s.patternProperties).filter((s)=>s !== undefined), patternProperties);
        result1.required = [
            ...baseSchema.map((s)=>s.required || []),
            required || []
        ].flat();
        if (result1.required && result1.required.length === 0) {
            result1.required = undefined;
        }
        const additionalPropArray = baseSchema.map((s)=>s.additionalProperties).filter((s)=>s !== undefined);
        if (additionalProperties) {
            additionalPropArray.push(additionalProperties);
        }
        if (additionalPropArray.length) {
            result1.additionalProperties = allOfSchema(...additionalPropArray);
        }
        const propNamesArray = baseSchema.map((s)=>s.propertyNames).filter((s)=>{
            if (typeof s !== "object") return true;
            if (s.tags === undefined) return true;
            if (s.tags["case-detection"] === true) {
                return false;
            }
            return true;
        }).filter((s)=>s !== undefined);
        if (propNamesArray.length === 1) {
            result1.propertyNames = propNamesArray[0];
        } else if (propNamesArray.length > 1) {
            result1.propertyNames = anyOfSchema(...propNamesArray);
        } else {
            delete result1.propertyNames;
        }
        result1.closed = closed || baseSchema.some((s)=>s.closed);
    } else {
        const caseConventionSchema = createCaseConventionSchema(properties);
        if (caseConventionSchema !== undefined) {
            propertyNames = caseConventionSchema;
            tags = {
                ...tags,
                ...caseConventionSchema.tags
            };
            tagsAreSet = true;
        }
        result1 = {
            ...internalId(),
            "type": "object",
            description
        };
        if (exhaustive) {
            result1.exhaustiveCompletions = true;
        }
        if (properties) {
            result1.properties = properties;
        }
        if (patternProperties) {
            result1.patternProperties = patternProperties;
        }
        if (required && required.length > 0) {
            result1.required = required;
        }
        result1.closed = closed;
        if (additionalProperties !== undefined) {
            result1.additionalProperties = additionalProperties;
        }
        if (propertyNames !== undefined) {
            result1.propertyNames = propertyNames;
        }
    }
    if (tagsAreSet) {
        result1.tags = tags;
    }
    return result1;
}
function arraySchema(items) {
    if (items) {
        return {
            ...internalId(),
            "type": "array",
            "description": `be an array of values, where each element must ${schemaDescription(items)}`,
            items
        };
    } else {
        return {
            ...internalId(),
            "type": "array",
            "description": `be an array of values`
        };
    }
}
function documentSchema(schema, doc) {
    const result1 = Object.assign({}, schema);
    result1.documentation = doc;
    return result1;
}
function describeSchema(schema, description) {
    const result1 = Object.assign({}, schema);
    result1.description = `be ${description}`;
    return result1;
}
function completeSchema(schema, ...completions) {
    const result1 = Object.assign({}, schema);
    const prevCompletions = (schema.completions || []).slice();
    prevCompletions.push(...completions);
    result1.completions = prevCompletions;
    return result1;
}
function completeSchemaOverwrite(schema, ...completions) {
    const result1 = Object.assign({}, schema);
    result1.completions = completions;
    return result1;
}
function idSchema(schema, id) {
    const result1 = Object.assign({}, schema);
    result1["$id"] = id;
    return result1;
}
function errorMessageSchema(schema, errorMessage) {
    return {
        ...schema,
        errorMessage
    };
}
function refSchema($ref, description) {
    return {
        ...internalId(),
        "type": "ref",
        $ref,
        description
    };
}
function valueSchema(val, description) {
    return {
        ...internalId(),
        "type": "enum",
        "enum": [
            val
        ],
        "description": description || `be ${JSON.stringify(val)}`
    };
}
const definitionsObject = {};
function hasSchemaDefinition(key) {
    return definitionsObject[key] !== undefined;
}
function getSchemaDefinition(key) {
    if (definitionsObject[key] === undefined) {
        throw new InternalError(`Schema ${key} not found.`);
    }
    return definitionsObject[key];
}
function setSchemaDefinition(schema) {
    if (schema.$id === undefined) {
        throw new InternalError("setSchemaDefinition needs $id");
    }
    if (definitionsObject[schema.$id] === undefined) {
        definitionsObject[schema.$id] = schema;
    }
}
function getSchemaDefinitionsObject() {
    return Object.assign({}, definitionsObject);
}
function expandAliasesFrom(lst, defs) {
    const aliases = defs;
    const result1 = [];
    lst = lst.slice();
    for(let i1 = 0; i1 < lst.length; ++i1){
        const el = lst[i1];
        if (el.startsWith("$")) {
            const v = aliases[el.slice(1)];
            if (v === undefined) {
                throw new InternalError(`${el} doesn't have an entry in the aliases map`);
            }
            lst.push(...v);
        } else {
            result1.push(el);
        }
    }
    return result1;
}
function resolveSchema(schema, visit, hasRef, next) {
    if (schema === false || schema === true) {
        return schema;
    }
    if (hasRef === undefined) {
        hasRef = (cursor)=>{
            return schemaCall(cursor, {
                ref: (_s)=>true
            }, (_s)=>false);
        };
    }
    if (!hasRef(schema)) {
        return schema;
    }
    if (visit === undefined) {
        visit = (_schema)=>{};
    }
    if (next === undefined) {
        next = (cursor)=>{
            const result1 = schemaCall(cursor, {
                ref: (s)=>getSchemaDefinition(s.$ref)
            });
            if (result1 === undefined) {
                throw new InternalError("couldn't resolve schema ${JSON.stringify(cursor)}");
            }
            return result1;
        };
    }
    let cursor1 = schema;
    let cursor2 = schema;
    let stopped = false;
    do {
        cursor1 = next(cursor1);
        visit(cursor1);
        if (hasRef(cursor2)) {
            cursor2 = next(cursor2);
        } else {
            stopped = true;
        }
        if (hasRef(cursor2)) {
            cursor2 = next(cursor2);
        } else {
            stopped = true;
        }
        if (!stopped && cursor1 === cursor2) {
            throw new Error(`reference cycle detected at ${JSON.stringify(cursor1)}`);
        }
    }while (hasRef(cursor1))
    return cursor1;
}
function matchAll1(str, regex) {
    let match;
    regex = new RegExp(regex);
    const result1 = [];
    while((match = regex.exec(str)) != null){
        result1.push(match);
    }
    return result1;
}
function rangedLines(text, includeNewLines = false) {
    const regex = /\r?\n/g;
    const result1 = [];
    let startOffset = 0;
    if (!includeNewLines) {
        for (const r of matchAll1(text, regex)){
            result1.push({
                substring: text.substring(startOffset, r.index),
                range: {
                    start: startOffset,
                    end: r.index
                }
            });
            startOffset = r.index + r[0].length;
        }
        result1.push({
            substring: text.substring(startOffset, text.length),
            range: {
                start: startOffset,
                end: text.length
            }
        });
        return result1;
    } else {
        const matches = matchAll1(text, regex);
        let prevOffset = 0;
        for (const r of matches){
            const stringEnd = r.index + r[0].length;
            result1.push({
                substring: text.substring(prevOffset, stringEnd),
                range: {
                    start: prevOffset,
                    end: stringEnd
                }
            });
            prevOffset = stringEnd;
        }
        result1.push({
            substring: text.substring(prevOffset, text.length),
            range: {
                start: prevOffset,
                end: text.length
            }
        });
        return result1;
    }
}
function mappedSubstring(source, start, end) {
    if (typeof source === "string") {
        source = asMappedString(source);
    }
    const value = source.value.substring(start, end);
    const mappedSource = source;
    return {
        value,
        fileName: mappedSource.fileName,
        map: (index, closest)=>{
            if (closest) {
                index = Math.max(0, Math.min(value.length, index - 1));
            }
            if (index === 0 && index === value.length) {
                return mappedSource.map(index + start, closest);
            }
            if (index < 0 || index >= value.length) {
                return undefined;
            }
            return mappedSource.map(index + start, closest);
        }
    };
}
function mappedString(source, pieces, fileName) {
    if (typeof source === "string") {
        source = asMappedString(source, fileName);
    }
    const mappedPieces = pieces.map((piece)=>{
        if (typeof piece === "string") {
            return asMappedString(piece);
        } else if (piece.value !== undefined) {
            return piece;
        } else {
            const { start, end } = piece;
            return mappedSubstring(source, start, end);
        }
    });
    return mappedConcat(mappedPieces);
}
function asMappedString(str, fileName) {
    if (typeof str === "string") {
        return {
            value: str,
            fileName,
            map: function(index, closest) {
                if (closest) {
                    index = Math.min(str.length - 1, Math.max(0, index));
                }
                if (index < 0 || index >= str.length) {
                    return undefined;
                }
                return {
                    index,
                    originalString: this
                };
            }
        };
    } else if (fileName !== undefined) {
        throw new InternalError("can't change the fileName of an existing MappedString");
    } else {
        return str;
    }
}
function mappedConcat(strings) {
    if (strings.length === 0) {
        return {
            value: "",
            map: (_index, _closest)=>undefined
        };
    }
    if (strings.every((s)=>typeof s === "string")) {
        return asMappedString(strings.join(""));
    }
    const mappedStrings = strings.map((s)=>{
        if (typeof s === "string") {
            return asMappedString(s);
        } else return s;
    });
    let currentOffset = 0;
    const offsets = [
        0
    ];
    for (const s of mappedStrings){
        currentOffset += s.value.length;
        offsets.push(currentOffset);
    }
    const value = mappedStrings.map((s)=>s.value).join("");
    return {
        value,
        map: (offset, closest)=>{
            if (closest) {
                offset = Math.max(0, Math.min(offset, value.length - 1));
            }
            if (offset === 0 && offset == value.length && mappedStrings.length) {
                return mappedStrings[0].map(0, closest);
            }
            if (offset < 0 || offset >= value.length) {
                return undefined;
            }
            const ix = glb(offsets, offset);
            const v = mappedStrings[ix];
            return v.map(offset - offsets[ix]);
        }
    };
}
function mappedIndexToLineCol(eitherText) {
    const text = asMappedString(eitherText);
    return function(offset) {
        const mapResult = text.map(offset, true);
        if (mapResult === undefined) {
            throw new InternalError("bad offset in mappedIndexRowCol");
        }
        const { index, originalString } = mapResult;
        return indexToLineCol(originalString.value)(index);
    };
}
function join4(mappedStrs, sep) {
    const innerStrings = [];
    const mappedSep = asMappedString(sep);
    for(let i1 = 0; i1 < mappedStrs.length; ++i1){
        const mappedStr = mappedStrs[i1];
        if (typeof mappedStr === "string") {
            innerStrings.push(asMappedString(mappedStr));
        } else {
            innerStrings.push(mappedStr);
        }
        if (i1 < mappedStrs.length) {
            innerStrings.push(mappedSep);
        }
    }
    return mappedConcat(innerStrings);
}
function mappedLines(str, keepNewLines = false) {
    const lines = rangedLines(str.value, keepNewLines);
    return lines.map((v)=>mappedString(str, [
            v.range
        ]));
}
function findSpaceStart(string) {
    const result1 = string.value.match(/^\s+/);
    if (result1 === null || result1.length === 0) {
        return 0;
    }
    return result1[0].length;
}
function findSpaceEnd(string) {
    const result1 = string.value.match(/\s+$/);
    if (result1 === null || result1.length === 0) {
        return 0;
    }
    return result1[0].length;
}
function mappedTrim(string) {
    const start = findSpaceStart(string);
    const end = findSpaceEnd(string);
    if (start === 0 && end === 0) {
        return string;
    }
    if (start > string.value.length - end) {
        return mappedSubstring(string, 0, 0);
    }
    return mappedSubstring(string, start, string.value.length - end);
}
Object.freeze({
    __proto__: null
});
class DataSet {
    constructor(raw2018, raw2019, raw2020, raw2021){
        this._raw2018 = raw2018;
        this._raw2019 = raw2019;
        this._raw2020 = raw2020;
        this._raw2021 = raw2021;
    }
    get es2018() {
        return this._set2018 || (this._set2018 = new Set(this._raw2018.split(" ")));
    }
    get es2019() {
        return this._set2019 || (this._set2019 = new Set(this._raw2019.split(" ")));
    }
    get es2020() {
        return this._set2020 || (this._set2020 = new Set(this._raw2020.split(" ")));
    }
    get es2021() {
        return this._set2021 || (this._set2021 = new Set(this._raw2021.split(" ")));
    }
}
new Set([
    "General_Category",
    "gc"
]);
new Set([
    "Script",
    "Script_Extensions",
    "sc",
    "scx"
]);
new DataSet("C Cased_Letter Cc Cf Close_Punctuation Cn Co Combining_Mark Connector_Punctuation Control Cs Currency_Symbol Dash_Punctuation Decimal_Number Enclosing_Mark Final_Punctuation Format Initial_Punctuation L LC Letter Letter_Number Line_Separator Ll Lm Lo Lowercase_Letter Lt Lu M Mark Math_Symbol Mc Me Mn Modifier_Letter Modifier_Symbol N Nd Nl No Nonspacing_Mark Number Open_Punctuation Other Other_Letter Other_Number Other_Punctuation Other_Symbol P Paragraph_Separator Pc Pd Pe Pf Pi Po Private_Use Ps Punctuation S Sc Separator Sk Sm So Space_Separator Spacing_Mark Surrogate Symbol Titlecase_Letter Unassigned Uppercase_Letter Z Zl Zp Zs cntrl digit punct", "", "", "");
new DataSet("Adlam Adlm Aghb Ahom Anatolian_Hieroglyphs Arab Arabic Armenian Armi Armn Avestan Avst Bali Balinese Bamu Bamum Bass Bassa_Vah Batak Batk Beng Bengali Bhaiksuki Bhks Bopo Bopomofo Brah Brahmi Brai Braille Bugi Buginese Buhd Buhid Cakm Canadian_Aboriginal Cans Cari Carian Caucasian_Albanian Chakma Cham Cher Cherokee Common Copt Coptic Cprt Cuneiform Cypriot Cyrillic Cyrl Deseret Deva Devanagari Dsrt Dupl Duployan Egyp Egyptian_Hieroglyphs Elba Elbasan Ethi Ethiopic Geor Georgian Glag Glagolitic Gonm Goth Gothic Gran Grantha Greek Grek Gujarati Gujr Gurmukhi Guru Han Hang Hangul Hani Hano Hanunoo Hatr Hatran Hebr Hebrew Hira Hiragana Hluw Hmng Hung Imperial_Aramaic Inherited Inscriptional_Pahlavi Inscriptional_Parthian Ital Java Javanese Kaithi Kali Kana Kannada Katakana Kayah_Li Khar Kharoshthi Khmer Khmr Khoj Khojki Khudawadi Knda Kthi Lana Lao Laoo Latin Latn Lepc Lepcha Limb Limbu Lina Linb Linear_A Linear_B Lisu Lyci Lycian Lydi Lydian Mahajani Mahj Malayalam Mand Mandaic Mani Manichaean Marc Marchen Masaram_Gondi Meetei_Mayek Mend Mende_Kikakui Merc Mero Meroitic_Cursive Meroitic_Hieroglyphs Miao Mlym Modi Mong Mongolian Mro Mroo Mtei Mult Multani Myanmar Mymr Nabataean Narb Nbat New_Tai_Lue Newa Nko Nkoo Nshu Nushu Ogam Ogham Ol_Chiki Olck Old_Hungarian Old_Italic Old_North_Arabian Old_Permic Old_Persian Old_South_Arabian Old_Turkic Oriya Orkh Orya Osage Osge Osma Osmanya Pahawh_Hmong Palm Palmyrene Pau_Cin_Hau Pauc Perm Phag Phags_Pa Phli Phlp Phnx Phoenician Plrd Prti Psalter_Pahlavi Qaac Qaai Rejang Rjng Runic Runr Samaritan Samr Sarb Saur Saurashtra Sgnw Sharada Shavian Shaw Shrd Sidd Siddham SignWriting Sind Sinh Sinhala Sora Sora_Sompeng Soyo Soyombo Sund Sundanese Sylo Syloti_Nagri Syrc Syriac Tagalog Tagb Tagbanwa Tai_Le Tai_Tham Tai_Viet Takr Takri Tale Talu Tamil Taml Tang Tangut Tavt Telu Telugu Tfng Tglg Thaa Thaana Thai Tibetan Tibt Tifinagh Tirh Tirhuta Ugar Ugaritic Vai Vaii Wara Warang_Citi Xpeo Xsux Yi Yiii Zanabazar_Square Zanb Zinh Zyyy", "Dogr Dogra Gong Gunjala_Gondi Hanifi_Rohingya Maka Makasar Medefaidrin Medf Old_Sogdian Rohg Sogd Sogdian Sogo", "Elym Elymaic Hmnp Nand Nandinagari Nyiakeng_Puachue_Hmong Wancho Wcho", "Chorasmian Chrs Diak Dives_Akuru Khitan_Small_Script Kits Yezi Yezidi");
new DataSet("AHex ASCII ASCII_Hex_Digit Alpha Alphabetic Any Assigned Bidi_C Bidi_Control Bidi_M Bidi_Mirrored CI CWCF CWCM CWKCF CWL CWT CWU Case_Ignorable Cased Changes_When_Casefolded Changes_When_Casemapped Changes_When_Lowercased Changes_When_NFKC_Casefolded Changes_When_Titlecased Changes_When_Uppercased DI Dash Default_Ignorable_Code_Point Dep Deprecated Dia Diacritic Emoji Emoji_Component Emoji_Modifier Emoji_Modifier_Base Emoji_Presentation Ext Extender Gr_Base Gr_Ext Grapheme_Base Grapheme_Extend Hex Hex_Digit IDC IDS IDSB IDST IDS_Binary_Operator IDS_Trinary_Operator ID_Continue ID_Start Ideo Ideographic Join_C Join_Control LOE Logical_Order_Exception Lower Lowercase Math NChar Noncharacter_Code_Point Pat_Syn Pat_WS Pattern_Syntax Pattern_White_Space QMark Quotation_Mark RI Radical Regional_Indicator SD STerm Sentence_Terminal Soft_Dotted Term Terminal_Punctuation UIdeo Unified_Ideograph Upper Uppercase VS Variation_Selector White_Space XIDC XIDS XID_Continue XID_Start space", "Extended_Pictographic", "", "EBase EComp EMod EPres ExtPict");
function navigateSchemaBySchemaPathSingle(schema, path) {
    const ensurePathFragment = (fragment, expected)=>{
        if (fragment !== expected) {
            throw new InternalError(`navigateSchemaBySchemaPathSingle: ${fragment} !== ${expected}`);
        }
    };
    const inner = (subschema, index)=>{
        subschema = resolveSchema(subschema);
        if (subschema === undefined) {
            throw new InternalError(`navigateSchemaBySchemaPathSingle: invalid path navigation`);
        }
        if (index === path.length) {
            return subschema;
        }
        const st = schemaType(subschema);
        switch(st){
            case "anyOf":
                ensurePathFragment(path[index], "anyOf");
                return inner(subschema.anyOf[path[index + 1]], index + 2);
            case "allOf":
                ensurePathFragment(path[index], "allOf");
                return inner(subschema.allOf[path[index + 1]], index + 2);
            case "array":
                ensurePathFragment(path[index], "array");
                return inner(subschema.arrayOf.schema, index + 2);
            case "object":
                ensurePathFragment(path[index], "object");
                if (path[index + 1] === "properties") {
                    return inner(subschema.properties[path[index + 2]], index + 3);
                } else if (path[index + 1] === "patternProperties") {
                    return inner(subschema.patternProperties[path[index + 2]], index + 3);
                } else if (path[index + 1] === "additionalProperties") {
                    return inner(subschema.additionalProperties, index + 2);
                } else {
                    throw new InternalError(`navigateSchemaBySchemaPathSingle: bad path fragment ${path[index]} in object navigation`);
                }
            default:
                throw new InternalError(`navigateSchemaBySchemaPathSingle: can't navigate schema type ${st}`);
        }
    };
    return inner(schema, 0);
}
function resolveDescription(s) {
    if (typeof s === "string") {
        return s;
    }
    const valueS = resolveSchema(s);
    if (valueS === false || valueS === true) {
        return "";
    }
    if (valueS.documentation === undefined) {
        return "";
    }
    if (typeof valueS.documentation === "string") {
        return valueS.documentation;
    }
    if (valueS.documentation.short) {
        return valueS.documentation.short;
    } else {
        return "";
    }
}
function schemaCompletions(s) {
    if (s === true || s === false) {
        return [];
    }
    let schema = resolveSchema(s);
    schema = resolveSchema(schema, (_schema)=>{}, (schema)=>{
        return schema.tags !== undefined && schema.tags["complete-from"] !== undefined;
    }, (schema)=>{
        return navigateSchemaBySchemaPathSingle(schema, schema.tags["complete-from"]);
    });
    if (schema === true || schema === false) {
        return [];
    }
    const normalize = (completions)=>{
        const result1 = (completions || []).map((c)=>{
            if (typeof c === "string") {
                return {
                    type: "value",
                    display: c,
                    value: c,
                    description: "",
                    suggest_on_accept: false,
                    schema
                };
            }
            return {
                ...c,
                description: resolveDescription(c.description),
                schema
            };
        });
        return result1;
    };
    if (schema.completions && schema.completions.length) {
        return normalize(schema.completions);
    }
    if (schema.tags && schema.tags.completions) {
        if (Array.isArray(schema.tags.completions) && schema.tags.completions.length) {
            return normalize(schema.tags.completions);
        } else {
            return normalize(Object.values(schema.tags.completions));
        }
    }
    return schemaCall(schema, {
        array: (s)=>{
            if (s.items) {
                return schemaCompletions(s.items);
            } else {
                return [];
            }
        },
        anyOf: (s)=>{
            return s.anyOf.map(schemaCompletions).flat();
        },
        allOf: (s)=>{
            return s.allOf.map(schemaCompletions).flat();
        },
        "object": (s)=>{
            s.cachedCompletions = getObjectCompletions(s);
            return normalize(s.cachedCompletions);
        }
    }, (_)=>[]);
}
function getObjectCompletions(s) {
    const completionsParam = s.tags && s.tags.completions || [];
    return schemaCall(s, {
        "object": (schema)=>{
            const properties = schema.properties;
            const objectKeys = completionsParam.length ? completionsParam : Object.getOwnPropertyNames(properties);
            const completions = [];
            for (const k of objectKeys){
                const schema = properties && properties[k];
                const maybeDescriptions = [];
                let hidden = false;
                if (schema !== undefined && schema !== true && schema !== false) {
                    if (schema.documentation) {
                        maybeDescriptions.push(schemaDocString(schema.documentation));
                    } else {
                        let described = false;
                        const visitor = (schema)=>{
                            if (schema === false || schema === true) {
                                return;
                            }
                            if (schema.hidden) {
                                hidden = true;
                            }
                            if (described) {
                                return;
                            }
                            if (schema.documentation) {
                                maybeDescriptions.push(schemaDocString(schema.documentation));
                                described = true;
                            }
                        };
                        try {
                            resolveSchema(schema, visitor);
                        } catch (_e) {}
                        if (!described) {
                            schemaDispatch(schema, {
                                ref: (schema)=>maybeDescriptions.push({
                                        $ref: schema.$ref
                                    })
                            });
                        }
                    }
                }
                if (hidden) {
                    continue;
                }
                let description = "";
                for (const md of maybeDescriptions){
                    if (md !== undefined) {
                        description = md;
                        break;
                    }
                }
                completions.push({
                    type: "key",
                    display: "",
                    value: `${k}: `,
                    description,
                    suggest_on_accept: true
                });
            }
            return completions;
        }
    }, (_)=>completionsParam.map((c)=>({
                type: "value",
                display: "",
                value: c,
                description: "",
                suggest_on_accept: false
            })));
}
function possibleSchemaKeys(schema) {
    const precomputedCompletions = schemaCompletions(schema).filter((c)=>c.type === "key").map((c)=>c.value.split(":")[0]);
    if (precomputedCompletions.length) {
        return precomputedCompletions;
    }
    const results = [];
    walkSchema(schema, {
        "object": (s)=>{
            results.push(...Object.keys(s.properties || {}));
            return true;
        },
        "array": (_s)=>true
    });
    return results;
}
function possibleSchemaValues(schema) {
    const results = [];
    walkSchema(schema, {
        "enum": (s)=>{
            results.push(...s["enum"].map(String));
            return true;
        },
        "array": (_s)=>true,
        "object": (_s)=>true
    });
    return results;
}
function walkSchema(schema, f) {
    const recur = {
        "anyOf": (ss)=>{
            for (const s of ss.anyOf){
                walkSchema(s, f);
            }
        },
        "allOf": (ss)=>{
            for (const s of ss.allOf){
                walkSchema(s, f);
            }
        },
        "array": (x)=>{
            if (x.items) {
                walkSchema(x.items, f);
            }
        },
        "object": (x)=>{
            if (x.properties) {
                for (const ss of Object.values(x.properties)){
                    walkSchema(ss, f);
                }
            }
            if (x.patternProperties) {
                for (const ss of Object.values(x.patternProperties)){
                    walkSchema(ss, f);
                }
            }
            if (x.propertyNames) {
                walkSchema(x.propertyNames, f);
            }
        }
    };
    if (typeof f === "function") {
        if (f(schema) === true) {
            return;
        }
    } else {
        if (schemaCall(schema, f, (_)=>false) === true) {
            return;
        }
    }
    schemaCall(schema, recur, (_)=>false);
}
function setDefaultErrorHandlers(validator) {
    validator.addHandler(ignoreExprViolations);
    validator.addHandler(expandEmptySpan);
    validator.addHandler(improveErrorHeadingForValueErrors);
    validator.addHandler(checkForTypeMismatch);
    validator.addHandler(checkForBadBoolean);
    validator.addHandler(checkForBadColon);
    validator.addHandler(checkForBadEquals);
    validator.addHandler(identifyKeyErrors);
    validator.addHandler(checkForNearbyCorrection);
    validator.addHandler(checkForNearbyRequired);
    validator.addHandler(schemaDefinedErrors);
}
function errorKeyword(error) {
    if (error.schemaPath.length === 0) {
        return "";
    }
    return String(error.schemaPath[error.schemaPath.length - 1]);
}
function schemaPathMatches(error, strs) {
    const schemaPath = error.schemaPath.slice(-strs.length);
    if (schemaPath.length !== strs.length) {
        return false;
    }
    return strs.every((str, i1)=>str === schemaPath[i1]);
}
function getBadKey(error) {
    if (error.schemaPath.indexOf("propertyNames") === -1 && error.schemaPath.indexOf("closed") === -1) {
        return undefined;
    }
    const result1 = error.violatingObject.result;
    if (typeof result1 !== "string") {
        throw new InternalError("propertyNames error has a violating non-string.");
    }
    return result1;
}
function getVerbatimInput(error) {
    return error.source.value;
}
function navigate(path, annotation, returnKey = false, pathIndex = 0) {
    if (annotation === undefined) {
        throw new Error("Can't navigate an undefined annotation");
    }
    if (pathIndex >= path.length) {
        return annotation;
    }
    if (annotation.kind === "mapping" || annotation.kind === "block_mapping") {
        const { components } = annotation;
        const searchKey = path[pathIndex];
        const lastKeyIndex = ~~((components.length - 1) / 2) * 2;
        for(let i1 = lastKeyIndex; i1 >= 0; i1 -= 2){
            const key = components[i1].result;
            if (key === searchKey) {
                if (returnKey && pathIndex === path.length - 1) {
                    return navigate(path, components[i1], returnKey, pathIndex + 1);
                } else {
                    return navigate(path, components[i1 + 1], returnKey, pathIndex + 1);
                }
            }
        }
        return annotation;
    } else if ([
        "sequence",
        "block_sequence",
        "flow_sequence"
    ].indexOf(annotation.kind) !== -1) {
        const searchKey = Number(path[pathIndex]);
        if (isNaN(searchKey) || searchKey < 0 || searchKey >= annotation.components.length) {
            return annotation;
        }
        return navigate(path, annotation.components[searchKey], returnKey, pathIndex + 1);
    } else {
        return annotation;
    }
}
function isEmptyValue(error) {
    const rawVerbatimInput = getVerbatimInput(error);
    return rawVerbatimInput.trim().length === 0;
}
function getLastFragment(instancePath) {
    if (instancePath.length === 0) {
        return undefined;
    }
    return instancePath[instancePath.length - 1];
}
function reindent(str) {
    const s = new Set();
    const ls = lines(str);
    for (const l1 of ls){
        const r = l1.match("^[ ]+");
        if (r) {
            s.add(r[0].length);
        }
    }
    if (s.size === 0) {
        return str;
    } else if (s.size === 1) {
        const v = Array.from(s)[0];
        const oldIndent = " ".repeat(v);
        if (v <= 2) {
            return str;
        }
        return ls.map((l1)=>l1.startsWith(oldIndent) ? l1.slice(v - 2) : l1).join("\n");
    } else {
        const [first, second] = Array.from(s);
        const oldIndent = " ".repeat(first);
        const newIndent = second - first;
        if (newIndent >= first) {
            return str;
        }
        return ls.map((l1)=>l1.startsWith(oldIndent) ? l1.slice(first - newIndent) : l1).join("\n");
    }
}
function ignoreExprViolations(error, _parse, _schema) {
    const { result: result1 } = error.violatingObject;
    if (typeof result1 !== "object" || Array.isArray(result1) || result1 === null || error.schemaPath.slice(-1)[0] !== "type") {
        return error;
    }
    if (result1.tag === "!expr" && typeof result1.value === "string") {
        return null;
    } else {
        return error;
    }
}
function formatHeadingForKeyError(_error, _parse, _schema, key) {
    return `property name ${blue3(key)} is invalid`;
}
function formatHeadingForValueError(error, _parse, _schema) {
    const rawVerbatimInput = reindent(getVerbatimInput(error));
    const rawLines = lines(rawVerbatimInput);
    let verbatimInput;
    if (rawLines.length > 4) {
        verbatimInput = quotedStringColor([
            ...rawLines.slice(0, 2),
            "...",
            ...rawLines.slice(-2)
        ].join("\n"));
    } else {
        verbatimInput = quotedStringColor(rawVerbatimInput);
    }
    const empty = isEmptyValue(error);
    const lastFragment = getLastFragment(error.instancePath);
    switch(typeof lastFragment){
        case "undefined":
            if (empty) {
                return "YAML value is missing.";
            } else {
                return `YAML value ${verbatimInput} must ${schemaDescription(error.schema)}.`;
            }
        case "number":
            if (empty) {
                return `Array entry ${lastFragment + 1} is empty but it must instead ${schemaDescription(error.schema)}.`;
            } else {
                return `Array entry ${lastFragment + 1} with value ${verbatimInput} failed to ${schemaDescription(error.schema)}.`;
            }
        case "string":
            {
                const formatLastFragment = '"' + blue3(lastFragment) + '"';
                if (empty) {
                    return `Field ${formatLastFragment} has empty value but it must instead ${schemaDescription(error.schema)}`;
                } else {
                    if (verbatimInput.indexOf("\n") !== -1) {
                        return `Field ${formatLastFragment} has value

${verbatimInput}

The value must instead ${schemaDescription(error.schema)}.`;
                    } else {
                        return `Field ${formatLastFragment} has value ${verbatimInput}, which must instead ${schemaDescription(error.schema)}`;
                    }
                }
            }
    }
}
function identifyKeyErrors(error, parse, schema) {
    if (error.schemaPath.indexOf("propertyNames") === -1 && error.schemaPath.indexOf("closed") === -1) {
        return error;
    }
    const badKey = getBadKey(error);
    if (badKey) {
        if (error.instancePath.length && error.instancePath[error.instancePath.length - 1] !== badKey) {
            addInstancePathInfo(error.niceError, [
                ...error.instancePath,
                badKey
            ]);
        } else {
            addInstancePathInfo(error.niceError, error.instancePath);
        }
        error.niceError.heading = formatHeadingForKeyError(error, parse, schema, badKey);
    }
    return error;
}
function improveErrorHeadingForValueErrors(error, parse, schema) {
    if (error.schemaPath.indexOf("propertyNames") !== -1 || error.schemaPath.indexOf("closed") !== -1 || errorKeyword(error) === "required") {
        return error;
    }
    return {
        ...error,
        niceError: {
            ...error.niceError,
            heading: formatHeadingForValueError(error, parse, schema)
        }
    };
}
function expandEmptySpan(error, parse, _schema) {
    if (error.location.start.line !== error.location.end.line || error.location.start.column !== error.location.end.column || !isEmptyValue(error) || typeof getLastFragment(error.instancePath) === "undefined") {
        return error;
    }
    const lastKey = navigate(error.instancePath, parse, true);
    const locF = mappedIndexToLineCol(parse.source);
    try {
        const location = {
            start: locF(lastKey.start),
            end: locF(lastKey.end)
        };
        return {
            ...error,
            location,
            niceError: {
                ...error.niceError,
                location
            }
        };
    } catch (_e) {
        return error;
    }
}
function checkForTypeMismatch(error, parse, schema) {
    const rawVerbatimInput = getVerbatimInput(error);
    const rawLines = lines(rawVerbatimInput);
    let verbatimInput;
    if (rawLines.length > 4) {
        verbatimInput = quotedStringColor([
            ...rawLines.slice(0, 2),
            "...",
            ...rawLines.slice(-2)
        ].join("\n"));
    } else {
        verbatimInput = quotedStringColor(rawVerbatimInput);
    }
    const goodType = (obj)=>{
        if (Array.isArray(obj)) {
            return "an array";
        }
        if (obj === null) {
            return "a null value";
        }
        return typeof obj;
    };
    if (errorKeyword(error) === "type" && rawVerbatimInput.length > 0) {
        const reindented = reindent(verbatimInput);
        const subject = reindented.indexOf("\n") === -1 ? `The value ${reindented} ` : `The value

${reindented}

`;
        const newError = {
            ...error.niceError,
            heading: formatHeadingForValueError(error, parse, schema),
            error: [
                `${subject}is of type ${goodType(error.violatingObject.result)}.`
            ],
            info: {},
            location: error.niceError.location
        };
        addInstancePathInfo(newError, error.instancePath);
        addFileInfo(newError, error.source);
        return {
            ...error,
            niceError: newError
        };
    }
    return error;
}
function checkForBadBoolean(error, parse, _schema) {
    const schema = error.schema;
    if (!(typeof error.violatingObject.result === "string" && errorKeyword(error) === "type" && schemaType(schema) === "boolean")) {
        return error;
    }
    const strValue = error.violatingObject.result;
    const verbatimInput = quotedStringColor(getVerbatimInput(error));
    const yesses = new Set("y|Y|yes|Yes|YES|true|True|TRUE|on|On|ON".split("|"));
    const nos = new Set("n|N|no|No|NO|false|False|FALSE|off|Off|OFF".split("|"));
    let fix;
    if (yesses.has(strValue)) {
        fix = true;
    } else if (nos.has(strValue)) {
        fix = false;
    } else {
        return error;
    }
    const errorMessage = `The value ${verbatimInput} is a string.`;
    const suggestion1 = `Quarto uses YAML 1.2, which interprets booleans strictly.`;
    const suggestion2 = `Try using ${quotedStringColor(String(fix))} instead.`;
    const newError = {
        heading: formatHeadingForValueError(error, parse, schema),
        error: [
            errorMessage
        ],
        info: {},
        location: error.niceError.location
    };
    addInstancePathInfo(newError, error.instancePath);
    addFileInfo(newError, error.source);
    newError.info["yaml-version-1.2"] = suggestion1;
    newError.info["suggestion-fix"] = suggestion2;
    return {
        ...error,
        niceError: newError
    };
}
function checkForBadColon(error, parse, schema) {
    if (typeof error.violatingObject.result !== "string") {
        return error;
    }
    if (!schemaPathMatches(error, [
        "object",
        "type"
    ])) {
        return error;
    }
    if (!error.violatingObject.result.match(/^.+:[^ ].*$/)) {
        return error;
    }
    const verbatimInput = quotedStringColor(getVerbatimInput(error));
    const errorMessage = `The value ${verbatimInput} is a string.`;
    const suggestion1 = `In YAML, key-value pairs in objects must be separated by a space.`;
    const suggestion2 = `Did you mean ${quotedStringColor(quotedStringColor(getVerbatimInput(error)).replace(/:/g, ": "))} instead?`;
    const newError = {
        heading: formatHeadingForValueError(error, parse, schema),
        error: [
            errorMessage
        ],
        info: {},
        location: error.niceError.location
    };
    addInstancePathInfo(newError, error.instancePath);
    addFileInfo(newError, error.source);
    newError.info["yaml-key-value-pairs"] = suggestion1;
    newError.info["suggestion-fix"] = suggestion2;
    return {
        ...error,
        niceError: newError
    };
}
function checkForBadEquals(error, parse, schema) {
    if (typeof error.violatingObject.result !== "string") {
        return error;
    }
    if (!schemaPathMatches(error, [
        "object",
        "type"
    ]) && !schemaPathMatches(error, [
        "object",
        "propertyNames",
        "string",
        "pattern"
    ])) {
        return error;
    }
    if (!error.violatingObject.result.match(/^.+ *= *.+$/)) {
        return error;
    }
    const verbatimInput = quotedStringColor(getVerbatimInput(error));
    const errorMessage = `The value ${verbatimInput} is a string.`;
    const suggestion1 = `In YAML, key-value pairs in objects must be separated by a colon and a space.`;
    const suggestion2 = `Did you mean ${quotedStringColor(quotedStringColor(getVerbatimInput(error)).replace(/ *= */g, ": "))} instead?`;
    const newError = {
        heading: formatHeadingForValueError(error, parse, schema),
        error: [
            errorMessage
        ],
        info: {},
        location: error.niceError.location
    };
    addInstancePathInfo(newError, error.instancePath);
    addFileInfo(newError, error.source);
    newError.info["yaml-key-value-pairs"] = suggestion1;
    newError.info["suggestion-fix"] = suggestion2;
    return {
        ...error,
        niceError: newError
    };
}
function createErrorFragments(error) {
    const rawVerbatimInput = getVerbatimInput(error);
    const verbatimInput = quotedStringColor(reindent(rawVerbatimInput));
    const pathFragments = error.instancePath.map((s)=>blue3(String(s)));
    return {
        location: locationString(error.location),
        fullPath: pathFragments.join(":"),
        key: pathFragments[pathFragments.length - 1],
        value: verbatimInput
    };
}
function schemaDefinedErrors(error, _parse, _schema) {
    const schema = error.schema;
    if (schema === true || schema === false) {
        return error;
    }
    if (schema.errorMessage === undefined) {
        return error;
    }
    if (typeof schema.errorMessage !== "string") {
        return error;
    }
    let result1 = schema.errorMessage;
    for (const [k, v] of Object.entries(createErrorFragments(error))){
        result1 = result1.replace("${" + k + "}", v);
    }
    return {
        ...error,
        niceError: {
            ...error.niceError,
            heading: result1
        }
    };
}
function checkForNearbyRequired(error, _parse, _schema) {
    const schema = error.schema;
    if (errorKeyword(error) !== "required") {
        return error;
    }
    const missingKeys = [];
    const errObj = error.violatingObject.result;
    const keys = Object.keys(errObj);
    schemaCall(schema, {
        object (s) {
            if (s.required === undefined) {
                throw new InternalError("required schema error without a required field");
            }
            for (const r of s.required){
                if (keys.indexOf(r) === -1) {
                    missingKeys.push(r);
                }
            }
        }
    }, (_)=>{
        throw new InternalError("required error on a non-object schema");
    });
    for (const missingKey of missingKeys){
        let bestCorrection;
        let bestDistance = Infinity;
        for (const correction of keys){
            const d = editDistance(correction, missingKey);
            if (d < bestDistance) {
                bestCorrection = [
                    correction
                ];
                bestDistance = d;
            } else if (d === bestDistance) {
                bestCorrection.push(correction);
                bestDistance = d;
            }
        }
        if (bestDistance > missingKey.length * 10 * 0.3) {
            continue;
        }
        const suggestions = bestCorrection.map((s)=>blue3(s));
        if (suggestions.length === 1) {
            error.niceError.info[`did-you-mean-key`] = `Is ${suggestions[0]} a typo of ${blue3(missingKey)}?`;
        } else if (suggestions.length === 2) {
            error.niceError.info[`did-you-mean-key`] = `Is ${suggestions[0]} or ${suggestions[1]} a typo of ${blue3(missingKey)}?`;
        } else {
            suggestions[suggestions.length - 1] = `or ${suggestions[suggestions.length - 1]}`;
            error.niceError.info[`did-you-mean-key`] = `Is one of ${suggestions.join(", ")} a typo of ${blue3(missingKey)}?`;
        }
    }
    return error;
}
function checkForNearbyCorrection(error, parse, _schema) {
    const schema = error.schema;
    const corrections = [];
    let errVal = "";
    let keyOrValue = "";
    const key = getBadKey(error);
    if (key) {
        errVal = key;
        corrections.push(...possibleSchemaKeys(schema));
        keyOrValue = "key";
    } else {
        const val = navigate(error.instancePath, parse);
        if (typeof val.result !== "string") {
            return error;
        }
        errVal = val.result;
        corrections.push(...possibleSchemaValues(schema));
        keyOrValue = "value";
    }
    if (corrections.length === 0) {
        return error;
    }
    let bestCorrection;
    let bestDistance = Infinity;
    for (const correction of corrections){
        const d = editDistance(correction, errVal);
        if (d < bestDistance) {
            bestCorrection = [
                correction
            ];
            bestDistance = d;
        } else if (d === bestDistance) {
            bestCorrection.push(correction);
            bestDistance = d;
        }
    }
    if (bestDistance > errVal.length * 10 * 0.3) {
        return error;
    }
    const suggestions = bestCorrection.map((s)=>blue3(s));
    if (suggestions.length === 1) {
        error.niceError.info[`did-you-mean-${keyOrValue}`] = `Did you mean ${suggestions[0]}?`;
    } else if (suggestions.length === 2) {
        error.niceError.info[`did-you-mean-${keyOrValue}`] = `Did you mean ${suggestions[0]} or ${suggestions[1]}?`;
    } else {
        suggestions[suggestions.length - 1] = `or ${suggestions[suggestions.length - 1]}`;
        error.niceError.info[`did-you-mean-${keyOrValue}`] = `Did you mean ${suggestions.join(", ")}?`;
    }
    return error;
}
function createSourceContext(src, location) {
    if (src.value.length === 0) {
        return "";
    }
    const startMapResult = src.map(location.start, true);
    const endMapResult = src.map(location.end, true);
    const locF = mappedIndexToLineCol(src);
    let sourceLocation;
    try {
        sourceLocation = {
            start: locF(location.start),
            end: locF(location.end)
        };
    } catch (_e) {
        sourceLocation = {
            start: {
                line: 0,
                column: 0
            },
            end: {
                line: 0,
                column: 0
            }
        };
    }
    if (startMapResult === undefined || endMapResult === undefined) {
        throw new InternalError(`createSourceContext called with bad location ${location.start}-${location.end}.`);
    }
    if (startMapResult.originalString !== endMapResult.originalString) {
        throw new InternalError("don't know how to create source context across different source files");
    }
    const originalString = startMapResult.originalString;
    const nLines = lines(originalString.value).length;
    const { start, end } = sourceLocation;
    const { prefixWidth, lines: formattedLines } = formatLineRange(originalString.value, Math.max(0, start.line - 1), Math.min(end.line + 1, nLines - 1));
    const contextLines = [];
    let mustPrintEllipsis = true;
    for (const { lineNumber, content, rawLine } of formattedLines){
        if (lineNumber < start.line || lineNumber > end.line) {
            if (rawLine.trim().length) {
                contextLines.push(content);
            }
        } else {
            if (lineNumber >= start.line + 2 && lineNumber <= end.line - 2) {
                if (mustPrintEllipsis) {
                    mustPrintEllipsis = false;
                    contextLines.push("...");
                }
            } else {
                const startColumn = lineNumber > start.line ? 0 : start.column;
                const endColumn = lineNumber < end.line ? rawLine.length : end.column;
                contextLines.push(content);
                contextLines.push(" ".repeat(prefixWidth + startColumn - 1) + "~".repeat(endColumn - startColumn + 1));
            }
        }
    }
    return contextLines.join("\n");
}
function createLocalizedError(obj) {
    const { violatingObject, instancePath, schemaPath, source, message, schema } = obj;
    const locF = mappedIndexToLineCol(source);
    let location;
    try {
        location = {
            start: locF(violatingObject.start),
            end: locF(violatingObject.end)
        };
    } catch (_e) {
        location = {
            start: {
                line: 0,
                column: 0
            },
            end: {
                line: 0,
                column: 0
            }
        };
    }
    const mapResult = source.map(violatingObject.start);
    const fileName = mapResult ? mapResult.originalString.fileName : undefined;
    return {
        source: mappedString(source, [
            {
                start: violatingObject.start,
                end: violatingObject.end
            }
        ]),
        violatingObject: violatingObject,
        instancePath,
        schemaPath,
        schema,
        message,
        location: location,
        niceError: {
            heading: message,
            error: [],
            info: {},
            fileName,
            location: location,
            sourceContext: createSourceContext(violatingObject.source, {
                start: violatingObject.start,
                end: violatingObject.end
            })
        }
    };
}
class ValidationContext {
    instancePath;
    root;
    nodeStack;
    currentNode;
    constructor(){
        this.instancePath = [];
        this.currentNode = {
            edge: "#",
            errors: [],
            children: []
        };
        this.nodeStack = [
            this.currentNode
        ];
        this.root = this.currentNode;
    }
    error(value, schema, message) {
        this.currentNode.errors.push({
            value,
            schema,
            message,
            instancePath: this.instancePath.slice(),
            schemaPath: this.nodeStack.map((node)=>node.edge)
        });
    }
    pushSchema(schemaPath) {
        const newNode = {
            edge: schemaPath,
            errors: [],
            children: []
        };
        this.currentNode.children.push(newNode);
        this.currentNode = newNode;
        this.nodeStack.push(newNode);
    }
    popSchema(success) {
        this.nodeStack.pop();
        this.currentNode = this.nodeStack[this.nodeStack.length - 1];
        if (success) {
            this.currentNode.children.pop();
        }
        return success;
    }
    pushInstance(instance) {
        this.instancePath.push(instance);
    }
    popInstance() {
        this.instancePath.pop();
    }
    withSchemaPath(schemaPath, chunk) {
        this.pushSchema(schemaPath);
        return this.popSchema(chunk());
    }
    validate(schema, source, value, pruneErrors = true) {
        if (validateGeneric(value, schema, this)) {
            return [];
        }
        return this.collectErrors(schema, source, value, pruneErrors);
    }
    collectErrors(_schema, source, _value, pruneErrors = true) {
        const inner = (node)=>{
            const result1 = [];
            if (node.edge === "anyOf" && pruneErrors) {
                const innerResults = node.children.map(inner);
                const isRequiredError = (e)=>e.schemaPath.indexOf("required") === e.schemaPath.length - 1;
                const isPropertyNamesError = (e)=>e.schemaPath.indexOf("propertyNames") !== -1;
                if (innerResults.some((el)=>el.length && isRequiredError(el[0])) && innerResults.some((el)=>el.length && isPropertyNamesError(el[0]))) {
                    return innerResults.filter((r)=>{
                        return r.length && r[0].schemaPath.slice(-1)[0] === "required";
                    })[0];
                }
                const errorTypeQuality = (e)=>{
                    const t = e.schemaPath.slice().reverse();
                    if (typeof e.schema === "object") {
                        if (e.schema.tags && e.schema.tags["error-importance"] && typeof e.schema.tags["error-importance"] === "number") {
                            return e.schema.tags["error-importance"];
                        }
                    }
                    if (e.schemaPath.indexOf("propertyNames") !== -1) {
                        return 10;
                    }
                    if (t[0] === "required") {
                        return 0;
                    }
                    if (t[0] === "type") {
                        if (t[1] === "null") {
                            return 10;
                        }
                        return 1;
                    }
                    return 1;
                };
                const errorComparator = (a, b)=>{
                    for(let i1 = 0; i1 < a.length; ++i1){
                        if (a[i1] < b[i1]) {
                            return -1;
                        }
                        if (a[i1] > b[i1]) {
                            return 1;
                        }
                    }
                    return 0;
                };
                let bestResults = [];
                let bestError = [
                    Infinity,
                    Infinity
                ];
                for (const resultGroup of innerResults){
                    let maxQuality = -Infinity;
                    let totalSpan = 0;
                    for (const result1 of resultGroup){
                        totalSpan += result1.value.end - result1.value.start;
                        maxQuality = Math.max(maxQuality, errorTypeQuality(result1));
                    }
                    const thisError = [
                        maxQuality,
                        totalSpan
                    ];
                    if (errorComparator(thisError, bestError) === -1) {
                        bestError = thisError;
                        bestResults = resultGroup;
                    }
                }
                return bestResults;
            } else {
                result1.push(...node.errors);
                for (const child of node.children){
                    result1.push(...inner(child));
                }
                return result1;
            }
        };
        const errors = inner(this.root);
        const result1 = errors.map((validationError)=>createLocalizedError({
                violatingObject: validationError.value,
                instancePath: validationError.instancePath,
                schemaPath: validationError.schemaPath,
                schema: validationError.schema,
                message: validationError.message,
                source
            }));
        return result1;
    }
}
function validateGeneric(value, s, context) {
    s = resolveSchema(s);
    const st = schemaType(s);
    return context.withSchemaPath(st, ()=>schemaCall(s, {
            "false": (schema)=>{
                context.error(value, schema, "false");
                return false;
            },
            "true": (_)=>true,
            "any": (schema)=>validateAny(value, schema, context),
            "boolean": (schema)=>validateBoolean(value, schema, context),
            "number": (schema)=>validateNumber(value, schema, context),
            "string": (schema)=>validateString(value, schema, context),
            "null": (schema)=>validateNull(value, schema, context),
            "enum": (schema)=>validateEnum(value, schema, context),
            "anyOf": (schema)=>validateAnyOf(value, schema, context),
            "allOf": (schema)=>validateAllOf(value, schema, context),
            "array": (schema)=>validateArray(value, schema, context),
            "object": (schema)=>validateObject(value, schema, context),
            "ref": (schema)=>validateGeneric(value, resolveSchema(schema), context)
        }));
}
function typeIsValid(value, schema, context, valid) {
    if (!valid) {
        return context.withSchemaPath("type", ()=>{
            context.error(value, schema, "type mismatch");
            return false;
        });
    }
    return valid;
}
function validateAny(_value, _schema, _context) {
    return true;
}
function validateBoolean(value, schema, context) {
    return typeIsValid(value, schema, context, typeof value.result === "boolean");
}
function validateNumber(value, schema, context) {
    if (!typeIsValid(value, schema, context, typeof value.result === "number")) {
        return false;
    }
    let result1 = true;
    if (schema.minimum !== undefined) {
        result1 = context.withSchemaPath("minimum", ()=>{
            const v = value.result;
            if (!(v >= schema.minimum)) {
                context.error(value, schema, `value ${value.result} is less than required minimum ${schema.minimum}`);
                return false;
            }
            return true;
        });
    }
    if (schema.maximum !== undefined) {
        result1 = context.withSchemaPath("maximum", ()=>{
            const v = value.result;
            if (!(v <= schema.maximum)) {
                context.error(value, schema, `value ${value.result} is greater than required maximum ${schema.maximum}`);
                return false;
            }
            return true;
        });
    }
    if (schema.exclusiveMinimum !== undefined) {
        result1 = context.withSchemaPath("exclusiveMinimum", ()=>{
            const v = value.result;
            if (!(v > schema.exclusiveMinimum)) {
                context.error(value, schema, `value ${value.result} is less than or equal to required (exclusive) minimum ${schema.exclusiveMinimum}`);
                return false;
            }
            return true;
        });
    }
    if (schema.exclusiveMaximum !== undefined) {
        result1 = context.withSchemaPath("exclusiveMaximum", ()=>{
            const v = value.result;
            if (!(v < schema.exclusiveMaximum)) {
                context.error(value, schema, `value ${value.result} is greater than or equal to required (exclusive) maximum ${schema.exclusiveMaximum}`);
                return false;
            }
            return true;
        });
    }
    return result1;
}
function validateString(value, schema, context) {
    if (!typeIsValid(value, schema, context, typeof value.result === "string")) {
        return false;
    }
    if (schema.pattern !== undefined) {
        if (schema.compiledPattern === undefined) {
            schema.compiledPattern = new RegExp(schema.pattern);
        }
        if (!value.result.match(schema.compiledPattern)) {
            return context.withSchemaPath("pattern", ()=>{
                context.error(value, schema, `value doesn't match pattern`);
                return false;
            });
        }
    }
    return true;
}
function validateNull(value, schema, context) {
    if (!typeIsValid(value, schema, context, value.result === null)) {
        return false;
    }
    return true;
}
function validateEnum(value, schema, context) {
    for (const enumValue of schema["enum"]){
        if (enumValue === value.result) {
            return true;
        }
    }
    context.error(value, schema, `must match one of the values`);
    return false;
}
function validateAnyOf(value, schema, context) {
    let passingSchemas = 0;
    for(let i1 = 0; i1 < schema.anyOf.length; ++i1){
        const subSchema = schema.anyOf[i1];
        context.withSchemaPath(i1, ()=>{
            if (validateGeneric(value, subSchema, context)) {
                passingSchemas++;
                return true;
            }
            return false;
        });
    }
    return passingSchemas > 0;
}
function validateAllOf(value, schema, context) {
    let passingSchemas = 0;
    for(let i1 = 0; i1 < schema.allOf.length; ++i1){
        const subSchema = schema.allOf[i1];
        context.withSchemaPath(i1, ()=>{
            if (validateGeneric(value, subSchema, context)) {
                passingSchemas++;
                return true;
            }
            return false;
        });
    }
    return passingSchemas === schema.allOf.length;
}
function validateArray(value, schema, context) {
    let result1 = true;
    if (!typeIsValid(value, schema, context, Array.isArray(value.result))) {
        return false;
    }
    const length = value.result.length;
    if (schema.minItems !== undefined && length < schema.minItems) {
        context.withSchemaPath("minItems", ()=>{
            context.error(value, schema, `array should have at least ${schema.minItems} items but has ${length} items instead`);
            return false;
        });
        result1 = false;
    }
    if (schema.maxItems !== undefined && length > schema.maxItems) {
        context.withSchemaPath("maxItems", ()=>{
            context.error(value, schema, `array should have at most ${schema.maxItems} items but has ${length} items instead`);
            return false;
        });
        result1 = false;
    }
    if (schema.items !== undefined) {
        result1 = context.withSchemaPath("items", ()=>{
            let result1 = true;
            for(let i1 = 0; i1 < value.components.length; ++i1){
                context.pushInstance(i1);
                result1 = validateGeneric(value.components[i1], schema.items, context) && result1;
                context.popInstance();
            }
            return result1;
        }) && result1;
    }
    return result1;
}
function validateObject(value, schema, context) {
    const isObject = typeof value.result === "object" && !Array.isArray(value.result) && value.result !== null;
    if (!typeIsValid(value, schema, context, isObject)) {
        return false;
    }
    let result1 = true;
    const ownProperties = new Set(Object.getOwnPropertyNames(value.result));
    const objResult = value.result;
    const locate = (key, keyOrValue = "value")=>{
        for(let i1 = 0; i1 < value.components.length; i1 += 2){
            if (String(value.components[i1].result) === key) {
                if (keyOrValue === "value") {
                    return value.components[i1 + 1];
                } else {
                    return value.components[i1];
                }
            }
        }
        throw new InternalError(`Couldn't locate key ${key}`);
    };
    const inspectedProps = new Set();
    if (schema.closed) {
        result1 = context.withSchemaPath("closed", ()=>{
            if (schema.properties === undefined) {
                throw new InternalError("Closed schemas need properties");
            }
            let innerResult = true;
            for (const key of ownProperties){
                if (!schema.properties[key]) {
                    context.error(locate(key, "key"), schema, `object has invalid field ${key}`);
                    innerResult = false;
                }
            }
            return innerResult;
        }) && result1;
    }
    if (schema.properties !== undefined) {
        result1 = context.withSchemaPath("properties", ()=>{
            let result1 = true;
            for (const [key, subSchema] of Object.entries(schema.properties)){
                if (ownProperties.has(key)) {
                    inspectedProps.add(key);
                    context.pushInstance(key);
                    result1 = context.withSchemaPath(key, ()=>validateGeneric(locate(key), subSchema, context)) && result1;
                    context.popInstance();
                }
            }
            return result1;
        }) && result1;
    }
    if (schema.patternProperties !== undefined) {
        result1 = context.withSchemaPath("patternProperties", ()=>{
            let result1 = true;
            for (const [key, subSchema] of Object.entries(schema.patternProperties)){
                if (schema.compiledPatterns === undefined) {
                    schema.compiledPatterns = {};
                }
                if (schema.compiledPatterns[key] === undefined) {
                    schema.compiledPatterns[key] = new RegExp(key);
                }
                const regexp = schema.compiledPatterns[key];
                for (const [objectKey, _val] of Object.entries(objResult)){
                    if (objectKey.match(regexp)) {
                        inspectedProps.add(objectKey);
                        context.pushInstance(objectKey);
                        result1 = context.withSchemaPath(key, ()=>validateGeneric(locate(objectKey), subSchema, context)) && result1;
                        context.popInstance();
                    }
                }
            }
            return result1;
        }) && result1;
    }
    if (schema.additionalProperties !== undefined) {
        result1 = context.withSchemaPath("additionalProperties", ()=>{
            return Object.keys(objResult).filter((objectKey)=>!inspectedProps.has(objectKey)).every((objectKey)=>validateGeneric(locate(objectKey), schema.additionalProperties, context));
        }) && result1;
    }
    if (schema.propertyNames !== undefined) {
        result1 = context.withSchemaPath("propertyNames", ()=>{
            return Array.from(ownProperties).every((key)=>validateGeneric(locate(key, "key"), schema.propertyNames, context));
        }) && result1;
    }
    if (schema.required !== undefined) {
        result1 = context.withSchemaPath("required", ()=>{
            let result1 = true;
            for (const reqKey of schema.required){
                if (!ownProperties.has(reqKey)) {
                    context.error(value, schema, `object is missing required property ${reqKey}`);
                    result1 = false;
                }
            }
            return result1;
        }) && result1;
    }
    return result1;
}
function validate(value, schema, source, pruneErrors = true) {
    const context = new ValidationContext();
    return context.validate(schema, source, value, pruneErrors);
}
class YAMLSchema {
    schema;
    errorHandlers;
    constructor(schema){
        this.errorHandlers = [];
        this.schema = schema;
    }
    addHandler(handler) {
        this.errorHandlers.push(handler);
    }
    transformErrors(annotation, errors) {
        return errors.map((error)=>{
            for (const handler of this.errorHandlers){
                const localError = handler(error, annotation, this.schema);
                if (localError === null) {
                    return null;
                }
                error = localError;
            }
            return error;
        }).filter((error)=>error !== null);
    }
    async validateParse(src, annotation, pruneErrors = true) {
        const validationErrors = validate(annotation, this.schema, src, pruneErrors);
        if (validationErrors.length) {
            const localizedErrors = this.transformErrors(annotation, validationErrors);
            return {
                result: annotation.result,
                errors: localizedErrors
            };
        } else {
            return {
                result: annotation.result,
                errors: []
            };
        }
    }
    reportErrorsInSource(result1, _src, message, error, log) {
        if (result1.errors.length) {
            if (message.length) {
                error(message);
            }
            for (const err of result1.errors){
                log(err.niceError);
            }
        }
        return result1;
    }
    async validateParseWithErrors(src, annotation, message, error, log) {
        const result1 = await this.validateParse(src, annotation);
        this.reportErrorsInSource(result1, src, message, error, log);
        return result1;
    }
}
const yamlValidators = {};
function getSchemaName(schema) {
    if (schema === true || schema === false) {
        throw new Error("Expected schema to be named");
    }
    let schemaName = schema["$id"];
    if (schemaName !== undefined) {
        return schemaName;
    }
    if (schemaType(schema) === "ref") {
        schemaName = schema["$ref"];
    }
    if (schemaName !== undefined) {
        return schemaName;
    }
    throw new Error("Expected schema to be named");
}
function getValidator(schema) {
    const schemaName = getSchemaName(schema);
    if (yamlValidators[schemaName]) {
        return yamlValidators[schemaName];
    }
    const validator = new YAMLSchema(schema);
    yamlValidators[schemaName] = validator;
    setDefaultErrorHandlers(validator);
    return validator;
}
async function withValidator(schema, fun) {
    let result1;
    let error;
    try {
        const validator = getValidator(schema);
        result1 = await fun(validator);
    } catch (e) {
        error = e;
    }
    if (error !== undefined) {
        throw error;
    }
    return result1;
}
function addValidatorErrorHandler(schema, handler) {
    return withValidator(schema, async (validator)=>{
        validator.addHandler(handler);
    });
}
function memoize1(f, keyMemoizer) {
    const memo = {};
    const inner = (...args)=>{
        const key = keyMemoizer(...args);
        const v = memo[key];
        if (v !== undefined) {
            return v;
        }
        memo[key] = f(...args);
        return memo[key];
    };
    return inner;
}
const regExpEscapeChars1 = [
    "!",
    "$",
    "(",
    ")",
    "*",
    "+",
    ".",
    "=",
    "?",
    "[",
    "\\",
    "^",
    "{",
    "|"
];
const rangeEscapeChars1 = [
    "-",
    "\\",
    "]"
];
function globToRegExp3(glob, { extended = true, globstar: globstarOption = true, caseInsensitive = false } = {}) {
    if (glob == "") {
        return /(?!)/;
    }
    const sep = "/+";
    const sepMaybe = "/*";
    const seps = [
        "/"
    ];
    const globstar = "(?:[^/]*(?:/|$)+)*";
    const wildcard = "[^/]*";
    const escapePrefix = "\\";
    let newLength = glob.length;
    for(; newLength > 1 && seps.includes(glob[newLength - 1]); newLength--);
    glob = glob.slice(0, newLength);
    let regExpString = "";
    for(let j = 0; j < glob.length;){
        let segment = "";
        const groupStack = [];
        let inRange = false;
        let inEscape = false;
        let endsWithSep = false;
        let i1 = j;
        for(; i1 < glob.length && !seps.includes(glob[i1]); i1++){
            if (inEscape) {
                inEscape = false;
                const escapeChars = inRange ? rangeEscapeChars1 : regExpEscapeChars1;
                segment += escapeChars.includes(glob[i1]) ? `\\${glob[i1]}` : glob[i1];
                continue;
            }
            if (glob[i1] == escapePrefix) {
                inEscape = true;
                continue;
            }
            if (glob[i1] == "[") {
                if (!inRange) {
                    inRange = true;
                    segment += "[";
                    if (glob[i1 + 1] == "!") {
                        i1++;
                        segment += "^";
                    } else if (glob[i1 + 1] == "^") {
                        i1++;
                        segment += "\\^";
                    }
                    continue;
                } else if (glob[i1 + 1] == ":") {
                    let k = i1 + 1;
                    let value = "";
                    while(glob[k + 1] != null && glob[k + 1] != ":"){
                        value += glob[k + 1];
                        k++;
                    }
                    if (glob[k + 1] == ":" && glob[k + 2] == "]") {
                        i1 = k + 2;
                        if (value == "alnum") segment += "\\dA-Za-z";
                        else if (value == "alpha") segment += "A-Za-z";
                        else if (value == "ascii") segment += "\x00-\x7F";
                        else if (value == "blank") segment += "\t ";
                        else if (value == "cntrl") segment += "\x00-\x1F\x7F";
                        else if (value == "digit") segment += "\\d";
                        else if (value == "graph") segment += "\x21-\x7E";
                        else if (value == "lower") segment += "a-z";
                        else if (value == "print") segment += "\x20-\x7E";
                        else if (value == "punct") {
                            segment += "!\"#$%&'()*+,\\-./:;<=>?@[\\\\\\]^_‘{|}~";
                        } else if (value == "space") segment += "\\s\v";
                        else if (value == "upper") segment += "A-Z";
                        else if (value == "word") segment += "\\w";
                        else if (value == "xdigit") segment += "\\dA-Fa-f";
                        continue;
                    }
                }
            }
            if (glob[i1] == "]" && inRange) {
                inRange = false;
                segment += "]";
                continue;
            }
            if (inRange) {
                if (glob[i1] == "\\") {
                    segment += `\\\\`;
                } else {
                    segment += glob[i1];
                }
                continue;
            }
            if (glob[i1] == ")" && groupStack.length > 0 && groupStack[groupStack.length - 1] != "BRACE") {
                segment += ")";
                const type = groupStack.pop();
                if (type == "!") {
                    segment += wildcard;
                } else if (type != "@") {
                    segment += type;
                }
                continue;
            }
            if (glob[i1] == "|" && groupStack.length > 0 && groupStack[groupStack.length - 1] != "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i1] == "+" && extended && glob[i1 + 1] == "(") {
                i1++;
                groupStack.push("+");
                segment += "(?:";
                continue;
            }
            if (glob[i1] == "@" && extended && glob[i1 + 1] == "(") {
                i1++;
                groupStack.push("@");
                segment += "(?:";
                continue;
            }
            if (glob[i1] == "?") {
                if (extended && glob[i1 + 1] == "(") {
                    i1++;
                    groupStack.push("?");
                    segment += "(?:";
                } else {
                    segment += ".";
                }
                continue;
            }
            if (glob[i1] == "!" && extended && glob[i1 + 1] == "(") {
                i1++;
                groupStack.push("!");
                segment += "(?!";
                continue;
            }
            if (glob[i1] == "{") {
                groupStack.push("BRACE");
                segment += "(?:";
                continue;
            }
            if (glob[i1] == "}" && groupStack[groupStack.length - 1] == "BRACE") {
                groupStack.pop();
                segment += ")";
                continue;
            }
            if (glob[i1] == "," && groupStack[groupStack.length - 1] == "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i1] == "*") {
                if (extended && glob[i1 + 1] == "(") {
                    i1++;
                    groupStack.push("*");
                    segment += "(?:";
                } else {
                    const prevChar = glob[i1 - 1];
                    let numStars = 1;
                    while(glob[i1 + 1] == "*"){
                        i1++;
                        numStars++;
                    }
                    const nextChar = glob[i1 + 1];
                    if (globstarOption && numStars == 2 && [
                        ...seps,
                        undefined
                    ].includes(prevChar) && [
                        ...seps,
                        undefined
                    ].includes(nextChar)) {
                        segment += globstar;
                        endsWithSep = true;
                    } else {
                        segment += wildcard;
                    }
                }
                continue;
            }
            segment += regExpEscapeChars1.includes(glob[i1]) ? `\\${glob[i1]}` : glob[i1];
        }
        if (groupStack.length > 0 || inRange || inEscape) {
            segment = "";
            for (const c of glob.slice(j, i1)){
                segment += regExpEscapeChars1.includes(c) ? `\\${c}` : c;
                endsWithSep = false;
            }
        }
        regExpString += segment;
        if (!endsWithSep) {
            regExpString += i1 < glob.length ? sep : sepMaybe;
            endsWithSep = true;
        }
        while(seps.includes(glob[i1]))i1++;
        if (!(i1 > j)) {
            throw new Error("Assertion failure: i > j (potential infinite loop)");
        }
        j = i1;
    }
    regExpString = `^${regExpString}$`;
    return new RegExp(regExpString, caseInsensitive ? "i" : "");
}
function isGlob1(str) {
    const chars = {
        "{": "}",
        "(": ")",
        "[": "]"
    };
    const regex = /\\(.)|(^!|\*|\?|[\].+)]\?|\[[^\\\]]+\]|\{[^\\}]+\}|\(\?[:!=][^\\)]+\)|\([^|]+\|[^\\)]+\))/;
    if (str === "") {
        return false;
    }
    let match;
    while(match = regex.exec(str)){
        if (match[2]) return true;
        let idx = match.index + match[0].length;
        const open = match[1];
        const close = open ? chars[open] : null;
        if (open && close) {
            const n = str.indexOf(close, idx);
            if (n !== -1) {
                idx = n + 1;
            }
        }
        str = str.slice(idx);
    }
    return false;
}
const _resources = {};
function setYamlIntelligenceResources(resources) {
    for (const [key, value] of Object.entries(resources)){
        _resources[key] = value;
    }
}
function exportYamlIntelligenceResources(pretty = false) {
    if (pretty) {
        return JSON.stringify(_resources, null, 2);
    } else {
        return JSON.stringify(_resources);
    }
}
function getYamlIntelligenceResource(filename) {
    if (_resources[filename] === undefined) {
        throw new InternalError(`getYamlIntelligenceResource called with missing resource ${filename}`);
    }
    return _resources[filename];
}
function expandResourceGlob(glob) {
    return Object.keys(_resources).filter((key)=>key.match(globToRegExp3(glob))).map((key)=>[
            key,
            getYamlIntelligenceResource(key)
        ]);
}
function fromEntries(iterable) {
    return [
        ...iterable
    ].reduce((obj, [key, val])=>{
        obj[key] = val;
        return obj;
    }, {});
}
let _parser;
function setTreeSitter(parser) {
    _parser = parser;
}
function getTreeSitterSync() {
    if (_parser === undefined) {
        throw new Error("tree-sitter uninitialized");
    }
    return _parser;
}
function isNothing(subject) {
    return typeof subject === "undefined" || subject === null;
}
function isObject1(subject) {
    return typeof subject === "object" && subject !== null;
}
function toArray(sequence) {
    if (Array.isArray(sequence)) return sequence;
    else if (isNothing(sequence)) return [];
    return [
        sequence
    ];
}
function extend(target, source) {
    var index, length, key, sourceKeys;
    if (source) {
        sourceKeys = Object.keys(source);
        for(index = 0, length = sourceKeys.length; index < length; index += 1){
            key = sourceKeys[index];
            target[key] = source[key];
        }
    }
    return target;
}
function repeat(string, count) {
    var result1 = "", cycle;
    for(cycle = 0; cycle < count; cycle += 1){
        result1 += string;
    }
    return result1;
}
function isNegativeZero(number) {
    return number === 0 && Number.NEGATIVE_INFINITY === 1 / number;
}
var isNothing_1 = isNothing;
var isObject_1 = isObject1;
var toArray_1 = toArray;
var repeat_1 = repeat;
var isNegativeZero_1 = isNegativeZero;
var extend_1 = extend;
var common1 = {
    isNothing: isNothing_1,
    isObject: isObject_1,
    toArray: toArray_1,
    repeat: repeat_1,
    isNegativeZero: isNegativeZero_1,
    extend: extend_1
};
function formatError(exception2, compact) {
    var where = "", message = exception2.reason || "(unknown reason)";
    if (!exception2.mark) return message;
    if (exception2.mark.name) {
        where += 'in "' + exception2.mark.name + '" ';
    }
    where += "(" + (exception2.mark.line + 1) + ":" + (exception2.mark.column + 1) + ")";
    if (!compact && exception2.mark.snippet) {
        where += "\n\n" + exception2.mark.snippet;
    }
    return message + " " + where;
}
function YAMLException$1(reason, mark) {
    Error.call(this);
    this.name = "YAMLException";
    this.reason = reason;
    this.mark = mark;
    this.message = formatError(this, false);
    if (Error.captureStackTrace) {
        Error.captureStackTrace(this, this.constructor);
    } else {
        this.stack = new Error().stack || "";
    }
}
YAMLException$1.prototype = Object.create(Error.prototype);
YAMLException$1.prototype.constructor = YAMLException$1;
YAMLException$1.prototype.toString = function toString(compact) {
    return this.name + ": " + formatError(this, compact);
};
var exception = YAMLException$1;
function getLine(buffer, lineStart, lineEnd, position, maxLineLength) {
    var head = "";
    var tail = "";
    var maxHalfLength = Math.floor(maxLineLength / 2) - 1;
    if (position - lineStart > maxHalfLength) {
        head = " ... ";
        lineStart = position - maxHalfLength + head.length;
    }
    if (lineEnd - position > maxHalfLength) {
        tail = " ...";
        lineEnd = position + maxHalfLength - tail.length;
    }
    return {
        str: head + buffer.slice(lineStart, lineEnd).replace(/\t/g, "\u2192") + tail,
        pos: position - lineStart + head.length
    };
}
function padStart(string, max) {
    return common1.repeat(" ", max - string.length) + string;
}
function makeSnippet(mark, options) {
    options = Object.create(options || null);
    if (!mark.buffer) return null;
    if (!options.maxLength) options.maxLength = 79;
    if (typeof options.indent !== "number") options.indent = 1;
    if (typeof options.linesBefore !== "number") options.linesBefore = 3;
    if (typeof options.linesAfter !== "number") options.linesAfter = 2;
    var re = /\r?\n|\r|\0/g;
    var lineStarts = [
        0
    ];
    var lineEnds = [];
    var match;
    var foundLineNo = -1;
    while(match = re.exec(mark.buffer)){
        lineEnds.push(match.index);
        lineStarts.push(match.index + match[0].length);
        if (mark.position <= match.index && foundLineNo < 0) {
            foundLineNo = lineStarts.length - 2;
        }
    }
    if (foundLineNo < 0) foundLineNo = lineStarts.length - 1;
    var result1 = "", i1, line;
    var lineNoLength = Math.min(mark.line + options.linesAfter, lineEnds.length).toString().length;
    var maxLineLength = options.maxLength - (options.indent + lineNoLength + 3);
    for(i1 = 1; i1 <= options.linesBefore; i1++){
        if (foundLineNo - i1 < 0) break;
        line = getLine(mark.buffer, lineStarts[foundLineNo - i1], lineEnds[foundLineNo - i1], mark.position - (lineStarts[foundLineNo] - lineStarts[foundLineNo - i1]), maxLineLength);
        result1 = common1.repeat(" ", options.indent) + padStart((mark.line - i1 + 1).toString(), lineNoLength) + " | " + line.str + "\n" + result1;
    }
    line = getLine(mark.buffer, lineStarts[foundLineNo], lineEnds[foundLineNo], mark.position, maxLineLength);
    result1 += common1.repeat(" ", options.indent) + padStart((mark.line + 1).toString(), lineNoLength) + " | " + line.str + "\n";
    result1 += common1.repeat("-", options.indent + lineNoLength + 3 + line.pos) + "^\n";
    for(i1 = 1; i1 <= options.linesAfter; i1++){
        if (foundLineNo + i1 >= lineEnds.length) break;
        line = getLine(mark.buffer, lineStarts[foundLineNo + i1], lineEnds[foundLineNo + i1], mark.position - (lineStarts[foundLineNo] - lineStarts[foundLineNo + i1]), maxLineLength);
        result1 += common1.repeat(" ", options.indent) + padStart((mark.line + i1 + 1).toString(), lineNoLength) + " | " + line.str + "\n";
    }
    return result1.replace(/\n$/, "");
}
var snippet = makeSnippet;
var TYPE_CONSTRUCTOR_OPTIONS = [
    "kind",
    "multi",
    "resolve",
    "construct",
    "instanceOf",
    "predicate",
    "represent",
    "representName",
    "defaultStyle",
    "styleAliases"
];
var YAML_NODE_KINDS = [
    "scalar",
    "sequence",
    "mapping"
];
function compileStyleAliases(map2) {
    var result1 = {};
    if (map2 !== null) {
        Object.keys(map2).forEach(function(style) {
            map2[style].forEach(function(alias) {
                result1[String(alias)] = style;
            });
        });
    }
    return result1;
}
function Type$1(tag, options) {
    options = options || {};
    Object.keys(options).forEach(function(name) {
        if (TYPE_CONSTRUCTOR_OPTIONS.indexOf(name) === -1) {
            throw new exception('Unknown option "' + name + '" is met in definition of "' + tag + '" YAML type.');
        }
    });
    this.options = options;
    this.tag = tag;
    this.kind = options["kind"] || null;
    this.resolve = options["resolve"] || function() {
        return true;
    };
    this.construct = options["construct"] || function(data) {
        return data;
    };
    this.instanceOf = options["instanceOf"] || null;
    this.predicate = options["predicate"] || null;
    this.represent = options["represent"] || null;
    this.representName = options["representName"] || null;
    this.defaultStyle = options["defaultStyle"] || null;
    this.multi = options["multi"] || false;
    this.styleAliases = compileStyleAliases(options["styleAliases"] || null);
    if (YAML_NODE_KINDS.indexOf(this.kind) === -1) {
        throw new exception('Unknown kind "' + this.kind + '" is specified for "' + tag + '" YAML type.');
    }
}
var type = Type$1;
function compileList(schema2, name) {
    var result1 = [];
    schema2[name].forEach(function(currentType) {
        var newIndex = result1.length;
        result1.forEach(function(previousType, previousIndex) {
            if (previousType.tag === currentType.tag && previousType.kind === currentType.kind && previousType.multi === currentType.multi) {
                newIndex = previousIndex;
            }
        });
        result1[newIndex] = currentType;
    });
    return result1;
}
function compileMap() {
    var result1 = {
        scalar: {},
        sequence: {},
        mapping: {},
        fallback: {},
        multi: {
            scalar: [],
            sequence: [],
            mapping: [],
            fallback: []
        }
    }, index, length;
    function collectType(type2) {
        if (type2.multi) {
            result1.multi[type2.kind].push(type2);
            result1.multi["fallback"].push(type2);
        } else {
            result1[type2.kind][type2.tag] = result1["fallback"][type2.tag] = type2;
        }
    }
    for(index = 0, length = arguments.length; index < length; index += 1){
        arguments[index].forEach(collectType);
    }
    return result1;
}
function Schema$1(definition) {
    return this.extend(definition);
}
Schema$1.prototype.extend = function extend2(definition) {
    var implicit = [];
    var explicit = [];
    if (definition instanceof type) {
        explicit.push(definition);
    } else if (Array.isArray(definition)) {
        explicit = explicit.concat(definition);
    } else if (definition && (Array.isArray(definition.implicit) || Array.isArray(definition.explicit))) {
        if (definition.implicit) implicit = implicit.concat(definition.implicit);
        if (definition.explicit) explicit = explicit.concat(definition.explicit);
    } else {
        throw new exception("Schema.extend argument should be a Type, [ Type ], or a schema definition ({ implicit: [...], explicit: [...] })");
    }
    implicit.forEach(function(type$1) {
        if (!(type$1 instanceof type)) {
            throw new exception("Specified list of YAML types (or a single Type object) contains a non-Type object.");
        }
        if (type$1.loadKind && type$1.loadKind !== "scalar") {
            throw new exception("There is a non-scalar type in the implicit list of a schema. Implicit resolving of such types is not supported.");
        }
        if (type$1.multi) {
            throw new exception("There is a multi type in the implicit list of a schema. Multi tags can only be listed as explicit.");
        }
    });
    explicit.forEach(function(type$1) {
        if (!(type$1 instanceof type)) {
            throw new exception("Specified list of YAML types (or a single Type object) contains a non-Type object.");
        }
    });
    var result1 = Object.create(Schema$1.prototype);
    result1.implicit = (this.implicit || []).concat(implicit);
    result1.explicit = (this.explicit || []).concat(explicit);
    result1.compiledImplicit = compileList(result1, "implicit");
    result1.compiledExplicit = compileList(result1, "explicit");
    result1.compiledTypeMap = compileMap(result1.compiledImplicit, result1.compiledExplicit);
    return result1;
};
var schema = Schema$1;
var str = new type("tag:yaml.org,2002:str", {
    kind: "scalar",
    construct: function(data) {
        return data !== null ? data : "";
    }
});
var seq = new type("tag:yaml.org,2002:seq", {
    kind: "sequence",
    construct: function(data) {
        return data !== null ? data : [];
    }
});
var map = new type("tag:yaml.org,2002:map", {
    kind: "mapping",
    construct: function(data) {
        return data !== null ? data : {};
    }
});
var failsafe = new schema({
    explicit: [
        str,
        seq,
        map
    ]
});
function resolveYamlNull(data) {
    if (data === null) return true;
    var max = data.length;
    return max === 1 && data === "~" || max === 4 && (data === "null" || data === "Null" || data === "NULL");
}
function constructYamlNull() {
    return null;
}
function isNull(object) {
    return object === null;
}
var _null = new type("tag:yaml.org,2002:null", {
    kind: "scalar",
    resolve: resolveYamlNull,
    construct: constructYamlNull,
    predicate: isNull,
    represent: {
        canonical: function() {
            return "~";
        },
        lowercase: function() {
            return "null";
        },
        uppercase: function() {
            return "NULL";
        },
        camelcase: function() {
            return "Null";
        },
        empty: function() {
            return "";
        }
    },
    defaultStyle: "lowercase"
});
function resolveYamlBoolean(data) {
    if (data === null) return false;
    var max = data.length;
    return max === 4 && (data === "true" || data === "True" || data === "TRUE") || max === 5 && (data === "false" || data === "False" || data === "FALSE");
}
function constructYamlBoolean(data) {
    return data === "true" || data === "True" || data === "TRUE";
}
function isBoolean(object) {
    return Object.prototype.toString.call(object) === "[object Boolean]";
}
var bool = new type("tag:yaml.org,2002:bool", {
    kind: "scalar",
    resolve: resolveYamlBoolean,
    construct: constructYamlBoolean,
    predicate: isBoolean,
    represent: {
        lowercase: function(object) {
            return object ? "true" : "false";
        },
        uppercase: function(object) {
            return object ? "TRUE" : "FALSE";
        },
        camelcase: function(object) {
            return object ? "True" : "False";
        }
    },
    defaultStyle: "lowercase"
});
function isHexCode(c) {
    return 48 <= c && c <= 57 || 65 <= c && c <= 70 || 97 <= c && c <= 102;
}
function isOctCode(c) {
    return 48 <= c && c <= 55;
}
function isDecCode(c) {
    return 48 <= c && c <= 57;
}
function resolveYamlInteger(data) {
    if (data === null) return false;
    var max = data.length, index = 0, hasDigits = false, ch;
    if (!max) return false;
    ch = data[index];
    if (ch === "-" || ch === "+") {
        ch = data[++index];
    }
    if (ch === "0") {
        if (index + 1 === max) return true;
        ch = data[++index];
        if (ch === "b") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (ch !== "0" && ch !== "1") return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        if (ch === "x") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (!isHexCode(data.charCodeAt(index))) return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        if (ch === "o") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (!isOctCode(data.charCodeAt(index))) return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
    }
    if (ch === "_") return false;
    for(; index < max; index++){
        ch = data[index];
        if (ch === "_") continue;
        if (!isDecCode(data.charCodeAt(index))) {
            return false;
        }
        hasDigits = true;
    }
    if (!hasDigits || ch === "_") return false;
    return true;
}
function constructYamlInteger(data) {
    var value = data, sign = 1, ch;
    if (value.indexOf("_") !== -1) {
        value = value.replace(/_/g, "");
    }
    ch = value[0];
    if (ch === "-" || ch === "+") {
        if (ch === "-") sign = -1;
        value = value.slice(1);
        ch = value[0];
    }
    if (value === "0") return 0;
    if (ch === "0") {
        if (value[1] === "b") return sign * parseInt(value.slice(2), 2);
        if (value[1] === "x") return sign * parseInt(value.slice(2), 16);
        if (value[1] === "o") return sign * parseInt(value.slice(2), 8);
    }
    return sign * parseInt(value, 10);
}
function isInteger(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && object % 1 === 0 && !common1.isNegativeZero(object);
}
var __int = new type("tag:yaml.org,2002:int", {
    kind: "scalar",
    resolve: resolveYamlInteger,
    construct: constructYamlInteger,
    predicate: isInteger,
    represent: {
        binary: function(obj) {
            return obj >= 0 ? "0b" + obj.toString(2) : "-0b" + obj.toString(2).slice(1);
        },
        octal: function(obj) {
            return obj >= 0 ? "0o" + obj.toString(8) : "-0o" + obj.toString(8).slice(1);
        },
        decimal: function(obj) {
            return obj.toString(10);
        },
        hexadecimal: function(obj) {
            return obj >= 0 ? "0x" + obj.toString(16).toUpperCase() : "-0x" + obj.toString(16).toUpperCase().slice(1);
        }
    },
    defaultStyle: "decimal",
    styleAliases: {
        binary: [
            2,
            "bin"
        ],
        octal: [
            8,
            "oct"
        ],
        decimal: [
            10,
            "dec"
        ],
        hexadecimal: [
            16,
            "hex"
        ]
    }
});
var YAML_FLOAT_PATTERN = new RegExp("^(?:[-+]?(?:[0-9][0-9_]*)(?:\\.[0-9_]*)?(?:[eE][-+]?[0-9]+)?|\\.[0-9_]+(?:[eE][-+]?[0-9]+)?|[-+]?\\.(?:inf|Inf|INF)|\\.(?:nan|NaN|NAN))$");
function resolveYamlFloat(data) {
    if (data === null) return false;
    if (!YAML_FLOAT_PATTERN.test(data) || data[data.length - 1] === "_") {
        return false;
    }
    return true;
}
function constructYamlFloat(data) {
    var value, sign;
    value = data.replace(/_/g, "").toLowerCase();
    sign = value[0] === "-" ? -1 : 1;
    if ("+-".indexOf(value[0]) >= 0) {
        value = value.slice(1);
    }
    if (value === ".inf") {
        return sign === 1 ? Number.POSITIVE_INFINITY : Number.NEGATIVE_INFINITY;
    } else if (value === ".nan") {
        return NaN;
    }
    return sign * parseFloat(value, 10);
}
var SCIENTIFIC_WITHOUT_DOT = /^[-+]?[0-9]+e/;
function representYamlFloat(object, style) {
    var res;
    if (isNaN(object)) {
        switch(style){
            case "lowercase":
                return ".nan";
            case "uppercase":
                return ".NAN";
            case "camelcase":
                return ".NaN";
        }
    } else if (Number.POSITIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return ".inf";
            case "uppercase":
                return ".INF";
            case "camelcase":
                return ".Inf";
        }
    } else if (Number.NEGATIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return "-.inf";
            case "uppercase":
                return "-.INF";
            case "camelcase":
                return "-.Inf";
        }
    } else if (common1.isNegativeZero(object)) {
        return "-0.0";
    }
    res = object.toString(10);
    return SCIENTIFIC_WITHOUT_DOT.test(res) ? res.replace("e", ".e") : res;
}
function isFloat(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && (object % 1 !== 0 || common1.isNegativeZero(object));
}
var __float = new type("tag:yaml.org,2002:float", {
    kind: "scalar",
    resolve: resolveYamlFloat,
    construct: constructYamlFloat,
    predicate: isFloat,
    represent: representYamlFloat,
    defaultStyle: "lowercase"
});
var json = failsafe.extend({
    implicit: [
        _null,
        bool,
        __int,
        __float
    ]
});
var core = json;
var YAML_DATE_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])-([0-9][0-9])-([0-9][0-9])$");
var YAML_TIMESTAMP_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])-([0-9][0-9]?)-([0-9][0-9]?)(?:[Tt]|[ \\t]+)([0-9][0-9]?):([0-9][0-9]):([0-9][0-9])(?:\\.([0-9]*))?(?:[ \\t]*(Z|([-+])([0-9][0-9]?)(?::([0-9][0-9]))?))?$");
function resolveYamlTimestamp(data) {
    if (data === null) return false;
    if (YAML_DATE_REGEXP.exec(data) !== null) return true;
    if (YAML_TIMESTAMP_REGEXP.exec(data) !== null) return true;
    return false;
}
function constructYamlTimestamp(data) {
    var match, year, month, day, hour, minute, second, fraction = 0, delta = null, tz_hour, tz_minute, date;
    match = YAML_DATE_REGEXP.exec(data);
    if (match === null) match = YAML_TIMESTAMP_REGEXP.exec(data);
    if (match === null) throw new Error("Date resolve error");
    year = +match[1];
    month = +match[2] - 1;
    day = +match[3];
    if (!match[4]) {
        return new Date(Date.UTC(year, month, day));
    }
    hour = +match[4];
    minute = +match[5];
    second = +match[6];
    if (match[7]) {
        fraction = match[7].slice(0, 3);
        while(fraction.length < 3){
            fraction += "0";
        }
        fraction = +fraction;
    }
    if (match[9]) {
        tz_hour = +match[10];
        tz_minute = +(match[11] || 0);
        delta = (tz_hour * 60 + tz_minute) * 6e4;
        if (match[9] === "-") delta = -delta;
    }
    date = new Date(Date.UTC(year, month, day, hour, minute, second, fraction));
    if (delta) date.setTime(date.getTime() - delta);
    return date;
}
function representYamlTimestamp(object) {
    return object.toISOString();
}
var timestamp = new type("tag:yaml.org,2002:timestamp", {
    kind: "scalar",
    resolve: resolveYamlTimestamp,
    construct: constructYamlTimestamp,
    instanceOf: Date,
    represent: representYamlTimestamp
});
function resolveYamlMerge(data) {
    return data === "<<" || data === null;
}
var merge = new type("tag:yaml.org,2002:merge", {
    kind: "scalar",
    resolve: resolveYamlMerge
});
var BASE64_MAP = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=\n\r";
function resolveYamlBinary(data) {
    if (data === null) return false;
    var code, idx, bitlen = 0, max = data.length, map2 = BASE64_MAP;
    for(idx = 0; idx < max; idx++){
        code = map2.indexOf(data.charAt(idx));
        if (code > 64) continue;
        if (code < 0) return false;
        bitlen += 6;
    }
    return bitlen % 8 === 0;
}
function constructYamlBinary(data) {
    var idx, tailbits, input = data.replace(/[\r\n=]/g, ""), max = input.length, map2 = BASE64_MAP, bits = 0, result1 = [];
    for(idx = 0; idx < max; idx++){
        if (idx % 4 === 0 && idx) {
            result1.push(bits >> 16 & 255);
            result1.push(bits >> 8 & 255);
            result1.push(bits & 255);
        }
        bits = bits << 6 | map2.indexOf(input.charAt(idx));
    }
    tailbits = max % 4 * 6;
    if (tailbits === 0) {
        result1.push(bits >> 16 & 255);
        result1.push(bits >> 8 & 255);
        result1.push(bits & 255);
    } else if (tailbits === 18) {
        result1.push(bits >> 10 & 255);
        result1.push(bits >> 2 & 255);
    } else if (tailbits === 12) {
        result1.push(bits >> 4 & 255);
    }
    return new Uint8Array(result1);
}
function representYamlBinary(object) {
    var result1 = "", bits = 0, idx, tail, max = object.length, map2 = BASE64_MAP;
    for(idx = 0; idx < max; idx++){
        if (idx % 3 === 0 && idx) {
            result1 += map2[bits >> 18 & 63];
            result1 += map2[bits >> 12 & 63];
            result1 += map2[bits >> 6 & 63];
            result1 += map2[bits & 63];
        }
        bits = (bits << 8) + object[idx];
    }
    tail = max % 3;
    if (tail === 0) {
        result1 += map2[bits >> 18 & 63];
        result1 += map2[bits >> 12 & 63];
        result1 += map2[bits >> 6 & 63];
        result1 += map2[bits & 63];
    } else if (tail === 2) {
        result1 += map2[bits >> 10 & 63];
        result1 += map2[bits >> 4 & 63];
        result1 += map2[bits << 2 & 63];
        result1 += map2[64];
    } else if (tail === 1) {
        result1 += map2[bits >> 2 & 63];
        result1 += map2[bits << 4 & 63];
        result1 += map2[64];
        result1 += map2[64];
    }
    return result1;
}
function isBinary(obj) {
    return Object.prototype.toString.call(obj) === "[object Uint8Array]";
}
var binary = new type("tag:yaml.org,2002:binary", {
    kind: "scalar",
    resolve: resolveYamlBinary,
    construct: constructYamlBinary,
    predicate: isBinary,
    represent: representYamlBinary
});
var _hasOwnProperty$3 = Object.prototype.hasOwnProperty;
var _toString$2 = Object.prototype.toString;
function resolveYamlOmap(data) {
    if (data === null) return true;
    var objectKeys = [], index, length, pair, pairKey, pairHasKey, object = data;
    for(index = 0, length = object.length; index < length; index += 1){
        pair = object[index];
        pairHasKey = false;
        if (_toString$2.call(pair) !== "[object Object]") return false;
        for(pairKey in pair){
            if (_hasOwnProperty$3.call(pair, pairKey)) {
                if (!pairHasKey) pairHasKey = true;
                else return false;
            }
        }
        if (!pairHasKey) return false;
        if (objectKeys.indexOf(pairKey) === -1) objectKeys.push(pairKey);
        else return false;
    }
    return true;
}
function constructYamlOmap(data) {
    return data !== null ? data : [];
}
var omap = new type("tag:yaml.org,2002:omap", {
    kind: "sequence",
    resolve: resolveYamlOmap,
    construct: constructYamlOmap
});
var _toString$1 = Object.prototype.toString;
function resolveYamlPairs(data) {
    if (data === null) return true;
    var index, length, pair, keys, result1, object = data;
    result1 = new Array(object.length);
    for(index = 0, length = object.length; index < length; index += 1){
        pair = object[index];
        if (_toString$1.call(pair) !== "[object Object]") return false;
        keys = Object.keys(pair);
        if (keys.length !== 1) return false;
        result1[index] = [
            keys[0],
            pair[keys[0]]
        ];
    }
    return true;
}
function constructYamlPairs(data) {
    if (data === null) return [];
    var index, length, pair, keys, result1, object = data;
    result1 = new Array(object.length);
    for(index = 0, length = object.length; index < length; index += 1){
        pair = object[index];
        keys = Object.keys(pair);
        result1[index] = [
            keys[0],
            pair[keys[0]]
        ];
    }
    return result1;
}
var pairs = new type("tag:yaml.org,2002:pairs", {
    kind: "sequence",
    resolve: resolveYamlPairs,
    construct: constructYamlPairs
});
var _hasOwnProperty$2 = Object.prototype.hasOwnProperty;
function resolveYamlSet(data) {
    if (data === null) return true;
    var key, object = data;
    for(key in object){
        if (_hasOwnProperty$2.call(object, key)) {
            if (object[key] !== null) return false;
        }
    }
    return true;
}
function constructYamlSet(data) {
    return data !== null ? data : {};
}
var set = new type("tag:yaml.org,2002:set", {
    kind: "mapping",
    resolve: resolveYamlSet,
    construct: constructYamlSet
});
var _default = core.extend({
    implicit: [
        timestamp,
        merge
    ],
    explicit: [
        binary,
        omap,
        pairs,
        set
    ]
});
var _hasOwnProperty$1 = Object.prototype.hasOwnProperty;
var CONTEXT_FLOW_IN = 1;
var CONTEXT_FLOW_OUT = 2;
var CONTEXT_BLOCK_IN = 3;
var CONTEXT_BLOCK_OUT = 4;
var CHOMPING_CLIP = 1;
var CHOMPING_STRIP = 2;
var CHOMPING_KEEP = 3;
var PATTERN_NON_PRINTABLE = /[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x84\x86-\x9F\uFFFE\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF]/;
var PATTERN_NON_ASCII_LINE_BREAKS = /[\x85\u2028\u2029]/;
var PATTERN_FLOW_INDICATORS = /[,\[\]\{\}]/;
var PATTERN_TAG_HANDLE = /^(?:!|!!|![a-z\-]+!)$/i;
var PATTERN_TAG_URI = /^(?:!|[^,\[\]\{\}])(?:%[0-9a-f]{2}|[0-9a-z\-#;\/\?:@&=\+\$,_\.!~\*'\(\)\[\]])*$/i;
function _class(obj) {
    return Object.prototype.toString.call(obj);
}
function is_EOL(c) {
    return c === 10 || c === 13;
}
function is_WHITE_SPACE(c) {
    return c === 9 || c === 32;
}
function is_WS_OR_EOL(c) {
    return c === 9 || c === 32 || c === 10 || c === 13;
}
function is_FLOW_INDICATOR(c) {
    return c === 44 || c === 91 || c === 93 || c === 123 || c === 125;
}
function fromHexCode(c) {
    var lc;
    if (48 <= c && c <= 57) {
        return c - 48;
    }
    lc = c | 32;
    if (97 <= lc && lc <= 102) {
        return lc - 97 + 10;
    }
    return -1;
}
function escapedHexLen(c) {
    if (c === 120) {
        return 2;
    }
    if (c === 117) {
        return 4;
    }
    if (c === 85) {
        return 8;
    }
    return 0;
}
function fromDecimalCode(c) {
    if (48 <= c && c <= 57) {
        return c - 48;
    }
    return -1;
}
function simpleEscapeSequence(c) {
    return c === 48 ? "\0" : c === 97 ? "\x07" : c === 98 ? "\b" : c === 116 ? "	" : c === 9 ? "	" : c === 110 ? "\n" : c === 118 ? "\v" : c === 102 ? "\f" : c === 114 ? "\r" : c === 101 ? "" : c === 32 ? " " : c === 34 ? '"' : c === 47 ? "/" : c === 92 ? "\\" : c === 78 ? "\x85" : c === 95 ? "\xA0" : c === 76 ? "\u2028" : c === 80 ? "\u2029" : "";
}
function charFromCodepoint(c) {
    if (c <= 65535) {
        return String.fromCharCode(c);
    }
    return String.fromCharCode((c - 65536 >> 10) + 55296, (c - 65536 & 1023) + 56320);
}
var simpleEscapeCheck = new Array(256);
var simpleEscapeMap = new Array(256);
for(var i1 = 0; i1 < 256; i1++){
    simpleEscapeCheck[i1] = simpleEscapeSequence(i1) ? 1 : 0;
    simpleEscapeMap[i1] = simpleEscapeSequence(i1);
}
function State$1(input, options) {
    this.input = input;
    this.filename = options["filename"] || null;
    this.schema = options["schema"] || _default;
    this.onWarning = options["onWarning"] || null;
    this.legacy = options["legacy"] || false;
    this.json = options["json"] || false;
    this.listener = options["listener"] || null;
    this.implicitTypes = this.schema.compiledImplicit;
    this.typeMap = this.schema.compiledTypeMap;
    this.length = input.length;
    this.position = 0;
    this.line = 0;
    this.lineStart = 0;
    this.lineIndent = 0;
    this.firstTabInLine = -1;
    this.documents = [];
}
function generateError(state, message) {
    var mark = {
        name: state.filename,
        buffer: state.input.slice(0, -1),
        position: state.position,
        line: state.line,
        column: state.position - state.lineStart
    };
    mark.snippet = snippet(mark);
    return new exception(message, mark);
}
function throwError(state, message) {
    throw generateError(state, message);
}
function throwWarning(state, message) {
    if (state.onWarning) {
        state.onWarning.call(null, generateError(state, message));
    }
}
var directiveHandlers = {
    YAML: function handleYamlDirective(state, name, args) {
        var match, major, minor;
        if (state.version !== null) {
            throwError(state, "duplication of %YAML directive");
        }
        if (args.length !== 1) {
            throwError(state, "YAML directive accepts exactly one argument");
        }
        match = /^([0-9]+)\.([0-9]+)$/.exec(args[0]);
        if (match === null) {
            throwError(state, "ill-formed argument of the YAML directive");
        }
        major = parseInt(match[1], 10);
        minor = parseInt(match[2], 10);
        if (major !== 1) {
            throwError(state, "unacceptable YAML version of the document");
        }
        state.version = args[0];
        state.checkLineBreaks = minor < 2;
        if (minor !== 1 && minor !== 2) {
            throwWarning(state, "unsupported YAML version of the document");
        }
    },
    TAG: function handleTagDirective(state, name, args) {
        var handle, prefix;
        if (args.length !== 2) {
            throwError(state, "TAG directive accepts exactly two arguments");
        }
        handle = args[0];
        prefix = args[1];
        if (!PATTERN_TAG_HANDLE.test(handle)) {
            throwError(state, "ill-formed tag handle (first argument) of the TAG directive");
        }
        if (_hasOwnProperty$1.call(state.tagMap, handle)) {
            throwError(state, 'there is a previously declared suffix for "' + handle + '" tag handle');
        }
        if (!PATTERN_TAG_URI.test(prefix)) {
            throwError(state, "ill-formed tag prefix (second argument) of the TAG directive");
        }
        try {
            prefix = decodeURIComponent(prefix);
        } catch (err) {
            throwError(state, "tag prefix is malformed: " + prefix);
        }
        state.tagMap[handle] = prefix;
    }
};
function captureSegment(state, start, end, checkJson) {
    var _position, _length, _character, _result;
    if (start < end) {
        _result = state.input.slice(start, end);
        if (checkJson) {
            for(_position = 0, _length = _result.length; _position < _length; _position += 1){
                _character = _result.charCodeAt(_position);
                if (!(_character === 9 || 32 <= _character && _character <= 1114111)) {
                    throwError(state, "expected valid JSON character");
                }
            }
        } else if (PATTERN_NON_PRINTABLE.test(_result)) {
            throwError(state, "the stream contains non-printable characters");
        }
        state.result += _result;
    }
}
function mergeMappings(state, destination, source, overridableKeys) {
    var sourceKeys, key, index, quantity;
    if (!common1.isObject(source)) {
        throwError(state, "cannot merge mappings; the provided source object is unacceptable");
    }
    sourceKeys = Object.keys(source);
    for(index = 0, quantity = sourceKeys.length; index < quantity; index += 1){
        key = sourceKeys[index];
        if (!_hasOwnProperty$1.call(destination, key)) {
            destination[key] = source[key];
            overridableKeys[key] = true;
        }
    }
}
function storeMappingPair(state, _result, overridableKeys, keyTag, keyNode, valueNode, startLine, startLineStart, startPos) {
    var index, quantity;
    if (Array.isArray(keyNode)) {
        keyNode = Array.prototype.slice.call(keyNode);
        for(index = 0, quantity = keyNode.length; index < quantity; index += 1){
            if (Array.isArray(keyNode[index])) {
                throwError(state, "nested arrays are not supported inside keys");
            }
            if (typeof keyNode === "object" && _class(keyNode[index]) === "[object Object]") {
                keyNode[index] = "[object Object]";
            }
        }
    }
    if (typeof keyNode === "object" && _class(keyNode) === "[object Object]") {
        keyNode = "[object Object]";
    }
    keyNode = String(keyNode);
    if (_result === null) {
        _result = {};
    }
    if (keyTag === "tag:yaml.org,2002:merge") {
        if (Array.isArray(valueNode)) {
            for(index = 0, quantity = valueNode.length; index < quantity; index += 1){
                mergeMappings(state, _result, valueNode[index], overridableKeys);
            }
        } else {
            mergeMappings(state, _result, valueNode, overridableKeys);
        }
    } else {
        if (!state.json && !_hasOwnProperty$1.call(overridableKeys, keyNode) && _hasOwnProperty$1.call(_result, keyNode)) {
            state.line = startLine || state.line;
            state.lineStart = startLineStart || state.lineStart;
            state.position = startPos || state.position;
            throwError(state, "duplicated mapping key");
        }
        if (keyNode === "__proto__") {
            Object.defineProperty(_result, keyNode, {
                configurable: true,
                enumerable: true,
                writable: true,
                value: valueNode
            });
        } else {
            _result[keyNode] = valueNode;
        }
        delete overridableKeys[keyNode];
    }
    return _result;
}
function readLineBreak(state) {
    var ch;
    ch = state.input.charCodeAt(state.position);
    if (ch === 10) {
        state.position++;
    } else if (ch === 13) {
        state.position++;
        if (state.input.charCodeAt(state.position) === 10) {
            state.position++;
        }
    } else {
        throwError(state, "a line break is expected");
    }
    state.line += 1;
    state.lineStart = state.position;
    state.firstTabInLine = -1;
}
function skipSeparationSpace(state, allowComments, checkIndent) {
    var lineBreaks = 0, ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        while(is_WHITE_SPACE(ch)){
            if (ch === 9 && state.firstTabInLine === -1) {
                state.firstTabInLine = state.position;
            }
            ch = state.input.charCodeAt(++state.position);
        }
        if (allowComments && ch === 35) {
            do {
                ch = state.input.charCodeAt(++state.position);
            }while (ch !== 10 && ch !== 13 && ch !== 0)
        }
        if (is_EOL(ch)) {
            readLineBreak(state);
            ch = state.input.charCodeAt(state.position);
            lineBreaks++;
            state.lineIndent = 0;
            while(ch === 32){
                state.lineIndent++;
                ch = state.input.charCodeAt(++state.position);
            }
        } else {
            break;
        }
    }
    if (checkIndent !== -1 && lineBreaks !== 0 && state.lineIndent < checkIndent) {
        throwWarning(state, "deficient indentation");
    }
    return lineBreaks;
}
function testDocumentSeparator(state) {
    var _position = state.position, ch;
    ch = state.input.charCodeAt(_position);
    if ((ch === 45 || ch === 46) && ch === state.input.charCodeAt(_position + 1) && ch === state.input.charCodeAt(_position + 2)) {
        _position += 3;
        ch = state.input.charCodeAt(_position);
        if (ch === 0 || is_WS_OR_EOL(ch)) {
            return true;
        }
    }
    return false;
}
function writeFoldedLines(state, count) {
    if (count === 1) {
        state.result += " ";
    } else if (count > 1) {
        state.result += common1.repeat("\n", count - 1);
    }
}
function readPlainScalar(state, nodeIndent, withinFlowCollection) {
    var preceding, following, captureStart, captureEnd, hasPendingContent, _line, _lineStart, _lineIndent, _kind = state.kind, _result = state.result, ch;
    ch = state.input.charCodeAt(state.position);
    if (is_WS_OR_EOL(ch) || is_FLOW_INDICATOR(ch) || ch === 35 || ch === 38 || ch === 42 || ch === 33 || ch === 124 || ch === 62 || ch === 39 || ch === 34 || ch === 37 || ch === 64 || ch === 96) {
        return false;
    }
    if (ch === 63 || ch === 45) {
        following = state.input.charCodeAt(state.position + 1);
        if (is_WS_OR_EOL(following) || withinFlowCollection && is_FLOW_INDICATOR(following)) {
            return false;
        }
    }
    state.kind = "scalar";
    state.result = "";
    captureStart = captureEnd = state.position;
    hasPendingContent = false;
    while(ch !== 0){
        if (ch === 58) {
            following = state.input.charCodeAt(state.position + 1);
            if (is_WS_OR_EOL(following) || withinFlowCollection && is_FLOW_INDICATOR(following)) {
                break;
            }
        } else if (ch === 35) {
            preceding = state.input.charCodeAt(state.position - 1);
            if (is_WS_OR_EOL(preceding)) {
                break;
            }
        } else if (state.position === state.lineStart && testDocumentSeparator(state) || withinFlowCollection && is_FLOW_INDICATOR(ch)) {
            break;
        } else if (is_EOL(ch)) {
            _line = state.line;
            _lineStart = state.lineStart;
            _lineIndent = state.lineIndent;
            skipSeparationSpace(state, false, -1);
            if (state.lineIndent >= nodeIndent) {
                hasPendingContent = true;
                ch = state.input.charCodeAt(state.position);
                continue;
            } else {
                state.position = captureEnd;
                state.line = _line;
                state.lineStart = _lineStart;
                state.lineIndent = _lineIndent;
                break;
            }
        }
        if (hasPendingContent) {
            captureSegment(state, captureStart, captureEnd, false);
            writeFoldedLines(state, state.line - _line);
            captureStart = captureEnd = state.position;
            hasPendingContent = false;
        }
        if (!is_WHITE_SPACE(ch)) {
            captureEnd = state.position + 1;
        }
        ch = state.input.charCodeAt(++state.position);
    }
    captureSegment(state, captureStart, captureEnd, false);
    if (state.result) {
        return true;
    }
    state.kind = _kind;
    state.result = _result;
    return false;
}
function readSingleQuotedScalar(state, nodeIndent) {
    var ch, captureStart, captureEnd;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 39) {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    state.position++;
    captureStart = captureEnd = state.position;
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        if (ch === 39) {
            captureSegment(state, captureStart, state.position, true);
            ch = state.input.charCodeAt(++state.position);
            if (ch === 39) {
                captureStart = state.position;
                state.position++;
                captureEnd = state.position;
            } else {
                return true;
            }
        } else if (is_EOL(ch)) {
            captureSegment(state, captureStart, captureEnd, true);
            writeFoldedLines(state, skipSeparationSpace(state, false, nodeIndent));
            captureStart = captureEnd = state.position;
        } else if (state.position === state.lineStart && testDocumentSeparator(state)) {
            throwError(state, "unexpected end of the document within a single quoted scalar");
        } else {
            state.position++;
            captureEnd = state.position;
        }
    }
    throwError(state, "unexpected end of the stream within a single quoted scalar");
}
function readDoubleQuotedScalar(state, nodeIndent) {
    var captureStart, captureEnd, hexLength, hexResult, tmp, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 34) {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    state.position++;
    captureStart = captureEnd = state.position;
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        if (ch === 34) {
            captureSegment(state, captureStart, state.position, true);
            state.position++;
            return true;
        } else if (ch === 92) {
            captureSegment(state, captureStart, state.position, true);
            ch = state.input.charCodeAt(++state.position);
            if (is_EOL(ch)) {
                skipSeparationSpace(state, false, nodeIndent);
            } else if (ch < 256 && simpleEscapeCheck[ch]) {
                state.result += simpleEscapeMap[ch];
                state.position++;
            } else if ((tmp = escapedHexLen(ch)) > 0) {
                hexLength = tmp;
                hexResult = 0;
                for(; hexLength > 0; hexLength--){
                    ch = state.input.charCodeAt(++state.position);
                    if ((tmp = fromHexCode(ch)) >= 0) {
                        hexResult = (hexResult << 4) + tmp;
                    } else {
                        throwError(state, "expected hexadecimal character");
                    }
                }
                state.result += charFromCodepoint(hexResult);
                state.position++;
            } else {
                throwError(state, "unknown escape sequence");
            }
            captureStart = captureEnd = state.position;
        } else if (is_EOL(ch)) {
            captureSegment(state, captureStart, captureEnd, true);
            writeFoldedLines(state, skipSeparationSpace(state, false, nodeIndent));
            captureStart = captureEnd = state.position;
        } else if (state.position === state.lineStart && testDocumentSeparator(state)) {
            throwError(state, "unexpected end of the document within a double quoted scalar");
        } else {
            state.position++;
            captureEnd = state.position;
        }
    }
    throwError(state, "unexpected end of the stream within a double quoted scalar");
}
function readFlowCollection(state, nodeIndent) {
    var readNext = true, _line, _lineStart, _pos, _tag = state.tag, _result, _anchor = state.anchor, following, terminator, isPair, isExplicitPair, isMapping, overridableKeys = Object.create(null), keyNode, keyTag, valueNode, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch === 91) {
        terminator = 93;
        isMapping = false;
        _result = [];
    } else if (ch === 123) {
        terminator = 125;
        isMapping = true;
        _result = {};
    } else {
        return false;
    }
    if (state.anchor !== null) {
        state.anchorMap[state.anchor] = _result;
    }
    ch = state.input.charCodeAt(++state.position);
    while(ch !== 0){
        skipSeparationSpace(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if (ch === terminator) {
            state.position++;
            state.tag = _tag;
            state.anchor = _anchor;
            state.kind = isMapping ? "mapping" : "sequence";
            state.result = _result;
            return true;
        } else if (!readNext) {
            throwError(state, "missed comma between flow collection entries");
        } else if (ch === 44) {
            throwError(state, "expected the node content, but found ','");
        }
        keyTag = keyNode = valueNode = null;
        isPair = isExplicitPair = false;
        if (ch === 63) {
            following = state.input.charCodeAt(state.position + 1);
            if (is_WS_OR_EOL(following)) {
                isPair = isExplicitPair = true;
                state.position++;
                skipSeparationSpace(state, true, nodeIndent);
            }
        }
        _line = state.line;
        _lineStart = state.lineStart;
        _pos = state.position;
        composeNode(state, nodeIndent, CONTEXT_FLOW_IN, false, true);
        keyTag = state.tag;
        keyNode = state.result;
        skipSeparationSpace(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if ((isExplicitPair || state.line === _line) && ch === 58) {
            isPair = true;
            ch = state.input.charCodeAt(++state.position);
            skipSeparationSpace(state, true, nodeIndent);
            composeNode(state, nodeIndent, CONTEXT_FLOW_IN, false, true);
            valueNode = state.result;
        }
        if (isMapping) {
            storeMappingPair(state, _result, overridableKeys, keyTag, keyNode, valueNode, _line, _lineStart, _pos);
        } else if (isPair) {
            _result.push(storeMappingPair(state, null, overridableKeys, keyTag, keyNode, valueNode, _line, _lineStart, _pos));
        } else {
            _result.push(keyNode);
        }
        skipSeparationSpace(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if (ch === 44) {
            readNext = true;
            ch = state.input.charCodeAt(++state.position);
        } else {
            readNext = false;
        }
    }
    throwError(state, "unexpected end of the stream within a flow collection");
}
function readBlockScalar(state, nodeIndent) {
    var captureStart, folding, chomping = CHOMPING_CLIP, didReadContent = false, detectedIndent = false, textIndent = nodeIndent, emptyLines = 0, atMoreIndented = false, tmp, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch === 124) {
        folding = false;
    } else if (ch === 62) {
        folding = true;
    } else {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    while(ch !== 0){
        ch = state.input.charCodeAt(++state.position);
        if (ch === 43 || ch === 45) {
            if (CHOMPING_CLIP === chomping) {
                chomping = ch === 43 ? CHOMPING_KEEP : CHOMPING_STRIP;
            } else {
                throwError(state, "repeat of a chomping mode identifier");
            }
        } else if ((tmp = fromDecimalCode(ch)) >= 0) {
            if (tmp === 0) {
                throwError(state, "bad explicit indentation width of a block scalar; it cannot be less than one");
            } else if (!detectedIndent) {
                textIndent = nodeIndent + tmp - 1;
                detectedIndent = true;
            } else {
                throwError(state, "repeat of an indentation width identifier");
            }
        } else {
            break;
        }
    }
    if (is_WHITE_SPACE(ch)) {
        do {
            ch = state.input.charCodeAt(++state.position);
        }while (is_WHITE_SPACE(ch))
        if (ch === 35) {
            do {
                ch = state.input.charCodeAt(++state.position);
            }while (!is_EOL(ch) && ch !== 0)
        }
    }
    while(ch !== 0){
        readLineBreak(state);
        state.lineIndent = 0;
        ch = state.input.charCodeAt(state.position);
        while((!detectedIndent || state.lineIndent < textIndent) && ch === 32){
            state.lineIndent++;
            ch = state.input.charCodeAt(++state.position);
        }
        if (!detectedIndent && state.lineIndent > textIndent) {
            textIndent = state.lineIndent;
        }
        if (is_EOL(ch)) {
            emptyLines++;
            continue;
        }
        if (state.lineIndent < textIndent) {
            if (chomping === CHOMPING_KEEP) {
                state.result += common1.repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (chomping === CHOMPING_CLIP) {
                if (didReadContent) {
                    state.result += "\n";
                }
            }
            break;
        }
        if (folding) {
            if (is_WHITE_SPACE(ch)) {
                atMoreIndented = true;
                state.result += common1.repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (atMoreIndented) {
                atMoreIndented = false;
                state.result += common1.repeat("\n", emptyLines + 1);
            } else if (emptyLines === 0) {
                if (didReadContent) {
                    state.result += " ";
                }
            } else {
                state.result += common1.repeat("\n", emptyLines);
            }
        } else {
            state.result += common1.repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
        }
        didReadContent = true;
        detectedIndent = true;
        emptyLines = 0;
        captureStart = state.position;
        while(!is_EOL(ch) && ch !== 0){
            ch = state.input.charCodeAt(++state.position);
        }
        captureSegment(state, captureStart, state.position, false);
    }
    return true;
}
function readBlockSequence(state, nodeIndent) {
    var _line, _tag = state.tag, _anchor = state.anchor, _result = [], following, detected = false, ch;
    if (state.firstTabInLine !== -1) return false;
    if (state.anchor !== null) {
        state.anchorMap[state.anchor] = _result;
    }
    ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        if (state.firstTabInLine !== -1) {
            state.position = state.firstTabInLine;
            throwError(state, "tab characters must not be used in indentation");
        }
        if (ch !== 45) {
            break;
        }
        following = state.input.charCodeAt(state.position + 1);
        if (!is_WS_OR_EOL(following)) {
            break;
        }
        detected = true;
        state.position++;
        if (skipSeparationSpace(state, true, -1)) {
            if (state.lineIndent <= nodeIndent) {
                _result.push(null);
                ch = state.input.charCodeAt(state.position);
                continue;
            }
        }
        _line = state.line;
        composeNode(state, nodeIndent, CONTEXT_BLOCK_IN, false, true);
        _result.push(state.result);
        skipSeparationSpace(state, true, -1);
        ch = state.input.charCodeAt(state.position);
        if ((state.line === _line || state.lineIndent > nodeIndent) && ch !== 0) {
            throwError(state, "bad indentation of a sequence entry");
        } else if (state.lineIndent < nodeIndent) {
            break;
        }
    }
    if (detected) {
        state.tag = _tag;
        state.anchor = _anchor;
        state.kind = "sequence";
        state.result = _result;
        return true;
    }
    return false;
}
function readBlockMapping(state, nodeIndent, flowIndent) {
    var following, allowCompact, _line, _keyLine, _keyLineStart, _keyPos, _tag = state.tag, _anchor = state.anchor, _result = {}, overridableKeys = Object.create(null), keyTag = null, keyNode = null, valueNode = null, atExplicitKey = false, detected = false, ch;
    if (state.firstTabInLine !== -1) return false;
    if (state.anchor !== null) {
        state.anchorMap[state.anchor] = _result;
    }
    ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        if (!atExplicitKey && state.firstTabInLine !== -1) {
            state.position = state.firstTabInLine;
            throwError(state, "tab characters must not be used in indentation");
        }
        following = state.input.charCodeAt(state.position + 1);
        _line = state.line;
        if ((ch === 63 || ch === 58) && is_WS_OR_EOL(following)) {
            if (ch === 63) {
                if (atExplicitKey) {
                    storeMappingPair(state, _result, overridableKeys, keyTag, keyNode, null, _keyLine, _keyLineStart, _keyPos);
                    keyTag = keyNode = valueNode = null;
                }
                detected = true;
                atExplicitKey = true;
                allowCompact = true;
            } else if (atExplicitKey) {
                atExplicitKey = false;
                allowCompact = true;
            } else {
                throwError(state, "incomplete explicit mapping pair; a key node is missed; or followed by a non-tabulated empty line");
            }
            state.position += 1;
            ch = following;
        } else {
            _keyLine = state.line;
            _keyLineStart = state.lineStart;
            _keyPos = state.position;
            if (!composeNode(state, flowIndent, CONTEXT_FLOW_OUT, false, true)) {
                break;
            }
            if (state.line === _line) {
                ch = state.input.charCodeAt(state.position);
                while(is_WHITE_SPACE(ch)){
                    ch = state.input.charCodeAt(++state.position);
                }
                if (ch === 58) {
                    ch = state.input.charCodeAt(++state.position);
                    if (!is_WS_OR_EOL(ch)) {
                        throwError(state, "a whitespace character is expected after the key-value separator within a block mapping");
                    }
                    if (atExplicitKey) {
                        storeMappingPair(state, _result, overridableKeys, keyTag, keyNode, null, _keyLine, _keyLineStart, _keyPos);
                        keyTag = keyNode = valueNode = null;
                    }
                    detected = true;
                    atExplicitKey = false;
                    allowCompact = false;
                    keyTag = state.tag;
                    keyNode = state.result;
                } else if (detected) {
                    throwError(state, "can not read an implicit mapping pair; a colon is missed");
                } else {
                    state.tag = _tag;
                    state.anchor = _anchor;
                    return true;
                }
            } else if (detected) {
                throwError(state, "can not read a block mapping entry; a multiline key may not be an implicit key");
            } else {
                state.tag = _tag;
                state.anchor = _anchor;
                return true;
            }
        }
        if (state.line === _line || state.lineIndent > nodeIndent) {
            if (atExplicitKey) {
                _keyLine = state.line;
                _keyLineStart = state.lineStart;
                _keyPos = state.position;
            }
            if (composeNode(state, nodeIndent, CONTEXT_BLOCK_OUT, true, allowCompact)) {
                if (atExplicitKey) {
                    keyNode = state.result;
                } else {
                    valueNode = state.result;
                }
            }
            if (!atExplicitKey) {
                storeMappingPair(state, _result, overridableKeys, keyTag, keyNode, valueNode, _keyLine, _keyLineStart, _keyPos);
                keyTag = keyNode = valueNode = null;
            }
            skipSeparationSpace(state, true, -1);
            ch = state.input.charCodeAt(state.position);
        }
        if ((state.line === _line || state.lineIndent > nodeIndent) && ch !== 0) {
            throwError(state, "bad indentation of a mapping entry");
        } else if (state.lineIndent < nodeIndent) {
            break;
        }
    }
    if (atExplicitKey) {
        storeMappingPair(state, _result, overridableKeys, keyTag, keyNode, null, _keyLine, _keyLineStart, _keyPos);
    }
    if (detected) {
        state.tag = _tag;
        state.anchor = _anchor;
        state.kind = "mapping";
        state.result = _result;
    }
    return detected;
}
function readTagProperty(state) {
    var _position, isVerbatim = false, isNamed = false, tagHandle, tagName, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 33) return false;
    if (state.tag !== null) {
        throwError(state, "duplication of a tag property");
    }
    ch = state.input.charCodeAt(++state.position);
    if (ch === 60) {
        isVerbatim = true;
        ch = state.input.charCodeAt(++state.position);
    } else if (ch === 33) {
        isNamed = true;
        tagHandle = "!!";
        ch = state.input.charCodeAt(++state.position);
    } else {
        tagHandle = "!";
    }
    _position = state.position;
    if (isVerbatim) {
        do {
            ch = state.input.charCodeAt(++state.position);
        }while (ch !== 0 && ch !== 62)
        if (state.position < state.length) {
            tagName = state.input.slice(_position, state.position);
            ch = state.input.charCodeAt(++state.position);
        } else {
            throwError(state, "unexpected end of the stream within a verbatim tag");
        }
    } else {
        while(ch !== 0 && !is_WS_OR_EOL(ch)){
            if (ch === 33) {
                if (!isNamed) {
                    tagHandle = state.input.slice(_position - 1, state.position + 1);
                    if (!PATTERN_TAG_HANDLE.test(tagHandle)) {
                        throwError(state, "named tag handle cannot contain such characters");
                    }
                    isNamed = true;
                    _position = state.position + 1;
                } else {
                    throwError(state, "tag suffix cannot contain exclamation marks");
                }
            }
            ch = state.input.charCodeAt(++state.position);
        }
        tagName = state.input.slice(_position, state.position);
        if (PATTERN_FLOW_INDICATORS.test(tagName)) {
            throwError(state, "tag suffix cannot contain flow indicator characters");
        }
    }
    if (tagName && !PATTERN_TAG_URI.test(tagName)) {
        throwError(state, "tag name cannot contain such characters: " + tagName);
    }
    try {
        tagName = decodeURIComponent(tagName);
    } catch (err) {
        throwError(state, "tag name is malformed: " + tagName);
    }
    if (isVerbatim) {
        state.tag = tagName;
    } else if (_hasOwnProperty$1.call(state.tagMap, tagHandle)) {
        state.tag = state.tagMap[tagHandle] + tagName;
    } else if (tagHandle === "!") {
        state.tag = "!" + tagName;
    } else if (tagHandle === "!!") {
        state.tag = "tag:yaml.org,2002:" + tagName;
    } else {
        throwError(state, 'undeclared tag handle "' + tagHandle + '"');
    }
    return true;
}
function readAnchorProperty(state) {
    var _position, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 38) return false;
    if (state.anchor !== null) {
        throwError(state, "duplication of an anchor property");
    }
    ch = state.input.charCodeAt(++state.position);
    _position = state.position;
    while(ch !== 0 && !is_WS_OR_EOL(ch) && !is_FLOW_INDICATOR(ch)){
        ch = state.input.charCodeAt(++state.position);
    }
    if (state.position === _position) {
        throwError(state, "name of an anchor node must contain at least one character");
    }
    state.anchor = state.input.slice(_position, state.position);
    return true;
}
function readAlias(state) {
    var _position, alias, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 42) return false;
    ch = state.input.charCodeAt(++state.position);
    _position = state.position;
    while(ch !== 0 && !is_WS_OR_EOL(ch) && !is_FLOW_INDICATOR(ch)){
        ch = state.input.charCodeAt(++state.position);
    }
    if (state.position === _position) {
        throwError(state, "name of an alias node must contain at least one character");
    }
    alias = state.input.slice(_position, state.position);
    if (!_hasOwnProperty$1.call(state.anchorMap, alias)) {
        throwError(state, 'unidentified alias "' + alias + '"');
    }
    state.result = state.anchorMap[alias];
    skipSeparationSpace(state, true, -1);
    return true;
}
function composeNode(state, parentIndent, nodeContext, allowToSeek, allowCompact) {
    var allowBlockStyles, allowBlockScalars, allowBlockCollections, indentStatus = 1, atNewLine = false, hasContent = false, typeIndex, typeQuantity, typeList, type2, flowIndent, blockIndent;
    if (state.listener !== null) {
        state.listener("open", state);
    }
    state.tag = null;
    state.anchor = null;
    state.kind = null;
    state.result = null;
    allowBlockStyles = allowBlockScalars = allowBlockCollections = CONTEXT_BLOCK_OUT === nodeContext || CONTEXT_BLOCK_IN === nodeContext;
    if (allowToSeek) {
        if (skipSeparationSpace(state, true, -1)) {
            atNewLine = true;
            if (state.lineIndent > parentIndent) {
                indentStatus = 1;
            } else if (state.lineIndent === parentIndent) {
                indentStatus = 0;
            } else if (state.lineIndent < parentIndent) {
                indentStatus = -1;
            }
        }
    }
    if (indentStatus === 1) {
        while(readTagProperty(state) || readAnchorProperty(state)){
            if (skipSeparationSpace(state, true, -1)) {
                atNewLine = true;
                allowBlockCollections = allowBlockStyles;
                if (state.lineIndent > parentIndent) {
                    indentStatus = 1;
                } else if (state.lineIndent === parentIndent) {
                    indentStatus = 0;
                } else if (state.lineIndent < parentIndent) {
                    indentStatus = -1;
                }
            } else {
                allowBlockCollections = false;
            }
        }
    }
    if (allowBlockCollections) {
        allowBlockCollections = atNewLine || allowCompact;
    }
    if (indentStatus === 1 || CONTEXT_BLOCK_OUT === nodeContext) {
        if (CONTEXT_FLOW_IN === nodeContext || CONTEXT_FLOW_OUT === nodeContext) {
            flowIndent = parentIndent;
        } else {
            flowIndent = parentIndent + 1;
        }
        blockIndent = state.position - state.lineStart;
        if (indentStatus === 1) {
            if (allowBlockCollections && (readBlockSequence(state, blockIndent) || readBlockMapping(state, blockIndent, flowIndent)) || readFlowCollection(state, flowIndent)) {
                hasContent = true;
            } else {
                if (allowBlockScalars && readBlockScalar(state, flowIndent) || readSingleQuotedScalar(state, flowIndent) || readDoubleQuotedScalar(state, flowIndent)) {
                    hasContent = true;
                } else if (readAlias(state)) {
                    hasContent = true;
                    if (state.tag !== null || state.anchor !== null) {
                        throwError(state, "alias node should not have any properties");
                    }
                } else if (readPlainScalar(state, flowIndent, CONTEXT_FLOW_IN === nodeContext)) {
                    hasContent = true;
                    if (state.tag === null) {
                        state.tag = "?";
                    }
                }
                if (state.anchor !== null) {
                    state.anchorMap[state.anchor] = state.result;
                }
            }
        } else if (indentStatus === 0) {
            hasContent = allowBlockCollections && readBlockSequence(state, blockIndent);
        }
    }
    if (state.tag === null) {
        if (state.anchor !== null) {
            state.anchorMap[state.anchor] = state.result;
        }
    } else if (state.tag === "?") {
        if (state.result !== null && state.kind !== "scalar") {
            throwError(state, 'unacceptable node kind for !<?> tag; it should be "scalar", not "' + state.kind + '"');
        }
        for(typeIndex = 0, typeQuantity = state.implicitTypes.length; typeIndex < typeQuantity; typeIndex += 1){
            type2 = state.implicitTypes[typeIndex];
            if (type2.resolve(state.result)) {
                state.result = type2.construct(state.result);
                state.tag = type2.tag;
                if (state.anchor !== null) {
                    state.anchorMap[state.anchor] = state.result;
                }
                break;
            }
        }
    } else if (state.tag !== "!") {
        if (_hasOwnProperty$1.call(state.typeMap[state.kind || "fallback"], state.tag)) {
            type2 = state.typeMap[state.kind || "fallback"][state.tag];
        } else {
            type2 = null;
            typeList = state.typeMap.multi[state.kind || "fallback"];
            for(typeIndex = 0, typeQuantity = typeList.length; typeIndex < typeQuantity; typeIndex += 1){
                if (state.tag.slice(0, typeList[typeIndex].tag.length) === typeList[typeIndex].tag) {
                    type2 = typeList[typeIndex];
                    break;
                }
            }
        }
        if (!type2) {
            throwError(state, "unknown tag !<" + state.tag + ">");
        }
        if (state.result !== null && type2.kind !== state.kind) {
            throwError(state, "unacceptable node kind for !<" + state.tag + '> tag; it should be "' + type2.kind + '", not "' + state.kind + '"');
        }
        if (!type2.resolve(state.result, state.tag)) {
            throwError(state, "cannot resolve a node with !<" + state.tag + "> explicit tag");
        } else {
            state.result = type2.construct(state.result, state.tag);
            if (state.anchor !== null) {
                state.anchorMap[state.anchor] = state.result;
            }
        }
    }
    if (state.listener !== null) {
        state.listener("close", state);
    }
    return state.tag !== null || state.anchor !== null || hasContent;
}
function readDocument(state) {
    var documentStart = state.position, _position, directiveName, directiveArgs, hasDirectives = false, ch;
    state.version = null;
    state.checkLineBreaks = state.legacy;
    state.tagMap = Object.create(null);
    state.anchorMap = Object.create(null);
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        skipSeparationSpace(state, true, -1);
        ch = state.input.charCodeAt(state.position);
        if (state.lineIndent > 0 || ch !== 37) {
            break;
        }
        hasDirectives = true;
        ch = state.input.charCodeAt(++state.position);
        _position = state.position;
        while(ch !== 0 && !is_WS_OR_EOL(ch)){
            ch = state.input.charCodeAt(++state.position);
        }
        directiveName = state.input.slice(_position, state.position);
        directiveArgs = [];
        if (directiveName.length < 1) {
            throwError(state, "directive name must not be less than one character in length");
        }
        while(ch !== 0){
            while(is_WHITE_SPACE(ch)){
                ch = state.input.charCodeAt(++state.position);
            }
            if (ch === 35) {
                do {
                    ch = state.input.charCodeAt(++state.position);
                }while (ch !== 0 && !is_EOL(ch))
                break;
            }
            if (is_EOL(ch)) break;
            _position = state.position;
            while(ch !== 0 && !is_WS_OR_EOL(ch)){
                ch = state.input.charCodeAt(++state.position);
            }
            directiveArgs.push(state.input.slice(_position, state.position));
        }
        if (ch !== 0) readLineBreak(state);
        if (_hasOwnProperty$1.call(directiveHandlers, directiveName)) {
            directiveHandlers[directiveName](state, directiveName, directiveArgs);
        } else {
            throwWarning(state, 'unknown document directive "' + directiveName + '"');
        }
    }
    skipSeparationSpace(state, true, -1);
    if (state.lineIndent === 0 && state.input.charCodeAt(state.position) === 45 && state.input.charCodeAt(state.position + 1) === 45 && state.input.charCodeAt(state.position + 2) === 45) {
        state.position += 3;
        skipSeparationSpace(state, true, -1);
    } else if (hasDirectives) {
        throwError(state, "directives end mark is expected");
    }
    composeNode(state, state.lineIndent - 1, CONTEXT_BLOCK_OUT, false, true);
    skipSeparationSpace(state, true, -1);
    if (state.checkLineBreaks && PATTERN_NON_ASCII_LINE_BREAKS.test(state.input.slice(documentStart, state.position))) {
        throwWarning(state, "non-ASCII line breaks are interpreted as content");
    }
    state.documents.push(state.result);
    if (state.position === state.lineStart && testDocumentSeparator(state)) {
        if (state.input.charCodeAt(state.position) === 46) {
            state.position += 3;
            skipSeparationSpace(state, true, -1);
        }
        return;
    }
    if (state.position < state.length - 1) {
        throwError(state, "end of the stream or a document separator is expected");
    } else {
        return;
    }
}
function loadDocuments(input, options) {
    input = String(input);
    options = options || {};
    if (input.length !== 0) {
        if (input.charCodeAt(input.length - 1) !== 10 && input.charCodeAt(input.length - 1) !== 13) {
            input += "\n";
        }
        if (input.charCodeAt(0) === 65279) {
            input = input.slice(1);
        }
    }
    var state = new State$1(input, options);
    var nullpos = input.indexOf("\0");
    if (nullpos !== -1) {
        state.position = nullpos;
        throwError(state, "null byte is not allowed in input");
    }
    state.input += "\0";
    while(state.input.charCodeAt(state.position) === 32){
        state.lineIndent += 1;
        state.position += 1;
    }
    while(state.position < state.length - 1){
        readDocument(state);
    }
    return state.documents;
}
function loadAll$1(input, iterator, options) {
    if (iterator !== null && typeof iterator === "object" && typeof options === "undefined") {
        options = iterator;
        iterator = null;
    }
    var documents = loadDocuments(input, options);
    if (typeof iterator !== "function") {
        return documents;
    }
    for(var index = 0, length = documents.length; index < length; index += 1){
        iterator(documents[index]);
    }
}
function load$1(input, options) {
    var documents = loadDocuments(input, options);
    if (documents.length === 0) {
        return void 0;
    } else if (documents.length === 1) {
        return documents[0];
    }
    throw new exception("expected a single document in the stream, but found more");
}
var loadAll_1 = loadAll$1;
var load_1 = load$1;
var loader = {
    loadAll: loadAll_1,
    load: load_1
};
var _toString = Object.prototype.toString;
var _hasOwnProperty = Object.prototype.hasOwnProperty;
var CHAR_BOM = 65279;
var CHAR_TAB = 9;
var CHAR_LINE_FEED = 10;
var CHAR_CARRIAGE_RETURN = 13;
var CHAR_SPACE = 32;
var CHAR_EXCLAMATION = 33;
var CHAR_DOUBLE_QUOTE = 34;
var CHAR_SHARP = 35;
var CHAR_PERCENT = 37;
var CHAR_AMPERSAND = 38;
var CHAR_SINGLE_QUOTE = 39;
var CHAR_ASTERISK = 42;
var CHAR_COMMA = 44;
var CHAR_MINUS = 45;
var CHAR_COLON = 58;
var CHAR_EQUALS = 61;
var CHAR_GREATER_THAN = 62;
var CHAR_QUESTION = 63;
var CHAR_COMMERCIAL_AT = 64;
var CHAR_LEFT_SQUARE_BRACKET = 91;
var CHAR_RIGHT_SQUARE_BRACKET = 93;
var CHAR_GRAVE_ACCENT = 96;
var CHAR_LEFT_CURLY_BRACKET = 123;
var CHAR_VERTICAL_LINE = 124;
var CHAR_RIGHT_CURLY_BRACKET = 125;
var ESCAPE_SEQUENCES = {};
ESCAPE_SEQUENCES[0] = "\\0";
ESCAPE_SEQUENCES[7] = "\\a";
ESCAPE_SEQUENCES[8] = "\\b";
ESCAPE_SEQUENCES[9] = "\\t";
ESCAPE_SEQUENCES[10] = "\\n";
ESCAPE_SEQUENCES[11] = "\\v";
ESCAPE_SEQUENCES[12] = "\\f";
ESCAPE_SEQUENCES[13] = "\\r";
ESCAPE_SEQUENCES[27] = "\\e";
ESCAPE_SEQUENCES[34] = '\\"';
ESCAPE_SEQUENCES[92] = "\\\\";
ESCAPE_SEQUENCES[133] = "\\N";
ESCAPE_SEQUENCES[160] = "\\_";
ESCAPE_SEQUENCES[8232] = "\\L";
ESCAPE_SEQUENCES[8233] = "\\P";
var DEPRECATED_BOOLEANS_SYNTAX = [
    "y",
    "Y",
    "yes",
    "Yes",
    "YES",
    "on",
    "On",
    "ON",
    "n",
    "N",
    "no",
    "No",
    "NO",
    "off",
    "Off",
    "OFF"
];
var DEPRECATED_BASE60_SYNTAX = /^[-+]?[0-9_]+(?::[0-9_]+)+(?:\.[0-9_]*)?$/;
function compileStyleMap(schema2, map2) {
    var result1, keys, index, length, tag, style, type2;
    if (map2 === null) return {};
    result1 = {};
    keys = Object.keys(map2);
    for(index = 0, length = keys.length; index < length; index += 1){
        tag = keys[index];
        style = String(map2[tag]);
        if (tag.slice(0, 2) === "!!") {
            tag = "tag:yaml.org,2002:" + tag.slice(2);
        }
        type2 = schema2.compiledTypeMap["fallback"][tag];
        if (type2 && _hasOwnProperty.call(type2.styleAliases, style)) {
            style = type2.styleAliases[style];
        }
        result1[tag] = style;
    }
    return result1;
}
function encodeHex(character) {
    var string, handle, length;
    string = character.toString(16).toUpperCase();
    if (character <= 255) {
        handle = "x";
        length = 2;
    } else if (character <= 65535) {
        handle = "u";
        length = 4;
    } else if (character <= 4294967295) {
        handle = "U";
        length = 8;
    } else {
        throw new exception("code point within a string may not be greater than 0xFFFFFFFF");
    }
    return "\\" + handle + common1.repeat("0", length - string.length) + string;
}
var QUOTING_TYPE_SINGLE = 1, QUOTING_TYPE_DOUBLE = 2;
function State1(options) {
    this.schema = options["schema"] || _default;
    this.indent = Math.max(1, options["indent"] || 2);
    this.noArrayIndent = options["noArrayIndent"] || false;
    this.skipInvalid = options["skipInvalid"] || false;
    this.flowLevel = common1.isNothing(options["flowLevel"]) ? -1 : options["flowLevel"];
    this.styleMap = compileStyleMap(this.schema, options["styles"] || null);
    this.sortKeys = options["sortKeys"] || false;
    this.lineWidth = options["lineWidth"] || 80;
    this.noRefs = options["noRefs"] || false;
    this.noCompatMode = options["noCompatMode"] || false;
    this.condenseFlow = options["condenseFlow"] || false;
    this.quotingType = options["quotingType"] === '"' ? QUOTING_TYPE_DOUBLE : QUOTING_TYPE_SINGLE;
    this.forceQuotes = options["forceQuotes"] || false;
    this.replacer = typeof options["replacer"] === "function" ? options["replacer"] : null;
    this.implicitTypes = this.schema.compiledImplicit;
    this.explicitTypes = this.schema.compiledExplicit;
    this.tag = null;
    this.result = "";
    this.duplicates = [];
    this.usedDuplicates = null;
}
function indentString(string, spaces) {
    var ind = common1.repeat(" ", spaces), position = 0, next = -1, result1 = "", line, length = string.length;
    while(position < length){
        next = string.indexOf("\n", position);
        if (next === -1) {
            line = string.slice(position);
            position = length;
        } else {
            line = string.slice(position, next + 1);
            position = next + 1;
        }
        if (line.length && line !== "\n") result1 += ind;
        result1 += line;
    }
    return result1;
}
function generateNextLine(state, level) {
    return "\n" + common1.repeat(" ", state.indent * level);
}
function testImplicitResolving(state, str2) {
    var index, length, type2;
    for(index = 0, length = state.implicitTypes.length; index < length; index += 1){
        type2 = state.implicitTypes[index];
        if (type2.resolve(str2)) {
            return true;
        }
    }
    return false;
}
function isWhitespace(c) {
    return c === CHAR_SPACE || c === CHAR_TAB;
}
function isPrintable(c) {
    return 32 <= c && c <= 126 || 161 <= c && c <= 55295 && c !== 8232 && c !== 8233 || 57344 <= c && c <= 65533 && c !== CHAR_BOM || 65536 <= c && c <= 1114111;
}
function isNsCharOrWhitespace(c) {
    return isPrintable(c) && c !== CHAR_BOM && c !== CHAR_CARRIAGE_RETURN && c !== CHAR_LINE_FEED;
}
function isPlainSafe(c, prev, inblock) {
    var cIsNsCharOrWhitespace = isNsCharOrWhitespace(c);
    var cIsNsChar = cIsNsCharOrWhitespace && !isWhitespace(c);
    return (inblock ? cIsNsCharOrWhitespace : cIsNsCharOrWhitespace && c !== CHAR_COMMA && c !== CHAR_LEFT_SQUARE_BRACKET && c !== CHAR_RIGHT_SQUARE_BRACKET && c !== CHAR_LEFT_CURLY_BRACKET && c !== CHAR_RIGHT_CURLY_BRACKET) && c !== CHAR_SHARP && !(prev === CHAR_COLON && !cIsNsChar) || isNsCharOrWhitespace(prev) && !isWhitespace(prev) && c === CHAR_SHARP || prev === CHAR_COLON && cIsNsChar;
}
function isPlainSafeFirst(c) {
    return isPrintable(c) && c !== CHAR_BOM && !isWhitespace(c) && c !== CHAR_MINUS && c !== CHAR_QUESTION && c !== CHAR_COLON && c !== CHAR_COMMA && c !== CHAR_LEFT_SQUARE_BRACKET && c !== CHAR_RIGHT_SQUARE_BRACKET && c !== CHAR_LEFT_CURLY_BRACKET && c !== CHAR_RIGHT_CURLY_BRACKET && c !== CHAR_SHARP && c !== CHAR_AMPERSAND && c !== CHAR_ASTERISK && c !== CHAR_EXCLAMATION && c !== CHAR_VERTICAL_LINE && c !== CHAR_EQUALS && c !== CHAR_GREATER_THAN && c !== CHAR_SINGLE_QUOTE && c !== CHAR_DOUBLE_QUOTE && c !== CHAR_PERCENT && c !== CHAR_COMMERCIAL_AT && c !== CHAR_GRAVE_ACCENT;
}
function isPlainSafeLast(c) {
    return !isWhitespace(c) && c !== CHAR_COLON;
}
function codePointAt(string, pos) {
    var first = string.charCodeAt(pos), second;
    if (first >= 55296 && first <= 56319 && pos + 1 < string.length) {
        second = string.charCodeAt(pos + 1);
        if (second >= 56320 && second <= 57343) {
            return (first - 55296) * 1024 + second - 56320 + 65536;
        }
    }
    return first;
}
function needIndentIndicator(string) {
    var leadingSpaceRe = /^\n* /;
    return leadingSpaceRe.test(string);
}
var STYLE_PLAIN = 1, STYLE_SINGLE = 2, STYLE_LITERAL = 3, STYLE_FOLDED = 4, STYLE_DOUBLE = 5;
function chooseScalarStyle(string, singleLineOnly, indentPerLevel, lineWidth, testAmbiguousType, quotingType, forceQuotes, inblock) {
    var i1;
    var __char = 0;
    var prevChar = null;
    var hasLineBreak = false;
    var hasFoldableLine = false;
    var shouldTrackWidth = lineWidth !== -1;
    var previousLineBreak = -1;
    var plain = isPlainSafeFirst(codePointAt(string, 0)) && isPlainSafeLast(codePointAt(string, string.length - 1));
    if (singleLineOnly || forceQuotes) {
        for(i1 = 0; i1 < string.length; __char >= 65536 ? i1 += 2 : i1++){
            __char = codePointAt(string, i1);
            if (!isPrintable(__char)) {
                return STYLE_DOUBLE;
            }
            plain = plain && isPlainSafe(__char, prevChar, inblock);
            prevChar = __char;
        }
    } else {
        for(i1 = 0; i1 < string.length; __char >= 65536 ? i1 += 2 : i1++){
            __char = codePointAt(string, i1);
            if (__char === CHAR_LINE_FEED) {
                hasLineBreak = true;
                if (shouldTrackWidth) {
                    hasFoldableLine = hasFoldableLine || i1 - previousLineBreak - 1 > lineWidth && string[previousLineBreak + 1] !== " ";
                    previousLineBreak = i1;
                }
            } else if (!isPrintable(__char)) {
                return STYLE_DOUBLE;
            }
            plain = plain && isPlainSafe(__char, prevChar, inblock);
            prevChar = __char;
        }
        hasFoldableLine = hasFoldableLine || shouldTrackWidth && i1 - previousLineBreak - 1 > lineWidth && string[previousLineBreak + 1] !== " ";
    }
    if (!hasLineBreak && !hasFoldableLine) {
        if (plain && !forceQuotes && !testAmbiguousType(string)) {
            return STYLE_PLAIN;
        }
        return quotingType === QUOTING_TYPE_DOUBLE ? STYLE_DOUBLE : STYLE_SINGLE;
    }
    if (indentPerLevel > 9 && needIndentIndicator(string)) {
        return STYLE_DOUBLE;
    }
    if (!forceQuotes) {
        return hasFoldableLine ? STYLE_FOLDED : STYLE_LITERAL;
    }
    return quotingType === QUOTING_TYPE_DOUBLE ? STYLE_DOUBLE : STYLE_SINGLE;
}
function writeScalar(state, string, level, iskey, inblock) {
    state.dump = function() {
        if (string.length === 0) {
            return state.quotingType === QUOTING_TYPE_DOUBLE ? '""' : "''";
        }
        if (!state.noCompatMode) {
            if (DEPRECATED_BOOLEANS_SYNTAX.indexOf(string) !== -1 || DEPRECATED_BASE60_SYNTAX.test(string)) {
                return state.quotingType === QUOTING_TYPE_DOUBLE ? '"' + string + '"' : "'" + string + "'";
            }
        }
        var indent = state.indent * Math.max(1, level);
        var lineWidth = state.lineWidth === -1 ? -1 : Math.max(Math.min(state.lineWidth, 40), state.lineWidth - indent);
        var singleLineOnly = iskey || state.flowLevel > -1 && level >= state.flowLevel;
        function testAmbiguity(string2) {
            return testImplicitResolving(state, string2);
        }
        switch(chooseScalarStyle(string, singleLineOnly, state.indent, lineWidth, testAmbiguity, state.quotingType, state.forceQuotes && !iskey, inblock)){
            case STYLE_PLAIN:
                return string;
            case STYLE_SINGLE:
                return "'" + string.replace(/'/g, "''") + "'";
            case STYLE_LITERAL:
                return "|" + blockHeader(string, state.indent) + dropEndingNewline(indentString(string, indent));
            case STYLE_FOLDED:
                return ">" + blockHeader(string, state.indent) + dropEndingNewline(indentString(foldString(string, lineWidth), indent));
            case STYLE_DOUBLE:
                return '"' + escapeString(string) + '"';
            default:
                throw new exception("impossible error: invalid scalar style");
        }
    }();
}
function blockHeader(string, indentPerLevel) {
    var indentIndicator = needIndentIndicator(string) ? String(indentPerLevel) : "";
    var clip = string[string.length - 1] === "\n";
    var keep = clip && (string[string.length - 2] === "\n" || string === "\n");
    var chomp = keep ? "+" : clip ? "" : "-";
    return indentIndicator + chomp + "\n";
}
function dropEndingNewline(string) {
    return string[string.length - 1] === "\n" ? string.slice(0, -1) : string;
}
function foldString(string, width) {
    var lineRe = /(\n+)([^\n]*)/g;
    var result1 = function() {
        var nextLF = string.indexOf("\n");
        nextLF = nextLF !== -1 ? nextLF : string.length;
        lineRe.lastIndex = nextLF;
        return foldLine(string.slice(0, nextLF), width);
    }();
    var prevMoreIndented = string[0] === "\n" || string[0] === " ";
    var moreIndented;
    var match;
    while(match = lineRe.exec(string)){
        var prefix = match[1], line = match[2];
        moreIndented = line[0] === " ";
        result1 += prefix + (!prevMoreIndented && !moreIndented && line !== "" ? "\n" : "") + foldLine(line, width);
        prevMoreIndented = moreIndented;
    }
    return result1;
}
function foldLine(line, width) {
    if (line === "" || line[0] === " ") return line;
    var breakRe = / [^ ]/g;
    var match;
    var start = 0, end, curr = 0, next = 0;
    var result1 = "";
    while(match = breakRe.exec(line)){
        next = match.index;
        if (next - start > width) {
            end = curr > start ? curr : next;
            result1 += "\n" + line.slice(start, end);
            start = end + 1;
        }
        curr = next;
    }
    result1 += "\n";
    if (line.length - start > width && curr > start) {
        result1 += line.slice(start, curr) + "\n" + line.slice(curr + 1);
    } else {
        result1 += line.slice(start);
    }
    return result1.slice(1);
}
function escapeString(string) {
    var result1 = "";
    var __char = 0;
    var escapeSeq;
    for(var i1 = 0; i1 < string.length; __char >= 65536 ? i1 += 2 : i1++){
        __char = codePointAt(string, i1);
        escapeSeq = ESCAPE_SEQUENCES[__char];
        if (!escapeSeq && isPrintable(__char)) {
            result1 += string[i1];
            if (__char >= 65536) result1 += string[i1 + 1];
        } else {
            result1 += escapeSeq || encodeHex(__char);
        }
    }
    return result1;
}
function writeFlowSequence(state, level, object) {
    var _result = "", _tag = state.tag, index, length, value;
    for(index = 0, length = object.length; index < length; index += 1){
        value = object[index];
        if (state.replacer) {
            value = state.replacer.call(object, String(index), value);
        }
        if (writeNode(state, level, value, false, false) || typeof value === "undefined" && writeNode(state, level, null, false, false)) {
            if (_result !== "") _result += "," + (!state.condenseFlow ? " " : "");
            _result += state.dump;
        }
    }
    state.tag = _tag;
    state.dump = "[" + _result + "]";
}
function writeBlockSequence(state, level, object, compact) {
    var _result = "", _tag = state.tag, index, length, value;
    for(index = 0, length = object.length; index < length; index += 1){
        value = object[index];
        if (state.replacer) {
            value = state.replacer.call(object, String(index), value);
        }
        if (writeNode(state, level + 1, value, true, true, false, true) || typeof value === "undefined" && writeNode(state, level + 1, null, true, true, false, true)) {
            if (!compact || _result !== "") {
                _result += generateNextLine(state, level);
            }
            if (state.dump && CHAR_LINE_FEED === state.dump.charCodeAt(0)) {
                _result += "-";
            } else {
                _result += "- ";
            }
            _result += state.dump;
        }
    }
    state.tag = _tag;
    state.dump = _result || "[]";
}
function writeFlowMapping(state, level, object) {
    var _result = "", _tag = state.tag, objectKeyList = Object.keys(object), index, length, objectKey, objectValue, pairBuffer;
    for(index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = "";
        if (_result !== "") pairBuffer += ", ";
        if (state.condenseFlow) pairBuffer += '"';
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (state.replacer) {
            objectValue = state.replacer.call(object, objectKey, objectValue);
        }
        if (!writeNode(state, level, objectKey, false, false)) {
            continue;
        }
        if (state.dump.length > 1024) pairBuffer += "? ";
        pairBuffer += state.dump + (state.condenseFlow ? '"' : "") + ":" + (state.condenseFlow ? "" : " ");
        if (!writeNode(state, level, objectValue, false, false)) {
            continue;
        }
        pairBuffer += state.dump;
        _result += pairBuffer;
    }
    state.tag = _tag;
    state.dump = "{" + _result + "}";
}
function writeBlockMapping(state, level, object, compact) {
    var _result = "", _tag = state.tag, objectKeyList = Object.keys(object), index, length, objectKey, objectValue, explicitPair, pairBuffer;
    if (state.sortKeys === true) {
        objectKeyList.sort();
    } else if (typeof state.sortKeys === "function") {
        objectKeyList.sort(state.sortKeys);
    } else if (state.sortKeys) {
        throw new exception("sortKeys must be a boolean or a function");
    }
    for(index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = "";
        if (!compact || _result !== "") {
            pairBuffer += generateNextLine(state, level);
        }
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (state.replacer) {
            objectValue = state.replacer.call(object, objectKey, objectValue);
        }
        if (!writeNode(state, level + 1, objectKey, true, true, true)) {
            continue;
        }
        explicitPair = state.tag !== null && state.tag !== "?" || state.dump && state.dump.length > 1024;
        if (explicitPair) {
            if (state.dump && CHAR_LINE_FEED === state.dump.charCodeAt(0)) {
                pairBuffer += "?";
            } else {
                pairBuffer += "? ";
            }
        }
        pairBuffer += state.dump;
        if (explicitPair) {
            pairBuffer += generateNextLine(state, level);
        }
        if (!writeNode(state, level + 1, objectValue, true, explicitPair)) {
            continue;
        }
        if (state.dump && CHAR_LINE_FEED === state.dump.charCodeAt(0)) {
            pairBuffer += ":";
        } else {
            pairBuffer += ": ";
        }
        pairBuffer += state.dump;
        _result += pairBuffer;
    }
    state.tag = _tag;
    state.dump = _result || "{}";
}
function detectType(state, object, explicit) {
    var _result, typeList, index, length, type2, style;
    typeList = explicit ? state.explicitTypes : state.implicitTypes;
    for(index = 0, length = typeList.length; index < length; index += 1){
        type2 = typeList[index];
        if ((type2.instanceOf || type2.predicate) && (!type2.instanceOf || typeof object === "object" && object instanceof type2.instanceOf) && (!type2.predicate || type2.predicate(object))) {
            if (explicit) {
                if (type2.multi && type2.representName) {
                    state.tag = type2.representName(object);
                } else {
                    state.tag = type2.tag;
                }
            } else {
                state.tag = "?";
            }
            if (type2.represent) {
                style = state.styleMap[type2.tag] || type2.defaultStyle;
                if (_toString.call(type2.represent) === "[object Function]") {
                    _result = type2.represent(object, style);
                } else if (_hasOwnProperty.call(type2.represent, style)) {
                    _result = type2.represent[style](object, style);
                } else {
                    throw new exception("!<" + type2.tag + '> tag resolver accepts not "' + style + '" style');
                }
                state.dump = _result;
            }
            return true;
        }
    }
    return false;
}
function writeNode(state, level, object, block, compact, iskey, isblockseq) {
    state.tag = null;
    state.dump = object;
    if (!detectType(state, object, false)) {
        detectType(state, object, true);
    }
    var type2 = _toString.call(state.dump);
    var inblock = block;
    var tagStr;
    if (block) {
        block = state.flowLevel < 0 || state.flowLevel > level;
    }
    var objectOrArray = type2 === "[object Object]" || type2 === "[object Array]", duplicateIndex, duplicate;
    if (objectOrArray) {
        duplicateIndex = state.duplicates.indexOf(object);
        duplicate = duplicateIndex !== -1;
    }
    if (state.tag !== null && state.tag !== "?" || duplicate || state.indent !== 2 && level > 0) {
        compact = false;
    }
    if (duplicate && state.usedDuplicates[duplicateIndex]) {
        state.dump = "*ref_" + duplicateIndex;
    } else {
        if (objectOrArray && duplicate && !state.usedDuplicates[duplicateIndex]) {
            state.usedDuplicates[duplicateIndex] = true;
        }
        if (type2 === "[object Object]") {
            if (block && Object.keys(state.dump).length !== 0) {
                writeBlockMapping(state, level, state.dump, compact);
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + state.dump;
                }
            } else {
                writeFlowMapping(state, level, state.dump);
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + " " + state.dump;
                }
            }
        } else if (type2 === "[object Array]") {
            if (block && state.dump.length !== 0) {
                if (state.noArrayIndent && !isblockseq && level > 0) {
                    writeBlockSequence(state, level - 1, state.dump, compact);
                } else {
                    writeBlockSequence(state, level, state.dump, compact);
                }
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + state.dump;
                }
            } else {
                writeFlowSequence(state, level, state.dump);
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + " " + state.dump;
                }
            }
        } else if (type2 === "[object String]") {
            if (state.tag !== "?") {
                writeScalar(state, state.dump, level, iskey, inblock);
            }
        } else if (type2 === "[object Undefined]") {
            return false;
        } else {
            if (state.skipInvalid) return false;
            throw new exception("unacceptable kind of an object to dump " + type2);
        }
        if (state.tag !== null && state.tag !== "?") {
            tagStr = encodeURI(state.tag[0] === "!" ? state.tag.slice(1) : state.tag).replace(/!/g, "%21");
            if (state.tag[0] === "!") {
                tagStr = "!" + tagStr;
            } else if (tagStr.slice(0, 18) === "tag:yaml.org,2002:") {
                tagStr = "!!" + tagStr.slice(18);
            } else {
                tagStr = "!<" + tagStr + ">";
            }
            state.dump = tagStr + " " + state.dump;
        }
    }
    return true;
}
function getDuplicateReferences(object, state) {
    var objects = [], duplicatesIndexes = [], index, length;
    inspectNode(object, objects, duplicatesIndexes);
    for(index = 0, length = duplicatesIndexes.length; index < length; index += 1){
        state.duplicates.push(objects[duplicatesIndexes[index]]);
    }
    state.usedDuplicates = new Array(length);
}
function inspectNode(object, objects, duplicatesIndexes) {
    var objectKeyList, index, length;
    if (object !== null && typeof object === "object") {
        index = objects.indexOf(object);
        if (index !== -1) {
            if (duplicatesIndexes.indexOf(index) === -1) {
                duplicatesIndexes.push(index);
            }
        } else {
            objects.push(object);
            if (Array.isArray(object)) {
                for(index = 0, length = object.length; index < length; index += 1){
                    inspectNode(object[index], objects, duplicatesIndexes);
                }
            } else {
                objectKeyList = Object.keys(object);
                for(index = 0, length = objectKeyList.length; index < length; index += 1){
                    inspectNode(object[objectKeyList[index]], objects, duplicatesIndexes);
                }
            }
        }
    }
}
function dump$1(input, options) {
    options = options || {};
    var state = new State1(options);
    if (!state.noRefs) getDuplicateReferences(input, state);
    var value = input;
    if (state.replacer) {
        value = state.replacer.call({
            "": value
        }, "", value);
    }
    if (writeNode(state, 0, value, true, true)) return state.dump + "\n";
    return "";
}
var dump_1 = dump$1;
var dumper = {
    dump: dump_1
};
function renamed(from, to) {
    return function() {
        throw new Error("Function yaml." + from + " is removed in js-yaml 4. Use yaml." + to + " instead, which is now safe by default.");
    };
}
var Type1 = type;
var Schema = schema;
var load = loader.load;
loader.loadAll;
dumper.dump;
renamed("safeLoad", "load");
renamed("safeLoadAll", "loadAll");
renamed("safeDump", "dump");
const QuartoJSONSchema = new Schema({
    implicit: [
        _null,
        bool,
        __int,
        __float
    ],
    include: [
        failsafe
    ],
    explicit: [
        new Type1("!expr", {
            kind: "scalar",
            construct (data) {
                const result1 = data !== null ? data : "";
                return {
                    value: result1,
                    tag: "!expr"
                };
            }
        })
    ]
});
function postProcessAnnotation(parse) {
    if (parse.components.length === 1 && parse.start === parse.components[0].start && parse.end === parse.components[0].end) {
        return postProcessAnnotation(parse.components[0]);
    } else {
        parse.components = parse.components.map(postProcessAnnotation);
        return parse;
    }
}
function jsYamlParseLenient(yml) {
    try {
        return load(yml, {
            schema: QuartoJSONSchema
        });
    } catch (_e) {
        return yml;
    }
}
function readAnnotatedYamlFromMappedString(mappedSource, lenient = false) {
    if (lenient) {
        const parser = getTreeSitterSync();
        const tree = parser.parse(mappedSource.value);
        const treeSitterAnnotation = buildTreeSitterAnnotation(tree, mappedSource);
        if (treeSitterAnnotation) {
            return treeSitterAnnotation;
        }
    }
    try {
        return buildJsYamlAnnotation(mappedSource);
    } catch (e) {
        if (e.name === "YAMLError") {
            e.name = "YAML Parsing";
        }
        const m = e.stack.split("\n")[0].match(/^.+ \((\d+):(\d+)\)$/);
        if (m) {
            const m1 = mappedSource.value.match(/([^\s]+):([^\s]+)/);
            if (m1 && e.reason.match(/a multiline key may not be an implicit key/)) {
                e.name = "YAML Parse Error";
                e.reason = "block has incorrect key formatting";
                const { originalString } = mappedSource.map(m1.index, true);
                const filename = originalString.fileName;
                const map = mappedSource.map(m1.index);
                const { line, column } = indexToLineCol(map.originalString.value)(map.index);
                const sourceContext = createSourceContext(mappedSource, {
                    start: m1.index + 1,
                    end: m1.index + m1[0].length
                });
                e.stack = `${e.reason} (${filename}, ${line + 1}:${column + 1})\n${sourceContext}`;
                e.message = e.stack;
                e.message = `${e.message}\n${tidyverseInfo("Is it possible you missed a space after a colon in the key-value mapping?")}`;
            } else {
                const f = lineColToIndex(mappedSource.value);
                const location = {
                    line: Number(m[1]) - 1,
                    column: Number(m[2] - 1)
                };
                const offset = f(location);
                const { originalString } = mappedSource.map(offset, true);
                const filename = originalString.fileName;
                const f2 = mappedIndexToLineCol(mappedSource);
                const { line, column } = f2(offset);
                const sourceContext = createSourceContext(mappedSource, {
                    start: offset,
                    end: offset + 1
                });
                e.stack = `${e.reason} (${filename}, ${line + 1}:${column + 1})\n${sourceContext}`;
                e.message = e.stack;
                if (mappedLines(mappedSource)[location.line].value.indexOf("!expr") !== -1 && e.reason.match(/bad indentation of a mapping entry/)) {
                    e.message = `${e.message}\n${tidyverseInfo("YAML tags like !expr must be followed by YAML strings.")}\n${tidyverseInfo("Is it possible you need to quote the value you passed to !expr ?")}`;
                }
            }
            e.stack = "";
        }
        throw e;
    }
}
function buildJsYamlAnnotation(mappedYaml) {
    const yml = mappedYaml.value;
    const stack = [];
    const results = [];
    function listener(what, state) {
        const { result: result1, position, kind } = state;
        if (what === "close") {
            const { position: openPosition, kind: openKind } = stack.pop();
            if (results.length > 0) {
                const last = results[results.length - 1];
                if (last.start === openPosition && last.end === position) {
                    return;
                }
            }
            const components = [];
            while(results.length > 0){
                const last = results[results.length - 1];
                if (last.end <= openPosition) {
                    break;
                }
                components.push(results.pop());
            }
            components.reverse();
            const rawRange = yml.substring(openPosition, position);
            const leftTrim = rawRange.length - rawRange.trimStart().length;
            const rightTrim = rawRange.length - rawRange.trimEnd().length;
            if (openKind === null && kind === null) {} else if (rawRange.trim().length === 0) {
                results.push({
                    start: position - rightTrim,
                    end: position - rightTrim,
                    result: result1,
                    components,
                    kind,
                    source: mappedYaml
                });
            } else {
                results.push({
                    start: openPosition + leftTrim,
                    end: position - rightTrim,
                    result: result1,
                    components,
                    kind,
                    source: mappedYaml
                });
            }
        } else {
            stack.push({
                position,
                kind
            });
        }
    }
    load(yml, {
        listener,
        schema: QuartoJSONSchema
    });
    if (results.length === 0) {
        return {
            start: 0,
            end: 0,
            result: null,
            kind: "null",
            components: [],
            source: mappedYaml
        };
    }
    if (results.length !== 1) {
        throw new InternalError(`Expected a single result, got ${results.length} instead`);
    }
    try {
        JSON.stringify(results[0]);
    } catch (e) {
        if (e.message.match("invalid string length")) {} else if (e.message.match(/circular structure/)) {
            throw new InternalError(`Circular structure detected in parsed yaml: ${e.message}`);
        } else {}
    }
    return postProcessAnnotation(results[0]);
}
function buildTreeSitterAnnotation(tree, mappedSource) {
    const errors = [];
    const singletonBuild = (node)=>{
        let tag = undefined;
        for (const child of node.children){
            if (child.type === "tag") {
                tag = child;
                continue;
            }
            if (child.type !== "comment") {
                const result1 = buildNode(child, node.endIndex);
                if (tag) {
                    return annotateTag(result1, tag, node);
                } else {
                    return result1;
                }
            }
        }
        return annotateEmpty(node.endIndex);
    };
    const buildNode = (node, endIndex)=>{
        if (node === null) {
            return annotateEmpty(endIndex === undefined ? -1 : endIndex);
        }
        if (dispatch[node.type] === undefined) {
            return annotateEmpty(endIndex || node.endIndex || -1);
        }
        return dispatch[node.type](node);
    };
    const annotateEmpty = (position)=>{
        return {
            start: position,
            end: position,
            result: null,
            kind: "<<EMPTY>>",
            components: [],
            source: mappedSource
        };
    };
    const annotate = (node, result1, components)=>{
        return {
            start: node.startIndex,
            end: node.endIndex,
            result: result1,
            kind: node.type,
            components,
            source: mappedSource
        };
    };
    const annotateTag = (innerParse, tagNode, outerNode)=>{
        const tagParse = annotate(tagNode, tagNode.text, []);
        const result1 = annotate(outerNode, {
            tag: tagNode.text,
            value: innerParse.result
        }, [
            tagParse,
            innerParse
        ]);
        return result1;
    };
    const buildPair = (node)=>{
        let key, value;
        const children = node.children.filter((n)=>n.type !== "comment");
        if (children.length === 3) {
            key = annotate(children[0], children[0].text, []);
            value = buildNode(children[2], node.endIndex);
        } else if (children.length === 2) {
            key = annotate(children[0], children[0].text, []);
            value = annotateEmpty(node.endIndex);
        } else {
            key = annotateEmpty(node.endIndex);
            value = annotateEmpty(node.endIndex);
        }
        return annotate(node, {
            key: key.result,
            value: value.result
        }, [
            key,
            value
        ]);
    };
    const dispatch = {
        "stream": singletonBuild,
        "document": singletonBuild,
        "block_node": singletonBuild,
        "flow_node": singletonBuild,
        "double_quote_scalar": (node)=>{
            return annotate(node, jsYamlParseLenient(node.text), []);
        },
        "single_quote_scalar": (node)=>{
            return annotate(node, jsYamlParseLenient(node.text), []);
        },
        "plain_scalar": (node)=>{
            return annotate(node, jsYamlParseLenient(node.text), []);
        },
        "block_scalar": (node)=>{
            return annotate(node, jsYamlParseLenient(node.text), []);
        },
        "block_sequence": (node)=>{
            const result1 = [], components = [];
            for(let i1 = 0; i1 < node.childCount; ++i1){
                const child = node.child(i1);
                if (child.type !== "block_sequence_item") {
                    continue;
                }
                const component = buildNode(child, node.endIndex);
                components.push(component);
                result1.push(component && component.result);
            }
            return annotate(node, result1, components);
        },
        "block_sequence_item": (node)=>{
            if (node.childCount < 2) {
                return annotateEmpty(node.endIndex);
            } else {
                return buildNode(node.child(1), node.endIndex);
            }
        },
        "flow_sequence": (node)=>{
            const result1 = [], components = [];
            for(let i1 = 0; i1 < node.childCount; ++i1){
                const child = node.child(i1);
                if (child.type !== "flow_node") {
                    continue;
                }
                const component = buildNode(child, node.endIndex);
                components.push(component);
                result1.push(component.result);
            }
            return annotate(node, result1, components);
        },
        "block_mapping": (node)=>{
            const result1 = {}, components = [];
            for(let i1 = 0; i1 < node.childCount; ++i1){
                const child = node.child(i1);
                let component;
                if (child.type === "ERROR") {
                    result1[child.text] = "<<ERROR>>";
                    const key = annotate(child, child.text, []);
                    const value = annotateEmpty(child.endIndex);
                    component = annotate(child, {
                        key: key.result,
                        value: value.result
                    }, [
                        key,
                        value
                    ]);
                } else if (child.type !== "block_mapping_pair") {
                    continue;
                } else {
                    component = buildNode(child, node.endIndex);
                }
                const { key, value } = component.result;
                result1[String(key)] = value;
                components.push(...component.components);
            }
            return annotate(node, result1, components);
        },
        "flow_pair": buildPair,
        "flow_mapping": (node)=>{
            const result1 = {}, components = [];
            for(let i1 = 0; i1 < node.childCount; ++i1){
                const child = node.child(i1);
                if (child.type === "flow_node") {
                    continue;
                }
                if (child.type === "flow_pair") {
                    const component = buildNode(child, node.endIndex);
                    const { key, value } = component.result;
                    result1[String(key)] = value;
                    components.push(...component.components);
                }
            }
            return annotate(node, result1, components);
        },
        "block_mapping_pair": buildPair
    };
    const result1 = buildNode(tree.rootNode, tree.rootNode.endIndex);
    if (errors.length) {
        result1.errors = errors;
    }
    const parsedSize = tree.rootNode.text.trim().length;
    const codeSize = mappedSource.value.trim().length;
    const lossage = parsedSize / codeSize;
    if (lossage < 0.95) {
        return null;
    }
    return result1;
}
function locateAnnotation(annotation, position, kind) {
    const originalSource = annotation.source;
    kind = kind || "value";
    for(let i1 = 0; i1 < position.length; ++i1){
        const value = position[i1];
        if (typeof value === "number") {
            const inner = annotation.components[value];
            if (inner === undefined) {
                throw new InternalError("invalid path for locateAnnotation");
            }
            annotation = inner;
        } else {
            let found = false;
            for(let j = 0; j < annotation.components.length; j += 2){
                if (originalSource.value.substring(annotation.components[j].start, annotation.components[j].end).trim() === value) {
                    if (i1 === position.length - 1) {
                        if (kind === "key") {
                            annotation = annotation.components[j];
                        } else {
                            annotation = annotation.components[j + 1];
                        }
                    }
                    found = true;
                    break;
                }
            }
            if (!found) {
                throw new InternalError("invalid path for locateAnnotation");
            }
        }
    }
    return annotation;
}
class ValidationError2 extends Error {
    validationErrors;
    constructor(msg, validationErrors){
        super([
            msg,
            ...validationErrors.map((e)=>tidyverseFormatError(e.niceError))
        ].join("\n\n"));
        Object.setPrototypeOf(this, ValidationError2.prototype);
        this.validationErrors = validationErrors;
    }
}
const isObject2 = (value)=>{
    const type = typeof value;
    return value !== null && (type === "object" || type === "function");
};
async function readAndValidateYamlFromMappedString(mappedYaml, schema, pruneErrors = true, lenient = false) {
    const annotation = await readAnnotatedYamlFromMappedString(mappedYaml, lenient);
    if (annotation === null) {
        throw new Error("Parse error in readAnnotatedYamlFromMappedString");
    }
    const validateYaml = !isObject2(annotation.result) || annotation.result["validate-yaml"] !== false;
    if (!validateYaml) {
        return {
            yaml: annotation.result,
            yamlValidationErrors: []
        };
    }
    const validate = async (validator)=>{
        const valResult = await validator.validateParse(mappedYaml, annotation, pruneErrors);
        return {
            yaml: annotation.result,
            yamlValidationErrors: valResult.errors
        };
    };
    if (typeof annotation.result === "object" && !Array.isArray(annotation.result)) {
        const preCheckResult = await withValidator(getSchemaDefinition("bad-parse-schema"), validate);
        if (preCheckResult.yamlValidationErrors.length !== 0) {
            return preCheckResult;
        }
    }
    const result1 = await withValidator(schema, validate);
    return result1;
}
function setBaseSchemaProperties(yaml, schema) {
    if (yaml.additionalCompletions) {
        schema = completeSchema(schema, ...yaml.additionalCompletions);
    }
    if (yaml.completions) {
        schema = completeSchemaOverwrite(schema, ...yaml.completions);
    }
    if (yaml.id) {
        schema = idSchema(schema, yaml.id);
    }
    if (yaml.hidden === true) {
        schema = completeSchemaOverwrite(schema);
        schema = tagSchema(schema, {
            "hidden": true
        });
    }
    if (yaml.tags) {
        schema = tagSchema(schema, yaml.tags);
    }
    if (yaml.description) {
        schema = tagSchema(schema, {
            description: yaml.description
        });
        if (typeof yaml.description === "string") {
            schema = documentSchema(schema, yaml.description);
        } else if (typeof yaml.description === "object") {
            schema = documentSchema(schema, yaml.description.short);
        }
    }
    const result1 = Object.assign({}, schema);
    if (yaml.errorDescription) {
        result1.description = yaml.errorDescription;
    }
    if (yaml.errorMessage) {
        result1.errorMessage = yaml.errorMessage;
    }
    return result1;
}
function convertFromNull(yaml) {
    return setBaseSchemaProperties(yaml["null"], nullSchema);
}
function convertFromSchema(yaml) {
    const schema = convertFromYaml(yaml.schema);
    return setBaseSchemaProperties(yaml, schema);
}
function convertFromString(yaml) {
    if (yaml["string"].pattern) {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml["string"], regexSchema(yaml["string"].pattern)));
    } else {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml["string"], stringSchema));
    }
}
function convertFromPattern(yaml) {
    if (typeof yaml.pattern === "string") {
        return setBaseSchemaProperties(yaml, regexSchema(yaml.pattern));
    } else {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.pattern, regexSchema(yaml.pattern.regex)));
    }
}
function convertFromPath(yaml) {
    return setBaseSchemaProperties(yaml["path"], stringSchema);
}
function convertFromNumber(yaml) {
    return setBaseSchemaProperties(yaml["number"], numberSchema);
}
function convertFromBoolean(yaml) {
    return setBaseSchemaProperties(yaml["boolean"], booleanSchema);
}
function convertFromRef(yaml) {
    return setBaseSchemaProperties(yaml, refSchema(yaml.ref, `be ${yaml.ref}`));
}
function convertFromMaybeArrayOf(yaml) {
    const inner = convertFromYaml(yaml.maybeArrayOf);
    const schema = tagSchema(anyOfSchema(inner, arraySchema(inner)), {
        "complete-from": [
            "anyOf",
            0
        ]
    });
    return setBaseSchemaProperties(yaml, schema);
}
function convertFromArrayOf(yaml) {
    if (yaml.arrayOf.schema) {
        const result1 = arraySchema(convertFromYaml(yaml.arrayOf.schema));
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.arrayOf, result1));
    } else {
        return setBaseSchemaProperties(yaml, arraySchema(convertFromYaml(yaml.arrayOf)));
    }
}
function convertFromAllOf(yaml) {
    if (yaml.allOf.schemas) {
        const inner = yaml.allOf.schemas.map((x)=>convertFromYaml(x));
        const schema = allOfSchema(...inner);
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.allOf, schema));
    } else {
        const inner = yaml.allOf.map((x)=>convertFromYaml(x));
        const schema = allOfSchema(...inner);
        return setBaseSchemaProperties(yaml, schema);
    }
}
function convertFromAnyOf(yaml) {
    if (yaml.anyOf.schemas) {
        const inner = yaml.anyOf.schemas.map((x)=>convertFromYaml(x));
        const schema = anyOfSchema(...inner);
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.anyOf, schema));
    } else {
        const inner = yaml.anyOf.map((x)=>convertFromYaml(x));
        const schema = anyOfSchema(...inner);
        return setBaseSchemaProperties(yaml, schema);
    }
}
function convertFromEnum(yaml) {
    const schema = yaml["enum"];
    if (schema.hasOwnProperty("values")) {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml["enum"], enumSchema(...schema.values)));
    } else {
        return setBaseSchemaProperties(yaml, enumSchema(...schema));
    }
}
function convertFromRecord(yaml) {
    if (yaml.record.properties) {
        const schema = convertFromObject({
            "object": {
                "properties": yaml.record.properties,
                "closed": true,
                "required": "all"
            }
        });
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.record, schema));
    } else {
        const schema = convertFromObject({
            "object": {
                "properties": yaml.record,
                "closed": true,
                "required": "all"
            }
        });
        return setBaseSchemaProperties(yaml, schema);
    }
}
function convertFromObject(yaml) {
    const schema = yaml["object"];
    const params = {};
    if (schema.namingConvention && typeof schema.namingConvention === "string") {
        switch(schema.namingConvention){
            case "ignore":
                params.namingConvention = "ignore";
                break;
            case "capitalizationCase":
                params.namingConvention = "capitalizationCase";
                break;
            case "capitalization-case":
                params.namingConvention = "capitalizationCase";
                break;
            case "capitalization_case":
                params.namingConvention = "capitalizationCase";
                break;
            case "underscoreCase":
                params.namingConvention = "underscore_case";
                break;
            case "underscore-case":
                params.namingConvention = "underscore_case";
                break;
            case "underscore_case":
                params.namingConvention = "underscore_case";
                break;
            case "dashCase":
                params.namingConvention = "dash-case";
                break;
            case "dash-case":
                params.namingConvention = "dash-case";
                break;
            case "dash_case":
                params.namingConvention = "dash-case";
                break;
            case "camelCase":
                params.namingConvention = "capitalizationCase";
                break;
            case "camel-case":
                params.namingConvention = "capitalizationCase";
                break;
            case "camel_case":
                params.namingConvention = "capitalizationCase";
                break;
            case "snakeCase":
                params.namingConvention = "underscore_case";
                break;
            case "snake-case":
                params.namingConvention = "underscore_case";
                break;
            case "snake_case":
                params.namingConvention = "underscore_case";
                break;
            case "kebabCase":
                params.namingConvention = "dash-case";
                break;
            case "kebab-case":
                params.namingConvention = "dash-case";
                break;
            case "kebab_case":
                params.namingConvention = "dash-case";
                break;
            default:
                throw new InternalError(`Unrecognized naming convention ${schema.namingConvention} should have failed validation`);
        }
    } else {
        params.namingConvention = schema.namingConvention;
    }
    if (schema.properties) {
        params.properties = fromEntries(Object.entries(schema.properties).map(([key, value])=>[
                key,
                convertFromYaml(value)
            ]));
    }
    if (schema.patternProperties) {
        params.patternProperties = fromEntries(Object.entries(schema.properties).map(([key, value])=>[
                key,
                convertFromYaml(value)
            ]));
    }
    if (schema.propertyNames !== undefined) {
        params.propertyNames = convertFromYaml(schema.propertyNames);
    } else if (schema.closed === true) {
        const objectKeys = Object.keys(params.properties || {});
        if (objectKeys.length === 0) {
            throw new Error("object schema `closed` requires field `properties`.");
        }
        if (params.namingConvention !== undefined && params.namingConvention !== "ignore") {
            throw new Error("object schema `closed` is only supported with namingConvention: `ignore`");
        }
        params.namingConvention = "ignore";
        params.closed = true;
    }
    if (schema.additionalProperties !== undefined) {
        if (schema.additionalProperties === false) {
            params.additionalProperties = false;
        } else {
            params.additionalProperties = convertFromYaml(schema.additionalProperties);
        }
    }
    if (schema["super"]) {
        if (Array.isArray(schema["super"])) {
            params.baseSchema = schema["super"].map((s)=>convertFromYaml(s));
        } else {
            params.baseSchema = convertFromYaml(schema["super"]);
        }
    }
    if (schema["required"] === "all") {
        params.required = Object.keys(schema.properties || {});
    } else if (schema["required"]) {
        params.required = schema["required"];
    }
    if (schema["completions"]) {
        params.completions = schema["completions"];
    }
    return setBaseSchemaProperties(yaml, setBaseSchemaProperties(schema, objectSchema(params)));
}
function lookup(yaml) {
    if (!hasSchemaDefinition(yaml.resolveRef)) {
        throw new Error(`lookup of key ${yaml.resolveRef} in definitions failed`);
    }
    return getSchemaDefinition(yaml.resolveRef);
}
function convertFromYaml(yaml) {
    const literalValues = [
        {
            val: "object",
            schema: objectSchema()
        },
        {
            val: "path",
            schema: stringSchema
        },
        {
            val: "string",
            schema: stringSchema
        },
        {
            val: "number",
            schema: numberSchema
        },
        {
            val: "boolean",
            schema: booleanSchema
        },
        {
            val: "any",
            schema: anySchema()
        },
        {
            val: null,
            schema: nullSchema
        }
    ];
    for (const { val, schema } of literalValues){
        if (yaml === val) {
            return schema;
        }
    }
    if (typeof yaml !== "object") {
        return valueSchema(yaml);
    }
    const schemaObjectKeyFunctions = [
        {
            key: "anyOf",
            value: convertFromAnyOf
        },
        {
            key: "allOf",
            value: convertFromAllOf
        },
        {
            key: "boolean",
            value: convertFromBoolean
        },
        {
            key: "arrayOf",
            value: convertFromArrayOf
        },
        {
            key: "enum",
            value: convertFromEnum
        },
        {
            key: "maybeArrayOf",
            value: convertFromMaybeArrayOf
        },
        {
            key: "null",
            value: convertFromNull
        },
        {
            key: "number",
            value: convertFromNumber
        },
        {
            key: "object",
            value: convertFromObject
        },
        {
            key: "path",
            value: convertFromPath
        },
        {
            key: "record",
            value: convertFromRecord
        },
        {
            key: "ref",
            value: convertFromRef
        },
        {
            key: "resolveRef",
            value: lookup
        },
        {
            key: "string",
            value: convertFromString
        },
        {
            key: "pattern",
            value: convertFromPattern
        },
        {
            key: "schema",
            value: convertFromSchema
        }
    ];
    for (const { key: objectKey, value: fun } of schemaObjectKeyFunctions){
        if (yaml[objectKey] !== undefined) {
            return fun(yaml);
        }
    }
    throw new InternalError("Cannot convert object; this should have failed validation.");
}
function objectSchemaFromFieldsObject(fields, exclude) {
    exclude = exclude || ((_key)=>false);
    const properties = {};
    convertFromFieldsObject(fields, properties);
    for (const key of Object.keys(properties)){
        if (exclude(key)) {
            delete properties[key];
        }
    }
    return objectSchema({
        properties
    });
}
function annotateSchemaFromField(field, schema) {
    if (field.enabled !== undefined) {
        schema = tagSchema(schema, {
            formats: field.enabled
        });
    }
    if (field.disabled !== undefined) {
        schema = tagSchema(schema, {
            formats: field.disabled.map((x)=>`!${x}`)
        });
    }
    if (field.tags) {
        schema = tagSchema(schema, field.tags);
    }
    if (field.description) {
        if (typeof field.description === "string") {
            schema = documentSchema(schema, field.description);
        } else if (typeof field.description === "object") {
            schema = documentSchema(schema, field.description.short);
        }
        schema = tagSchema(schema, {
            description: field.description
        });
    }
    if (field.hidden) {
        schema = tagSchema(schema, {
            "hidden": true
        });
    }
    return schema;
}
function schemaFromField(entry) {
    const schema = convertFromYaml(entry.schema);
    return annotateSchemaFromField(entry, schema);
}
function convertFromFieldsObject(yaml, obj) {
    const result1 = obj || {};
    for (const field of yaml){
        let schema = convertFromYaml(field.schema);
        schema = annotateSchemaFromField(field, schema);
        result1[field.name] = schema;
        if (field.alias) {
            result1[field.alias] = schema;
        }
    }
    return result1;
}
function schemaFieldsFromGlob(globPath, testFun) {
    const result1 = [];
    testFun = testFun || ((_e, _p)=>true);
    for (const [file, fields] of expandResourceGlob(globPath)){
        for (const field of fields){
            const fieldName = field.name;
            const schemaId = `quarto-resource-${file.split("/").slice(-1)[0].slice(0, -4)}-${fieldName}`;
            if (testFun(field, file)) {
                result1.push({
                    schemaId,
                    field
                });
            }
        }
    }
    return result1;
}
memoize1(()=>{
    const groups = getYamlIntelligenceResource("schema/groups.yml");
    const result1 = [];
    for (const [topLevel, sub] of Object.entries(groups)){
        for (const key of Object.keys(sub)){
            result1.push(`${topLevel}-${key}`);
        }
    }
    return result1;
}, ()=>"const");
function objectRefSchemaFromContextGlob(contextGlob, testFun) {
    const regexp = globToRegExp3(contextGlob);
    return objectRefSchemaFromGlob("schema/{document,cell}-*.yml", (field, path)=>{
        if (testFun !== undefined && !testFun(field, path)) {
            return false;
        }
        const pathContext = path.split("/").slice(-1)[0].slice(0, -4);
        const schemaContexts = field !== undefined && field.tags !== undefined && field.tags.contexts || [];
        if (pathContext.match(regexp)) {
            return true;
        }
        return schemaContexts.some((c)=>c.match(regexp));
    });
}
function objectRefSchemaFromGlob(glob, testFun) {
    const properties = {};
    for (const { schemaId, field } of schemaFieldsFromGlob(glob, testFun)){
        const schema = refSchema(schemaId, schemaId);
        properties[field.name] = schema;
        if (field.alias) {
            properties[field.alias] = schema;
        }
    }
    return objectSchema({
        properties
    });
}
async function buildResourceSchemas() {
    const path = "schema/{cell-*,document-*,project}.yml";
    for (const [file, fields] of expandResourceGlob(path)){
        const yaml = fields;
        const entries = Object.entries(convertFromFieldsObject(yaml));
        for (const [fieldName, fieldSchema] of entries){
            const schemaId = `quarto-resource-${file.split("/").slice(-1)[0].slice(0, -4)}-${fieldName}`;
            const schema = idSchema(fieldSchema, schemaId);
            setSchemaDefinition(schema);
            await withValidator(schema, async (_validator)=>{});
        }
    }
}
function defineCached(thunk, schemaId) {
    let schema;
    return async ()=>{
        if (hasSchemaDefinition(schemaId)) {
            schema = getSchemaDefinition(schemaId);
            return refSchema(schema.$id, schema.description || `be a {schema['$id'] as string}`);
        }
        const result1 = await thunk();
        const { errorHandlers } = result1;
        schema = result1.schema;
        if (schemaId !== schema.$id) {
            schema = idSchema(schema, schemaId);
        }
        define1(schema);
        for (const fun of errorHandlers){
            addValidatorErrorHandler(schema, fun);
        }
        return refSchema(schema.$id, schema.description || `be a {schema['$id']}`);
    };
}
function define1(schema) {
    if (schema !== true && schema !== false && schema.$id && !hasSchemaDefinition(schema.$id)) {
        setSchemaDefinition(schema);
    }
}
async function loadDefaultSchemaDefinitions() {
    await loadSchemaDefinitions(getYamlIntelligenceResource("schema/definitions.yml"));
    await buildResourceSchemas();
}
async function loadSchemaDefinitions(yaml) {
    await Promise.all(yaml.map(async (yamlSchema)=>{
        const schema = convertFromYaml(yamlSchema);
        if (schema.$id === undefined) {
            throw new InternalError(`Unnamed schema in definitions`);
        }
        setSchemaDefinition(schema);
    }));
}
function getFormatExecuteOptionsSchema() {
    const schema = idSchema(objectRefSchemaFromContextGlob("document-execute"), "front-matter-execute");
    define1(schema);
    return refSchema("front-matter-execute", "be a front-matter-execute object");
}
let formatAliases = undefined;
function getFormatAliases() {
    if (formatAliases !== undefined) {
        return formatAliases;
    }
    formatAliases = getYamlIntelligenceResource("schema/format-aliases.yml").aliases;
    return formatAliases;
}
function expandFormatAliases(lst) {
    return expandAliasesFrom(lst, getFormatAliases());
}
function useSchema(schema, format) {
    const formats = schema && schema.tags && schema.tags.formats;
    if (formats === undefined) {
        return true;
    }
    const disabled = formats.filter((f)=>f.startsWith("!")).map((f)=>f.slice(1));
    const enabled = formats.filter((f)=>!f.startsWith("!"));
    if (disabled.length > 0 && expandFormatAliases(disabled).indexOf(format) !== -1) {
        return false;
    }
    if (enabled.length > 0 && expandFormatAliases(enabled).indexOf(format) === -1) {
        return false;
    }
    return true;
}
function getFormatSchema(format) {
    const schema = objectRefSchemaFromContextGlob("document-*", (field)=>{
        const schema = schemaFromField(field);
        return useSchema(schema, format);
    });
    return anyOfSchema(schema, enumSchema("default"));
}
function pandocFormatsResource() {
    return getYamlIntelligenceResource("pandoc/formats.yml");
}
async function makeFrontMatterFormatSchema(nonStrict = false) {
    const hideFormat = (format)=>{
        const hideList = [
            "html",
            "epub",
            "docbook"
        ];
        const hidden = hideList.some((h)=>format.startsWith(h) && format.length > h.length);
        return {
            name: format,
            hidden
        };
    };
    const formatSchemaDescriptorList = (await pandocFormatsResource()).concat("md", "hugo", "dashboard", "email").map((format)=>{
        const { name, hidden } = hideFormat(format);
        return {
            regex: `^(.+-)?${name}([-+].+)?$`,
            schema: getFormatSchema(name),
            name,
            hidden
        };
    });
    const formatSchemas = formatSchemaDescriptorList.map(({ regex, schema })=>[
            regex,
            schema
        ]);
    const plusFormatStringSchemas = formatSchemaDescriptorList.map(({ regex, name, hidden })=>{
        const schema = regexSchema(regex, `be '${name}'`);
        if (hidden) {
            return schema;
        }
        return completeSchema(schema, name);
    });
    const luaFilenameS = regexSchema("^.+\.lua$");
    plusFormatStringSchemas.push(luaFilenameS);
    const completionsObject = fromEntries(formatSchemaDescriptorList.filter(({ hidden })=>!hidden).map(({ name })=>[
            name,
            {
                type: "key",
                display: name,
                value: `${name}: `,
                description: `be '${name}'`,
                suggest_on_accept: true
            }
        ]));
    return errorMessageSchema(anyOfSchema(describeSchema(anyOfSchema(...plusFormatStringSchemas), "the name of a pandoc-supported output format"), objectSchema({
        propertyNames: luaFilenameS
    }), allOfSchema(objectSchema({
        patternProperties: fromEntries(formatSchemas),
        completions: completionsObject,
        additionalProperties: nonStrict
    }))), "${value} is not a valid output format.");
}
const getFrontMatterFormatSchema = defineCached(async ()=>{
    return {
        schema: await makeFrontMatterFormatSchema(),
        errorHandlers: []
    };
}, "front-matter-format");
defineCached(async ()=>{
    return {
        schema: await makeFrontMatterFormatSchema(true),
        errorHandlers: []
    };
}, "front-matter-format-nonstrict");
const getFrontMatterSchema = defineCached(async ()=>{
    const executeObjSchema = await getFormatExecuteOptionsSchema();
    return {
        schema: anyOfSchema(nullSchema, allOfSchema(objectSchema({
            properties: {
                execute: executeObjSchema,
                format: await getFrontMatterFormatSchema()
            },
            description: "be a Quarto YAML front matter object"
        }), objectRefSchemaFromContextGlob("document-*", (field)=>field.name !== "format"), executeObjSchema, refSchema("quarto-dev-schema", ""))),
        errorHandlers: []
    };
}, "front-matter");
function Diff() {}
Diff.prototype = {
    diff: function diff(oldString, newString) {
        var options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
        var callback = options.callback;
        if (typeof options === 'function') {
            callback = options;
            options = {};
        }
        this.options = options;
        var self1 = this;
        function done(value) {
            return value;
        }
        oldString = this.castInput(oldString);
        newString = this.castInput(newString);
        oldString = this.removeEmpty(this.tokenize(oldString));
        newString = this.removeEmpty(this.tokenize(newString));
        var newLen = newString.length, oldLen = oldString.length;
        var editLength = 1;
        var maxEditLength = newLen + oldLen;
        if (options.maxEditLength) {
            maxEditLength = Math.min(maxEditLength, options.maxEditLength);
        }
        var bestPath = [
            {
                newPos: -1,
                components: []
            }
        ];
        var oldPos = this.extractCommon(bestPath[0], newString, oldString, 0);
        if (bestPath[0].newPos + 1 >= newLen && oldPos + 1 >= oldLen) {
            return done([
                {
                    value: this.join(newString),
                    count: newString.length
                }
            ]);
        }
        function execEditLength() {
            for(var diagonalPath = -1 * editLength; diagonalPath <= editLength; diagonalPath += 2){
                var basePath = void 0;
                var addPath = bestPath[diagonalPath - 1], removePath = bestPath[diagonalPath + 1], _oldPos = (removePath ? removePath.newPos : 0) - diagonalPath;
                if (addPath) {
                    bestPath[diagonalPath - 1] = undefined;
                }
                var canAdd = addPath && addPath.newPos + 1 < newLen, canRemove = removePath && 0 <= _oldPos && _oldPos < oldLen;
                if (!canAdd && !canRemove) {
                    bestPath[diagonalPath] = undefined;
                    continue;
                }
                if (!canAdd || canRemove && addPath.newPos < removePath.newPos) {
                    basePath = clonePath(removePath);
                    self1.pushComponent(basePath.components, undefined, true);
                } else {
                    basePath = addPath;
                    basePath.newPos++;
                    self1.pushComponent(basePath.components, true, undefined);
                }
                _oldPos = self1.extractCommon(basePath, newString, oldString, diagonalPath);
                if (basePath.newPos + 1 >= newLen && _oldPos + 1 >= oldLen) {
                    return done(buildValues(self1, basePath.components, newString, oldString, self1.useLongestToken));
                } else {
                    bestPath[diagonalPath] = basePath;
                }
            }
            editLength++;
        }
        while(editLength <= maxEditLength){
            var ret = execEditLength();
            if (callback && callback(ret, editLength, maxEditLength)) {
                return ret;
            }
            if (ret) {
                return ret;
            }
        }
    },
    pushComponent: function pushComponent(components, added, removed) {
        var last = components[components.length - 1];
        if (last && last.added === added && last.removed === removed) {
            components[components.length - 1] = {
                count: last.count + 1,
                added: added,
                removed: removed
            };
        } else {
            components.push({
                count: 1,
                added: added,
                removed: removed
            });
        }
    },
    extractCommon: function extractCommon(basePath, newString, oldString, diagonalPath) {
        var newLen = newString.length, oldLen = oldString.length, newPos = basePath.newPos, oldPos = newPos - diagonalPath, commonCount = 0;
        while(newPos + 1 < newLen && oldPos + 1 < oldLen && this.equals(newString[newPos + 1], oldString[oldPos + 1])){
            newPos++;
            oldPos++;
            commonCount++;
        }
        if (commonCount) {
            basePath.components.push({
                count: commonCount
            });
        }
        basePath.newPos = newPos;
        return oldPos;
    },
    equals: function equals(left, right) {
        if (this.options.comparator) {
            return this.options.comparator(left, right);
        } else {
            return left === right || this.options.ignoreCase && left.toLowerCase() === right.toLowerCase();
        }
    },
    removeEmpty: function removeEmpty(array) {
        var ret = [];
        for(var i1 = 0; i1 < array.length; i1++){
            if (array[i1]) {
                ret.push(array[i1]);
            }
        }
        return ret;
    },
    castInput: function castInput(value) {
        return value;
    },
    tokenize: function tokenize(value) {
        return value.split('');
    },
    join: function join(chars) {
        return chars.join('');
    }
};
function buildValues(diff, components, newString, oldString, useLongestToken) {
    var componentPos = 0, componentLen = components.length, newPos = 0, oldPos = 0;
    for(; componentPos < componentLen; componentPos++){
        var component = components[componentPos];
        if (!component.removed) {
            if (!component.added && useLongestToken) {
                var value = newString.slice(newPos, newPos + component.count);
                value = value.map(function(value, i1) {
                    var oldValue = oldString[oldPos + i1];
                    return oldValue.length > value.length ? oldValue : value;
                });
                component.value = diff.join(value);
            } else {
                component.value = diff.join(newString.slice(newPos, newPos + component.count));
            }
            newPos += component.count;
            if (!component.added) {
                oldPos += component.count;
            }
        } else {
            component.value = diff.join(oldString.slice(oldPos, oldPos + component.count));
            oldPos += component.count;
            if (componentPos && components[componentPos - 1].added) {
                var tmp = components[componentPos - 1];
                components[componentPos - 1] = components[componentPos];
                components[componentPos] = tmp;
            }
        }
    }
    var lastComponent = components[componentLen - 1];
    if (componentLen > 1 && typeof lastComponent.value === 'string' && (lastComponent.added || lastComponent.removed) && diff.equals('', lastComponent.value)) {
        components[componentLen - 2].value += lastComponent.value;
        components.pop();
    }
    return components;
}
function clonePath(path) {
    return {
        newPos: path.newPos,
        components: path.components.slice(0)
    };
}
new Diff();
var extendedWordChars = /^[A-Za-z\xC0-\u02C6\u02C8-\u02D7\u02DE-\u02FF\u1E00-\u1EFF]+$/;
var reWhitespace1 = /\S/;
var wordDiff = new Diff();
wordDiff.equals = function(left, right) {
    if (this.options.ignoreCase) {
        left = left.toLowerCase();
        right = right.toLowerCase();
    }
    return left === right || this.options.ignoreWhitespace && !reWhitespace1.test(left) && !reWhitespace1.test(right);
};
wordDiff.tokenize = function(value) {
    var tokens = value.split(/([^\S\r\n]+|[()[\]{}'"\r\n]|\b)/);
    for(var i1 = 0; i1 < tokens.length - 1; i1++){
        if (!tokens[i1 + 1] && tokens[i1 + 2] && extendedWordChars.test(tokens[i1]) && extendedWordChars.test(tokens[i1 + 2])) {
            tokens[i1] += tokens[i1 + 2];
            tokens.splice(i1 + 1, 2);
            i1--;
        }
    }
    return tokens;
};
var lineDiff = new Diff();
lineDiff.tokenize = function(value) {
    var retLines = [], linesAndNewlines = value.split(/(\n|\r\n)/);
    if (!linesAndNewlines[linesAndNewlines.length - 1]) {
        linesAndNewlines.pop();
    }
    for(var i1 = 0; i1 < linesAndNewlines.length; i1++){
        var line = linesAndNewlines[i1];
        if (i1 % 2 && !this.options.newlineIsToken) {
            retLines[retLines.length - 1] += line;
        } else {
            if (this.options.ignoreWhitespace) {
                line = line.trim();
            }
            retLines.push(line);
        }
    }
    return retLines;
};
function diffLines(oldStr, newStr, callback) {
    return lineDiff.diff(oldStr, newStr, callback);
}
var sentenceDiff = new Diff();
sentenceDiff.tokenize = function(value) {
    return value.split(/(\S.+?[.!?])(?=\s+|$)/);
};
var cssDiff = new Diff();
cssDiff.tokenize = function(value) {
    return value.split(/([{}:;,]|\s+)/);
};
function _typeof(obj) {
    "@babel/helpers - typeof";
    if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
        _typeof = function(obj) {
            return typeof obj;
        };
    } else {
        _typeof = function(obj) {
            return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
        };
    }
    return _typeof(obj);
}
var objectPrototypeToString = Object.prototype.toString;
var jsonDiff = new Diff();
jsonDiff.useLongestToken = true;
jsonDiff.tokenize = lineDiff.tokenize;
jsonDiff.castInput = function(value) {
    var _this$options = this.options, undefinedReplacement = _this$options.undefinedReplacement, _this$options$stringi = _this$options.stringifyReplacer, stringifyReplacer = _this$options$stringi === void 0 ? function(k, v) {
        return typeof v === 'undefined' ? undefinedReplacement : v;
    } : _this$options$stringi;
    return typeof value === 'string' ? value : JSON.stringify(canonicalize(value, null, null, stringifyReplacer), stringifyReplacer, '  ');
};
jsonDiff.equals = function(left, right) {
    return Diff.prototype.equals.call(jsonDiff, left.replace(/,([\r\n])/g, '$1'), right.replace(/,([\r\n])/g, '$1'));
};
function canonicalize(obj, stack, replacementStack, replacer, key) {
    stack = stack || [];
    replacementStack = replacementStack || [];
    if (replacer) {
        obj = replacer(key, obj);
    }
    var i1;
    for(i1 = 0; i1 < stack.length; i1 += 1){
        if (stack[i1] === obj) {
            return replacementStack[i1];
        }
    }
    var canonicalizedObj;
    if ('[object Array]' === objectPrototypeToString.call(obj)) {
        stack.push(obj);
        canonicalizedObj = new Array(obj.length);
        replacementStack.push(canonicalizedObj);
        for(i1 = 0; i1 < obj.length; i1 += 1){
            canonicalizedObj[i1] = canonicalize(obj[i1], stack, replacementStack, replacer, key);
        }
        stack.pop();
        replacementStack.pop();
        return canonicalizedObj;
    }
    if (obj && obj.toJSON) {
        obj = obj.toJSON();
    }
    if (_typeof(obj) === 'object' && obj !== null) {
        stack.push(obj);
        canonicalizedObj = {};
        replacementStack.push(canonicalizedObj);
        var sortedKeys = [], _key;
        for(_key in obj){
            if (obj.hasOwnProperty(_key)) {
                sortedKeys.push(_key);
            }
        }
        sortedKeys.sort();
        for(i1 = 0; i1 < sortedKeys.length; i1 += 1){
            _key = sortedKeys[i1];
            canonicalizedObj[_key] = canonicalize(obj[_key], stack, replacementStack, replacer, _key);
        }
        stack.pop();
        replacementStack.pop();
    } else {
        canonicalizedObj = obj;
    }
    return canonicalizedObj;
}
var arrayDiff = new Diff();
arrayDiff.tokenize = function(value) {
    return value.slice();
};
arrayDiff.join = arrayDiff.removeEmpty = function(value) {
    return value;
};
const nodeStack = [
    {
        name: "global",
        children: [
            {
                name: "deno-init",
                start: 0,
                end: performance.now(),
                elapsed: performance.now(),
                children: []
            }
        ],
        start: 0,
        end: 0
    }
];
function withTiming(name, fun) {
    push(name);
    try {
        const result1 = fun();
        return result1;
    } finally{
        pop();
    }
}
async function withTimingAsync(name, fun) {
    push(name);
    try {
        const result1 = await fun();
        pop();
        return result1;
    } catch (e) {
        pop();
        throw e;
    }
}
function push(name) {
    nodeStack.push({
        name,
        start: performance.now(),
        end: 0,
        children: []
    });
}
function pop(at) {
    if (nodeStack.length <= 1) {
        throw new Error("Can't pop top node");
    }
    const node = nodeStack.pop();
    node.end = at ?? performance.now();
    node.elapsed = node.end - node.start;
    nodeStack[nodeStack.length - 1].children.push(node);
}
function getData() {
    nodeStack[0].end = performance.now();
    nodeStack[0].elapsed = nodeStack[0].end - nodeStack[0].start;
    return nodeStack[0];
}
function insertExplicitTimingEntries(start, end, entries, groupName) {
    const [theirStart, ourStart] = start;
    const [theirEnd, _] = end;
    nodeStack.push({
        name: groupName,
        start: ourStart,
        end: 0,
        children: []
    });
    let prevTime = ourStart;
    if (entries.length && entries[0].name === "_start") {
        prevTime = entries[0].time - theirStart + ourStart;
        entries = entries.slice(1);
    }
    for (const entry of entries){
        nodeStack.push({
            name: entry.name,
            start: prevTime,
            end: 0,
            children: []
        });
        prevTime = entry.time - theirStart + ourStart;
        pop(prevTime);
    }
    prevTime = theirEnd - theirStart + ourStart;
    pop(prevTime);
}
async function getLuaTiming() {
    if (Deno.env.get("QUARTO_PROFILER_OUTPUT")) {
        return [
            Number((await execProcess({
                cmd: [
                    "python",
                    "-c",
                    "import time; print(time.time() * 1000)"
                ],
                stdout: "piped"
            })).stdout),
            performance.now()
        ];
    } else {
        return [
            0,
            0
        ];
    }
}
function mappedDiff(source, target) {
    return withTiming("mapped-diff", ()=>{
        const sourceLineRanges = rangedLines(source.value).map((x)=>x.range);
        let sourceCursor = 0;
        const resultChunks = [];
        const started = performance.now();
        const diffResult = diffLines(source.value, target, ()=>{
            const now = performance.now();
            if (now - started > 200) {
                return true;
            }
        });
        if (diffResult === undefined) {
            debug("The computation used to determine source line information from the engine timed out.\nLine number information will be unavailable and/or inaccurate.");
            return asMappedString(target);
        }
        for (const action of diffResult){
            if (action.removed) {
                sourceCursor += action.count;
            } else if (action.added) {
                resultChunks.push(action.value);
            } else {
                const start = sourceLineRanges[sourceCursor].start;
                const nextCursor = sourceCursor + action.count;
                const end = nextCursor < sourceLineRanges.length ? sourceLineRanges[nextCursor].start : sourceLineRanges[sourceLineRanges.length - 1].end;
                sourceCursor = nextCursor;
                resultChunks.push({
                    start,
                    end
                });
            }
        }
        return mappedString(source, resultChunks, source.fileName);
    });
}
function mappedStringFromFile(filename) {
    const value = Deno.readTextFileSync(filename);
    if (filename.startsWith("/")) {
        filename = relative3(Deno.cwd(), filename);
    }
    return asMappedString(value, filename);
}
function isNothing1(subject) {
    return typeof subject === "undefined" || subject === null;
}
function isObject3(subject) {
    return typeof subject === "object" && subject !== null;
}
function toArray1(sequence) {
    if (Array.isArray(sequence)) return sequence;
    else if (isNothing1(sequence)) return [];
    return [
        sequence
    ];
}
function extend1(target, source) {
    var index, length, key, sourceKeys;
    if (source) {
        sourceKeys = Object.keys(source);
        for(index = 0, length = sourceKeys.length; index < length; index += 1){
            key = sourceKeys[index];
            target[key] = source[key];
        }
    }
    return target;
}
function repeat1(string, count) {
    var result1 = "", cycle;
    for(cycle = 0; cycle < count; cycle += 1){
        result1 += string;
    }
    return result1;
}
function isNegativeZero1(number) {
    return number === 0 && Number.NEGATIVE_INFINITY === 1 / number;
}
var isNothing_11 = isNothing1;
var isObject_11 = isObject3;
var toArray_11 = toArray1;
var repeat_11 = repeat1;
var isNegativeZero_11 = isNegativeZero1;
var extend_11 = extend1;
var common2 = {
    isNothing: isNothing_11,
    isObject: isObject_11,
    toArray: toArray_11,
    repeat: repeat_11,
    isNegativeZero: isNegativeZero_11,
    extend: extend_11
};
function formatError1(exception2, compact) {
    var where = "", message = exception2.reason || "(unknown reason)";
    if (!exception2.mark) return message;
    if (exception2.mark.name) {
        where += 'in "' + exception2.mark.name + '" ';
    }
    where += "(" + (exception2.mark.line + 1) + ":" + (exception2.mark.column + 1) + ")";
    if (!compact && exception2.mark.snippet) {
        where += "\n\n" + exception2.mark.snippet;
    }
    return message + " " + where;
}
function YAMLException$11(reason, mark) {
    Error.call(this);
    this.name = "YAMLException";
    this.reason = reason;
    this.mark = mark;
    this.message = formatError1(this, false);
    if (Error.captureStackTrace) {
        Error.captureStackTrace(this, this.constructor);
    } else {
        this.stack = new Error().stack || "";
    }
}
YAMLException$11.prototype = Object.create(Error.prototype);
YAMLException$11.prototype.constructor = YAMLException$11;
YAMLException$11.prototype.toString = function toString(compact) {
    return this.name + ": " + formatError1(this, compact);
};
var exception1 = YAMLException$11;
function getLine1(buffer, lineStart, lineEnd, position, maxLineLength) {
    var head = "";
    var tail = "";
    var maxHalfLength = Math.floor(maxLineLength / 2) - 1;
    if (position - lineStart > maxHalfLength) {
        head = " ... ";
        lineStart = position - maxHalfLength + head.length;
    }
    if (lineEnd - position > maxHalfLength) {
        tail = " ...";
        lineEnd = position + maxHalfLength - tail.length;
    }
    return {
        str: head + buffer.slice(lineStart, lineEnd).replace(/\t/g, "\u2192") + tail,
        pos: position - lineStart + head.length
    };
}
function padStart1(string, max) {
    return common2.repeat(" ", max - string.length) + string;
}
function makeSnippet1(mark, options) {
    options = Object.create(options || null);
    if (!mark.buffer) return null;
    if (!options.maxLength) options.maxLength = 79;
    if (typeof options.indent !== "number") options.indent = 1;
    if (typeof options.linesBefore !== "number") options.linesBefore = 3;
    if (typeof options.linesAfter !== "number") options.linesAfter = 2;
    var re = /\r?\n|\r|\0/g;
    var lineStarts = [
        0
    ];
    var lineEnds = [];
    var match;
    var foundLineNo = -1;
    while(match = re.exec(mark.buffer)){
        lineEnds.push(match.index);
        lineStarts.push(match.index + match[0].length);
        if (mark.position <= match.index && foundLineNo < 0) {
            foundLineNo = lineStarts.length - 2;
        }
    }
    if (foundLineNo < 0) foundLineNo = lineStarts.length - 1;
    var result1 = "", i1, line;
    var lineNoLength = Math.min(mark.line + options.linesAfter, lineEnds.length).toString().length;
    var maxLineLength = options.maxLength - (options.indent + lineNoLength + 3);
    for(i1 = 1; i1 <= options.linesBefore; i1++){
        if (foundLineNo - i1 < 0) break;
        line = getLine1(mark.buffer, lineStarts[foundLineNo - i1], lineEnds[foundLineNo - i1], mark.position - (lineStarts[foundLineNo] - lineStarts[foundLineNo - i1]), maxLineLength);
        result1 = common2.repeat(" ", options.indent) + padStart1((mark.line - i1 + 1).toString(), lineNoLength) + " | " + line.str + "\n" + result1;
    }
    line = getLine1(mark.buffer, lineStarts[foundLineNo], lineEnds[foundLineNo], mark.position, maxLineLength);
    result1 += common2.repeat(" ", options.indent) + padStart1((mark.line + 1).toString(), lineNoLength) + " | " + line.str + "\n";
    result1 += common2.repeat("-", options.indent + lineNoLength + 3 + line.pos) + "^\n";
    for(i1 = 1; i1 <= options.linesAfter; i1++){
        if (foundLineNo + i1 >= lineEnds.length) break;
        line = getLine1(mark.buffer, lineStarts[foundLineNo + i1], lineEnds[foundLineNo + i1], mark.position - (lineStarts[foundLineNo] - lineStarts[foundLineNo + i1]), maxLineLength);
        result1 += common2.repeat(" ", options.indent) + padStart1((mark.line + i1 + 1).toString(), lineNoLength) + " | " + line.str + "\n";
    }
    return result1.replace(/\n$/, "");
}
var snippet1 = makeSnippet1;
var TYPE_CONSTRUCTOR_OPTIONS1 = [
    "kind",
    "multi",
    "resolve",
    "construct",
    "instanceOf",
    "predicate",
    "represent",
    "representName",
    "defaultStyle",
    "styleAliases"
];
var YAML_NODE_KINDS1 = [
    "scalar",
    "sequence",
    "mapping"
];
function compileStyleAliases1(map2) {
    var result1 = {};
    if (map2 !== null) {
        Object.keys(map2).forEach(function(style) {
            map2[style].forEach(function(alias) {
                result1[String(alias)] = style;
            });
        });
    }
    return result1;
}
function Type$11(tag, options) {
    options = options || {};
    Object.keys(options).forEach(function(name) {
        if (TYPE_CONSTRUCTOR_OPTIONS1.indexOf(name) === -1) {
            throw new exception1('Unknown option "' + name + '" is met in definition of "' + tag + '" YAML type.');
        }
    });
    this.options = options;
    this.tag = tag;
    this.kind = options["kind"] || null;
    this.resolve = options["resolve"] || function() {
        return true;
    };
    this.construct = options["construct"] || function(data) {
        return data;
    };
    this.instanceOf = options["instanceOf"] || null;
    this.predicate = options["predicate"] || null;
    this.represent = options["represent"] || null;
    this.representName = options["representName"] || null;
    this.defaultStyle = options["defaultStyle"] || null;
    this.multi = options["multi"] || false;
    this.styleAliases = compileStyleAliases1(options["styleAliases"] || null);
    if (YAML_NODE_KINDS1.indexOf(this.kind) === -1) {
        throw new exception1('Unknown kind "' + this.kind + '" is specified for "' + tag + '" YAML type.');
    }
}
var type1 = Type$11;
function compileList1(schema2, name) {
    var result1 = [];
    schema2[name].forEach(function(currentType) {
        var newIndex = result1.length;
        result1.forEach(function(previousType, previousIndex) {
            if (previousType.tag === currentType.tag && previousType.kind === currentType.kind && previousType.multi === currentType.multi) {
                newIndex = previousIndex;
            }
        });
        result1[newIndex] = currentType;
    });
    return result1;
}
function compileMap1() {
    var result1 = {
        scalar: {},
        sequence: {},
        mapping: {},
        fallback: {},
        multi: {
            scalar: [],
            sequence: [],
            mapping: [],
            fallback: []
        }
    }, index, length;
    function collectType(type2) {
        if (type2.multi) {
            result1.multi[type2.kind].push(type2);
            result1.multi["fallback"].push(type2);
        } else {
            result1[type2.kind][type2.tag] = result1["fallback"][type2.tag] = type2;
        }
    }
    for(index = 0, length = arguments.length; index < length; index += 1){
        arguments[index].forEach(collectType);
    }
    return result1;
}
function Schema$11(definition) {
    return this.extend(definition);
}
Schema$11.prototype.extend = function extend2(definition) {
    var implicit = [];
    var explicit = [];
    if (definition instanceof type1) {
        explicit.push(definition);
    } else if (Array.isArray(definition)) {
        explicit = explicit.concat(definition);
    } else if (definition && (Array.isArray(definition.implicit) || Array.isArray(definition.explicit))) {
        if (definition.implicit) implicit = implicit.concat(definition.implicit);
        if (definition.explicit) explicit = explicit.concat(definition.explicit);
    } else {
        throw new exception1("Schema.extend argument should be a Type, [ Type ], or a schema definition ({ implicit: [...], explicit: [...] })");
    }
    implicit.forEach(function(type$1) {
        if (!(type$1 instanceof type1)) {
            throw new exception1("Specified list of YAML types (or a single Type object) contains a non-Type object.");
        }
        if (type$1.loadKind && type$1.loadKind !== "scalar") {
            throw new exception1("There is a non-scalar type in the implicit list of a schema. Implicit resolving of such types is not supported.");
        }
        if (type$1.multi) {
            throw new exception1("There is a multi type in the implicit list of a schema. Multi tags can only be listed as explicit.");
        }
    });
    explicit.forEach(function(type$1) {
        if (!(type$1 instanceof type1)) {
            throw new exception1("Specified list of YAML types (or a single Type object) contains a non-Type object.");
        }
    });
    var result1 = Object.create(Schema$11.prototype);
    result1.implicit = (this.implicit || []).concat(implicit);
    result1.explicit = (this.explicit || []).concat(explicit);
    result1.compiledImplicit = compileList1(result1, "implicit");
    result1.compiledExplicit = compileList1(result1, "explicit");
    result1.compiledTypeMap = compileMap1(result1.compiledImplicit, result1.compiledExplicit);
    return result1;
};
var schema1 = Schema$11;
var str1 = new type1("tag:yaml.org,2002:str", {
    kind: "scalar",
    construct: function(data) {
        return data !== null ? data : "";
    }
});
var seq1 = new type1("tag:yaml.org,2002:seq", {
    kind: "sequence",
    construct: function(data) {
        return data !== null ? data : [];
    }
});
var map1 = new type1("tag:yaml.org,2002:map", {
    kind: "mapping",
    construct: function(data) {
        return data !== null ? data : {};
    }
});
var failsafe1 = new schema1({
    explicit: [
        str1,
        seq1,
        map1
    ]
});
function resolveYamlNull1(data) {
    if (data === null) return true;
    var max = data.length;
    return max === 1 && data === "~" || max === 4 && (data === "null" || data === "Null" || data === "NULL");
}
function constructYamlNull1() {
    return null;
}
function isNull1(object) {
    return object === null;
}
var _null1 = new type1("tag:yaml.org,2002:null", {
    kind: "scalar",
    resolve: resolveYamlNull1,
    construct: constructYamlNull1,
    predicate: isNull1,
    represent: {
        canonical: function() {
            return "~";
        },
        lowercase: function() {
            return "null";
        },
        uppercase: function() {
            return "NULL";
        },
        camelcase: function() {
            return "Null";
        },
        empty: function() {
            return "";
        }
    },
    defaultStyle: "lowercase"
});
function resolveYamlBoolean1(data) {
    if (data === null) return false;
    var max = data.length;
    return max === 4 && (data === "true" || data === "True" || data === "TRUE") || max === 5 && (data === "false" || data === "False" || data === "FALSE");
}
function constructYamlBoolean1(data) {
    return data === "true" || data === "True" || data === "TRUE";
}
function isBoolean1(object) {
    return Object.prototype.toString.call(object) === "[object Boolean]";
}
var bool1 = new type1("tag:yaml.org,2002:bool", {
    kind: "scalar",
    resolve: resolveYamlBoolean1,
    construct: constructYamlBoolean1,
    predicate: isBoolean1,
    represent: {
        lowercase: function(object) {
            return object ? "true" : "false";
        },
        uppercase: function(object) {
            return object ? "TRUE" : "FALSE";
        },
        camelcase: function(object) {
            return object ? "True" : "False";
        }
    },
    defaultStyle: "lowercase"
});
function isHexCode1(c) {
    return 48 <= c && c <= 57 || 65 <= c && c <= 70 || 97 <= c && c <= 102;
}
function isOctCode1(c) {
    return 48 <= c && c <= 55;
}
function isDecCode1(c) {
    return 48 <= c && c <= 57;
}
function resolveYamlInteger1(data) {
    if (data === null) return false;
    var max = data.length, index = 0, hasDigits = false, ch;
    if (!max) return false;
    ch = data[index];
    if (ch === "-" || ch === "+") {
        ch = data[++index];
    }
    if (ch === "0") {
        if (index + 1 === max) return true;
        ch = data[++index];
        if (ch === "b") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (ch !== "0" && ch !== "1") return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        if (ch === "x") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (!isHexCode1(data.charCodeAt(index))) return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        if (ch === "o") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (!isOctCode1(data.charCodeAt(index))) return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
    }
    if (ch === "_") return false;
    for(; index < max; index++){
        ch = data[index];
        if (ch === "_") continue;
        if (!isDecCode1(data.charCodeAt(index))) {
            return false;
        }
        hasDigits = true;
    }
    if (!hasDigits || ch === "_") return false;
    return true;
}
function constructYamlInteger1(data) {
    var value = data, sign = 1, ch;
    if (value.indexOf("_") !== -1) {
        value = value.replace(/_/g, "");
    }
    ch = value[0];
    if (ch === "-" || ch === "+") {
        if (ch === "-") sign = -1;
        value = value.slice(1);
        ch = value[0];
    }
    if (value === "0") return 0;
    if (ch === "0") {
        if (value[1] === "b") return sign * parseInt(value.slice(2), 2);
        if (value[1] === "x") return sign * parseInt(value.slice(2), 16);
        if (value[1] === "o") return sign * parseInt(value.slice(2), 8);
    }
    return sign * parseInt(value, 10);
}
function isInteger1(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && object % 1 === 0 && !common2.isNegativeZero(object);
}
var __int1 = new type1("tag:yaml.org,2002:int", {
    kind: "scalar",
    resolve: resolveYamlInteger1,
    construct: constructYamlInteger1,
    predicate: isInteger1,
    represent: {
        binary: function(obj) {
            return obj >= 0 ? "0b" + obj.toString(2) : "-0b" + obj.toString(2).slice(1);
        },
        octal: function(obj) {
            return obj >= 0 ? "0o" + obj.toString(8) : "-0o" + obj.toString(8).slice(1);
        },
        decimal: function(obj) {
            return obj.toString(10);
        },
        hexadecimal: function(obj) {
            return obj >= 0 ? "0x" + obj.toString(16).toUpperCase() : "-0x" + obj.toString(16).toUpperCase().slice(1);
        }
    },
    defaultStyle: "decimal",
    styleAliases: {
        binary: [
            2,
            "bin"
        ],
        octal: [
            8,
            "oct"
        ],
        decimal: [
            10,
            "dec"
        ],
        hexadecimal: [
            16,
            "hex"
        ]
    }
});
var YAML_FLOAT_PATTERN1 = new RegExp("^(?:[-+]?(?:[0-9][0-9_]*)(?:\\.[0-9_]*)?(?:[eE][-+]?[0-9]+)?|\\.[0-9_]+(?:[eE][-+]?[0-9]+)?|[-+]?\\.(?:inf|Inf|INF)|\\.(?:nan|NaN|NAN))$");
function resolveYamlFloat1(data) {
    if (data === null) return false;
    if (!YAML_FLOAT_PATTERN1.test(data) || data[data.length - 1] === "_") {
        return false;
    }
    return true;
}
function constructYamlFloat1(data) {
    var value, sign;
    value = data.replace(/_/g, "").toLowerCase();
    sign = value[0] === "-" ? -1 : 1;
    if ("+-".indexOf(value[0]) >= 0) {
        value = value.slice(1);
    }
    if (value === ".inf") {
        return sign === 1 ? Number.POSITIVE_INFINITY : Number.NEGATIVE_INFINITY;
    } else if (value === ".nan") {
        return NaN;
    }
    return sign * parseFloat(value, 10);
}
var SCIENTIFIC_WITHOUT_DOT1 = /^[-+]?[0-9]+e/;
function representYamlFloat1(object, style) {
    var res;
    if (isNaN(object)) {
        switch(style){
            case "lowercase":
                return ".nan";
            case "uppercase":
                return ".NAN";
            case "camelcase":
                return ".NaN";
        }
    } else if (Number.POSITIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return ".inf";
            case "uppercase":
                return ".INF";
            case "camelcase":
                return ".Inf";
        }
    } else if (Number.NEGATIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return "-.inf";
            case "uppercase":
                return "-.INF";
            case "camelcase":
                return "-.Inf";
        }
    } else if (common2.isNegativeZero(object)) {
        return "-0.0";
    }
    res = object.toString(10);
    return SCIENTIFIC_WITHOUT_DOT1.test(res) ? res.replace("e", ".e") : res;
}
function isFloat1(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && (object % 1 !== 0 || common2.isNegativeZero(object));
}
var __float1 = new type1("tag:yaml.org,2002:float", {
    kind: "scalar",
    resolve: resolveYamlFloat1,
    construct: constructYamlFloat1,
    predicate: isFloat1,
    represent: representYamlFloat1,
    defaultStyle: "lowercase"
});
var json1 = failsafe1.extend({
    implicit: [
        _null1,
        bool1,
        __int1,
        __float1
    ]
});
var core1 = json1;
var YAML_DATE_REGEXP1 = new RegExp("^([0-9][0-9][0-9][0-9])-([0-9][0-9])-([0-9][0-9])$");
var YAML_TIMESTAMP_REGEXP1 = new RegExp("^([0-9][0-9][0-9][0-9])-([0-9][0-9]?)-([0-9][0-9]?)(?:[Tt]|[ \\t]+)([0-9][0-9]?):([0-9][0-9]):([0-9][0-9])(?:\\.([0-9]*))?(?:[ \\t]*(Z|([-+])([0-9][0-9]?)(?::([0-9][0-9]))?))?$");
function resolveYamlTimestamp1(data) {
    if (data === null) return false;
    if (YAML_DATE_REGEXP1.exec(data) !== null) return true;
    if (YAML_TIMESTAMP_REGEXP1.exec(data) !== null) return true;
    return false;
}
function constructYamlTimestamp1(data) {
    var match, year, month, day, hour, minute, second, fraction = 0, delta = null, tz_hour, tz_minute, date;
    match = YAML_DATE_REGEXP1.exec(data);
    if (match === null) match = YAML_TIMESTAMP_REGEXP1.exec(data);
    if (match === null) throw new Error("Date resolve error");
    year = +match[1];
    month = +match[2] - 1;
    day = +match[3];
    if (!match[4]) {
        return new Date(Date.UTC(year, month, day));
    }
    hour = +match[4];
    minute = +match[5];
    second = +match[6];
    if (match[7]) {
        fraction = match[7].slice(0, 3);
        while(fraction.length < 3){
            fraction += "0";
        }
        fraction = +fraction;
    }
    if (match[9]) {
        tz_hour = +match[10];
        tz_minute = +(match[11] || 0);
        delta = (tz_hour * 60 + tz_minute) * 6e4;
        if (match[9] === "-") delta = -delta;
    }
    date = new Date(Date.UTC(year, month, day, hour, minute, second, fraction));
    if (delta) date.setTime(date.getTime() - delta);
    return date;
}
function representYamlTimestamp1(object) {
    return object.toISOString();
}
var timestamp1 = new type1("tag:yaml.org,2002:timestamp", {
    kind: "scalar",
    resolve: resolveYamlTimestamp1,
    construct: constructYamlTimestamp1,
    instanceOf: Date,
    represent: representYamlTimestamp1
});
function resolveYamlMerge1(data) {
    return data === "<<" || data === null;
}
var merge1 = new type1("tag:yaml.org,2002:merge", {
    kind: "scalar",
    resolve: resolveYamlMerge1
});
var BASE64_MAP1 = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=\n\r";
function resolveYamlBinary1(data) {
    if (data === null) return false;
    var code, idx, bitlen = 0, max = data.length, map2 = BASE64_MAP1;
    for(idx = 0; idx < max; idx++){
        code = map2.indexOf(data.charAt(idx));
        if (code > 64) continue;
        if (code < 0) return false;
        bitlen += 6;
    }
    return bitlen % 8 === 0;
}
function constructYamlBinary1(data) {
    var idx, tailbits, input = data.replace(/[\r\n=]/g, ""), max = input.length, map2 = BASE64_MAP1, bits = 0, result1 = [];
    for(idx = 0; idx < max; idx++){
        if (idx % 4 === 0 && idx) {
            result1.push(bits >> 16 & 255);
            result1.push(bits >> 8 & 255);
            result1.push(bits & 255);
        }
        bits = bits << 6 | map2.indexOf(input.charAt(idx));
    }
    tailbits = max % 4 * 6;
    if (tailbits === 0) {
        result1.push(bits >> 16 & 255);
        result1.push(bits >> 8 & 255);
        result1.push(bits & 255);
    } else if (tailbits === 18) {
        result1.push(bits >> 10 & 255);
        result1.push(bits >> 2 & 255);
    } else if (tailbits === 12) {
        result1.push(bits >> 4 & 255);
    }
    return new Uint8Array(result1);
}
function representYamlBinary1(object) {
    var result1 = "", bits = 0, idx, tail, max = object.length, map2 = BASE64_MAP1;
    for(idx = 0; idx < max; idx++){
        if (idx % 3 === 0 && idx) {
            result1 += map2[bits >> 18 & 63];
            result1 += map2[bits >> 12 & 63];
            result1 += map2[bits >> 6 & 63];
            result1 += map2[bits & 63];
        }
        bits = (bits << 8) + object[idx];
    }
    tail = max % 3;
    if (tail === 0) {
        result1 += map2[bits >> 18 & 63];
        result1 += map2[bits >> 12 & 63];
        result1 += map2[bits >> 6 & 63];
        result1 += map2[bits & 63];
    } else if (tail === 2) {
        result1 += map2[bits >> 10 & 63];
        result1 += map2[bits >> 4 & 63];
        result1 += map2[bits << 2 & 63];
        result1 += map2[64];
    } else if (tail === 1) {
        result1 += map2[bits >> 2 & 63];
        result1 += map2[bits << 4 & 63];
        result1 += map2[64];
        result1 += map2[64];
    }
    return result1;
}
function isBinary1(obj) {
    return Object.prototype.toString.call(obj) === "[object Uint8Array]";
}
var binary1 = new type1("tag:yaml.org,2002:binary", {
    kind: "scalar",
    resolve: resolveYamlBinary1,
    construct: constructYamlBinary1,
    predicate: isBinary1,
    represent: representYamlBinary1
});
var _hasOwnProperty$31 = Object.prototype.hasOwnProperty;
var _toString$21 = Object.prototype.toString;
function resolveYamlOmap1(data) {
    if (data === null) return true;
    var objectKeys = [], index, length, pair, pairKey, pairHasKey, object = data;
    for(index = 0, length = object.length; index < length; index += 1){
        pair = object[index];
        pairHasKey = false;
        if (_toString$21.call(pair) !== "[object Object]") return false;
        for(pairKey in pair){
            if (_hasOwnProperty$31.call(pair, pairKey)) {
                if (!pairHasKey) pairHasKey = true;
                else return false;
            }
        }
        if (!pairHasKey) return false;
        if (objectKeys.indexOf(pairKey) === -1) objectKeys.push(pairKey);
        else return false;
    }
    return true;
}
function constructYamlOmap1(data) {
    return data !== null ? data : [];
}
var omap1 = new type1("tag:yaml.org,2002:omap", {
    kind: "sequence",
    resolve: resolveYamlOmap1,
    construct: constructYamlOmap1
});
var _toString$11 = Object.prototype.toString;
function resolveYamlPairs1(data) {
    if (data === null) return true;
    var index, length, pair, keys, result1, object = data;
    result1 = new Array(object.length);
    for(index = 0, length = object.length; index < length; index += 1){
        pair = object[index];
        if (_toString$11.call(pair) !== "[object Object]") return false;
        keys = Object.keys(pair);
        if (keys.length !== 1) return false;
        result1[index] = [
            keys[0],
            pair[keys[0]]
        ];
    }
    return true;
}
function constructYamlPairs1(data) {
    if (data === null) return [];
    var index, length, pair, keys, result1, object = data;
    result1 = new Array(object.length);
    for(index = 0, length = object.length; index < length; index += 1){
        pair = object[index];
        keys = Object.keys(pair);
        result1[index] = [
            keys[0],
            pair[keys[0]]
        ];
    }
    return result1;
}
var pairs1 = new type1("tag:yaml.org,2002:pairs", {
    kind: "sequence",
    resolve: resolveYamlPairs1,
    construct: constructYamlPairs1
});
var _hasOwnProperty$21 = Object.prototype.hasOwnProperty;
function resolveYamlSet1(data) {
    if (data === null) return true;
    var key, object = data;
    for(key in object){
        if (_hasOwnProperty$21.call(object, key)) {
            if (object[key] !== null) return false;
        }
    }
    return true;
}
function constructYamlSet1(data) {
    return data !== null ? data : {};
}
var set1 = new type1("tag:yaml.org,2002:set", {
    kind: "mapping",
    resolve: resolveYamlSet1,
    construct: constructYamlSet1
});
var _default1 = core1.extend({
    implicit: [
        timestamp1,
        merge1
    ],
    explicit: [
        binary1,
        omap1,
        pairs1,
        set1
    ]
});
var _hasOwnProperty$11 = Object.prototype.hasOwnProperty;
var CONTEXT_FLOW_IN1 = 1;
var CONTEXT_FLOW_OUT1 = 2;
var CONTEXT_BLOCK_IN1 = 3;
var CONTEXT_BLOCK_OUT1 = 4;
var CHOMPING_CLIP1 = 1;
var CHOMPING_STRIP1 = 2;
var CHOMPING_KEEP1 = 3;
var PATTERN_NON_PRINTABLE1 = /[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x84\x86-\x9F\uFFFE\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF]/;
var PATTERN_NON_ASCII_LINE_BREAKS1 = /[\x85\u2028\u2029]/;
var PATTERN_FLOW_INDICATORS1 = /[,\[\]\{\}]/;
var PATTERN_TAG_HANDLE1 = /^(?:!|!!|![a-z\-]+!)$/i;
var PATTERN_TAG_URI1 = /^(?:!|[^,\[\]\{\}])(?:%[0-9a-f]{2}|[0-9a-z\-#;\/\?:@&=\+\$,_\.!~\*'\(\)\[\]])*$/i;
function _class1(obj) {
    return Object.prototype.toString.call(obj);
}
function is_EOL1(c) {
    return c === 10 || c === 13;
}
function is_WHITE_SPACE1(c) {
    return c === 9 || c === 32;
}
function is_WS_OR_EOL1(c) {
    return c === 9 || c === 32 || c === 10 || c === 13;
}
function is_FLOW_INDICATOR1(c) {
    return c === 44 || c === 91 || c === 93 || c === 123 || c === 125;
}
function fromHexCode1(c) {
    var lc;
    if (48 <= c && c <= 57) {
        return c - 48;
    }
    lc = c | 32;
    if (97 <= lc && lc <= 102) {
        return lc - 97 + 10;
    }
    return -1;
}
function escapedHexLen1(c) {
    if (c === 120) {
        return 2;
    }
    if (c === 117) {
        return 4;
    }
    if (c === 85) {
        return 8;
    }
    return 0;
}
function fromDecimalCode1(c) {
    if (48 <= c && c <= 57) {
        return c - 48;
    }
    return -1;
}
function simpleEscapeSequence1(c) {
    return c === 48 ? "\0" : c === 97 ? "\x07" : c === 98 ? "\b" : c === 116 ? "	" : c === 9 ? "	" : c === 110 ? "\n" : c === 118 ? "\v" : c === 102 ? "\f" : c === 114 ? "\r" : c === 101 ? "" : c === 32 ? " " : c === 34 ? '"' : c === 47 ? "/" : c === 92 ? "\\" : c === 78 ? "\x85" : c === 95 ? "\xA0" : c === 76 ? "\u2028" : c === 80 ? "\u2029" : "";
}
function charFromCodepoint1(c) {
    if (c <= 65535) {
        return String.fromCharCode(c);
    }
    return String.fromCharCode((c - 65536 >> 10) + 55296, (c - 65536 & 1023) + 56320);
}
var simpleEscapeCheck1 = new Array(256);
var simpleEscapeMap1 = new Array(256);
for(var i2 = 0; i2 < 256; i2++){
    simpleEscapeCheck1[i2] = simpleEscapeSequence1(i2) ? 1 : 0;
    simpleEscapeMap1[i2] = simpleEscapeSequence1(i2);
}
function State$11(input, options) {
    this.input = input;
    this.filename = options["filename"] || null;
    this.schema = options["schema"] || _default1;
    this.onWarning = options["onWarning"] || null;
    this.legacy = options["legacy"] || false;
    this.json = options["json"] || false;
    this.listener = options["listener"] || null;
    this.implicitTypes = this.schema.compiledImplicit;
    this.typeMap = this.schema.compiledTypeMap;
    this.length = input.length;
    this.position = 0;
    this.line = 0;
    this.lineStart = 0;
    this.lineIndent = 0;
    this.firstTabInLine = -1;
    this.documents = [];
}
function generateError1(state, message) {
    var mark = {
        name: state.filename,
        buffer: state.input.slice(0, -1),
        position: state.position,
        line: state.line,
        column: state.position - state.lineStart
    };
    mark.snippet = snippet1(mark);
    return new exception1(message, mark);
}
function throwError1(state, message) {
    throw generateError1(state, message);
}
function throwWarning1(state, message) {
    if (state.onWarning) {
        state.onWarning.call(null, generateError1(state, message));
    }
}
var directiveHandlers1 = {
    YAML: function handleYamlDirective(state, name, args) {
        var match, major, minor;
        if (state.version !== null) {
            throwError1(state, "duplication of %YAML directive");
        }
        if (args.length !== 1) {
            throwError1(state, "YAML directive accepts exactly one argument");
        }
        match = /^([0-9]+)\.([0-9]+)$/.exec(args[0]);
        if (match === null) {
            throwError1(state, "ill-formed argument of the YAML directive");
        }
        major = parseInt(match[1], 10);
        minor = parseInt(match[2], 10);
        if (major !== 1) {
            throwError1(state, "unacceptable YAML version of the document");
        }
        state.version = args[0];
        state.checkLineBreaks = minor < 2;
        if (minor !== 1 && minor !== 2) {
            throwWarning1(state, "unsupported YAML version of the document");
        }
    },
    TAG: function handleTagDirective(state, name, args) {
        var handle, prefix;
        if (args.length !== 2) {
            throwError1(state, "TAG directive accepts exactly two arguments");
        }
        handle = args[0];
        prefix = args[1];
        if (!PATTERN_TAG_HANDLE1.test(handle)) {
            throwError1(state, "ill-formed tag handle (first argument) of the TAG directive");
        }
        if (_hasOwnProperty$11.call(state.tagMap, handle)) {
            throwError1(state, 'there is a previously declared suffix for "' + handle + '" tag handle');
        }
        if (!PATTERN_TAG_URI1.test(prefix)) {
            throwError1(state, "ill-formed tag prefix (second argument) of the TAG directive");
        }
        try {
            prefix = decodeURIComponent(prefix);
        } catch (err) {
            throwError1(state, "tag prefix is malformed: " + prefix);
        }
        state.tagMap[handle] = prefix;
    }
};
function captureSegment1(state, start, end, checkJson) {
    var _position, _length, _character, _result;
    if (start < end) {
        _result = state.input.slice(start, end);
        if (checkJson) {
            for(_position = 0, _length = _result.length; _position < _length; _position += 1){
                _character = _result.charCodeAt(_position);
                if (!(_character === 9 || 32 <= _character && _character <= 1114111)) {
                    throwError1(state, "expected valid JSON character");
                }
            }
        } else if (PATTERN_NON_PRINTABLE1.test(_result)) {
            throwError1(state, "the stream contains non-printable characters");
        }
        state.result += _result;
    }
}
function mergeMappings1(state, destination, source, overridableKeys) {
    var sourceKeys, key, index, quantity;
    if (!common2.isObject(source)) {
        throwError1(state, "cannot merge mappings; the provided source object is unacceptable");
    }
    sourceKeys = Object.keys(source);
    for(index = 0, quantity = sourceKeys.length; index < quantity; index += 1){
        key = sourceKeys[index];
        if (!_hasOwnProperty$11.call(destination, key)) {
            destination[key] = source[key];
            overridableKeys[key] = true;
        }
    }
}
function storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, valueNode, startLine, startLineStart, startPos) {
    var index, quantity;
    if (Array.isArray(keyNode)) {
        keyNode = Array.prototype.slice.call(keyNode);
        for(index = 0, quantity = keyNode.length; index < quantity; index += 1){
            if (Array.isArray(keyNode[index])) {
                throwError1(state, "nested arrays are not supported inside keys");
            }
            if (typeof keyNode === "object" && _class1(keyNode[index]) === "[object Object]") {
                keyNode[index] = "[object Object]";
            }
        }
    }
    if (typeof keyNode === "object" && _class1(keyNode) === "[object Object]") {
        keyNode = "[object Object]";
    }
    keyNode = String(keyNode);
    if (_result === null) {
        _result = {};
    }
    if (keyTag === "tag:yaml.org,2002:merge") {
        if (Array.isArray(valueNode)) {
            for(index = 0, quantity = valueNode.length; index < quantity; index += 1){
                mergeMappings1(state, _result, valueNode[index], overridableKeys);
            }
        } else {
            mergeMappings1(state, _result, valueNode, overridableKeys);
        }
    } else {
        if (!state.json && !_hasOwnProperty$11.call(overridableKeys, keyNode) && _hasOwnProperty$11.call(_result, keyNode)) {
            state.line = startLine || state.line;
            state.lineStart = startLineStart || state.lineStart;
            state.position = startPos || state.position;
            throwError1(state, "duplicated mapping key");
        }
        if (keyNode === "__proto__") {
            Object.defineProperty(_result, keyNode, {
                configurable: true,
                enumerable: true,
                writable: true,
                value: valueNode
            });
        } else {
            _result[keyNode] = valueNode;
        }
        delete overridableKeys[keyNode];
    }
    return _result;
}
function readLineBreak1(state) {
    var ch;
    ch = state.input.charCodeAt(state.position);
    if (ch === 10) {
        state.position++;
    } else if (ch === 13) {
        state.position++;
        if (state.input.charCodeAt(state.position) === 10) {
            state.position++;
        }
    } else {
        throwError1(state, "a line break is expected");
    }
    state.line += 1;
    state.lineStart = state.position;
    state.firstTabInLine = -1;
}
function skipSeparationSpace1(state, allowComments, checkIndent) {
    var lineBreaks = 0, ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        while(is_WHITE_SPACE1(ch)){
            if (ch === 9 && state.firstTabInLine === -1) {
                state.firstTabInLine = state.position;
            }
            ch = state.input.charCodeAt(++state.position);
        }
        if (allowComments && ch === 35) {
            do {
                ch = state.input.charCodeAt(++state.position);
            }while (ch !== 10 && ch !== 13 && ch !== 0)
        }
        if (is_EOL1(ch)) {
            readLineBreak1(state);
            ch = state.input.charCodeAt(state.position);
            lineBreaks++;
            state.lineIndent = 0;
            while(ch === 32){
                state.lineIndent++;
                ch = state.input.charCodeAt(++state.position);
            }
        } else {
            break;
        }
    }
    if (checkIndent !== -1 && lineBreaks !== 0 && state.lineIndent < checkIndent) {
        throwWarning1(state, "deficient indentation");
    }
    return lineBreaks;
}
function testDocumentSeparator1(state) {
    var _position = state.position, ch;
    ch = state.input.charCodeAt(_position);
    if ((ch === 45 || ch === 46) && ch === state.input.charCodeAt(_position + 1) && ch === state.input.charCodeAt(_position + 2)) {
        _position += 3;
        ch = state.input.charCodeAt(_position);
        if (ch === 0 || is_WS_OR_EOL1(ch)) {
            return true;
        }
    }
    return false;
}
function writeFoldedLines1(state, count) {
    if (count === 1) {
        state.result += " ";
    } else if (count > 1) {
        state.result += common2.repeat("\n", count - 1);
    }
}
function readPlainScalar1(state, nodeIndent, withinFlowCollection) {
    var preceding, following, captureStart, captureEnd, hasPendingContent, _line, _lineStart, _lineIndent, _kind = state.kind, _result = state.result, ch;
    ch = state.input.charCodeAt(state.position);
    if (is_WS_OR_EOL1(ch) || is_FLOW_INDICATOR1(ch) || ch === 35 || ch === 38 || ch === 42 || ch === 33 || ch === 124 || ch === 62 || ch === 39 || ch === 34 || ch === 37 || ch === 64 || ch === 96) {
        return false;
    }
    if (ch === 63 || ch === 45) {
        following = state.input.charCodeAt(state.position + 1);
        if (is_WS_OR_EOL1(following) || withinFlowCollection && is_FLOW_INDICATOR1(following)) {
            return false;
        }
    }
    state.kind = "scalar";
    state.result = "";
    captureStart = captureEnd = state.position;
    hasPendingContent = false;
    while(ch !== 0){
        if (ch === 58) {
            following = state.input.charCodeAt(state.position + 1);
            if (is_WS_OR_EOL1(following) || withinFlowCollection && is_FLOW_INDICATOR1(following)) {
                break;
            }
        } else if (ch === 35) {
            preceding = state.input.charCodeAt(state.position - 1);
            if (is_WS_OR_EOL1(preceding)) {
                break;
            }
        } else if (state.position === state.lineStart && testDocumentSeparator1(state) || withinFlowCollection && is_FLOW_INDICATOR1(ch)) {
            break;
        } else if (is_EOL1(ch)) {
            _line = state.line;
            _lineStart = state.lineStart;
            _lineIndent = state.lineIndent;
            skipSeparationSpace1(state, false, -1);
            if (state.lineIndent >= nodeIndent) {
                hasPendingContent = true;
                ch = state.input.charCodeAt(state.position);
                continue;
            } else {
                state.position = captureEnd;
                state.line = _line;
                state.lineStart = _lineStart;
                state.lineIndent = _lineIndent;
                break;
            }
        }
        if (hasPendingContent) {
            captureSegment1(state, captureStart, captureEnd, false);
            writeFoldedLines1(state, state.line - _line);
            captureStart = captureEnd = state.position;
            hasPendingContent = false;
        }
        if (!is_WHITE_SPACE1(ch)) {
            captureEnd = state.position + 1;
        }
        ch = state.input.charCodeAt(++state.position);
    }
    captureSegment1(state, captureStart, captureEnd, false);
    if (state.result) {
        return true;
    }
    state.kind = _kind;
    state.result = _result;
    return false;
}
function readSingleQuotedScalar1(state, nodeIndent) {
    var ch, captureStart, captureEnd;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 39) {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    state.position++;
    captureStart = captureEnd = state.position;
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        if (ch === 39) {
            captureSegment1(state, captureStart, state.position, true);
            ch = state.input.charCodeAt(++state.position);
            if (ch === 39) {
                captureStart = state.position;
                state.position++;
                captureEnd = state.position;
            } else {
                return true;
            }
        } else if (is_EOL1(ch)) {
            captureSegment1(state, captureStart, captureEnd, true);
            writeFoldedLines1(state, skipSeparationSpace1(state, false, nodeIndent));
            captureStart = captureEnd = state.position;
        } else if (state.position === state.lineStart && testDocumentSeparator1(state)) {
            throwError1(state, "unexpected end of the document within a single quoted scalar");
        } else {
            state.position++;
            captureEnd = state.position;
        }
    }
    throwError1(state, "unexpected end of the stream within a single quoted scalar");
}
function readDoubleQuotedScalar1(state, nodeIndent) {
    var captureStart, captureEnd, hexLength, hexResult, tmp, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 34) {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    state.position++;
    captureStart = captureEnd = state.position;
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        if (ch === 34) {
            captureSegment1(state, captureStart, state.position, true);
            state.position++;
            return true;
        } else if (ch === 92) {
            captureSegment1(state, captureStart, state.position, true);
            ch = state.input.charCodeAt(++state.position);
            if (is_EOL1(ch)) {
                skipSeparationSpace1(state, false, nodeIndent);
            } else if (ch < 256 && simpleEscapeCheck1[ch]) {
                state.result += simpleEscapeMap1[ch];
                state.position++;
            } else if ((tmp = escapedHexLen1(ch)) > 0) {
                hexLength = tmp;
                hexResult = 0;
                for(; hexLength > 0; hexLength--){
                    ch = state.input.charCodeAt(++state.position);
                    if ((tmp = fromHexCode1(ch)) >= 0) {
                        hexResult = (hexResult << 4) + tmp;
                    } else {
                        throwError1(state, "expected hexadecimal character");
                    }
                }
                state.result += charFromCodepoint1(hexResult);
                state.position++;
            } else {
                throwError1(state, "unknown escape sequence");
            }
            captureStart = captureEnd = state.position;
        } else if (is_EOL1(ch)) {
            captureSegment1(state, captureStart, captureEnd, true);
            writeFoldedLines1(state, skipSeparationSpace1(state, false, nodeIndent));
            captureStart = captureEnd = state.position;
        } else if (state.position === state.lineStart && testDocumentSeparator1(state)) {
            throwError1(state, "unexpected end of the document within a double quoted scalar");
        } else {
            state.position++;
            captureEnd = state.position;
        }
    }
    throwError1(state, "unexpected end of the stream within a double quoted scalar");
}
function readFlowCollection1(state, nodeIndent) {
    var readNext = true, _line, _lineStart, _pos, _tag = state.tag, _result, _anchor = state.anchor, following, terminator, isPair, isExplicitPair, isMapping, overridableKeys = Object.create(null), keyNode, keyTag, valueNode, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch === 91) {
        terminator = 93;
        isMapping = false;
        _result = [];
    } else if (ch === 123) {
        terminator = 125;
        isMapping = true;
        _result = {};
    } else {
        return false;
    }
    if (state.anchor !== null) {
        state.anchorMap[state.anchor] = _result;
    }
    ch = state.input.charCodeAt(++state.position);
    while(ch !== 0){
        skipSeparationSpace1(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if (ch === terminator) {
            state.position++;
            state.tag = _tag;
            state.anchor = _anchor;
            state.kind = isMapping ? "mapping" : "sequence";
            state.result = _result;
            return true;
        } else if (!readNext) {
            throwError1(state, "missed comma between flow collection entries");
        } else if (ch === 44) {
            throwError1(state, "expected the node content, but found ','");
        }
        keyTag = keyNode = valueNode = null;
        isPair = isExplicitPair = false;
        if (ch === 63) {
            following = state.input.charCodeAt(state.position + 1);
            if (is_WS_OR_EOL1(following)) {
                isPair = isExplicitPair = true;
                state.position++;
                skipSeparationSpace1(state, true, nodeIndent);
            }
        }
        _line = state.line;
        _lineStart = state.lineStart;
        _pos = state.position;
        composeNode1(state, nodeIndent, CONTEXT_FLOW_IN1, false, true);
        keyTag = state.tag;
        keyNode = state.result;
        skipSeparationSpace1(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if ((isExplicitPair || state.line === _line) && ch === 58) {
            isPair = true;
            ch = state.input.charCodeAt(++state.position);
            skipSeparationSpace1(state, true, nodeIndent);
            composeNode1(state, nodeIndent, CONTEXT_FLOW_IN1, false, true);
            valueNode = state.result;
        }
        if (isMapping) {
            storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, valueNode, _line, _lineStart, _pos);
        } else if (isPair) {
            _result.push(storeMappingPair1(state, null, overridableKeys, keyTag, keyNode, valueNode, _line, _lineStart, _pos));
        } else {
            _result.push(keyNode);
        }
        skipSeparationSpace1(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if (ch === 44) {
            readNext = true;
            ch = state.input.charCodeAt(++state.position);
        } else {
            readNext = false;
        }
    }
    throwError1(state, "unexpected end of the stream within a flow collection");
}
function readBlockScalar1(state, nodeIndent) {
    var captureStart, folding, chomping = CHOMPING_CLIP1, didReadContent = false, detectedIndent = false, textIndent = nodeIndent, emptyLines = 0, atMoreIndented = false, tmp, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch === 124) {
        folding = false;
    } else if (ch === 62) {
        folding = true;
    } else {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    while(ch !== 0){
        ch = state.input.charCodeAt(++state.position);
        if (ch === 43 || ch === 45) {
            if (CHOMPING_CLIP1 === chomping) {
                chomping = ch === 43 ? CHOMPING_KEEP1 : CHOMPING_STRIP1;
            } else {
                throwError1(state, "repeat of a chomping mode identifier");
            }
        } else if ((tmp = fromDecimalCode1(ch)) >= 0) {
            if (tmp === 0) {
                throwError1(state, "bad explicit indentation width of a block scalar; it cannot be less than one");
            } else if (!detectedIndent) {
                textIndent = nodeIndent + tmp - 1;
                detectedIndent = true;
            } else {
                throwError1(state, "repeat of an indentation width identifier");
            }
        } else {
            break;
        }
    }
    if (is_WHITE_SPACE1(ch)) {
        do {
            ch = state.input.charCodeAt(++state.position);
        }while (is_WHITE_SPACE1(ch))
        if (ch === 35) {
            do {
                ch = state.input.charCodeAt(++state.position);
            }while (!is_EOL1(ch) && ch !== 0)
        }
    }
    while(ch !== 0){
        readLineBreak1(state);
        state.lineIndent = 0;
        ch = state.input.charCodeAt(state.position);
        while((!detectedIndent || state.lineIndent < textIndent) && ch === 32){
            state.lineIndent++;
            ch = state.input.charCodeAt(++state.position);
        }
        if (!detectedIndent && state.lineIndent > textIndent) {
            textIndent = state.lineIndent;
        }
        if (is_EOL1(ch)) {
            emptyLines++;
            continue;
        }
        if (state.lineIndent < textIndent) {
            if (chomping === CHOMPING_KEEP1) {
                state.result += common2.repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (chomping === CHOMPING_CLIP1) {
                if (didReadContent) {
                    state.result += "\n";
                }
            }
            break;
        }
        if (folding) {
            if (is_WHITE_SPACE1(ch)) {
                atMoreIndented = true;
                state.result += common2.repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (atMoreIndented) {
                atMoreIndented = false;
                state.result += common2.repeat("\n", emptyLines + 1);
            } else if (emptyLines === 0) {
                if (didReadContent) {
                    state.result += " ";
                }
            } else {
                state.result += common2.repeat("\n", emptyLines);
            }
        } else {
            state.result += common2.repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
        }
        didReadContent = true;
        detectedIndent = true;
        emptyLines = 0;
        captureStart = state.position;
        while(!is_EOL1(ch) && ch !== 0){
            ch = state.input.charCodeAt(++state.position);
        }
        captureSegment1(state, captureStart, state.position, false);
    }
    return true;
}
function readBlockSequence1(state, nodeIndent) {
    var _line, _tag = state.tag, _anchor = state.anchor, _result = [], following, detected = false, ch;
    if (state.firstTabInLine !== -1) return false;
    if (state.anchor !== null) {
        state.anchorMap[state.anchor] = _result;
    }
    ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        if (state.firstTabInLine !== -1) {
            state.position = state.firstTabInLine;
            throwError1(state, "tab characters must not be used in indentation");
        }
        if (ch !== 45) {
            break;
        }
        following = state.input.charCodeAt(state.position + 1);
        if (!is_WS_OR_EOL1(following)) {
            break;
        }
        detected = true;
        state.position++;
        if (skipSeparationSpace1(state, true, -1)) {
            if (state.lineIndent <= nodeIndent) {
                _result.push(null);
                ch = state.input.charCodeAt(state.position);
                continue;
            }
        }
        _line = state.line;
        composeNode1(state, nodeIndent, CONTEXT_BLOCK_IN1, false, true);
        _result.push(state.result);
        skipSeparationSpace1(state, true, -1);
        ch = state.input.charCodeAt(state.position);
        if ((state.line === _line || state.lineIndent > nodeIndent) && ch !== 0) {
            throwError1(state, "bad indentation of a sequence entry");
        } else if (state.lineIndent < nodeIndent) {
            break;
        }
    }
    if (detected) {
        state.tag = _tag;
        state.anchor = _anchor;
        state.kind = "sequence";
        state.result = _result;
        return true;
    }
    return false;
}
function readBlockMapping1(state, nodeIndent, flowIndent) {
    var following, allowCompact, _line, _keyLine, _keyLineStart, _keyPos, _tag = state.tag, _anchor = state.anchor, _result = {}, overridableKeys = Object.create(null), keyTag = null, keyNode = null, valueNode = null, atExplicitKey = false, detected = false, ch;
    if (state.firstTabInLine !== -1) return false;
    if (state.anchor !== null) {
        state.anchorMap[state.anchor] = _result;
    }
    ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        if (!atExplicitKey && state.firstTabInLine !== -1) {
            state.position = state.firstTabInLine;
            throwError1(state, "tab characters must not be used in indentation");
        }
        following = state.input.charCodeAt(state.position + 1);
        _line = state.line;
        if ((ch === 63 || ch === 58) && is_WS_OR_EOL1(following)) {
            if (ch === 63) {
                if (atExplicitKey) {
                    storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, null, _keyLine, _keyLineStart, _keyPos);
                    keyTag = keyNode = valueNode = null;
                }
                detected = true;
                atExplicitKey = true;
                allowCompact = true;
            } else if (atExplicitKey) {
                atExplicitKey = false;
                allowCompact = true;
            } else {
                throwError1(state, "incomplete explicit mapping pair; a key node is missed; or followed by a non-tabulated empty line");
            }
            state.position += 1;
            ch = following;
        } else {
            _keyLine = state.line;
            _keyLineStart = state.lineStart;
            _keyPos = state.position;
            if (!composeNode1(state, flowIndent, CONTEXT_FLOW_OUT1, false, true)) {
                break;
            }
            if (state.line === _line) {
                ch = state.input.charCodeAt(state.position);
                while(is_WHITE_SPACE1(ch)){
                    ch = state.input.charCodeAt(++state.position);
                }
                if (ch === 58) {
                    ch = state.input.charCodeAt(++state.position);
                    if (!is_WS_OR_EOL1(ch)) {
                        throwError1(state, "a whitespace character is expected after the key-value separator within a block mapping");
                    }
                    if (atExplicitKey) {
                        storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, null, _keyLine, _keyLineStart, _keyPos);
                        keyTag = keyNode = valueNode = null;
                    }
                    detected = true;
                    atExplicitKey = false;
                    allowCompact = false;
                    keyTag = state.tag;
                    keyNode = state.result;
                } else if (detected) {
                    throwError1(state, "can not read an implicit mapping pair; a colon is missed");
                } else {
                    state.tag = _tag;
                    state.anchor = _anchor;
                    return true;
                }
            } else if (detected) {
                throwError1(state, "can not read a block mapping entry; a multiline key may not be an implicit key");
            } else {
                state.tag = _tag;
                state.anchor = _anchor;
                return true;
            }
        }
        if (state.line === _line || state.lineIndent > nodeIndent) {
            if (atExplicitKey) {
                _keyLine = state.line;
                _keyLineStart = state.lineStart;
                _keyPos = state.position;
            }
            if (composeNode1(state, nodeIndent, CONTEXT_BLOCK_OUT1, true, allowCompact)) {
                if (atExplicitKey) {
                    keyNode = state.result;
                } else {
                    valueNode = state.result;
                }
            }
            if (!atExplicitKey) {
                storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, valueNode, _keyLine, _keyLineStart, _keyPos);
                keyTag = keyNode = valueNode = null;
            }
            skipSeparationSpace1(state, true, -1);
            ch = state.input.charCodeAt(state.position);
        }
        if ((state.line === _line || state.lineIndent > nodeIndent) && ch !== 0) {
            throwError1(state, "bad indentation of a mapping entry");
        } else if (state.lineIndent < nodeIndent) {
            break;
        }
    }
    if (atExplicitKey) {
        storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, null, _keyLine, _keyLineStart, _keyPos);
    }
    if (detected) {
        state.tag = _tag;
        state.anchor = _anchor;
        state.kind = "mapping";
        state.result = _result;
    }
    return detected;
}
function readTagProperty1(state) {
    var _position, isVerbatim = false, isNamed = false, tagHandle, tagName, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 33) return false;
    if (state.tag !== null) {
        throwError1(state, "duplication of a tag property");
    }
    ch = state.input.charCodeAt(++state.position);
    if (ch === 60) {
        isVerbatim = true;
        ch = state.input.charCodeAt(++state.position);
    } else if (ch === 33) {
        isNamed = true;
        tagHandle = "!!";
        ch = state.input.charCodeAt(++state.position);
    } else {
        tagHandle = "!";
    }
    _position = state.position;
    if (isVerbatim) {
        do {
            ch = state.input.charCodeAt(++state.position);
        }while (ch !== 0 && ch !== 62)
        if (state.position < state.length) {
            tagName = state.input.slice(_position, state.position);
            ch = state.input.charCodeAt(++state.position);
        } else {
            throwError1(state, "unexpected end of the stream within a verbatim tag");
        }
    } else {
        while(ch !== 0 && !is_WS_OR_EOL1(ch)){
            if (ch === 33) {
                if (!isNamed) {
                    tagHandle = state.input.slice(_position - 1, state.position + 1);
                    if (!PATTERN_TAG_HANDLE1.test(tagHandle)) {
                        throwError1(state, "named tag handle cannot contain such characters");
                    }
                    isNamed = true;
                    _position = state.position + 1;
                } else {
                    throwError1(state, "tag suffix cannot contain exclamation marks");
                }
            }
            ch = state.input.charCodeAt(++state.position);
        }
        tagName = state.input.slice(_position, state.position);
        if (PATTERN_FLOW_INDICATORS1.test(tagName)) {
            throwError1(state, "tag suffix cannot contain flow indicator characters");
        }
    }
    if (tagName && !PATTERN_TAG_URI1.test(tagName)) {
        throwError1(state, "tag name cannot contain such characters: " + tagName);
    }
    try {
        tagName = decodeURIComponent(tagName);
    } catch (err) {
        throwError1(state, "tag name is malformed: " + tagName);
    }
    if (isVerbatim) {
        state.tag = tagName;
    } else if (_hasOwnProperty$11.call(state.tagMap, tagHandle)) {
        state.tag = state.tagMap[tagHandle] + tagName;
    } else if (tagHandle === "!") {
        state.tag = "!" + tagName;
    } else if (tagHandle === "!!") {
        state.tag = "tag:yaml.org,2002:" + tagName;
    } else {
        throwError1(state, 'undeclared tag handle "' + tagHandle + '"');
    }
    return true;
}
function readAnchorProperty1(state) {
    var _position, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 38) return false;
    if (state.anchor !== null) {
        throwError1(state, "duplication of an anchor property");
    }
    ch = state.input.charCodeAt(++state.position);
    _position = state.position;
    while(ch !== 0 && !is_WS_OR_EOL1(ch) && !is_FLOW_INDICATOR1(ch)){
        ch = state.input.charCodeAt(++state.position);
    }
    if (state.position === _position) {
        throwError1(state, "name of an anchor node must contain at least one character");
    }
    state.anchor = state.input.slice(_position, state.position);
    return true;
}
function readAlias1(state) {
    var _position, alias, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 42) return false;
    ch = state.input.charCodeAt(++state.position);
    _position = state.position;
    while(ch !== 0 && !is_WS_OR_EOL1(ch) && !is_FLOW_INDICATOR1(ch)){
        ch = state.input.charCodeAt(++state.position);
    }
    if (state.position === _position) {
        throwError1(state, "name of an alias node must contain at least one character");
    }
    alias = state.input.slice(_position, state.position);
    if (!_hasOwnProperty$11.call(state.anchorMap, alias)) {
        throwError1(state, 'unidentified alias "' + alias + '"');
    }
    state.result = state.anchorMap[alias];
    skipSeparationSpace1(state, true, -1);
    return true;
}
function composeNode1(state, parentIndent, nodeContext, allowToSeek, allowCompact) {
    var allowBlockStyles, allowBlockScalars, allowBlockCollections, indentStatus = 1, atNewLine = false, hasContent = false, typeIndex, typeQuantity, typeList, type2, flowIndent, blockIndent;
    if (state.listener !== null) {
        state.listener("open", state);
    }
    state.tag = null;
    state.anchor = null;
    state.kind = null;
    state.result = null;
    allowBlockStyles = allowBlockScalars = allowBlockCollections = CONTEXT_BLOCK_OUT1 === nodeContext || CONTEXT_BLOCK_IN1 === nodeContext;
    if (allowToSeek) {
        if (skipSeparationSpace1(state, true, -1)) {
            atNewLine = true;
            if (state.lineIndent > parentIndent) {
                indentStatus = 1;
            } else if (state.lineIndent === parentIndent) {
                indentStatus = 0;
            } else if (state.lineIndent < parentIndent) {
                indentStatus = -1;
            }
        }
    }
    if (indentStatus === 1) {
        while(readTagProperty1(state) || readAnchorProperty1(state)){
            if (skipSeparationSpace1(state, true, -1)) {
                atNewLine = true;
                allowBlockCollections = allowBlockStyles;
                if (state.lineIndent > parentIndent) {
                    indentStatus = 1;
                } else if (state.lineIndent === parentIndent) {
                    indentStatus = 0;
                } else if (state.lineIndent < parentIndent) {
                    indentStatus = -1;
                }
            } else {
                allowBlockCollections = false;
            }
        }
    }
    if (allowBlockCollections) {
        allowBlockCollections = atNewLine || allowCompact;
    }
    if (indentStatus === 1 || CONTEXT_BLOCK_OUT1 === nodeContext) {
        if (CONTEXT_FLOW_IN1 === nodeContext || CONTEXT_FLOW_OUT1 === nodeContext) {
            flowIndent = parentIndent;
        } else {
            flowIndent = parentIndent + 1;
        }
        blockIndent = state.position - state.lineStart;
        if (indentStatus === 1) {
            if (allowBlockCollections && (readBlockSequence1(state, blockIndent) || readBlockMapping1(state, blockIndent, flowIndent)) || readFlowCollection1(state, flowIndent)) {
                hasContent = true;
            } else {
                if (allowBlockScalars && readBlockScalar1(state, flowIndent) || readSingleQuotedScalar1(state, flowIndent) || readDoubleQuotedScalar1(state, flowIndent)) {
                    hasContent = true;
                } else if (readAlias1(state)) {
                    hasContent = true;
                    if (state.tag !== null || state.anchor !== null) {
                        throwError1(state, "alias node should not have any properties");
                    }
                } else if (readPlainScalar1(state, flowIndent, CONTEXT_FLOW_IN1 === nodeContext)) {
                    hasContent = true;
                    if (state.tag === null) {
                        state.tag = "?";
                    }
                }
                if (state.anchor !== null) {
                    state.anchorMap[state.anchor] = state.result;
                }
            }
        } else if (indentStatus === 0) {
            hasContent = allowBlockCollections && readBlockSequence1(state, blockIndent);
        }
    }
    if (state.tag === null) {
        if (state.anchor !== null) {
            state.anchorMap[state.anchor] = state.result;
        }
    } else if (state.tag === "?") {
        if (state.result !== null && state.kind !== "scalar") {
            throwError1(state, 'unacceptable node kind for !<?> tag; it should be "scalar", not "' + state.kind + '"');
        }
        for(typeIndex = 0, typeQuantity = state.implicitTypes.length; typeIndex < typeQuantity; typeIndex += 1){
            type2 = state.implicitTypes[typeIndex];
            if (type2.resolve(state.result)) {
                state.result = type2.construct(state.result);
                state.tag = type2.tag;
                if (state.anchor !== null) {
                    state.anchorMap[state.anchor] = state.result;
                }
                break;
            }
        }
    } else if (state.tag !== "!") {
        if (_hasOwnProperty$11.call(state.typeMap[state.kind || "fallback"], state.tag)) {
            type2 = state.typeMap[state.kind || "fallback"][state.tag];
        } else {
            type2 = null;
            typeList = state.typeMap.multi[state.kind || "fallback"];
            for(typeIndex = 0, typeQuantity = typeList.length; typeIndex < typeQuantity; typeIndex += 1){
                if (state.tag.slice(0, typeList[typeIndex].tag.length) === typeList[typeIndex].tag) {
                    type2 = typeList[typeIndex];
                    break;
                }
            }
        }
        if (!type2) {
            throwError1(state, "unknown tag !<" + state.tag + ">");
        }
        if (state.result !== null && type2.kind !== state.kind) {
            throwError1(state, "unacceptable node kind for !<" + state.tag + '> tag; it should be "' + type2.kind + '", not "' + state.kind + '"');
        }
        if (!type2.resolve(state.result, state.tag)) {
            throwError1(state, "cannot resolve a node with !<" + state.tag + "> explicit tag");
        } else {
            state.result = type2.construct(state.result, state.tag);
            if (state.anchor !== null) {
                state.anchorMap[state.anchor] = state.result;
            }
        }
    }
    if (state.listener !== null) {
        state.listener("close", state);
    }
    return state.tag !== null || state.anchor !== null || hasContent;
}
function readDocument1(state) {
    var documentStart = state.position, _position, directiveName, directiveArgs, hasDirectives = false, ch;
    state.version = null;
    state.checkLineBreaks = state.legacy;
    state.tagMap = Object.create(null);
    state.anchorMap = Object.create(null);
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        skipSeparationSpace1(state, true, -1);
        ch = state.input.charCodeAt(state.position);
        if (state.lineIndent > 0 || ch !== 37) {
            break;
        }
        hasDirectives = true;
        ch = state.input.charCodeAt(++state.position);
        _position = state.position;
        while(ch !== 0 && !is_WS_OR_EOL1(ch)){
            ch = state.input.charCodeAt(++state.position);
        }
        directiveName = state.input.slice(_position, state.position);
        directiveArgs = [];
        if (directiveName.length < 1) {
            throwError1(state, "directive name must not be less than one character in length");
        }
        while(ch !== 0){
            while(is_WHITE_SPACE1(ch)){
                ch = state.input.charCodeAt(++state.position);
            }
            if (ch === 35) {
                do {
                    ch = state.input.charCodeAt(++state.position);
                }while (ch !== 0 && !is_EOL1(ch))
                break;
            }
            if (is_EOL1(ch)) break;
            _position = state.position;
            while(ch !== 0 && !is_WS_OR_EOL1(ch)){
                ch = state.input.charCodeAt(++state.position);
            }
            directiveArgs.push(state.input.slice(_position, state.position));
        }
        if (ch !== 0) readLineBreak1(state);
        if (_hasOwnProperty$11.call(directiveHandlers1, directiveName)) {
            directiveHandlers1[directiveName](state, directiveName, directiveArgs);
        } else {
            throwWarning1(state, 'unknown document directive "' + directiveName + '"');
        }
    }
    skipSeparationSpace1(state, true, -1);
    if (state.lineIndent === 0 && state.input.charCodeAt(state.position) === 45 && state.input.charCodeAt(state.position + 1) === 45 && state.input.charCodeAt(state.position + 2) === 45) {
        state.position += 3;
        skipSeparationSpace1(state, true, -1);
    } else if (hasDirectives) {
        throwError1(state, "directives end mark is expected");
    }
    composeNode1(state, state.lineIndent - 1, CONTEXT_BLOCK_OUT1, false, true);
    skipSeparationSpace1(state, true, -1);
    if (state.checkLineBreaks && PATTERN_NON_ASCII_LINE_BREAKS1.test(state.input.slice(documentStart, state.position))) {
        throwWarning1(state, "non-ASCII line breaks are interpreted as content");
    }
    state.documents.push(state.result);
    if (state.position === state.lineStart && testDocumentSeparator1(state)) {
        if (state.input.charCodeAt(state.position) === 46) {
            state.position += 3;
            skipSeparationSpace1(state, true, -1);
        }
        return;
    }
    if (state.position < state.length - 1) {
        throwError1(state, "end of the stream or a document separator is expected");
    } else {
        return;
    }
}
function loadDocuments1(input, options) {
    input = String(input);
    options = options || {};
    if (input.length !== 0) {
        if (input.charCodeAt(input.length - 1) !== 10 && input.charCodeAt(input.length - 1) !== 13) {
            input += "\n";
        }
        if (input.charCodeAt(0) === 65279) {
            input = input.slice(1);
        }
    }
    var state = new State$11(input, options);
    var nullpos = input.indexOf("\0");
    if (nullpos !== -1) {
        state.position = nullpos;
        throwError1(state, "null byte is not allowed in input");
    }
    state.input += "\0";
    while(state.input.charCodeAt(state.position) === 32){
        state.lineIndent += 1;
        state.position += 1;
    }
    while(state.position < state.length - 1){
        readDocument1(state);
    }
    return state.documents;
}
function loadAll$11(input, iterator, options) {
    if (iterator !== null && typeof iterator === "object" && typeof options === "undefined") {
        options = iterator;
        iterator = null;
    }
    var documents = loadDocuments1(input, options);
    if (typeof iterator !== "function") {
        return documents;
    }
    for(var index = 0, length = documents.length; index < length; index += 1){
        iterator(documents[index]);
    }
}
function load$11(input, options) {
    var documents = loadDocuments1(input, options);
    if (documents.length === 0) {
        return void 0;
    } else if (documents.length === 1) {
        return documents[0];
    }
    throw new exception1("expected a single document in the stream, but found more");
}
var loadAll_11 = loadAll$11;
var load_11 = load$11;
var loader1 = {
    loadAll: loadAll_11,
    load: load_11
};
var _toString1 = Object.prototype.toString;
var _hasOwnProperty1 = Object.prototype.hasOwnProperty;
var CHAR_BOM1 = 65279;
var CHAR_TAB1 = 9;
var CHAR_LINE_FEED1 = 10;
var CHAR_CARRIAGE_RETURN1 = 13;
var CHAR_SPACE1 = 32;
var CHAR_EXCLAMATION1 = 33;
var CHAR_DOUBLE_QUOTE1 = 34;
var CHAR_SHARP1 = 35;
var CHAR_PERCENT1 = 37;
var CHAR_AMPERSAND1 = 38;
var CHAR_SINGLE_QUOTE1 = 39;
var CHAR_ASTERISK1 = 42;
var CHAR_COMMA1 = 44;
var CHAR_MINUS1 = 45;
var CHAR_COLON1 = 58;
var CHAR_EQUALS1 = 61;
var CHAR_GREATER_THAN1 = 62;
var CHAR_QUESTION1 = 63;
var CHAR_COMMERCIAL_AT1 = 64;
var CHAR_LEFT_SQUARE_BRACKET1 = 91;
var CHAR_RIGHT_SQUARE_BRACKET1 = 93;
var CHAR_GRAVE_ACCENT1 = 96;
var CHAR_LEFT_CURLY_BRACKET1 = 123;
var CHAR_VERTICAL_LINE1 = 124;
var CHAR_RIGHT_CURLY_BRACKET1 = 125;
var ESCAPE_SEQUENCES1 = {};
ESCAPE_SEQUENCES1[0] = "\\0";
ESCAPE_SEQUENCES1[7] = "\\a";
ESCAPE_SEQUENCES1[8] = "\\b";
ESCAPE_SEQUENCES1[9] = "\\t";
ESCAPE_SEQUENCES1[10] = "\\n";
ESCAPE_SEQUENCES1[11] = "\\v";
ESCAPE_SEQUENCES1[12] = "\\f";
ESCAPE_SEQUENCES1[13] = "\\r";
ESCAPE_SEQUENCES1[27] = "\\e";
ESCAPE_SEQUENCES1[34] = '\\"';
ESCAPE_SEQUENCES1[92] = "\\\\";
ESCAPE_SEQUENCES1[133] = "\\N";
ESCAPE_SEQUENCES1[160] = "\\_";
ESCAPE_SEQUENCES1[8232] = "\\L";
ESCAPE_SEQUENCES1[8233] = "\\P";
var DEPRECATED_BOOLEANS_SYNTAX1 = [
    "y",
    "Y",
    "yes",
    "Yes",
    "YES",
    "on",
    "On",
    "ON",
    "n",
    "N",
    "no",
    "No",
    "NO",
    "off",
    "Off",
    "OFF"
];
var DEPRECATED_BASE60_SYNTAX1 = /^[-+]?[0-9_]+(?::[0-9_]+)+(?:\.[0-9_]*)?$/;
function compileStyleMap1(schema2, map2) {
    var result1, keys, index, length, tag, style, type2;
    if (map2 === null) return {};
    result1 = {};
    keys = Object.keys(map2);
    for(index = 0, length = keys.length; index < length; index += 1){
        tag = keys[index];
        style = String(map2[tag]);
        if (tag.slice(0, 2) === "!!") {
            tag = "tag:yaml.org,2002:" + tag.slice(2);
        }
        type2 = schema2.compiledTypeMap["fallback"][tag];
        if (type2 && _hasOwnProperty1.call(type2.styleAliases, style)) {
            style = type2.styleAliases[style];
        }
        result1[tag] = style;
    }
    return result1;
}
function encodeHex1(character) {
    var string, handle, length;
    string = character.toString(16).toUpperCase();
    if (character <= 255) {
        handle = "x";
        length = 2;
    } else if (character <= 65535) {
        handle = "u";
        length = 4;
    } else if (character <= 4294967295) {
        handle = "U";
        length = 8;
    } else {
        throw new exception1("code point within a string may not be greater than 0xFFFFFFFF");
    }
    return "\\" + handle + common2.repeat("0", length - string.length) + string;
}
var QUOTING_TYPE_SINGLE1 = 1, QUOTING_TYPE_DOUBLE1 = 2;
function State2(options) {
    this.schema = options["schema"] || _default1;
    this.indent = Math.max(1, options["indent"] || 2);
    this.noArrayIndent = options["noArrayIndent"] || false;
    this.skipInvalid = options["skipInvalid"] || false;
    this.flowLevel = common2.isNothing(options["flowLevel"]) ? -1 : options["flowLevel"];
    this.styleMap = compileStyleMap1(this.schema, options["styles"] || null);
    this.sortKeys = options["sortKeys"] || false;
    this.lineWidth = options["lineWidth"] || 80;
    this.noRefs = options["noRefs"] || false;
    this.noCompatMode = options["noCompatMode"] || false;
    this.condenseFlow = options["condenseFlow"] || false;
    this.quotingType = options["quotingType"] === '"' ? QUOTING_TYPE_DOUBLE1 : QUOTING_TYPE_SINGLE1;
    this.forceQuotes = options["forceQuotes"] || false;
    this.replacer = typeof options["replacer"] === "function" ? options["replacer"] : null;
    this.implicitTypes = this.schema.compiledImplicit;
    this.explicitTypes = this.schema.compiledExplicit;
    this.tag = null;
    this.result = "";
    this.duplicates = [];
    this.usedDuplicates = null;
}
function indentString1(string, spaces) {
    var ind = common2.repeat(" ", spaces), position = 0, next = -1, result1 = "", line, length = string.length;
    while(position < length){
        next = string.indexOf("\n", position);
        if (next === -1) {
            line = string.slice(position);
            position = length;
        } else {
            line = string.slice(position, next + 1);
            position = next + 1;
        }
        if (line.length && line !== "\n") result1 += ind;
        result1 += line;
    }
    return result1;
}
function generateNextLine1(state, level) {
    return "\n" + common2.repeat(" ", state.indent * level);
}
function testImplicitResolving1(state, str2) {
    var index, length, type2;
    for(index = 0, length = state.implicitTypes.length; index < length; index += 1){
        type2 = state.implicitTypes[index];
        if (type2.resolve(str2)) {
            return true;
        }
    }
    return false;
}
function isWhitespace1(c) {
    return c === CHAR_SPACE1 || c === CHAR_TAB1;
}
function isPrintable1(c) {
    return 32 <= c && c <= 126 || 161 <= c && c <= 55295 && c !== 8232 && c !== 8233 || 57344 <= c && c <= 65533 && c !== CHAR_BOM1 || 65536 <= c && c <= 1114111;
}
function isNsCharOrWhitespace1(c) {
    return isPrintable1(c) && c !== CHAR_BOM1 && c !== CHAR_CARRIAGE_RETURN1 && c !== CHAR_LINE_FEED1;
}
function isPlainSafe1(c, prev, inblock) {
    var cIsNsCharOrWhitespace = isNsCharOrWhitespace1(c);
    var cIsNsChar = cIsNsCharOrWhitespace && !isWhitespace1(c);
    return (inblock ? cIsNsCharOrWhitespace : cIsNsCharOrWhitespace && c !== CHAR_COMMA1 && c !== CHAR_LEFT_SQUARE_BRACKET1 && c !== CHAR_RIGHT_SQUARE_BRACKET1 && c !== CHAR_LEFT_CURLY_BRACKET1 && c !== CHAR_RIGHT_CURLY_BRACKET1) && c !== CHAR_SHARP1 && !(prev === CHAR_COLON1 && !cIsNsChar) || isNsCharOrWhitespace1(prev) && !isWhitespace1(prev) && c === CHAR_SHARP1 || prev === CHAR_COLON1 && cIsNsChar;
}
function isPlainSafeFirst1(c) {
    return isPrintable1(c) && c !== CHAR_BOM1 && !isWhitespace1(c) && c !== CHAR_MINUS1 && c !== CHAR_QUESTION1 && c !== CHAR_COLON1 && c !== CHAR_COMMA1 && c !== CHAR_LEFT_SQUARE_BRACKET1 && c !== CHAR_RIGHT_SQUARE_BRACKET1 && c !== CHAR_LEFT_CURLY_BRACKET1 && c !== CHAR_RIGHT_CURLY_BRACKET1 && c !== CHAR_SHARP1 && c !== CHAR_AMPERSAND1 && c !== CHAR_ASTERISK1 && c !== CHAR_EXCLAMATION1 && c !== CHAR_VERTICAL_LINE1 && c !== CHAR_EQUALS1 && c !== CHAR_GREATER_THAN1 && c !== CHAR_SINGLE_QUOTE1 && c !== CHAR_DOUBLE_QUOTE1 && c !== CHAR_PERCENT1 && c !== CHAR_COMMERCIAL_AT1 && c !== CHAR_GRAVE_ACCENT1;
}
function isPlainSafeLast1(c) {
    return !isWhitespace1(c) && c !== CHAR_COLON1;
}
function codePointAt1(string, pos) {
    var first = string.charCodeAt(pos), second;
    if (first >= 55296 && first <= 56319 && pos + 1 < string.length) {
        second = string.charCodeAt(pos + 1);
        if (second >= 56320 && second <= 57343) {
            return (first - 55296) * 1024 + second - 56320 + 65536;
        }
    }
    return first;
}
function needIndentIndicator1(string) {
    var leadingSpaceRe = /^\n* /;
    return leadingSpaceRe.test(string);
}
var STYLE_PLAIN1 = 1, STYLE_SINGLE1 = 2, STYLE_LITERAL1 = 3, STYLE_FOLDED1 = 4, STYLE_DOUBLE1 = 5;
function chooseScalarStyle1(string, singleLineOnly, indentPerLevel, lineWidth, testAmbiguousType, quotingType, forceQuotes, inblock) {
    var i1;
    var __char = 0;
    var prevChar = null;
    var hasLineBreak = false;
    var hasFoldableLine = false;
    var shouldTrackWidth = lineWidth !== -1;
    var previousLineBreak = -1;
    var plain = isPlainSafeFirst1(codePointAt1(string, 0)) && isPlainSafeLast1(codePointAt1(string, string.length - 1));
    if (singleLineOnly || forceQuotes) {
        for(i1 = 0; i1 < string.length; __char >= 65536 ? i1 += 2 : i1++){
            __char = codePointAt1(string, i1);
            if (!isPrintable1(__char)) {
                return STYLE_DOUBLE1;
            }
            plain = plain && isPlainSafe1(__char, prevChar, inblock);
            prevChar = __char;
        }
    } else {
        for(i1 = 0; i1 < string.length; __char >= 65536 ? i1 += 2 : i1++){
            __char = codePointAt1(string, i1);
            if (__char === CHAR_LINE_FEED1) {
                hasLineBreak = true;
                if (shouldTrackWidth) {
                    hasFoldableLine = hasFoldableLine || i1 - previousLineBreak - 1 > lineWidth && string[previousLineBreak + 1] !== " ";
                    previousLineBreak = i1;
                }
            } else if (!isPrintable1(__char)) {
                return STYLE_DOUBLE1;
            }
            plain = plain && isPlainSafe1(__char, prevChar, inblock);
            prevChar = __char;
        }
        hasFoldableLine = hasFoldableLine || shouldTrackWidth && i1 - previousLineBreak - 1 > lineWidth && string[previousLineBreak + 1] !== " ";
    }
    if (!hasLineBreak && !hasFoldableLine) {
        if (plain && !forceQuotes && !testAmbiguousType(string)) {
            return STYLE_PLAIN1;
        }
        return quotingType === QUOTING_TYPE_DOUBLE1 ? STYLE_DOUBLE1 : STYLE_SINGLE1;
    }
    if (indentPerLevel > 9 && needIndentIndicator1(string)) {
        return STYLE_DOUBLE1;
    }
    if (!forceQuotes) {
        return hasFoldableLine ? STYLE_FOLDED1 : STYLE_LITERAL1;
    }
    return quotingType === QUOTING_TYPE_DOUBLE1 ? STYLE_DOUBLE1 : STYLE_SINGLE1;
}
function writeScalar1(state, string, level, iskey, inblock) {
    state.dump = function() {
        if (string.length === 0) {
            return state.quotingType === QUOTING_TYPE_DOUBLE1 ? '""' : "''";
        }
        if (!state.noCompatMode) {
            if (DEPRECATED_BOOLEANS_SYNTAX1.indexOf(string) !== -1 || DEPRECATED_BASE60_SYNTAX1.test(string)) {
                return state.quotingType === QUOTING_TYPE_DOUBLE1 ? '"' + string + '"' : "'" + string + "'";
            }
        }
        var indent = state.indent * Math.max(1, level);
        var lineWidth = state.lineWidth === -1 ? -1 : Math.max(Math.min(state.lineWidth, 40), state.lineWidth - indent);
        var singleLineOnly = iskey || state.flowLevel > -1 && level >= state.flowLevel;
        function testAmbiguity(string2) {
            return testImplicitResolving1(state, string2);
        }
        switch(chooseScalarStyle1(string, singleLineOnly, state.indent, lineWidth, testAmbiguity, state.quotingType, state.forceQuotes && !iskey, inblock)){
            case STYLE_PLAIN1:
                return string;
            case STYLE_SINGLE1:
                return "'" + string.replace(/'/g, "''") + "'";
            case STYLE_LITERAL1:
                return "|" + blockHeader1(string, state.indent) + dropEndingNewline1(indentString1(string, indent));
            case STYLE_FOLDED1:
                return ">" + blockHeader1(string, state.indent) + dropEndingNewline1(indentString1(foldString1(string, lineWidth), indent));
            case STYLE_DOUBLE1:
                return '"' + escapeString1(string) + '"';
            default:
                throw new exception1("impossible error: invalid scalar style");
        }
    }();
}
function blockHeader1(string, indentPerLevel) {
    var indentIndicator = needIndentIndicator1(string) ? String(indentPerLevel) : "";
    var clip = string[string.length - 1] === "\n";
    var keep = clip && (string[string.length - 2] === "\n" || string === "\n");
    var chomp = keep ? "+" : clip ? "" : "-";
    return indentIndicator + chomp + "\n";
}
function dropEndingNewline1(string) {
    return string[string.length - 1] === "\n" ? string.slice(0, -1) : string;
}
function foldString1(string, width) {
    var lineRe = /(\n+)([^\n]*)/g;
    var result1 = function() {
        var nextLF = string.indexOf("\n");
        nextLF = nextLF !== -1 ? nextLF : string.length;
        lineRe.lastIndex = nextLF;
        return foldLine1(string.slice(0, nextLF), width);
    }();
    var prevMoreIndented = string[0] === "\n" || string[0] === " ";
    var moreIndented;
    var match;
    while(match = lineRe.exec(string)){
        var prefix = match[1], line = match[2];
        moreIndented = line[0] === " ";
        result1 += prefix + (!prevMoreIndented && !moreIndented && line !== "" ? "\n" : "") + foldLine1(line, width);
        prevMoreIndented = moreIndented;
    }
    return result1;
}
function foldLine1(line, width) {
    if (line === "" || line[0] === " ") return line;
    var breakRe = / [^ ]/g;
    var match;
    var start = 0, end, curr = 0, next = 0;
    var result1 = "";
    while(match = breakRe.exec(line)){
        next = match.index;
        if (next - start > width) {
            end = curr > start ? curr : next;
            result1 += "\n" + line.slice(start, end);
            start = end + 1;
        }
        curr = next;
    }
    result1 += "\n";
    if (line.length - start > width && curr > start) {
        result1 += line.slice(start, curr) + "\n" + line.slice(curr + 1);
    } else {
        result1 += line.slice(start);
    }
    return result1.slice(1);
}
function escapeString1(string) {
    var result1 = "";
    var __char = 0;
    var escapeSeq;
    for(var i1 = 0; i1 < string.length; __char >= 65536 ? i1 += 2 : i1++){
        __char = codePointAt1(string, i1);
        escapeSeq = ESCAPE_SEQUENCES1[__char];
        if (!escapeSeq && isPrintable1(__char)) {
            result1 += string[i1];
            if (__char >= 65536) result1 += string[i1 + 1];
        } else {
            result1 += escapeSeq || encodeHex1(__char);
        }
    }
    return result1;
}
function writeFlowSequence1(state, level, object) {
    var _result = "", _tag = state.tag, index, length, value;
    for(index = 0, length = object.length; index < length; index += 1){
        value = object[index];
        if (state.replacer) {
            value = state.replacer.call(object, String(index), value);
        }
        if (writeNode1(state, level, value, false, false) || typeof value === "undefined" && writeNode1(state, level, null, false, false)) {
            if (_result !== "") _result += "," + (!state.condenseFlow ? " " : "");
            _result += state.dump;
        }
    }
    state.tag = _tag;
    state.dump = "[" + _result + "]";
}
function writeBlockSequence1(state, level, object, compact) {
    var _result = "", _tag = state.tag, index, length, value;
    for(index = 0, length = object.length; index < length; index += 1){
        value = object[index];
        if (state.replacer) {
            value = state.replacer.call(object, String(index), value);
        }
        if (writeNode1(state, level + 1, value, true, true, false, true) || typeof value === "undefined" && writeNode1(state, level + 1, null, true, true, false, true)) {
            if (!compact || _result !== "") {
                _result += generateNextLine1(state, level);
            }
            if (state.dump && CHAR_LINE_FEED1 === state.dump.charCodeAt(0)) {
                _result += "-";
            } else {
                _result += "- ";
            }
            _result += state.dump;
        }
    }
    state.tag = _tag;
    state.dump = _result || "[]";
}
function writeFlowMapping1(state, level, object) {
    var _result = "", _tag = state.tag, objectKeyList = Object.keys(object), index, length, objectKey, objectValue, pairBuffer;
    for(index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = "";
        if (_result !== "") pairBuffer += ", ";
        if (state.condenseFlow) pairBuffer += '"';
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (state.replacer) {
            objectValue = state.replacer.call(object, objectKey, objectValue);
        }
        if (!writeNode1(state, level, objectKey, false, false)) {
            continue;
        }
        if (state.dump.length > 1024) pairBuffer += "? ";
        pairBuffer += state.dump + (state.condenseFlow ? '"' : "") + ":" + (state.condenseFlow ? "" : " ");
        if (!writeNode1(state, level, objectValue, false, false)) {
            continue;
        }
        pairBuffer += state.dump;
        _result += pairBuffer;
    }
    state.tag = _tag;
    state.dump = "{" + _result + "}";
}
function writeBlockMapping1(state, level, object, compact) {
    var _result = "", _tag = state.tag, objectKeyList = Object.keys(object), index, length, objectKey, objectValue, explicitPair, pairBuffer;
    if (state.sortKeys === true) {
        objectKeyList.sort();
    } else if (typeof state.sortKeys === "function") {
        objectKeyList.sort(state.sortKeys);
    } else if (state.sortKeys) {
        throw new exception1("sortKeys must be a boolean or a function");
    }
    for(index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = "";
        if (!compact || _result !== "") {
            pairBuffer += generateNextLine1(state, level);
        }
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (state.replacer) {
            objectValue = state.replacer.call(object, objectKey, objectValue);
        }
        if (!writeNode1(state, level + 1, objectKey, true, true, true)) {
            continue;
        }
        explicitPair = state.tag !== null && state.tag !== "?" || state.dump && state.dump.length > 1024;
        if (explicitPair) {
            if (state.dump && CHAR_LINE_FEED1 === state.dump.charCodeAt(0)) {
                pairBuffer += "?";
            } else {
                pairBuffer += "? ";
            }
        }
        pairBuffer += state.dump;
        if (explicitPair) {
            pairBuffer += generateNextLine1(state, level);
        }
        if (!writeNode1(state, level + 1, objectValue, true, explicitPair)) {
            continue;
        }
        if (state.dump && CHAR_LINE_FEED1 === state.dump.charCodeAt(0)) {
            pairBuffer += ":";
        } else {
            pairBuffer += ": ";
        }
        pairBuffer += state.dump;
        _result += pairBuffer;
    }
    state.tag = _tag;
    state.dump = _result || "{}";
}
function detectType1(state, object, explicit) {
    var _result, typeList, index, length, type2, style;
    typeList = explicit ? state.explicitTypes : state.implicitTypes;
    for(index = 0, length = typeList.length; index < length; index += 1){
        type2 = typeList[index];
        if ((type2.instanceOf || type2.predicate) && (!type2.instanceOf || typeof object === "object" && object instanceof type2.instanceOf) && (!type2.predicate || type2.predicate(object))) {
            if (explicit) {
                if (type2.multi && type2.representName) {
                    state.tag = type2.representName(object);
                } else {
                    state.tag = type2.tag;
                }
            } else {
                state.tag = "?";
            }
            if (type2.represent) {
                style = state.styleMap[type2.tag] || type2.defaultStyle;
                if (_toString1.call(type2.represent) === "[object Function]") {
                    _result = type2.represent(object, style);
                } else if (_hasOwnProperty1.call(type2.represent, style)) {
                    _result = type2.represent[style](object, style);
                } else {
                    throw new exception1("!<" + type2.tag + '> tag resolver accepts not "' + style + '" style');
                }
                state.dump = _result;
            }
            return true;
        }
    }
    return false;
}
function writeNode1(state, level, object, block, compact, iskey, isblockseq) {
    state.tag = null;
    state.dump = object;
    if (!detectType1(state, object, false)) {
        detectType1(state, object, true);
    }
    var type2 = _toString1.call(state.dump);
    var inblock = block;
    var tagStr;
    if (block) {
        block = state.flowLevel < 0 || state.flowLevel > level;
    }
    var objectOrArray = type2 === "[object Object]" || type2 === "[object Array]", duplicateIndex, duplicate;
    if (objectOrArray) {
        duplicateIndex = state.duplicates.indexOf(object);
        duplicate = duplicateIndex !== -1;
    }
    if (state.tag !== null && state.tag !== "?" || duplicate || state.indent !== 2 && level > 0) {
        compact = false;
    }
    if (duplicate && state.usedDuplicates[duplicateIndex]) {
        state.dump = "*ref_" + duplicateIndex;
    } else {
        if (objectOrArray && duplicate && !state.usedDuplicates[duplicateIndex]) {
            state.usedDuplicates[duplicateIndex] = true;
        }
        if (type2 === "[object Object]") {
            if (block && Object.keys(state.dump).length !== 0) {
                writeBlockMapping1(state, level, state.dump, compact);
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + state.dump;
                }
            } else {
                writeFlowMapping1(state, level, state.dump);
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + " " + state.dump;
                }
            }
        } else if (type2 === "[object Array]") {
            if (block && state.dump.length !== 0) {
                if (state.noArrayIndent && !isblockseq && level > 0) {
                    writeBlockSequence1(state, level - 1, state.dump, compact);
                } else {
                    writeBlockSequence1(state, level, state.dump, compact);
                }
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + state.dump;
                }
            } else {
                writeFlowSequence1(state, level, state.dump);
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + " " + state.dump;
                }
            }
        } else if (type2 === "[object String]") {
            if (state.tag !== "?") {
                writeScalar1(state, state.dump, level, iskey, inblock);
            }
        } else if (type2 === "[object Undefined]") {
            return false;
        } else {
            if (state.skipInvalid) return false;
            throw new exception1("unacceptable kind of an object to dump " + type2);
        }
        if (state.tag !== null && state.tag !== "?") {
            tagStr = encodeURI(state.tag[0] === "!" ? state.tag.slice(1) : state.tag).replace(/!/g, "%21");
            if (state.tag[0] === "!") {
                tagStr = "!" + tagStr;
            } else if (tagStr.slice(0, 18) === "tag:yaml.org,2002:") {
                tagStr = "!!" + tagStr.slice(18);
            } else {
                tagStr = "!<" + tagStr + ">";
            }
            state.dump = tagStr + " " + state.dump;
        }
    }
    return true;
}
function getDuplicateReferences1(object, state) {
    var objects = [], duplicatesIndexes = [], index, length;
    inspectNode1(object, objects, duplicatesIndexes);
    for(index = 0, length = duplicatesIndexes.length; index < length; index += 1){
        state.duplicates.push(objects[duplicatesIndexes[index]]);
    }
    state.usedDuplicates = new Array(length);
}
function inspectNode1(object, objects, duplicatesIndexes) {
    var objectKeyList, index, length;
    if (object !== null && typeof object === "object") {
        index = objects.indexOf(object);
        if (index !== -1) {
            if (duplicatesIndexes.indexOf(index) === -1) {
                duplicatesIndexes.push(index);
            }
        } else {
            objects.push(object);
            if (Array.isArray(object)) {
                for(index = 0, length = object.length; index < length; index += 1){
                    inspectNode1(object[index], objects, duplicatesIndexes);
                }
            } else {
                objectKeyList = Object.keys(object);
                for(index = 0, length = objectKeyList.length; index < length; index += 1){
                    inspectNode1(object[objectKeyList[index]], objects, duplicatesIndexes);
                }
            }
        }
    }
}
function dump$11(input, options) {
    options = options || {};
    var state = new State2(options);
    if (!state.noRefs) getDuplicateReferences1(input, state);
    var value = input;
    if (state.replacer) {
        value = state.replacer.call({
            "": value
        }, "", value);
    }
    if (writeNode1(state, 0, value, true, true)) return state.dump + "\n";
    return "";
}
var dump_11 = dump$11;
var dumper1 = {
    dump: dump_11
};
function renamed1(from, to) {
    return function() {
        throw new Error("Function yaml." + from + " is removed in js-yaml 4. Use yaml." + to + " instead, which is now safe by default.");
    };
}
var Type2 = type1;
var Schema1 = schema1;
var FAILSAFE_SCHEMA = failsafe1;
var load1 = loader1.load;
loader1.loadAll;
var dump = dumper1.dump;
var types = {
    binary: binary1,
    float: __float1,
    map: map1,
    null: _null1,
    pairs: pairs1,
    set: set1,
    timestamp: timestamp1,
    bool: bool1,
    int: __int1,
    merge: merge1,
    omap: omap1,
    seq: seq1,
    str: str1
};
renamed1("safeLoad", "load");
renamed1("safeLoadAll", "loadAll");
renamed1("safeDump", "dump");
const { bool: bool2, float: __float2, int: __int2, null: nil } = types;
const kRegExBeginYAML = /^---[ \t]*$/;
const kRegExEndYAML = /^(?:---|\.\.\.)([ \t]*)$/;
const kRegExYAML = /(^)(---[ \t]*[\r\n]+(?![ \t]*[\r\n]+)[\W\w]*?[\r\n]+(?:---|\.\.\.))([ \t]*)$/gm;
const kRegxHTMLComment = /<!--[\W\w]*?-->/gm;
const kRegexFencedCode = /^([\t >]*`{3,})[^`\n]*\n[\W\w]*?\n\1\s*$/gm;
function isYamlPath(file) {
    return [
        ".yaml",
        ".yml"
    ].includes(extname2(file));
}
function readYaml(file) {
    if (existsSync(file)) {
        const decoder = new TextDecoder("utf-8");
        const yml = Deno.readFileSync(file);
        const result1 = parseWithNiceErrors(decoder.decode(yml));
        try {
            JSON.stringify(result1);
            return result1;
        } catch (e) {
            throw new Error(`Circular structures not allowed.\nFile ${file}\n${e.message.split("\n").slice(1).join("\n")}`);
        }
    } else {
        throw new Error(`YAML file ${file} not found.`);
    }
}
function readYamlFromString(yml) {
    return parseWithNiceErrors(yml);
}
function readYamlFromMarkdown(markdown) {
    if (markdown) {
        markdown = normalizeNewlines(markdown);
        markdown = markdown.replaceAll(kRegxHTMLComment, "");
        markdown = markdown.replaceAll(kRegexFencedCode, "");
        let yaml = "";
        kRegExYAML.lastIndex = 0;
        let match = kRegExYAML.exec(markdown);
        while(match != null){
            let yamlBlock = removeYamlDelimiters(match[2]);
            yamlBlock = lines(yamlBlock).map((x)=>x.trimEnd()).join("\n");
            if (!yamlBlock.match(/^\n\s*\n/) && !yamlBlock.match(/^\n\s*\n---/m) && yamlBlock.trim().length > 0) {
                parseWithNiceErrors(yamlBlock, {
                    json: true,
                    schema: QuartoJSONSchema1
                });
                yaml += yamlBlock;
            }
            match = kRegExYAML.exec(markdown);
        }
        kRegExYAML.lastIndex = 0;
        const metadata = parseWithNiceErrors(yaml, {
            json: true,
            schema: QuartoJSONSchema1
        });
        return metadata || {};
    } else {
        return {};
    }
}
function partitionYamlFrontMatter(markdown) {
    const mdLines = lines(markdown.trimLeft());
    if (mdLines.length < 3 || !mdLines[0].match(kRegExBeginYAML)) {
        return null;
    } else if (mdLines[1].trim().length === 0 || mdLines[1].match(kRegExEndYAML)) {
        return null;
    } else {
        const endYamlPos = mdLines.findIndex((line, index)=>index > 0 && line.match(kRegExEndYAML));
        if (endYamlPos === -1) {
            return null;
        } else {
            return {
                yaml: mdLines.slice(0, endYamlPos + 1).join("\n"),
                markdown: "\n" + mdLines.slice(endYamlPos + 1).join("\n")
            };
        }
    }
}
function removeYamlDelimiters(yaml) {
    return yaml.replace(/^---/, "").replace(/---\s*$/, "");
}
class YAMLValidationError extends ErrorEx {
    constructor(message){
        super("YAMLValidationError", message, false, false);
    }
}
const QuartoJSONSchema1 = new Schema1({
    implicit: [
        nil,
        bool2,
        __int2,
        __float2
    ],
    include: [
        FAILSAFE_SCHEMA
    ],
    explicit: [
        new Type2("!expr", {
            kind: "scalar",
            construct (data) {
                const result1 = data !== null ? data : "";
                return {
                    value: result1,
                    tag: "!expr"
                };
            }
        })
    ]
});
function parseWithNiceErrors(content, options) {
    try {
        return load1(content, options || {
            json: true,
            schema: QuartoJSONSchema1
        });
    } catch (e) {
        throw improveYamlParseErrorMessage(e);
    }
}
function improveYamlParseErrorMessage(e) {
    if (e.message.match(/unknown tag/)) {
        e.message = `${e.message}\nDid you try to use a '!' in a YAML string? If so, you need to add explicit quotes to your string.`;
    }
    return e;
}
function mergeConfigs(config, ...configs) {
    config = cloneDeep1(config);
    configs = cloneDeep1(configs);
    return __VIRTUAL_FILE154(config, ...configs, mergeArrayCustomizer);
}
function mergeArrayCustomizer(objValue, srcValue) {
    if (__VIRTUAL_FILE53(objValue) || __VIRTUAL_FILE53(srcValue)) {
        if (!objValue) {
            return srcValue;
        } else if (!srcValue) {
            return objValue;
        } else {
            if (!__VIRTUAL_FILE53(objValue)) {
                objValue = [
                    objValue
                ];
            }
            if (!__VIRTUAL_FILE53(srcValue)) {
                srcValue = [
                    srcValue
                ];
            }
        }
        const combined = objValue.concat(srcValue);
        return __VIRTUAL_FILE210(combined, (value)=>{
            if (typeof value === "function") {
                return globalThis.crypto.randomUUID();
            } else {
                return JSON.stringify(value);
            }
        });
    }
}
function camelToKebab(camel) {
    camel = camel.replace(/([A-Z])([A-Z]+)$/g, function(_, p1, p2) {
        return p1 + p2.toLowerCase();
    });
    camel = camel.replaceAll(/([A-Z])([A-Z]+)([A-Z])/g, function(_, p1, p2, p3) {
        return p1 + p2.toLowerCase() + p3;
    });
    const kebab = [];
    for(let i1 = 0; i1 < camel.length; i1++){
        const ch = camel.charAt(i1);
        if (ch === ch.toUpperCase() && !/^\d+/.test(ch)) {
            if (i1 > 0) {
                kebab.push("-");
            }
            kebab.push(ch.toLowerCase());
        } else {
            kebab.push(ch);
        }
    }
    return kebab.join("");
}
function kebabToCamel(kebab, leadingUppercase = false) {
    const camel = [];
    for(let i1 = 0; i1 < kebab.length; i1++){
        const ch = kebab.charAt(i1);
        if (ch === "-") {
            camel.push(kebab.charAt(++i1).toUpperCase());
        } else if (i1 === 0 && leadingUppercase) {
            camel.push(ch.toUpperCase());
        } else {
            camel.push(ch);
        }
    }
    return camel.join("");
}
const kMetadataFormat = "format";
const kDisplayName = "display-name";
const kExtensionName = "extension-name";
const kTargetFormat = "target-format";
const kBaseFormat = "base-format";
const kIdentifierDefaults = "indentifier";
const kRenderDefaults = "render";
const kExecuteDefaults = "execute";
const kPandocDefaults = "pandoc";
const kLanguageDefaults = "language";
const kPandocMetadata = "metadata";
const kFigWidth = "fig-width";
const kFigHeight = "fig-height";
const kFigFormat = "fig-format";
const kFigDpi = "fig-dpi";
const kFigAsp = "fig-asp";
const kMermaidFormat = "mermaid-format";
const kDfPrint = "df-print";
const kCache = "cache";
const kFreeze = "freeze";
const kEngine = "engine";
const kEval = "eval";
const kEcho = "echo";
const kOutput = "output";
const kWarning = "warning";
const kError = "error";
const kInclude = "include";
const kResources = "resources";
const kFormatResources = "format-resources";
const kSupporting = "supporting";
const kIpynbProduceSourceNotebook = "produce-source-notebook";
const kEnableCrossRef = "enable-crossref";
const kFormatLinks = "format-links";
const kNotebookLinks = "notebook-links";
const kOtherLinks = "other-links";
const kCodeLinks = "code-links";
const kNotebookSubarticles = "notebook-subarticles";
const kNotebookView = "notebook-view";
const kNotebookViewStyle = "notebook-view-style";
const kNotebookPreserveCells = "notebook-preserve-cells";
const kClearCellOptions = "clear-cell-options";
const kDownloadUrl = "download-url";
const kLightbox = "lightbox";
const kCanonicalUrl = "canonical-url";
const kMath = "math";
const kNotebookPreviewOptions = "notebook-preview-options";
const kKeepHidden = "keep-hidden";
const kRemoveHidden = "remove-hidden";
const kClearHiddenClasses = "clear-hidden-classes";
const kUnrollMarkdownCells = "unroll-markdown-cells";
const kExecuteEnabled = "enabled";
const kExecuteIpynb = "ipynb";
const kExecuteDaemon = "daemon";
const kExecuteDaemonRestart = "daemon-restart";
const kExecuteDebug = "debug";
const kIpynbFilter = "ipynb-filter";
const kIpynbFilters = "ipynb-filters";
const kIpynbShellInteractivity = "ipynb-shell-interactivity";
const kIPynbTitleBlockTemplate = "ipynb-title-block";
const kPlotlyConnected = "plotly-connected";
const kJatsSubarticleId = "jats-subarticle-id";
const kShortcodes = "shortcodes";
const kKeepMd = "keep-md";
const kKeepTex = "keep-tex";
const kKeepTyp = "keep-typ";
const kKeepIpynb = "keep-ipynb";
const kKeepSource = "keep-source";
const kVariant = "variant";
const kOutputExt = "output-ext";
const kOutputDivs = "output-divs";
const kPageWidth = "page-width";
const kFigAlign = "fig-align";
const kFigEnv = "fig-env";
const kFigPos = "fig-pos";
const kCodeFold = "code-fold";
const kCodeLineNumbers = "code-line-numbers";
const kCodeOverflow = "code-overflow";
const kCodeLink = "code-link";
const kCodeTools = "code-tools";
const kTblCap = "tbl-cap";
const kTblColwidths = "tbl-colwidths";
const kMergeIncludes = "merge-includes";
const kInlineIncludes = "inline-includes";
const kPreserveYaml = "preserve-yaml";
const kPreferHtml = "prefer-html";
const kSelfContainedMath = "self-contained-math";
const kBodyClasses = "body-classes";
const kBrand = "brand";
const kLatexAutoMk = "latex-auto-mk";
const kLatexAutoInstall = "latex-auto-install";
const kLatexMinRuns = "latex-min-runs";
const kLatexMaxRuns = "latex-max-runs";
const kLatexClean = "latex-clean";
const kLatexInputPaths = "latex-input-paths";
const kLatexMakeIndex = "latex-makeindex";
const kLatexMakeIndexOpts = "latex-makeindex-opts";
const kLatexTinyTex = "latex-tinytex";
const kLatexTlmgrOpts = "latex-tlmgr-opts";
const kLatexOutputDir = "latex-output-dir";
const kLinkExternalIcon = "link-external-icon";
const kLinkExternalNewwindow = "link-external-newwindow";
const kLinkExternalFilter = "link-external-filter";
const kQuartoVersion = "quarto-version";
const kQuartoRequired = "quarto-required";
const kPreviewMode = "preview-mode";
const kPreviewModeRaw = "raw";
const kFontPaths = "font-paths";
const kHtmlTableProcessing = "html-table-processing";
const kHtmlPreTagProcessing = "html-pre-tag-processing";
const kCssPropertyProcessing = "css-property-processing";
const kUseRsvgConvert = "use-rsvg-convert";
const kValidateYaml = "validate-yaml";
const kIdentifierDefaultsKeys = [
    kTargetFormat,
    kDisplayName,
    kExtensionName
];
const kExecuteDefaultsKeys = [
    kFigWidth,
    kFigHeight,
    kFigFormat,
    kFigDpi,
    kFigAsp,
    kMermaidFormat,
    kDfPrint,
    kError,
    kEval,
    kEngine,
    kCache,
    kFreeze,
    kEcho,
    kOutput,
    kWarning,
    kInclude,
    kKeepMd,
    kKeepIpynb,
    kExecuteEnabled,
    kExecuteIpynb,
    kExecuteDaemon,
    kExecuteDaemonRestart,
    kExecuteDebug,
    kIpynbFilter,
    kIpynbFilters,
    kIpynbShellInteractivity,
    kPlotlyConnected
];
const kRenderDefaultsKeys = [
    kKeepTex,
    kKeepTyp,
    kKeepSource,
    kKeepHidden,
    kRemoveHidden,
    kClearHiddenClasses,
    kVariant,
    kOutputExt,
    kOutputDivs,
    kPreferHtml,
    kPageWidth,
    kFigAlign,
    kFigPos,
    kFigEnv,
    kCodeFold,
    kCodeLink,
    kCodeLineNumbers,
    kCodeOverflow,
    kCodeTools,
    kShortcodes,
    kTblColwidths,
    kInlineIncludes,
    kPreserveYaml,
    kMergeIncludes,
    kSelfContainedMath,
    kLatexAutoMk,
    kLatexAutoInstall,
    kLatexMinRuns,
    kLatexMaxRuns,
    kLatexClean,
    kLatexInputPaths,
    kLatexMakeIndex,
    kLatexMakeIndexOpts,
    kLatexTlmgrOpts,
    kLatexOutputDir,
    kLatexTinyTex,
    kLinkExternalIcon,
    kLinkExternalNewwindow,
    kLinkExternalFilter,
    kFormatResources,
    kFormatLinks,
    kIpynbProduceSourceNotebook,
    kNotebookLinks,
    kNotebookSubarticles,
    kNotebookView,
    kNotebookViewStyle,
    kNotebookPreserveCells,
    kClearCellOptions,
    kHtmlTableProcessing,
    kValidateYaml,
    kCanonicalUrl,
    kBodyClasses
];
const kTocTitleDocument = "toc-title-document";
const kTocTitleWebsite = "toc-title-website";
const kRelatedFormatsTitle = "related-formats-title";
const kRelatedNotebooksTitle = "related-notebooks-title";
const kOtherLinksTitle = "other-links-title";
const kCodeLinksTitle = "code-links-title";
const kLaunchDevContainerTitle = "launch-dev-container-title";
const kLaunchBinderTitle = "launch-binder-title";
const kSourceNotebookPrefix = "source-notebooks-prefix";
const kCalloutTipCaption = "callout-tip-title";
const kCalloutNoteCaption = "callout-note-title";
const kCalloutWarningCaption = "callout-warning-title";
const kCalloutImportantCaption = "callout-important-title";
const kCalloutCautionCaption = "callout-caution-title";
const kCalloutTipPrefix = "callout-tip-prefix";
const kCalloutNotePrefix = "callout-note-prefix";
const kCalloutWarningPrefix = "callout-warning-prefix";
const kCalloutImportantPrefix = "callout-important-prefix";
const kCalloutCautionPrefix = "callout-caution-prefix";
const kSectionTitleAbstract = "section-title-abstract";
const kSectionTitleFootnotes = "section-title-footnotes";
const kSectionTitleReferences = "section-title-references";
const kSectionTitleAppendices = "section-title-appendices";
const kSectionTitleReuse = "section-title-reuse";
const kSectionTitleCopyright = "section-title-copyright";
const kSectionTitleCitation = "section-title-citation";
const kAppendixAttributionBibTex = "appendix-attribution-bibtex";
const kAppendixAttributionCiteAs = "appendix-attribution-cite-as";
const kAppendixViewLicense = "appendix-view-license";
const kTitleBlockAuthorSingle = "title-block-author-single";
const kTitleBlockAuthorPlural = "title-block-author-plural";
const kTitleBlockAffiliationSingle = "title-block-affiliation-single";
const kTitleBlockAffiliationPlural = "title-block-affiliation-plural";
const kTitleBlockPublished = "title-block-published";
const kTitleBlockModified = "title-block-modified";
const kTitleBlockKeywords = "title-block-keywords";
const kCodeSummary = "code-summary";
const kCodeLine = "code-line";
const kCodeLines = "code-lines";
const kCodeToolsMenuCaption = "code-tools-menu-caption";
const kCodeToolsShowAllCode = "code-tools-show-all-code";
const kCodeToolsHideAllCode = "code-tools-hide-all-code";
const kCodeToolsViewSource = "code-tools-view-source";
const kCodeToolsSourceCode = "code-tools-source-code";
const kSearchNoResultsText = "search-no-results-text";
const kSearchLabel = "search-label";
const kToolsShare = "tools-share";
const kToolsDownload = "tools-download";
const kToggleSection = "toggle-section";
const kToggleSidebar = "toggle-sidebar";
const kToggleDarkMode = "toggle-dark-mode";
const kToggleReaderMode = "toggle-reader-mode";
const kToggleNavigation = "toggle-navigation";
const kCopyButtonTooltip = "copy-button-tooltip";
const kCopyButtonTooltipSuccess = "copy-button-tooltip-success";
const kBackToTop = "back-to-top";
const kRepoActionLinksEdit = "repo-action-links-edit";
const kRepoActionLinksSource = "repo-action-links-source";
const kRepoActionLinksIssue = "repo-action-links-issue";
const kSearchMatchingDocumentsText = "search-matching-documents-text";
const kSearchCopyLinkTitle = "search-copy-link-title";
const kSearchHideMatchesText = "search-hide-matches-text";
const kSearchMoreMatchText = "search-more-match-text";
const kSearchMoreMatchesText = "search-more-matches-text";
const kSearchClearButtonTitle = "search-clear-button-title";
const kSearchTextPlaceholder = "search-text-placeholder";
const kSearchDetatchedCancelButtonTitle = "search-detached-cancel-button-title";
const kSearchSubmitButtonTitle = "search-submit-button-title";
const kCrossrefFigTitle = "crossref-fig-title";
const kCrossrefTblTitle = "crossref-tbl-title";
const kCrossrefLstTitle = "crossref-lst-title";
const kCrossrefThmTitle = "crossref-thm-title";
const kCrossrefLemTitle = "crossref-lem-title";
const kCrossrefCorTitle = "crossref-cor-title";
const kCrossrefPrfTitle = "crossref-prp-title";
const kCrossrefCnjTitle = "crossref-cnj-title";
const kCrossrefDefTitle = "crossref-def-title";
const kCrossrefExmTitle = "crossref-exm-title";
const kCrossrefExrTitle = "crossref-exr-title";
const kCrossrefFigPrefix = "crossref-fig-prefix";
const kCrossrefTblPrefix = "crossref-tbl-prefix";
const kCrossrefLstPrefix = "crossref-lst-prefix";
const kCrossrefChPrefix = "crossref-ch-prefix";
const kCrossrefApxPrefix = "crossref-apx-prefix";
const kCrossrefSecPrefix = "crossref-sec-prefix";
const kCrossrefEqPrefix = "crossref-eq-prefix";
const kCrossrefThmPrefix = "crossref-thm-prefix";
const kCrossrefLemPrefix = "crossref-lem-prefix";
const kCrossrefCorPrefix = "crossref-cor-prefix";
const kCrossrefPrpPrefix = "crossref-prp-prefix";
const kCrossrefCnjPrefix = "crossref-cnj-prefix";
const kCrossrefDefPrefix = "crossref-def-prefix";
const kCrossrefExmPrefix = "crossref-exm-prefix";
const kCrossrefExrPrefix = "crossref-exr-prefix";
const kCrossrefLofTitle = "crossref-lof-title";
const kCrossrefLotTitle = "crossref-lot-title";
const kCrossrefLolTitle = "crossref-lol-title";
const kEnvironmentProofTitle = "environment-proof-title";
const kEnvironmentRemarkTitle = "environment-remark-title";
const kEnvironmentSolutionTitle = "environment-solution-title";
const kListingPageOrderBy = "listing-page-order-by";
const kListingPageFilter = "listing-page-filter";
const kListingPageOrderByDefault = "listing-page-order-by-default";
const kListingPageOrderByDateAsc = "listing-page-order-by-date-asc";
const kListingPageOrderByDateDesc = "listing-page-order-by-date-desc";
const kListingPageOrderByNumberAsc = "listing-page-order-by-number-asc";
const kListingPageOrderByNumberDesc = "listing-page-order-by-number-desc";
const kListingPageFieldDate = "listing-page-field-date";
const kListingPageFieldTitle = "listing-page-field-title";
const kListingPageFieldDescription = "listing-page-field-description";
const kListingPageFieldAuthor = "listing-page-field-author";
const kListingPageFieldFileName = "listing-page-field-filename";
const kListingPageFieldFileModified = "listing-page-field-filemodified";
const kListingPageFieldSubtitle = "listing-page-field-subtitle";
const kListingPageFieldReadingTime = "listing-page-field-readingtime";
const kListingPageFieldWordCount = "listing-page-field-wordcount";
const kListingPageFieldCategories = "listing-page-field-categories";
const kListingPageMinutesCompact = "listing-page-minutes-compact";
const kListingPageWords = "listing-page-words";
const kListingPageCategoryAll = "listing-page-category-all";
const kListingPageNoMatches = "listing-page-no-matches";
const kNotebookPreviewDownload = "notebook-preview-download";
const kNotebookPreviewDownloadSrc = "notebook-preview-download-src";
const kNotebookPreviewBack = "notebook-preview-back";
const kArticleNotebookLabel = "article-notebook-label";
const kManuscriptMecaBundle = "manuscript-meca-bundle";
const kDraftLabel = "draft";
const kLanguageDefaultsKeys = [
    kTocTitleDocument,
    kTocTitleWebsite,
    kRelatedFormatsTitle,
    kOtherLinksTitle,
    kCodeLinksTitle,
    kLaunchDevContainerTitle,
    kLaunchBinderTitle,
    kRelatedNotebooksTitle,
    kSourceNotebookPrefix,
    kCalloutTipCaption,
    kCalloutNoteCaption,
    kCalloutWarningCaption,
    kCalloutImportantCaption,
    kCalloutCautionCaption,
    kCalloutTipPrefix,
    kCalloutNotePrefix,
    kCalloutWarningPrefix,
    kCalloutImportantPrefix,
    kCalloutCautionPrefix,
    kSectionTitleAbstract,
    kSectionTitleFootnotes,
    kSectionTitleReferences,
    kSectionTitleAppendices,
    kSectionTitleReuse,
    kSectionTitleCopyright,
    kSectionTitleCitation,
    kAppendixAttributionBibTex,
    kAppendixAttributionCiteAs,
    kAppendixViewLicense,
    kTitleBlockAuthorSingle,
    kTitleBlockPublished,
    kTitleBlockModified,
    kTitleBlockKeywords,
    kCodeSummary,
    kCodeLine,
    kCodeLines,
    kCodeToolsMenuCaption,
    kCodeToolsShowAllCode,
    kCodeToolsHideAllCode,
    kCodeToolsViewSource,
    kCodeToolsSourceCode,
    kToolsShare,
    kToolsDownload,
    kSearchNoResultsText,
    kSearchLabel,
    kToggleDarkMode,
    kToggleNavigation,
    kToggleReaderMode,
    kToggleSidebar,
    kToggleSection,
    kCopyButtonTooltip,
    kCopyButtonTooltipSuccess,
    kBackToTop,
    kRepoActionLinksEdit,
    kRepoActionLinksSource,
    kRepoActionLinksIssue,
    kSearchMatchingDocumentsText,
    kSearchCopyLinkTitle,
    kSearchHideMatchesText,
    kSearchMoreMatchText,
    kSearchMoreMatchesText,
    kSearchClearButtonTitle,
    kSearchTextPlaceholder,
    kSearchDetatchedCancelButtonTitle,
    kSearchSubmitButtonTitle,
    kCrossrefFigTitle,
    kCrossrefTblTitle,
    kCrossrefLstTitle,
    kCrossrefThmTitle,
    kCrossrefLemTitle,
    kCrossrefCorTitle,
    kCrossrefPrfTitle,
    kCrossrefCnjTitle,
    kCrossrefDefTitle,
    kCrossrefExmTitle,
    kCrossrefExrTitle,
    kCrossrefFigPrefix,
    kCrossrefTblPrefix,
    kCrossrefLstPrefix,
    kCrossrefChPrefix,
    kCrossrefApxPrefix,
    kCrossrefSecPrefix,
    kCrossrefEqPrefix,
    kCrossrefThmPrefix,
    kCrossrefLemPrefix,
    kCrossrefCorPrefix,
    kCrossrefPrpPrefix,
    kCrossrefCnjPrefix,
    kCrossrefDefPrefix,
    kCrossrefExmPrefix,
    kCrossrefExrPrefix,
    kCrossrefLofTitle,
    kCrossrefLotTitle,
    kCrossrefLolTitle,
    kEnvironmentProofTitle,
    kEnvironmentRemarkTitle,
    kEnvironmentSolutionTitle,
    kListingPageFilter,
    kListingPageOrderBy,
    kListingPageOrderByDefault,
    kListingPageOrderByDateAsc,
    kListingPageOrderByDateDesc,
    kListingPageOrderByNumberAsc,
    kListingPageOrderByNumberDesc,
    kListingPageFieldDate,
    kListingPageFieldTitle,
    kListingPageFieldDescription,
    kListingPageFieldAuthor,
    kListingPageFieldFileName,
    kListingPageFieldFileModified,
    kListingPageFieldSubtitle,
    kListingPageFieldReadingTime,
    kListingPageFieldWordCount,
    kListingPageFieldCategories,
    kListingPageMinutesCompact,
    kListingPageWords,
    kListingPageCategoryAll,
    kListingPageNoMatches,
    kTitleBlockAuthorSingle,
    kTitleBlockAuthorPlural,
    kTitleBlockAffiliationSingle,
    kTitleBlockAffiliationPlural,
    kTitleBlockPublished,
    kTitleBlockModified,
    kTitleBlockKeywords,
    kNotebookPreviewDownload,
    kNotebookPreviewDownloadSrc,
    kNotebookPreviewBack,
    kArticleNotebookLabel,
    kDraftLabel,
    kManuscriptMecaBundle
];
const kTo = "to";
const kFrom = "from";
const kReader = "reader";
const kWriter = "writer";
const kOutputFile = "output-file";
const kInputFiles = "input-files";
const kMarkdownHeadings = "markdown-headings";
const kTemplate = "template";
const kWrap = "wrap";
const kColumns = "columns";
const kStandalone = "standalone";
const kSelfContained = "self-contained";
const kEmbedResources = "embed-resources";
const kIncludeBeforeBody = "include-before-body";
const kIncludeAfterBody = "include-after-body";
const kIncludeInHeader = "include-in-header";
const kResourcePath = "resource-path";
const kCiteproc = "citeproc";
const kCiteMethod = "cite-method";
const kFilters = "filters";
const kQuartoFilters = "quarto-filters";
const kFilterParams = "filter-params";
const kPdfEngine = "pdf-engine";
const kNotebooks = "notebooks";
const kPdfEngineOpts = "pdf-engine-opts";
const kPdfEngineOpt = "pdf-engine-opt";
const kListings = "listings";
const kNumberSections = "number-sections";
const kSectionNumbering = "section-numbering";
const kNumberOffset = "number-offset";
const kShiftHeadingLevelBy = "shift-heading-level-by";
const kNumberDepth = "number-depth";
const kTopLevelDivision = "top-level-division";
const kPaperSize = "papersize";
const kLogFile = "log-file";
const kHighlightStyle = "highlight-style";
const kDefaultImageExtension = "default-image-extension";
const kLogo = "logo";
const kLinkColor = "linkcolor";
const kColorLinks = "colorlinks";
const kVariables = "variables";
const kMetadataFile = "metadata-file";
const kMetadataFiles = "metadata-files";
const kSyntaxDefinitions = "syntax-definitions";
const kSyntaxDefinition = "syntax-definition";
const kReferenceDoc = "reference-doc";
const kHtmlMathMethod = "html-math-method";
const kToc = "toc";
const kTocDepth = "toc-depth";
const kTocIndent = "toc-indent";
const kTableOfContents = "table-of-contents";
const kSectionDivs = "section-divs";
const kEPubCoverImage = "epub-cover-image";
const kReferenceLocation = "reference-location";
const kCitationLocation = "citation-location";
const kQuartoVarsKey = "_quarto-vars";
const kQuartoTemplateParams = "quarto-template-params";
const kRevealJsScripts = "reveal-jsscripts";
const kQuartoInternal = "quarto-internal";
const kTitle = "title";
const kSubtitle = "subtitle";
const kAuthor = "author";
const kDate = "date";
const kDateFormat = "date-format";
const kDateModified = "date-modified";
const kDoi = "doi";
const kAbstract = "abstract";
const kAbstractTitle = "abstract-title";
const kDescription = "description";
const kHideDescription = "hide-description";
const kTocTitle = "toc-title";
const kTocLocation = "toc-location";
const kTocExpand = "toc-expand";
const kLang = "lang";
const kOrder = "order";
const kDisableArticleLayout = "disable-article-layout";
const kAuthors = "authors";
const kInstitute = "institute";
const kInstitutes = "institutes";
const kServer = "server";
const kPageTitle = "pagetitle";
const kTitlePrefix = "title-prefix";
const kCsl = "csl";
const kNoCite = "nocite";
const kCss = "css";
const kBibliography = "bibliography";
const kReferences = "references";
const kHeaderIncludes = "header-includes";
const kIncludeBefore = "include-before";
const kIncludeAfter = "include-after";
const kLinkCitations = "link-citations";
const kDocumentClass = "documentclass";
const kClassOption = "classoption";
const kSlideLevel = "slide-level";
const kTheme = "theme";
const kCrossref = "crossref";
const kCrossrefChapters = "chapters";
const kCrossrefLabels = "labels";
const kCrossrefAppendixTitle = "appendix-title";
const kCrossrefAppendixDelim = "appendix-delim";
const kCrossrefChaptersAppendix = "chapters-appendix";
const kCrossrefChaptersAlpha = "chapters-alpha";
const kCrossrefChapterId = "chapter-id";
const kGrid = "grid";
const kContentMode = "content-mode";
const kFigResponsive = "fig-responsive";
const kOutputLocation = "output-location";
const kCapLoc = "cap-location";
const kFigCapLoc = "fig-cap-location";
const kTblCapLoc = "tbl-cap-location";
const kCapTop = "top";
const kCapBottom = "bottom";
const kPositionedRefs = "positioned-refs";
const kPandocDefaultsKeys = [
    kTo,
    kFrom,
    kReader,
    kWriter,
    kOutputFile,
    kInputFiles,
    "defaults",
    kTemplate,
    kStandalone,
    kSelfContained,
    kEmbedResources,
    kVariables,
    "metadata",
    kMetadataFiles,
    kMetadataFile,
    kIncludeBeforeBody,
    kIncludeAfterBody,
    kIncludeInHeader,
    kResourcePath,
    kCiteproc,
    kCiteMethod,
    "citation-abbreviations",
    "filters",
    "file-scope",
    "data-dir",
    "verbosity",
    kLogFile,
    kTopLevelDivision,
    "abbreviations",
    kPdfEngine,
    kPdfEngineOpts,
    kPdfEngineOpt,
    kWrap,
    kColumns,
    "dpi",
    "extract-media",
    kToc,
    kTableOfContents,
    kTocDepth,
    kNumberSections,
    kNumberOffset,
    kShiftHeadingLevelBy,
    kSectionDivs,
    "identifier-prefix",
    kTitlePrefix,
    "eol",
    "strip-comments",
    "indented-code-classes",
    "ascii",
    kDefaultImageExtension,
    kHighlightStyle,
    kSyntaxDefinitions,
    kSyntaxDefinition,
    kListings,
    kReferenceDoc,
    kHtmlMathMethod,
    "email-obfuscation",
    "tab-stop",
    "preserve-tabs",
    "incremental",
    kSlideLevel,
    "epub-subdirectory",
    "epub-metadata",
    "epub-fonts",
    "epub-chapter-level",
    kEPubCoverImage,
    "reference-links",
    kReferenceLocation,
    kMarkdownHeadings,
    "track-changes",
    "html-q-tags",
    "css",
    "ipynb-output",
    "request-headers",
    "fail-if-warnings",
    "dump-args",
    "ignore-args",
    "trace"
];
const kCellCollapsed = "collapsed";
const kCellAutoscroll = "autoscroll";
const kCellDeletable = "deletable";
const kCellFormat = "format";
const kCellName = "name";
const kCellLinesToNext = "lines_to_next_cell";
const kCellLanguage = "language";
const kCellSlideshow = "slideshow";
const kCellSlideshowSlideType = "slide_type";
const kCellRawMimeType = "raw_mimetype";
const kCellUserExpressions = "user_expressions";
const kCellId = "id";
const kCellLabel = "label";
const kCellFigCap = "fig-cap";
const kCellFigSubCap = "fig-subcap";
const kCellFigScap = "fig-scap";
const kCellFigColumn = "fig-column";
const kCellTblColumn = "tbl-column";
const kCellFigLink = "fig-link";
const kCellFigAlign = "fig-align";
const kCellFigEnv = "fig-env";
const kCellFigPos = "fig-pos";
const kCellFigAlt = "fig-alt";
const kCellLstLabel = "lst-label";
const kCellLstCap = "lst-cap";
const kCellClasses = "classes";
const kCellPanel = "panel";
const kCellColumn = "column";
const kCellOutWidth = "out-width";
const kCellOutHeight = "out-height";
const kCellMdIndent = "md-indent";
const kCellWidth = "width";
const kCellHeight = "height";
const kCellColab = "colab";
const kCellColabType = "colab_type";
const kCellColbOutputId = "outputId";
const kLayoutAlign = "layout-align";
const kLayoutVAlign = "layout-valign";
const kLayoutNcol = "layout-ncol";
const kLayoutNrow = "layout-nrow";
const kLayout = "layout";
const kCliffyImplicitCwd = "5a6d2e4f-f9a2-43bc-8019-8149fbb76c85";
const kSourceMappingRegexes = [
    /^\/\/#\s*sourceMappingURL\=.*\.map$/gm,
    /\/\*\# sourceMappingURL=.* \*\//g
];
const kFormatIdentifier = "format-identifier";
const kRenderFileLifetime = "render-file";
const kRenderServicesLifetime = "render-services";
const kQuartoPre = "pre-quarto";
const kQuartoPost = "post-render";
async function readAndValidateYamlFromFile(file, schema, errorMessage, defaultContents) {
    if (!existsSync(file)) {
        throw new Error(`YAML file ${file} not found.`);
    }
    let shortFileName = file;
    if (shortFileName.startsWith("/")) {
        shortFileName = relative3(Deno.cwd(), shortFileName);
    }
    let fileContents = Deno.readTextFileSync(file).trimEnd();
    if (fileContents.trim().length === 0 && defaultContents) {
        fileContents = defaultContents;
    }
    const contents = asMappedString(fileContents, shortFileName);
    const { yaml, yamlValidationErrors } = await readAndValidateYamlFromMappedString(contents, schema);
    if (yamlValidationErrors.length) {
        throw new ValidationError2(errorMessage, yamlValidationErrors);
    }
    return yaml;
}
const kGfmCommonmarkExtensions = [
    "+autolink_bare_uris",
    "+emoji",
    "+footnotes",
    "+gfm_auto_identifiers",
    "+pipe_tables",
    "+strikeout",
    "+task_lists",
    "+tex_math_dollars"
];
const kGfmCommonmarkVariant = kGfmCommonmarkExtensions.join("");
`commonmark${kGfmCommonmarkVariant}`;
const kQmdExtensions = [
    ".qmd"
];
const kMarkdownEngine = "markdown";
const kKnitrEngine = "knitr";
const kJupyterEngine = "jupyter";
const kJuliaEngine = "julia";
async function includedMetadata(dir, baseMetadata, schema) {
    const yamlFiles = [];
    const metadataFile = baseMetadata[kMetadataFile];
    if (metadataFile) {
        yamlFiles.push(join2(dir, metadataFile));
    }
    const metadataFiles = baseMetadata[kMetadataFiles];
    if (metadataFiles && Array.isArray(metadataFiles)) {
        metadataFiles.forEach((file)=>yamlFiles.push(join2(dir, file)));
    }
    const filesMetadata = await Promise.all(yamlFiles.map(async (yamlFile)=>{
        if (existsSync(yamlFile)) {
            try {
                const yaml = await readAndValidateYamlFromFile(yamlFile, schema, `Validation of metadata file ${yamlFile} failed.`);
                return yaml;
            } catch (e) {
                error("\nError reading metadata file from " + yamlFile + "\n");
                throw e;
            }
        } else {
            return undefined;
        }
    }));
    return {
        metadata: mergeFormatMetadata({}, ...filesMetadata),
        files: yamlFiles
    };
}
function formatFromMetadata(baseFormat, to, debug) {
    const typedFormat = {
        identifier: {},
        render: {},
        execute: {},
        pandoc: {},
        language: {},
        metadata: {}
    };
    let format = typedFormat;
    const configFormats = baseFormat.metadata[kMetadataFormat];
    if (configFormats instanceof Object) {
        const configFormat = configFormats[to];
        if (configFormat === "default" || configFormat === true) {
            format = metadataAsFormat({});
        } else if (configFormat instanceof Object) {
            format = metadataAsFormat(configFormat);
        }
    }
    const mergedFormat = mergeFormatMetadata(baseFormat, format);
    if (debug) {
        mergedFormat.execute[kKeepMd] = true;
        mergedFormat.render[kKeepTex] = true;
        mergedFormat.render[kKeepTyp] = true;
    }
    return mergedFormat;
}
function formatKeys(metadata) {
    if (typeof metadata[kMetadataFormat] === "string") {
        return [
            metadata[kMetadataFormat]
        ];
    } else if (metadata[kMetadataFormat] instanceof Object) {
        return Object.keys(metadata[kMetadataFormat]).filter((key)=>{
            const format = metadata[kMetadataFormat][key];
            return format !== null && format !== false;
        });
    } else {
        return [];
    }
}
function isQuartoMetadata(key) {
    return kRenderDefaultsKeys.includes(key) || kExecuteDefaultsKeys.includes(key) || kPandocDefaultsKeys.includes(key) || kLanguageDefaultsKeys.includes(key) || [
        kKnitrEngine,
        kJupyterEngine
    ].includes(key);
}
function isIncludeMetadata(key) {
    return [
        kHeaderIncludes,
        kIncludeBefore,
        kIncludeAfter
    ].includes(key);
}
function metadataAsFormat(metadata) {
    const typedFormat = {
        identifier: {},
        render: {},
        execute: {},
        pandoc: {},
        language: {},
        metadata: {}
    };
    const format = typedFormat;
    Object.keys(metadata).forEach((key)=>{
        if ([
            kIdentifierDefaults,
            kRenderDefaults,
            kExecuteDefaults,
            kPandocDefaults,
            kLanguageDefaults,
            kPandocMetadata
        ].includes(key)) {
            if (typeof metadata[key] == "boolean") {
                if (key === kExecuteDefaults) {
                    format[key] = format[key] || {};
                    format[kExecuteDefaults][kExecuteEnabled] = metadata[key];
                }
            } else {
                format[key] = {
                    ...format[key],
                    ...metadata[key]
                };
            }
        } else {
            if (kIdentifierDefaultsKeys.includes(key)) {
                format.identifier[key] = metadata[key];
            } else if (kRenderDefaultsKeys.includes(key)) {
                format.render[key] = metadata[key];
            } else if (kExecuteDefaultsKeys.includes(key)) {
                format.execute[key] = metadata[key];
            } else if (kPandocDefaultsKeys.includes(key)) {
                format.pandoc[key] = metadata[key];
            } else {
                format.metadata[key] = metadata[key];
            }
        }
    });
    if (typeof format.metadata[kServer] === "string") {
        format.metadata[kServer] = {
            type: format.metadata[kServer]
        };
    }
    const filter = format.execute[kIpynbFilter];
    if (typeof filter === "string") {
        typedFormat.execute[kIpynbFilters] = typedFormat.execute[kIpynbFilters] || [];
        typedFormat.execute[kIpynbFilters]?.push(filter);
        delete typedFormat.execute[kIpynbFilter];
    }
    if (typeof typedFormat.render.variant === "string") {
        typedFormat.render.variant = typedFormat.render.variant.replace(/^gfm/, kGfmCommonmarkVariant);
    }
    return typedFormat;
}
function metadataGetDeep(metadata, property) {
    let values = [];
    __VIRTUAL_FILE145(metadata, (value, key)=>{
        if (key === property) {
            values.push(value);
        } else if (__VIRTUAL_FILE18(value)) {
            values = values.concat(metadataGetDeep(value, property));
        }
    });
    return values;
}
function mergeFormatMetadata(config, ...configs) {
    const kUnmergeableKeys = [
        kTblColwidths
    ];
    const kBooleanDisableArrays = [
        kCodeLinks,
        kOtherLinks
    ];
    return mergeConfigsCustomized((objValue, srcValue, key)=>{
        if (kUnmergeableKeys.includes(key)) {
            return srcValue;
        } else if (key === kVariant) {
            return mergePandocVariant(objValue, srcValue);
        } else if (kBooleanDisableArrays.includes(key)) {
            return mergeDisablableArray(objValue, srcValue);
        } else {
            return undefined;
        }
    }, config, ...configs);
}
function mergeProjectMetadata(config, ...configs) {
    const kExandableStringKeys = [
        "contents"
    ];
    return mergeConfigsCustomized((objValue, srcValue, key)=>{
        if (kExandableStringKeys.includes(key) && typeof objValue === "string") {
            return srcValue;
        } else {
            return undefined;
        }
    }, config, ...configs);
}
function mergeConfigsCustomized(customizer, config, ...configs) {
    config = cloneDeep1(config);
    configs = cloneDeep1(configs);
    return __VIRTUAL_FILE154(config, ...configs, (objValue, srcValue, key)=>{
        const custom = customizer(objValue, srcValue, key);
        if (custom !== undefined) {
            return custom;
        } else {
            return mergeArrayCustomizer(objValue, srcValue);
        }
    });
}
function mergeDisablableArray(objValue, srcValue) {
    if (Array.isArray(objValue) && Array.isArray(srcValue)) {
        return mergeArrayCustomizer(objValue, srcValue);
    } else {
        if (srcValue === false) {
            return [];
        } else {
            const srcArr = srcValue !== undefined ? Array.isArray(srcValue) ? srcValue : [
                srcValue
            ] : [];
            const objArr = objValue !== undefined ? Array.isArray(objValue) ? objValue : [
                objValue
            ] : [];
            return mergeArrayCustomizer(objArr, srcArr);
        }
    }
}
function mergePandocVariant(objValue, srcValue) {
    if (typeof objValue === "string" && typeof srcValue === "string" && objValue !== srcValue) {
        const extensions = {};
        [
            ...parsePandocVariant(objValue),
            ...parsePandocVariant(srcValue)
        ].forEach((extension)=>{
            extensions[extension.name] = extension.enabled;
        });
        return Object.keys(extensions).map((name)=>`${extensions[name] ? "+" : "-"}${name}`).join("");
    } else {
        return undefined;
    }
}
function parsePandocVariant(variant) {
    variant = variant.split("\n").join();
    const extensions = [];
    const re = /([+-])([a-z_]+)/g;
    let match = re.exec(variant);
    while(match){
        extensions.push({
            name: match[2],
            enabled: match[1] === "+"
        });
        match = re.exec(variant);
    }
    return extensions;
}
function removeFlags(flags, remove) {
    let removeNext = false;
    return flags.reduce((args, arg)=>{
        if (!remove.has(arg)) {
            if (!removeNext) {
                args.push(arg);
            }
            removeNext = false;
        } else {
            removeNext = remove.get(arg);
        }
        return args;
    }, new Array());
}
const kStdOut = "-";
async function parseRenderFlags(args) {
    const flags = {};
    const argsStack = [
        ...args
    ];
    let arg = argsStack.shift();
    while(arg !== undefined){
        const equalSignIndex = arg.indexOf("=");
        if (arg.startsWith("--") && equalSignIndex > 0) {
            argsStack.unshift(arg.slice(equalSignIndex + 1));
            arg = arg.slice(0, equalSignIndex);
        }
        switch(arg){
            case "-t":
            case "--to":
                arg = argsStack.shift();
                if (arg && !arg.startsWith("-")) {
                    flags.to = arg;
                }
                break;
            case "-o":
            case "--output":
                arg = argsStack.shift();
                if (!arg || arg.startsWith("-")) {
                    flags.output = kStdOut;
                } else {
                    if (arg.match(SEPARATOR_PATTERN2)) {
                        throw new Error("--output option cannot specify a relative or absolute path");
                    }
                    flags.output = arg;
                }
                break;
            case "--output-dir":
                arg = argsStack.shift();
                flags.outputDir = arg;
                break;
            case "--site-url":
                arg = argsStack.shift();
                flags.siteUrl = arg;
                break;
            case "--standalone":
                flags[kStandalone] = true;
                arg = argsStack.shift();
                break;
            case "--self-contained":
                flags[kSelfContained] = true;
                arg = argsStack.shift();
                break;
            case "--embed-resources":
                flags[kEmbedResources] = true;
                arg = argsStack.shift();
                break;
            case "--pdf-engine":
                arg = argsStack.shift();
                flags.pdfEngine = arg;
                break;
            case "--pdf-engine-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.pdfEngineOpts = flags.pdfEngineOpts || [];
                    flags.pdfEngineOpts.push(arg);
                }
                break;
            case "--latex-makeindex-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.makeIndexOpts = flags.makeIndexOpts || [];
                    flags.makeIndexOpts.push(arg);
                }
                break;
            case "--latex-tlmgr-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.tlmgrOpts = flags.tlmgrOpts || [];
                    flags.tlmgrOpts.push(arg);
                }
                break;
            case "--natbib":
                arg = argsStack.shift();
                flags.natbib = true;
                break;
            case "--biblatex":
                arg = argsStack.shift();
                flags.biblatex = true;
                break;
            case `--${kToc}`:
            case `--${kTableOfContents}`:
                arg = argsStack.shift();
                flags.toc = true;
                break;
            case "--listings":
                arg = argsStack.shift();
                flags[kListings] = true;
                break;
            case "--number-sections":
                arg = argsStack.shift();
                flags[kNumberSections] = true;
                break;
            case "--number-offset":
                arg = argsStack.shift();
                flags[kNumberSections] = true;
                flags[kNumberOffset] = parseNumbers("--number-offset", arg);
                break;
            case "--top-level-division":
                arg = argsStack.shift();
                flags[kTopLevelDivision] = arg;
                break;
            case "--shift-heading-level-by":
                arg = argsStack.shift();
                flags[kShiftHeadingLevelBy] = arg;
                break;
            case "--include-in-header":
            case "--include-before-body":
            case "--include-after-body":
                {
                    const include = arg.replace("^--", "");
                    const includeFlags = flags;
                    includeFlags[include] = includeFlags[include] || [];
                    arg = argsStack.shift();
                    includeFlags[include].push(arg);
                    break;
                }
            case "--mathjax":
                flags.mathjax = true;
                arg = argsStack.shift();
                break;
            case "--katex":
                flags.katex = true;
                arg = argsStack.shift();
                break;
            case "--mathml":
                flags.mathml = true;
                arg = argsStack.shift();
                break;
            case "--gladtex":
                flags.gladtex = true;
                arg = argsStack.shift();
                break;
            case "--webtex":
                flags.webtex = true;
                arg = argsStack.shift();
                break;
            case "--execute":
                flags.execute = true;
                arg = argsStack.shift();
                break;
            case "--no-execute":
                flags.execute = false;
                arg = argsStack.shift();
                break;
            case "--execute-params":
                arg = argsStack.shift();
                flags.paramsFile = arg;
                break;
            case "--execute-dir":
                arg = argsStack.shift();
                if (arg) {
                    if (isAbsolute2(arg)) {
                        flags.executeDir = arg;
                    } else {
                        flags.executeDir = normalizePath(arg);
                    }
                }
                break;
            case "--execute-daemon":
                arg = argsStack.shift();
                flags.executeDaemon = parseInt(arg, 10);
                if (isNaN(flags.executeDaemon)) {
                    delete flags.executeDaemon;
                }
                break;
            case "--no-execute-daemon":
                arg = argsStack.shift();
                flags.executeDaemon = 0;
                break;
            case "--execute-daemon-restart":
                arg = argsStack.shift();
                flags.executeDaemonRestart = true;
                break;
            case "--execute-debug":
                arg = argsStack.shift();
                flags.executeDebug = true;
                break;
            case "--use-freezer":
                arg = argsStack.shift();
                flags.useFreezer = true;
                break;
            case "--cache":
                arg = argsStack.shift();
                flags.executeCache = true;
                break;
            case "--no-cache":
                arg = argsStack.shift();
                flags.executeCache = false;
                break;
            case "--cache-refresh":
                arg = argsStack.shift();
                flags.executeCache = "refresh";
                break;
            case "--clean":
                arg = argsStack.shift();
                flags.clean = true;
                break;
            case "--no-clean":
                arg = argsStack.shift();
                flags.clean = false;
                break;
            case "--debug":
                flags.debug = true;
                arg = argsStack.shift();
                break;
            case "-P":
            case "--execute-param":
                arg = argsStack.shift();
                if (arg) {
                    const param = parseMetadataFlagValue(arg);
                    if (param) {
                        if (param.value !== undefined) {
                            flags.params = flags.params || {};
                            flags.params[param.name] = param.value;
                        }
                    }
                }
                break;
            case "-M":
            case "--metadata":
                arg = argsStack.shift();
                if (arg) {
                    const metadata = parseMetadataFlagValue(arg);
                    if (metadata) {
                        if (metadata.value !== undefined) {
                            if (isQuartoMetadata(metadata.name)) {
                                flags.metadata = flags.metadata || {};
                                flags.metadata[metadata.name] = metadata.value;
                            } else {
                                flags.pandocMetadata = flags.pandocMetadata || {};
                                flags.pandocMetadata[metadata.name] = metadata.value;
                            }
                        }
                    }
                }
                break;
            case "--metadata-file":
                arg = argsStack.shift();
                if (arg) {
                    if (existsSync(arg)) {
                        const metadata = await readYamlFromString(Deno.readTextFileSync(arg));
                        flags.metadata = {
                            ...flags.metadata,
                            ...metadata
                        };
                    }
                }
                break;
            case "--reference-location":
                arg = argsStack.shift();
                if (arg) {
                    flags[kReferenceLocation] = arg;
                }
                break;
            default:
                arg = argsStack.shift();
                break;
        }
    }
    if (flags.clean === undefined) {
        flags.clean = true;
    }
    return flags;
}
function havePandocArg(pandocArgs, arg) {
    return pandocArgs.indexOf(arg) !== -1;
}
function replacePandocArg(pandocArgs, arg, value) {
    const newArgs = [
        ...pandocArgs
    ];
    const argIndex = pandocArgs.indexOf(arg);
    if (argIndex !== -1) {
        newArgs[argIndex + 1] = value;
    } else {
        newArgs.push(arg);
        newArgs.push(value);
    }
    return newArgs;
}
function getPandocArg(pandocArgs, arg) {
    const argIndex = pandocArgs.indexOf(arg);
    if (argIndex !== -1 && argIndex + 1 < pandocArgs.length) {
        return pandocArgs[argIndex + 1];
    } else {
        return undefined;
    }
}
function replacePandocOutputArg(pandocArgs, output) {
    if (havePandocArg(pandocArgs, "--output")) {
        return replacePandocArg(pandocArgs, "--output", output);
    } else if (havePandocArg(pandocArgs, "-o")) {
        return replacePandocArg(pandocArgs, "-o", output);
    } else {
        return pandocArgs;
    }
}
function fixupPandocArgs(pandocArgs, flags) {
    pandocArgs = pandocArgs.reduce((args, arg, index)=>{
        args.push(arg);
        if (flags.output === kStdOut && pandocArgs[index + 1] !== kStdOut && (arg === "-o" || arg === "--output")) {
            args.push(kStdOut);
        }
        return args;
    }, new Array());
    const removeArgs = new Map();
    removeArgs.set("--output-dir", true);
    removeArgs.set("--site-url", true);
    removeArgs.set("--execute", false);
    removeArgs.set("--no-execute", false);
    removeArgs.set("-P", true);
    removeArgs.set("--execute-param", true);
    removeArgs.set("--execute-params", true);
    removeArgs.set("--execute-dir", true);
    removeArgs.set("--execute-daemon", true);
    removeArgs.set("--no-execute-daemon", false);
    removeArgs.set("--execute-daemon-restart", false);
    removeArgs.set("--execute-debug", false);
    removeArgs.set("--use-freezer", false);
    removeArgs.set("--cache", false);
    removeArgs.set("--no-cache", false);
    removeArgs.set("--cache-refresh", false);
    removeArgs.set("--clean", false);
    removeArgs.set("--no-clean", false);
    removeArgs.set("--debug", false);
    removeArgs.set("--metadata-file", true);
    removeArgs.set("--latex-makeindex-opt", true);
    removeArgs.set("--latex-tlmgr-opt", true);
    removeArgs.set("--log", true);
    removeArgs.set("--l", true);
    removeArgs.set("--log-level", true);
    removeArgs.set("--ll", true);
    removeArgs.set("--log-format", true);
    removeArgs.set("--lf", true);
    removeArgs.set("--quiet", false);
    removeArgs.set("--q", false);
    removeArgs.set("--profile", true);
    pandocArgs = removePandocArgs(pandocArgs, removeArgs);
    return removeQuartoMetadataFlags(pandocArgs);
}
function removePandocArgs(pandocArgs, removeArgs) {
    return removeFlags(pandocArgs, removeArgs);
}
function removePandocToArg(args) {
    const removeArgs = new Map();
    removeArgs.set("--to", true);
    removeArgs.set("-t", true);
    return removePandocArgs(args, removeArgs);
}
function removePandocTo(renderOptions) {
    renderOptions = cloneDeep1(renderOptions);
    delete renderOptions.flags?.to;
    if (renderOptions.pandocArgs) {
        renderOptions.pandocArgs = removePandocToArg(renderOptions.pandocArgs);
    }
    return renderOptions;
}
function removeQuartoMetadataFlags(pandocArgs) {
    const args = [];
    for(let i1 = 0; i1 < pandocArgs.length; i1++){
        const arg = pandocArgs[i1];
        if (arg === "--metadata" || arg === "-M") {
            const flagValue = parseMetadataFlagValue(pandocArgs[i1 + 1] || "");
            if (flagValue !== undefined && (isQuartoMetadata(flagValue.name) || kQuartoForwardedMetadataFields.includes(flagValue.name))) {
                i1++;
            } else {
                args.push(arg);
            }
        } else {
            args.push(arg);
        }
    }
    return args;
}
const kQuartoForwardedMetadataFields = [
    kAuthor,
    kDate
];
function parseMetadataFlagValue(arg) {
    const match = arg.match(/^([^=:]+)[=:](.*)$/);
    if (match) {
        return {
            name: match[1],
            value: readYamlFromString(match[2])
        };
    }
    return undefined;
}
function resolveParams(params, paramsFile) {
    if (params || paramsFile) {
        params = params || {};
        if (paramsFile) {
            params = mergeConfigs(readYaml(paramsFile), params);
        }
        return params;
    } else {
        return undefined;
    }
}
function parseNumbers(flag, value) {
    if (value) {
        const numbers = value.split(/,/).map((number)=>parseInt(number.trim(), 10)).filter((number)=>!isNaN(number));
        if (numbers.length > 0) {
            return numbers;
        }
    }
    throw new Error(`Invalid value for ${flag} (should be a comma separated list of numbers)`);
}
const importMeta = {
    url: "https://deno.land/x/deno_dom@v0.1.41/build/deno-wasm/deno-wasm.js",
    main: false
};
let wasm;
let WASM_VECTOR_LEN = 0;
let cachegetUint8Memory0 = null;
function getUint8Memory0() {
    if (cachegetUint8Memory0 === null || cachegetUint8Memory0.buffer !== wasm.memory.buffer) {
        cachegetUint8Memory0 = new Uint8Array(wasm.memory.buffer);
    }
    return cachegetUint8Memory0;
}
let cachedTextEncoder = new TextEncoder("utf-8");
const encodeString = typeof cachedTextEncoder.encodeInto === "function" ? function(arg, view) {
    return cachedTextEncoder.encodeInto(arg, view);
} : function(arg, view) {
    const buf = cachedTextEncoder.encode(arg);
    view.set(buf);
    return {
        read: arg.length,
        written: buf.length
    };
};
function passStringToWasm0(arg, malloc, realloc) {
    if (realloc === undefined) {
        const buf = cachedTextEncoder.encode(arg);
        const ptr = malloc(buf.length);
        getUint8Memory0().subarray(ptr, ptr + buf.length).set(buf);
        WASM_VECTOR_LEN = buf.length;
        return ptr;
    }
    let len = arg.length;
    let ptr = malloc(len);
    const mem = getUint8Memory0();
    let offset = 0;
    for(; offset < len; offset++){
        const code = arg.charCodeAt(offset);
        if (code > 0x7F) break;
        mem[ptr + offset] = code;
    }
    if (offset !== len) {
        if (offset !== 0) {
            arg = arg.slice(offset);
        }
        ptr = realloc(ptr, len, len = offset + arg.length * 3);
        const view = getUint8Memory0().subarray(ptr + offset, ptr + len);
        const ret = encodeString(arg, view);
        offset += ret.written;
    }
    WASM_VECTOR_LEN = offset;
    return ptr;
}
let cachegetInt32Memory0 = null;
function getInt32Memory0() {
    if (cachegetInt32Memory0 === null || cachegetInt32Memory0.buffer !== wasm.memory.buffer) {
        cachegetInt32Memory0 = new Int32Array(wasm.memory.buffer);
    }
    return cachegetInt32Memory0;
}
let cachedTextDecoder = new TextDecoder("utf-8", {
    ignoreBOM: true,
    fatal: true
});
cachedTextDecoder.decode();
function getStringFromWasm0(ptr, len) {
    return cachedTextDecoder.decode(getUint8Memory0().subarray(ptr, ptr + len));
}
function parse1(html) {
    try {
        const retptr = wasm.__wbindgen_add_to_stack_pointer(-16);
        var ptr0 = passStringToWasm0(html, wasm.__wbindgen_malloc, wasm.__wbindgen_realloc);
        var len0 = WASM_VECTOR_LEN;
        wasm.parse(retptr, ptr0, len0);
        var r0 = getInt32Memory0()[retptr / 4 + 0];
        var r1 = getInt32Memory0()[retptr / 4 + 1];
        return getStringFromWasm0(r0, r1);
    } finally{
        wasm.__wbindgen_add_to_stack_pointer(16);
        wasm.__wbindgen_free(r0, r1);
    }
}
function parse_frag(html, context_local_name) {
    try {
        const retptr = wasm.__wbindgen_add_to_stack_pointer(-16);
        var ptr0 = passStringToWasm0(html, wasm.__wbindgen_malloc, wasm.__wbindgen_realloc);
        var len0 = WASM_VECTOR_LEN;
        var ptr1 = passStringToWasm0(context_local_name, wasm.__wbindgen_malloc, wasm.__wbindgen_realloc);
        var len1 = WASM_VECTOR_LEN;
        wasm.parse_frag(retptr, ptr0, len0, ptr1, len1);
        var r0 = getInt32Memory0()[retptr / 4 + 0];
        var r1 = getInt32Memory0()[retptr / 4 + 1];
        return getStringFromWasm0(r0, r1);
    } finally{
        wasm.__wbindgen_add_to_stack_pointer(16);
        wasm.__wbindgen_free(r0, r1);
    }
}
async function load2(module1, imports) {
    if (typeof Response === "function" && module1 instanceof Response) {
        if (typeof WebAssembly.instantiateStreaming === "function") {
            try {
                return await WebAssembly.instantiateStreaming(module1, imports);
            } catch (e) {
                if (module1.headers.get("Content-Type") != "application/wasm") {
                    console.warn("`WebAssembly.instantiateStreaming` failed because your server does not serve wasm with `application/wasm` MIME type. Falling back to `WebAssembly.instantiate` which is slower. Original error:\n", e);
                } else {
                    throw e;
                }
            }
        }
        const bytes = await module1.arrayBuffer();
        return await WebAssembly.instantiate(bytes, imports);
    } else {
        const instance = await WebAssembly.instantiate(module1, imports);
        if (instance instanceof WebAssembly.Instance) {
            return {
                instance,
                module: module1
            };
        } else {
            return instance;
        }
    }
}
async function init(input) {
    if (typeof input === "undefined") {
        input = new URL("deno-wasm_bg.wasm", importMeta.url);
    }
    const imports = {};
    imports.env = {
        now () {}
    };
    input = Uint8Array.from(atob("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