// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/**
 * {@linkcode encode} and {@linkcode decode} for
 * [base58](https://en.wikipedia.org/wiki/Binary-to-text_encoding#Base58) encoding.
 *
 * This module is browser compatible.
 *
 * @module
 */ // deno-fmt-ignore
const mapBase58 = {
    "1": 0,
    "2": 1,
    "3": 2,
    "4": 3,
    "5": 4,
    "6": 5,
    "7": 6,
    "8": 7,
    "9": 8,
    A: 9,
    B: 10,
    C: 11,
    D: 12,
    E: 13,
    F: 14,
    G: 15,
    H: 16,
    J: 17,
    K: 18,
    L: 19,
    M: 20,
    N: 21,
    P: 22,
    Q: 23,
    R: 24,
    S: 25,
    T: 26,
    U: 27,
    V: 28,
    W: 29,
    X: 30,
    Y: 31,
    Z: 32,
    a: 33,
    b: 34,
    c: 35,
    d: 36,
    e: 37,
    f: 38,
    g: 39,
    h: 40,
    i: 41,
    j: 42,
    k: 43,
    m: 44,
    n: 45,
    o: 46,
    p: 47,
    q: 48,
    r: 49,
    s: 50,
    t: 51,
    u: 52,
    v: 53,
    w: 54,
    x: 55,
    y: 56,
    z: 57
};
const base58alphabet = "123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz".split("");
/**
 * Encodes a given Uint8Array, ArrayBuffer or string into draft-mspotny-base58-03 RFC base58 representation:
 * https://tools.ietf.org/id/draft-msporny-base58-01.html#rfc.section.1
 *
 * @param {ArrayBuffer | string} data
 *
 * @returns {string} Encoded value
 */ export function encode(data) {
    const uint8tData = typeof data === "string" ? new TextEncoder().encode(data) : data instanceof Uint8Array ? data : new Uint8Array(data);
    let length = 0;
    let zeroes = 0;
    // Counting leading zeroes
    let index = 0;
    while(uint8tData[index] === 0){
        zeroes++;
        index++;
    }
    const notZeroUint8Data = uint8tData.slice(index);
    const size = Math.round(uint8tData.length * 138 / 100 + 1);
    const b58Encoding = [];
    notZeroUint8Data.forEach((byte)=>{
        let i = 0;
        let carry = byte;
        for(let reverse_iterator = size - 1; (carry > 0 || i < length) && reverse_iterator !== -1; reverse_iterator--, i++){
            carry += (b58Encoding[reverse_iterator] || 0) * 256;
            b58Encoding[reverse_iterator] = Math.round(carry % 58);
            carry = Math.floor(carry / 58);
        }
        length = i;
    });
    const strResult = Array.from({
        length: b58Encoding.length + zeroes
    });
    if (zeroes > 0) {
        strResult.fill("1", 0, zeroes);
    }
    b58Encoding.forEach((byteValue)=>strResult.push(base58alphabet[byteValue]));
    return strResult.join("");
}
/**
 * Decodes a given b58 string according to draft-mspotny-base58-03 RFC base58 representation:
 * https://tools.ietf.org/id/draft-msporny-base58-01.html#rfc.section.1
 *
 * @param {string} b58
 *
 * @returns {Uint8Array} Decoded value
 */ export function decode(b58) {
    const splittedInput = b58.trim().split("");
    let length = 0;
    let ones = 0;
    // Counting leading ones
    let index = 0;
    while(splittedInput[index] === "1"){
        ones++;
        index++;
    }
    const notZeroData = splittedInput.slice(index);
    const size = Math.round(b58.length * 733 / 1000 + 1);
    const output = [];
    notZeroData.forEach((char, idx)=>{
        let carry = mapBase58[char];
        let i = 0;
        if (carry === undefined) {
            throw new Error(`Invalid base58 char at index ${idx} with value ${char}`);
        }
        for(let reverse_iterator = size - 1; (carry > 0 || i < length) && reverse_iterator !== 0; --reverse_iterator, ++i){
            carry += 58 * (output[reverse_iterator - 1] || 0);
            output[reverse_iterator - 1] = Math.round(carry % 256);
            carry = Math.floor(carry / 256);
        }
        length = i;
    });
    const validOutput = output.filter((item)=>item ?? false);
    if (ones > 0) {
        const onesResult = Array.from({
            length: ones
        }).fill(0, 0, ones);
        return new Uint8Array([
            ...onesResult,
            ...validOutput
        ]);
    }
    return new Uint8Array(validOutput);
}
//# sourceMappingURL=data:application/json;base64,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