// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/**
 * Extensions to the
 * [Web Crypto](https://developer.mozilla.org/en-US/docs/Web/API/Web_Crypto_API)
 * supporting additional encryption APIs.
 *
 * Provides additional digest algorithms that are not part of the WebCrypto
 * standard as well as a `subtle.digest` and `subtle.digestSync` methods. It
 * also provide a `subtle.timingSafeEqual()` method to compare array buffers
 * or data views in a way that isn't prone to timing based attacks.
 *
 * The "polyfill" delegates to `WebCrypto` where possible.
 *
 * The {@linkcode KeyStack} export implements the {@linkcode KeyRing} interface
 * for managing rotatable keys for signing data to prevent tampering, like with
 * HTTP cookies.
 *
 * @module
 */ import { digestAlgorithms as wasmDigestAlgorithms, instantiateWasm } from "./_wasm/mod.ts";
import { timingSafeEqual } from "./timing_safe_equal.ts";
import { fnv } from "./_fnv/index.ts";
export { KeyStack } from "./keystack.ts";
export { toHashString } from "./util.ts";
/**
 * A copy of the global WebCrypto interface, with methods bound so they're
 * safe to re-export.
 */ const webCrypto = ((crypto)=>({
        getRandomValues: crypto.getRandomValues?.bind(crypto),
        randomUUID: crypto.randomUUID?.bind(crypto),
        subtle: {
            decrypt: crypto.subtle?.decrypt?.bind(crypto.subtle),
            deriveBits: crypto.subtle?.deriveBits?.bind(crypto.subtle),
            deriveKey: crypto.subtle?.deriveKey?.bind(crypto.subtle),
            digest: crypto.subtle?.digest?.bind(crypto.subtle),
            encrypt: crypto.subtle?.encrypt?.bind(crypto.subtle),
            exportKey: crypto.subtle?.exportKey?.bind(crypto.subtle),
            generateKey: crypto.subtle?.generateKey?.bind(crypto.subtle),
            importKey: crypto.subtle?.importKey?.bind(crypto.subtle),
            sign: crypto.subtle?.sign?.bind(crypto.subtle),
            unwrapKey: crypto.subtle?.unwrapKey?.bind(crypto.subtle),
            verify: crypto.subtle?.verify?.bind(crypto.subtle),
            wrapKey: crypto.subtle?.wrapKey?.bind(crypto.subtle)
        }
    }))(globalThis.crypto);
const bufferSourceBytes = (data)=>{
    let bytes;
    if (data instanceof Uint8Array) {
        bytes = data;
    } else if (ArrayBuffer.isView(data)) {
        bytes = new Uint8Array(data.buffer, data.byteOffset, data.byteLength);
    } else if (data instanceof ArrayBuffer) {
        bytes = new Uint8Array(data);
    }
    return bytes;
};
/**
 * An wrapper for WebCrypto adding support for additional non-standard
 * algorithms, but delegating to the runtime WebCrypto implementation whenever
 * possible.
 */ const stdCrypto = ((x)=>x)({
    ...webCrypto,
    subtle: {
        ...webCrypto.subtle,
        /**
     * Polyfills stream support until the Web Crypto API does so:
     * @see {@link https://github.com/wintercg/proposal-webcrypto-streams}
     */ async digest (algorithm, data) {
            const { name , length  } = normalizeAlgorithm(algorithm);
            const bytes = bufferSourceBytes(data);
            if (FNVAlgorithms.includes(name)) {
                return fnv(name, bytes);
            }
            // We delegate to WebCrypto whenever possible,
            if (// if the algorithm is supported by the WebCrypto standard,
            webCryptoDigestAlgorithms.includes(name) && // and the data is a single buffer,
            bytes) {
                return webCrypto.subtle.digest(algorithm, bytes);
            } else if (wasmDigestAlgorithms.includes(name)) {
                if (bytes) {
                    // Otherwise, we use our bundled Wasm implementation via digestSync
                    // if it supports the algorithm.
                    return stdCrypto.subtle.digestSync(algorithm, bytes);
                } else if (data[Symbol.iterator]) {
                    return stdCrypto.subtle.digestSync(algorithm, data);
                } else if (data[Symbol.asyncIterator]) {
                    const wasmCrypto = instantiateWasm();
                    const context = new wasmCrypto.DigestContext(name);
                    for await (const chunk of data){
                        const chunkBytes = bufferSourceBytes(chunk);
                        if (!chunkBytes) {
                            throw new TypeError("data contained chunk of the wrong type");
                        }
                        context.update(chunkBytes);
                    }
                    return context.digestAndDrop(length).buffer;
                } else {
                    throw new TypeError("data must be a BufferSource or [Async]Iterable<BufferSource>");
                }
            } else if (webCrypto.subtle?.digest) {
                // (TypeScript type definitions prohibit this case.) If they're trying
                // to call an algorithm we don't recognize, pass it along to WebCrypto
                // in case it's a non-standard algorithm supported by the the runtime
                // they're using.
                return webCrypto.subtle.digest(algorithm, data);
            } else {
                throw new TypeError(`unsupported digest algorithm: ${algorithm}`);
            }
        },
        digestSync (algorithm, data) {
            algorithm = normalizeAlgorithm(algorithm);
            const bytes = bufferSourceBytes(data);
            if (FNVAlgorithms.includes(algorithm.name)) {
                return fnv(algorithm.name, bytes);
            }
            const wasmCrypto = instantiateWasm();
            if (bytes) {
                return wasmCrypto.digest(algorithm.name, bytes, algorithm.length).buffer;
            } else if (data[Symbol.iterator]) {
                const context = new wasmCrypto.DigestContext(algorithm.name);
                for (const chunk of data){
                    const chunkBytes = bufferSourceBytes(chunk);
                    if (!chunkBytes) {
                        throw new TypeError("data contained chunk of the wrong type");
                    }
                    context.update(chunkBytes);
                }
                return context.digestAndDrop(algorithm.length).buffer;
            } else {
                throw new TypeError("data must be a BufferSource or Iterable<BufferSource>");
            }
        },
        // TODO(@kitsonk): rework when https://github.com/w3c/webcrypto/issues/270 resolved
        timingSafeEqual
    }
});
const FNVAlgorithms = [
    "FNV32",
    "FNV32A",
    "FNV64",
    "FNV64A"
];
/** Digest algorithms supported by WebCrypto. */ const webCryptoDigestAlgorithms = [
    "SHA-384",
    "SHA-256",
    "SHA-512",
    // insecure (length-extendable and collidable):
    "SHA-1"
];
const normalizeAlgorithm = (algorithm)=>typeof algorithm === "string" ? {
        name: algorithm.toUpperCase()
    } : {
        ...algorithm,
        name: algorithm.name.toUpperCase()
    };
export { stdCrypto as crypto };
//# sourceMappingURL=data:application/json;base64,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