// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/*!
 * Ported and modified from: https://github.com/beatgammit/tar-js and
 * licensed as:
 *
 * (The MIT License)
 *
 * Copyright (c) 2011 T. Jameson Little
 * Copyright (c) 2019 Jun Kato
 * Copyright (c) 2018-2022 the Deno authors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */ /**
 * Provides a `Tar` and `Untar` classes for compressing and decompressing
 * arbitrary data.
 *
 * ## Examples
 *
 * ### Tar
 *
 * ```ts
 * import { Tar } from "https://deno.land/std@$STD_VERSION/archive/tar.ts";
 * import { Buffer } from "https://deno.land/std@$STD_VERSION/io/buffer.ts";
 * import { copy } from "https://deno.land/std@$STD_VERSION/streams/conversion.ts";
 *
 * const tar = new Tar();
 * const content = new TextEncoder().encode("Deno.land");
 * await tar.append("deno.txt", {
 *   reader: new Buffer(content),
 *   contentSize: content.byteLength,
 * });
 *
 * // Or specifying a filePath.
 * await tar.append("land.txt", {
 *   filePath: "./land.txt",
 * });
 *
 * // use tar.getReader() to read the contents.
 *
 * const writer = await Deno.open("./out.tar", { write: true, create: true });
 * await copy(tar.getReader(), writer);
 * writer.close();
 * ```
 *
 * ### Untar
 *
 * ```ts
 * import { Untar } from "https://deno.land/std@$STD_VERSION/archive/tar.ts";
 * import { ensureFile } from "https://deno.land/std@$STD_VERSION/fs/ensure_file.ts";
 * import { ensureDir } from "https://deno.land/std@$STD_VERSION/fs/ensure_dir.ts";
 * import { copy } from "https://deno.land/std@$STD_VERSION/streams/conversion.ts";
 *
 * const reader = await Deno.open("./out.tar", { read: true });
 * const untar = new Untar(reader);
 *
 * for await (const entry of untar) {
 *   console.log(entry); // metadata
 *
 *   if (entry.type === "directory") {
 *     await ensureDir(entry.fileName);
 *     continue;
 *   }
 *
 *   await ensureFile(entry.fileName);
 *   const file = await Deno.open(entry.fileName, { write: true });
 *   // <entry> is a reader.
 *   await copy(entry, file);
 * }
 * reader.close();
 * ```
 *
 * @module
 */ import { MultiReader } from "../io/readers.ts";
import { Buffer, PartialReadError } from "../io/buffer.ts";
import { assert } from "../_util/asserts.ts";
import { readAll } from "../streams/conversion.ts";
const recordSize = 512;
const ustar = "ustar\u000000";
// https://pubs.opengroup.org/onlinepubs/9699919799/utilities/pax.html#tag_20_92_13_06
// eight checksum bytes taken to be ascii spaces (decimal value 32)
const initialChecksum = 8 * 32;
async function readBlock(reader, p) {
    let bytesRead = 0;
    while(bytesRead < p.length){
        const rr = await reader.read(p.subarray(bytesRead));
        if (rr === null) {
            if (bytesRead === 0) {
                return null;
            } else {
                throw new PartialReadError();
            }
        }
        bytesRead += rr;
    }
    return bytesRead;
}
/**
 * Simple file reader
 */ class FileReader {
    #file;
    constructor(filePath){
        this.filePath = filePath;
    }
    async read(p) {
        if (!this.#file) {
            this.#file = await Deno.open(this.filePath, {
                read: true
            });
        }
        const res = await this.#file.read(p);
        if (res === null) {
            this.#file.close();
            this.#file = undefined;
        }
        return res;
    }
    filePath;
}
/**
 * Remove the trailing null codes
 * @param buffer
 */ function trim(buffer) {
    const index = buffer.findIndex((v)=>v === 0);
    if (index < 0) return buffer;
    return buffer.subarray(0, index);
}
/**
 * Initialize Uint8Array of the specified length filled with 0
 * @param length
 */ function clean(length) {
    const buffer = new Uint8Array(length);
    buffer.fill(0, 0, length - 1);
    return buffer;
}
function pad(num, bytes, base = 8) {
    const numString = num.toString(base);
    return "000000000000".substr(numString.length + 12 - bytes) + numString;
}
var FileTypes;
(function(FileTypes) {
    FileTypes[FileTypes["file"] = 0] = "file";
    FileTypes[FileTypes["link"] = 1] = "link";
    FileTypes[FileTypes["symlink"] = 2] = "symlink";
    FileTypes[FileTypes["character-device"] = 3] = "character-device";
    FileTypes[FileTypes["block-device"] = 4] = "block-device";
    FileTypes[FileTypes["directory"] = 5] = "directory";
    FileTypes[FileTypes["fifo"] = 6] = "fifo";
    FileTypes[FileTypes["contiguous-file"] = 7] = "contiguous-file";
})(FileTypes || (FileTypes = {}));
/*
struct posix_header {           // byte offset
  char name[100];               //   0
  char mode[8];                 // 100
  char uid[8];                  // 108
  char gid[8];                  // 116
  char size[12];                // 124
  char mtime[12];               // 136
  char chksum[8];               // 148
  char typeflag;                // 156
  char linkname[100];           // 157
  char magic[6];                // 257
  char version[2];              // 263
  char uname[32];               // 265
  char gname[32];               // 297
  char devmajor[8];             // 329
  char devminor[8];             // 337
  char prefix[155];             // 345
                                // 500
};
*/ const ustarStructure = [
    {
        field: "fileName",
        length: 100
    },
    {
        field: "fileMode",
        length: 8
    },
    {
        field: "uid",
        length: 8
    },
    {
        field: "gid",
        length: 8
    },
    {
        field: "fileSize",
        length: 12
    },
    {
        field: "mtime",
        length: 12
    },
    {
        field: "checksum",
        length: 8
    },
    {
        field: "type",
        length: 1
    },
    {
        field: "linkName",
        length: 100
    },
    {
        field: "ustar",
        length: 8
    },
    {
        field: "owner",
        length: 32
    },
    {
        field: "group",
        length: 32
    },
    {
        field: "majorNumber",
        length: 8
    },
    {
        field: "minorNumber",
        length: 8
    },
    {
        field: "fileNamePrefix",
        length: 155
    },
    {
        field: "padding",
        length: 12
    }
];
/**
 * Create header for a file in a tar archive
 */ function formatHeader(data) {
    const encoder = new TextEncoder(), buffer = clean(512);
    let offset = 0;
    ustarStructure.forEach(function(value) {
        const entry = encoder.encode(data[value.field] || "");
        buffer.set(entry, offset);
        offset += value.length; // space it out with nulls
    });
    return buffer;
}
/**
 * Parse file header in a tar archive
 * @param length
 */ function parseHeader(buffer) {
    const data = {};
    let offset = 0;
    ustarStructure.forEach(function(value) {
        const arr = buffer.subarray(offset, offset + value.length);
        data[value.field] = arr;
        offset += value.length;
    });
    return data;
}
/**
 * A class to create a tar archive
 */ export class Tar {
    data;
    constructor(){
        this.data = [];
    }
    /**
   * Append a file to this tar archive
   * @param fn file name
   *                 e.g., test.txt; use slash for directory separators
   * @param opts options
   */ async append(fn, opts) {
        if (typeof fn !== "string") {
            throw new Error("file name not specified");
        }
        let fileName = fn;
        // separate file name into two parts if needed
        let fileNamePrefix;
        if (fileName.length > 100) {
            let i = fileName.length;
            while(i >= 0){
                i = fileName.lastIndexOf("/", i);
                if (i <= 155) {
                    fileNamePrefix = fileName.substr(0, i);
                    fileName = fileName.substr(i + 1);
                    break;
                }
                i--;
            }
            const errMsg = "ustar format does not allow a long file name (length of [file name" + "prefix] + / + [file name] must be shorter than 256 bytes)";
            if (i < 0 || fileName.length > 100) {
                throw new Error(errMsg);
            } else {
                assert(fileNamePrefix != null);
                if (fileNamePrefix.length > 155) {
                    throw new Error(errMsg);
                }
            }
        }
        opts = opts || {};
        // set meta data
        let info;
        if (opts.filePath) {
            info = await Deno.stat(opts.filePath);
            if (info.isDirectory) {
                info.size = 0;
                opts.reader = new Buffer();
            }
        }
        const mode = opts.fileMode || info && info.mode || parseInt("777", 8) & 0xfff, mtime = Math.floor(opts.mtime ?? (info?.mtime ?? new Date()).valueOf() / 1000), uid = opts.uid || 0, gid = opts.gid || 0;
        if (typeof opts.owner === "string" && opts.owner.length >= 32) {
            throw new Error("ustar format does not allow owner name length >= 32 bytes");
        }
        if (typeof opts.group === "string" && opts.group.length >= 32) {
            throw new Error("ustar format does not allow group name length >= 32 bytes");
        }
        const fileSize = info?.size ?? opts.contentSize;
        assert(fileSize != null, "fileSize must be set");
        const type = opts.type ? FileTypes[opts.type] : info?.isDirectory ? FileTypes.directory : FileTypes.file;
        const tarData = {
            fileName,
            fileNamePrefix,
            fileMode: pad(mode, 7),
            uid: pad(uid, 7),
            gid: pad(gid, 7),
            fileSize: pad(fileSize, 11),
            mtime: pad(mtime, 11),
            checksum: "        ",
            type: type.toString(),
            ustar,
            owner: opts.owner || "",
            group: opts.group || "",
            filePath: opts.filePath,
            reader: opts.reader
        };
        // calculate the checksum
        let checksum = 0;
        const encoder = new TextEncoder();
        Object.keys(tarData).filter((key)=>[
                "filePath",
                "reader"
            ].indexOf(key) < 0).forEach(function(key) {
            checksum += encoder.encode(tarData[key]).reduce((p, c)=>p + c, 0);
        });
        tarData.checksum = pad(checksum, 6) + "\u0000 ";
        this.data.push(tarData);
    }
    /**
   * Get a Reader instance for this tar data
   */ getReader() {
        const readers = [];
        this.data.forEach((tarData)=>{
            let { reader  } = tarData;
            const { filePath  } = tarData;
            const headerArr = formatHeader(tarData);
            readers.push(new Buffer(headerArr));
            if (!reader) {
                assert(filePath != null);
                reader = new FileReader(filePath);
            }
            readers.push(reader);
            // to the nearest multiple of recordSize
            assert(tarData.fileSize != null, "fileSize must be set");
            readers.push(new Buffer(clean(recordSize - (parseInt(tarData.fileSize, 8) % recordSize || recordSize))));
        });
        // append 2 empty records
        readers.push(new Buffer(clean(recordSize * 2)));
        return new MultiReader(readers);
    }
}
class TarEntry {
    #header;
    #reader;
    #size;
    #read = 0;
    #consumed = false;
    #entrySize;
    constructor(meta, header, reader){
        Object.assign(this, meta);
        this.#header = header;
        this.#reader = reader;
        // File Size
        this.#size = this.fileSize || 0;
        // Entry Size
        const blocks = Math.ceil(this.#size / recordSize);
        this.#entrySize = blocks * recordSize;
    }
    get consumed() {
        return this.#consumed;
    }
    async read(p) {
        // Bytes left for entry
        const entryBytesLeft = this.#entrySize - this.#read;
        const bufSize = Math.min(// bufSize can't be greater than p.length nor bytes left in the entry
        p.length, entryBytesLeft);
        if (entryBytesLeft <= 0) {
            this.#consumed = true;
            return null;
        }
        const block = new Uint8Array(bufSize);
        const n = await readBlock(this.#reader, block);
        const bytesLeft = this.#size - this.#read;
        this.#read += n || 0;
        if (n === null || bytesLeft <= 0) {
            if (n === null) this.#consumed = true;
            return null;
        }
        // Remove zero filled
        const offset = bytesLeft < n ? bytesLeft : n;
        p.set(block.subarray(0, offset), 0);
        return offset < 0 ? n - Math.abs(offset) : offset;
    }
    async discard() {
        // Discard current entry
        if (this.#consumed) return;
        this.#consumed = true;
        if (typeof this.#reader.seek === "function") {
            await this.#reader.seek(this.#entrySize - this.#read, Deno.SeekMode.Current);
            this.#read = this.#entrySize;
        } else {
            await readAll(this);
        }
    }
}
/**
 * A class to extract a tar archive
 */ export class Untar {
    reader;
    block;
    #entry;
    constructor(reader){
        this.reader = reader;
        this.block = new Uint8Array(recordSize);
    }
    #checksum(header) {
        let sum = initialChecksum;
        for(let i = 0; i < 512; i++){
            if (i >= 148 && i < 156) {
                continue;
            }
            sum += header[i];
        }
        return sum;
    }
    async #getHeader() {
        await readBlock(this.reader, this.block);
        const header1 = parseHeader(this.block);
        // calculate the checksum
        const decoder = new TextDecoder();
        const checksum = this.#checksum(this.block);
        if (parseInt(decoder.decode(header1.checksum), 8) !== checksum) {
            if (checksum === initialChecksum) {
                // EOF
                return null;
            }
            throw new Error("checksum error");
        }
        const magic = decoder.decode(header1.ustar);
        if (magic.indexOf("ustar")) {
            throw new Error(`unsupported archive format: ${magic}`);
        }
        return header1;
    }
    #getMetadata(header2) {
        const decoder1 = new TextDecoder();
        // get meta data
        const meta = {
            fileName: decoder1.decode(trim(header2.fileName))
        };
        const fileNamePrefix = trim(header2.fileNamePrefix);
        if (fileNamePrefix.byteLength > 0) {
            meta.fileName = decoder1.decode(fileNamePrefix) + "/" + meta.fileName;
        }
        [
            "fileMode",
            "mtime",
            "uid",
            "gid"
        ].forEach((key)=>{
            const arr = trim(header2[key]);
            if (arr.byteLength > 0) {
                meta[key] = parseInt(decoder1.decode(arr), 8);
            }
        });
        [
            "owner",
            "group",
            "type"
        ].forEach((key)=>{
            const arr = trim(header2[key]);
            if (arr.byteLength > 0) {
                meta[key] = decoder1.decode(arr);
            }
        });
        meta.fileSize = parseInt(decoder1.decode(header2.fileSize), 8);
        meta.type = FileTypes[parseInt(meta.type)] ?? meta.type;
        return meta;
    }
    async extract() {
        if (this.#entry && !this.#entry.consumed) {
            // If entry body was not read, discard the body
            // so we can read the next entry.
            await this.#entry.discard();
        }
        const header = await this.#getHeader();
        if (header === null) return null;
        const meta = this.#getMetadata(header);
        this.#entry = new TarEntry(meta, header, this.reader);
        return this.#entry;
    }
    async *[Symbol.asyncIterator]() {
        while(true){
            const entry = await this.extract();
            if (entry === null) return;
            yield entry;
        }
    }
}
export { TarEntry };
//# sourceMappingURL=data:application/json;base64,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