// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/**
 * Functions for encoding typed integers in array buffers.
 *
 * @module
 */ import { instantiate } from "./_wasm/lib/deno_std_wasm_varint.generated.mjs";
const U32MAX = 4_294_967_295;
const U64MAX = 18_446_744_073_709_551_615n;
/**
 * Encodes the given `number` into `Uint8Array` with LEB128. The number needs to be in the range of `0` and `0xffffffff`.
 * ```ts
 * import { encodeU32 } from "https://deno.land/std@$STD_VERSION/encoding/varint/mod.ts";
 *
 * const encodedValue = encodeU32(42);
 * // Do something with the encoded value
 * ```
 */ export function encodeU32(val) {
    if (!Number.isInteger(val)) throw new TypeError("Floats are not supported");
    if (val < 0) throw new RangeError("Signed integers are not supported");
    if (val > U32MAX) {
        throw new RangeError(`The given number exceeds the limit of unsigned integer: ${val}`);
    }
    const wasm = instantiate();
    return wasm.encode_u32(val);
}
/**
 * Encodes the given `BigInt` into `Uint8Array` with LEB128. The number needs to be in the range of `0` and `0xffffffffffffffff`.
 * ```ts
 * import { encodeU64 } from "https://deno.land/std@$STD_VERSION/encoding/varint/mod.ts";
 *
 * const encodedValue = encodeU64(42n);
 * // Do something with the encoded value
 * ```
 */ export function encodeU64(val) {
    if (val < 0) throw new RangeError("Signed integers are not supported");
    if (val > U64MAX) {
        throw new RangeError(`The given number exceeds the limit of unsigned long integer: ${val}`);
    }
    const wasm = instantiate();
    return wasm.encode_u64(val);
}
/**
 * Decodes the given `Uint8Array` into a `number` with LEB128.
 * ```ts
 * import { decodeU32 } from "https://deno.land/std@$STD_VERSION/encoding/varint/mod.ts";
 * const bytes = Uint8Array.from([221, 199, 1]);
 * const decodedValue = decodeU32(bytes);
 *
 * // Do something with the decoded value
 * console.log(decodedValue === 25565);
 * ```
 */ export function decodeU32(val) {
    if (val.length > 5) throw RangeError("Too many bytes");
    const wasm = instantiate();
    try {
        return wasm.decode_u32(val);
    } catch  {
        throw new RangeError(`Bad varint: ${val}`);
    }
}
/**
 * Decodes the given `Uint8Array` into a `BigInt` with LEB128.
 * ```ts
 * import { decodeU64 } from "https://deno.land/std@$STD_VERSION/encoding/varint/mod.ts";
 * const bytes = Uint8Array.from([221, 199, 1]);
 * const decodedValue = decodeU64(bytes);
 *
 * // Do something with the decoded value
 * console.log(decodedValue === 25565n);
 * ```
 */ export function decodeU64(val) {
    if (val.length > 10) throw RangeError("Too many bytes");
    const wasm = instantiate();
    try {
        return wasm.decode_u64(val);
    } catch  {
        throw new RangeError(`Bad varint: ${val}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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