// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
import { getLevelByName, LogLevels } from "./levels.ts";
import { blue, bold, red, yellow } from "../fmt/colors.ts";
import { exists, existsSync } from "../fs/exists.ts";
import { BufWriterSync } from "../io/buffer.ts";
const DEFAULT_FORMATTER = "{levelName} {msg}";
export class BaseHandler {
    level;
    levelName;
    formatter;
    constructor(levelName, options = {}){
        this.level = getLevelByName(levelName);
        this.levelName = levelName;
        this.formatter = options.formatter || DEFAULT_FORMATTER;
    }
    handle(logRecord) {
        if (this.level > logRecord.level) return;
        const msg = this.format(logRecord);
        return this.log(msg);
    }
    format(logRecord) {
        if (this.formatter instanceof Function) {
            return this.formatter(logRecord);
        }
        return this.formatter.replace(/{([^\s}]+)}/g, (match, p1)=>{
            const value = logRecord[p1];
            // do not interpolate missing values
            if (value == null) {
                return match;
            }
            return String(value);
        });
    }
    log(_msg) {}
    setup() {}
    destroy() {}
}
export class ConsoleHandler extends BaseHandler {
    format(logRecord) {
        let msg = super.format(logRecord);
        switch(logRecord.level){
            case LogLevels.INFO:
                msg = blue(msg);
                break;
            case LogLevels.WARNING:
                msg = yellow(msg);
                break;
            case LogLevels.ERROR:
                msg = red(msg);
                break;
            case LogLevels.CRITICAL:
                msg = bold(red(msg));
                break;
            default:
                break;
        }
        return msg;
    }
    log(msg) {
        console.log(msg);
    }
}
export class WriterHandler extends BaseHandler {
    _writer;
    #encoder = new TextEncoder();
}
export class FileHandler extends WriterHandler {
    _file;
    _buf;
    _filename;
    _mode;
    _openOptions;
    _encoder = new TextEncoder();
    #unloadCallback = (()=>{
        this.destroy();
    }).bind(this);
    constructor(levelName, options){
        super(levelName, options);
        this._filename = options.filename;
        // default to append mode, write only
        this._mode = options.mode ? options.mode : "a";
        this._openOptions = {
            createNew: this._mode === "x",
            create: this._mode !== "x",
            append: this._mode === "a",
            truncate: this._mode !== "a",
            write: true
        };
    }
    setup() {
        this._file = Deno.openSync(this._filename, this._openOptions);
        this._writer = this._file;
        this._buf = new BufWriterSync(this._file);
        addEventListener("unload", this.#unloadCallback);
    }
    handle(logRecord) {
        super.handle(logRecord);
        // Immediately flush if log level is higher than ERROR
        if (logRecord.level > LogLevels.ERROR) {
            this.flush();
        }
    }
    log(msg) {
        if (this._encoder.encode(msg).byteLength + 1 > this._buf.available()) {
            this.flush();
        }
        this._buf.writeSync(this._encoder.encode(msg + "\n"));
    }
    flush() {
        if (this._buf?.buffered() > 0) {
            this._buf.flush();
        }
    }
    destroy() {
        this.flush();
        this._file?.close();
        this._file = undefined;
        removeEventListener("unload", this.#unloadCallback);
    }
}
export class RotatingFileHandler extends FileHandler {
    #maxBytes;
    #maxBackupCount;
    #currentFileSize = 0;
    constructor(levelName, options){
        super(levelName, options);
        this.#maxBytes = options.maxBytes;
        this.#maxBackupCount = options.maxBackupCount;
    }
    async setup() {
        if (this.#maxBytes < 1) {
            this.destroy();
            throw new Error("maxBytes cannot be less than 1");
        }
        if (this.#maxBackupCount < 1) {
            this.destroy();
            throw new Error("maxBackupCount cannot be less than 1");
        }
        await super.setup();
        if (this._mode === "w") {
            // Remove old backups too as it doesn't make sense to start with a clean
            // log file, but old backups
            for(let i = 1; i <= this.#maxBackupCount; i++){
                try {
                    await Deno.remove(this._filename + "." + i);
                } catch (error) {
                    if (!(error instanceof Deno.errors.NotFound)) {
                        throw error;
                    }
                }
            }
        } else if (this._mode === "x") {
            // Throw if any backups also exist
            for(let i1 = 1; i1 <= this.#maxBackupCount; i1++){
                if (await exists(this._filename + "." + i1)) {
                    this.destroy();
                    throw new Deno.errors.AlreadyExists("Backup log file " + this._filename + "." + i1 + " already exists");
                }
            }
        } else {
            this.#currentFileSize = (await Deno.stat(this._filename)).size;
        }
    }
    log(msg) {
        const msgByteLength = this._encoder.encode(msg).byteLength + 1;
        if (this.#currentFileSize + msgByteLength > this.#maxBytes) {
            this.rotateLogFiles();
            this.#currentFileSize = 0;
        }
        super.log(msg);
        this.#currentFileSize += msgByteLength;
    }
    rotateLogFiles() {
        this._buf.flush();
        this._file.close();
        for(let i = this.#maxBackupCount - 1; i >= 0; i--){
            const source = this._filename + (i === 0 ? "" : "." + i);
            const dest = this._filename + "." + (i + 1);
            if (existsSync(source)) {
                Deno.renameSync(source, dest);
            }
        }
        this._file = Deno.openSync(this._filename, this._openOptions);
        this._writer = this._file;
        this._buf = new BufWriterSync(this._file);
    }
}
//# sourceMappingURL=data:application/json;base64,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