// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/**
 * Logging library with the support for terminal and file outputs. Also provides
 * interfaces for building custom loggers.
 *
 * @module
 */ import { Logger } from "./logger.ts";
import { BaseHandler, ConsoleHandler, FileHandler, RotatingFileHandler, WriterHandler } from "./handlers.ts";
import { assert } from "../_util/asserts.ts";
export { LogLevels } from "./levels.ts";
export { Logger } from "./logger.ts";
export class LoggerConfig {
    level;
    handlers;
}
const DEFAULT_LEVEL = "INFO";
const DEFAULT_CONFIG = {
    handlers: {
        default: new ConsoleHandler(DEFAULT_LEVEL)
    },
    loggers: {
        default: {
            level: DEFAULT_LEVEL,
            handlers: [
                "default"
            ]
        }
    }
};
const state = {
    handlers: new Map(),
    loggers: new Map(),
    config: DEFAULT_CONFIG
};
export const handlers = {
    BaseHandler,
    ConsoleHandler,
    WriterHandler,
    FileHandler,
    RotatingFileHandler
};
/** Get a logger instance. If not specified `name`, get the default logger.  */ export function getLogger(name) {
    if (!name) {
        const d = state.loggers.get("default");
        assert(d != null, `"default" logger must be set for getting logger without name`);
        return d;
    }
    const result = state.loggers.get(name);
    if (!result) {
        const logger = new Logger(name, "NOTSET", {
            handlers: []
        });
        state.loggers.set(name, logger);
        return logger;
    }
    return result;
}
export function debug(msg, ...args) {
    // Assist TS compiler with pass-through generic type
    if (msg instanceof Function) {
        return getLogger("default").debug(msg, ...args);
    }
    return getLogger("default").debug(msg, ...args);
}
export function info(msg, ...args) {
    // Assist TS compiler with pass-through generic type
    if (msg instanceof Function) {
        return getLogger("default").info(msg, ...args);
    }
    return getLogger("default").info(msg, ...args);
}
export function warning(msg, ...args) {
    // Assist TS compiler with pass-through generic type
    if (msg instanceof Function) {
        return getLogger("default").warning(msg, ...args);
    }
    return getLogger("default").warning(msg, ...args);
}
export function error(msg, ...args) {
    // Assist TS compiler with pass-through generic type
    if (msg instanceof Function) {
        return getLogger("default").error(msg, ...args);
    }
    return getLogger("default").error(msg, ...args);
}
export function critical(msg, ...args) {
    // Assist TS compiler with pass-through generic type
    if (msg instanceof Function) {
        return getLogger("default").critical(msg, ...args);
    }
    return getLogger("default").critical(msg, ...args);
}
/** Setup logger config. */ export function setup(config) {
    state.config = {
        handlers: {
            ...DEFAULT_CONFIG.handlers,
            ...config.handlers
        },
        loggers: {
            ...DEFAULT_CONFIG.loggers,
            ...config.loggers
        }
    };
    // tear down existing handlers
    state.handlers.forEach((handler)=>{
        handler.destroy();
    });
    state.handlers.clear();
    // setup handlers
    const handlers = state.config.handlers || {};
    for(const handlerName in handlers){
        const handler = handlers[handlerName];
        handler.setup();
        state.handlers.set(handlerName, handler);
    }
    // remove existing loggers
    state.loggers.clear();
    // setup loggers
    const loggers = state.config.loggers || {};
    for(const loggerName in loggers){
        const loggerConfig = loggers[loggerName];
        const handlerNames = loggerConfig.handlers || [];
        const handlers1 = [];
        handlerNames.forEach((handlerName)=>{
            const handler = state.handlers.get(handlerName);
            if (handler) {
                handlers1.push(handler);
            }
        });
        const levelName = loggerConfig.level || DEFAULT_LEVEL;
        const logger = new Logger(loggerName, levelName, {
            handlers: handlers1
        });
        state.loggers.set(loggerName, logger);
    }
}
setup(DEFAULT_CONFIG);
//# sourceMappingURL=data:application/json;base64,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