// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { deepMerge } from "../../collections/deep_merge.ts";
export class TOMLParseError extends Error {
}
export class Scanner {
    #whitespace;
    #position;
    constructor(source){
        this.source = source;
        this.#whitespace = /[ \t]/;
        this.#position = 0;
    }
    /**
   * Get current character
   * @param index - relative index from current position
   */ char(index = 0) {
        return this.source[this.#position + index] ?? "";
    }
    /**
   * Get sliced string
   * @param start - start position relative from current position
   * @param end - end position relative from current position
   */ slice(start, end) {
        return this.source.slice(this.#position + start, this.#position + end);
    }
    /**
   * Move position to next
   */ next(count) {
        if (typeof count === "number") {
            for(let i = 0; i < count; i++){
                this.#position++;
            }
        } else {
            this.#position++;
        }
    }
    /**
   * Move position until current char is not a whitespace, EOL, or comment.
   * @param options.inline - skip only whitespaces
   */ nextUntilChar(options = {
        comment: true
    }) {
        if (options.inline) {
            while(this.#whitespace.test(this.char()) && !this.eof()){
                this.next();
            }
        } else {
            while(!this.eof()){
                const char = this.char();
                if (this.#whitespace.test(char) || this.isCurrentCharEOL()) {
                    this.next();
                } else if (options.comment && this.char() === "#") {
                    // entering comment
                    while(!this.isCurrentCharEOL() && !this.eof()){
                        this.next();
                    }
                } else {
                    break;
                }
            }
        }
        // Invalid if current char is other kinds of whitespace
        if (!this.isCurrentCharEOL() && /\s/.test(this.char())) {
            const escaped = "\\u" + this.char().charCodeAt(0).toString(16);
            throw new TOMLParseError(`Contains invalid whitespaces: \`${escaped}\``);
        }
    }
    /**
   * Position reached EOF or not
   */ eof() {
        return this.position() >= this.source.length;
    }
    /**
   * Get current position
   */ position() {
        return this.#position;
    }
    isCurrentCharEOL() {
        return this.char() === "\n" || this.slice(0, 2) === "\r\n";
    }
    source;
}
// -----------------------
// Utilities
// -----------------------
function success(body) {
    return {
        ok: true,
        body
    };
}
function failure() {
    return {
        ok: false
    };
}
export const Utils = {
    unflat (keys, values = {}, cObj) {
        const out = {};
        if (keys.length === 0) {
            return cObj;
        } else {
            if (!cObj) {
                cObj = values;
            }
            const key = keys[keys.length - 1];
            if (typeof key === "string") {
                out[key] = cObj;
            }
            return this.unflat(keys.slice(0, -1), values, out);
        }
    },
    deepAssignWithTable (target, table) {
        if (table.key.length === 0) {
            throw new Error("Unexpected key length");
        }
        const value = target[table.key[0]];
        if (typeof value === "undefined") {
            Object.assign(target, this.unflat(table.key, table.type === "Table" ? table.value : [
                table.value
            ]));
        } else if (Array.isArray(value)) {
            if (table.type === "TableArray" && table.key.length === 1) {
                value.push(table.value);
            } else {
                const last = value[value.length - 1];
                Utils.deepAssignWithTable(last, {
                    type: table.type,
                    key: table.key.slice(1),
                    value: table.value
                });
            }
        } else if (typeof value === "object" && value !== null) {
            Utils.deepAssignWithTable(value, {
                type: table.type,
                key: table.key.slice(1),
                value: table.value
            });
        } else {
            throw new Error("Unexpected assign");
        }
    }
};
// ---------------------------------
// Parser combinators and generators
// ---------------------------------
function or(parsers) {
    return function Or(scanner) {
        for (const parse of parsers){
            const result = parse(scanner);
            if (result.ok) {
                return result;
            }
        }
        return failure();
    };
}
function join(parser, separator) {
    const Separator = character(separator);
    return function Join(scanner) {
        const first = parser(scanner);
        if (!first.ok) {
            return failure();
        }
        const out = [
            first.body
        ];
        while(!scanner.eof()){
            if (!Separator(scanner).ok) {
                break;
            }
            const result = parser(scanner);
            if (result.ok) {
                out.push(result.body);
            } else {
                throw new TOMLParseError(`Invalid token after "${separator}"`);
            }
        }
        return success(out);
    };
}
function kv(keyParser, separator, valueParser) {
    const Separator = character(separator);
    return function Kv(scanner) {
        const key = keyParser(scanner);
        if (!key.ok) {
            return failure();
        }
        const sep = Separator(scanner);
        if (!sep.ok) {
            throw new TOMLParseError(`key/value pair doesn't have "${separator}"`);
        }
        const value = valueParser(scanner);
        if (!value.ok) {
            throw new TOMLParseError(`Value of key/value pair is invalid data format`);
        }
        return success(Utils.unflat(key.body, value.body));
    };
}
function merge(parser) {
    return function Merge(scanner) {
        const result = parser(scanner);
        if (!result.ok) {
            return failure();
        }
        let body = {};
        for (const record of result.body){
            if (typeof body === "object" && body !== null) {
                // deno-lint-ignore no-explicit-any
                body = deepMerge(body, record);
            }
        }
        return success(body);
    };
}
function repeat(parser) {
    return function Repeat(scanner) {
        const body = [];
        while(!scanner.eof()){
            const result = parser(scanner);
            if (result.ok) {
                body.push(result.body);
            } else {
                break;
            }
            scanner.nextUntilChar();
        }
        if (body.length === 0) {
            return failure();
        }
        return success(body);
    };
}
function surround(left, parser, right) {
    const Left = character(left);
    const Right = character(right);
    return function Surround(scanner) {
        if (!Left(scanner).ok) {
            return failure();
        }
        const result = parser(scanner);
        if (!result.ok) {
            throw new TOMLParseError(`Invalid token after "${left}"`);
        }
        if (!Right(scanner).ok) {
            throw new TOMLParseError(`Not closed by "${right}" after started with "${left}"`);
        }
        return success(result.body);
    };
}
function character(str) {
    return function character(scanner) {
        scanner.nextUntilChar({
            inline: true
        });
        if (scanner.slice(0, str.length) === str) {
            scanner.next(str.length);
        } else {
            return failure();
        }
        scanner.nextUntilChar({
            inline: true
        });
        return success(undefined);
    };
}
// -----------------------
// Parser components
// -----------------------
const Patterns = {
    BARE_KEY: /[A-Za-z0-9_-]/,
    FLOAT: /[0-9_\.e+\-]/i,
    END_OF_VALUE: /[ \t\r\n#,}]/
};
export function BareKey(scanner) {
    scanner.nextUntilChar({
        inline: true
    });
    if (!scanner.char() || !Patterns.BARE_KEY.test(scanner.char())) {
        return failure();
    }
    const acc = [];
    while(scanner.char() && Patterns.BARE_KEY.test(scanner.char())){
        acc.push(scanner.char());
        scanner.next();
    }
    const key = acc.join("");
    return success(key);
}
function EscapeSequence(scanner) {
    if (scanner.char() === "\\") {
        scanner.next();
        // See https://toml.io/en/v1.0.0-rc.3#string
        switch(scanner.char()){
            case "b":
                scanner.next();
                return success("\b");
            case "t":
                scanner.next();
                return success("\t");
            case "n":
                scanner.next();
                return success("\n");
            case "f":
                scanner.next();
                return success("\f");
            case "r":
                scanner.next();
                return success("\r");
            case "u":
            case "U":
                {
                    // Unicode character
                    const codePointLen = scanner.char() === "u" ? 4 : 6;
                    const codePoint = parseInt("0x" + scanner.slice(1, 1 + codePointLen), 16);
                    const str = String.fromCodePoint(codePoint);
                    scanner.next(codePointLen + 1);
                    return success(str);
                }
            case '"':
                scanner.next();
                return success('"');
            case "\\":
                scanner.next();
                return success("\\");
            default:
                scanner.next();
                return success(scanner.char());
        }
    } else {
        return failure();
    }
}
export function BasicString(scanner) {
    scanner.nextUntilChar({
        inline: true
    });
    if (scanner.char() === '"') {
        scanner.next();
    } else {
        return failure();
    }
    const acc = [];
    while(scanner.char() !== '"' && !scanner.eof()){
        if (scanner.char() === "\n") {
            throw new TOMLParseError("Single-line string cannot contain EOL");
        }
        const escapedChar = EscapeSequence(scanner);
        if (escapedChar.ok) {
            acc.push(escapedChar.body);
        } else {
            acc.push(scanner.char());
            scanner.next();
        }
    }
    if (scanner.eof()) {
        throw new TOMLParseError(`Single-line string is not closed:\n${acc.join("")}`);
    }
    scanner.next(); // skip last '""
    return success(acc.join(""));
}
export function LiteralString(scanner) {
    scanner.nextUntilChar({
        inline: true
    });
    if (scanner.char() === "'") {
        scanner.next();
    } else {
        return failure();
    }
    const acc = [];
    while(scanner.char() !== "'" && !scanner.eof()){
        if (scanner.char() === "\n") {
            throw new TOMLParseError("Single-line string cannot contain EOL");
        }
        acc.push(scanner.char());
        scanner.next();
    }
    if (scanner.eof()) {
        throw new TOMLParseError(`Single-line string is not closed:\n${acc.join("")}`);
    }
    scanner.next(); // skip last "'"
    return success(acc.join(""));
}
export function MultilineBasicString(scanner) {
    scanner.nextUntilChar({
        inline: true
    });
    if (scanner.slice(0, 3) === '"""') {
        scanner.next(3);
    } else {
        return failure();
    }
    if (scanner.char() === "\n") {
        // The first newline is trimmed
        scanner.next();
    }
    const acc = [];
    while(scanner.slice(0, 3) !== '"""' && !scanner.eof()){
        // line ending backslash
        if (scanner.slice(0, 2) === "\\\n") {
            scanner.next();
            scanner.nextUntilChar({
                comment: false
            });
            continue;
        }
        const escapedChar = EscapeSequence(scanner);
        if (escapedChar.ok) {
            acc.push(escapedChar.body);
        } else {
            acc.push(scanner.char());
            scanner.next();
        }
    }
    if (scanner.eof()) {
        throw new TOMLParseError(`Multi-line string is not closed:\n${acc.join("")}`);
    }
    // if ends with 4 `"`, push the fist `"` to string
    if (scanner.char(3) === '"') {
        acc.push('"');
        scanner.next();
    }
    scanner.next(3); // skip last '""""
    return success(acc.join(""));
}
export function MultilineLiteralString(scanner) {
    scanner.nextUntilChar({
        inline: true
    });
    if (scanner.slice(0, 3) === "'''") {
        scanner.next(3);
    } else {
        return failure();
    }
    if (scanner.char() === "\n") {
        // The first newline is trimmed
        scanner.next();
    }
    const acc = [];
    while(scanner.slice(0, 3) !== "'''" && !scanner.eof()){
        acc.push(scanner.char());
        scanner.next();
    }
    if (scanner.eof()) {
        throw new TOMLParseError(`Multi-line string is not closed:\n${acc.join("")}`);
    }
    // if ends with 4 `'`, push the fist `'` to string
    if (scanner.char(3) === "'") {
        acc.push("'");
        scanner.next();
    }
    scanner.next(3); // skip last "'''"
    return success(acc.join(""));
}
const symbolPairs = [
    [
        "true",
        true
    ],
    [
        "false",
        false
    ],
    [
        "inf",
        Infinity
    ],
    [
        "+inf",
        Infinity
    ],
    [
        "-inf",
        -Infinity
    ],
    [
        "nan",
        NaN
    ],
    [
        "+nan",
        NaN
    ],
    [
        "-nan",
        NaN
    ]
];
export function Symbols(scanner) {
    scanner.nextUntilChar({
        inline: true
    });
    const found = symbolPairs.find(([str])=>scanner.slice(0, str.length) === str);
    if (!found) {
        return failure();
    }
    const [str, value] = found;
    scanner.next(str.length);
    return success(value);
}
export const DottedKey = join(or([
    BareKey,
    BasicString,
    LiteralString
]), ".");
export function Integer(scanner) {
    scanner.nextUntilChar({
        inline: true
    });
    // If binary / octal / hex
    const first2 = scanner.slice(0, 2);
    if (first2.length === 2 && /0(?:x|o|b)/i.test(first2)) {
        scanner.next(2);
        const acc = [
            first2
        ];
        while(/[0-9a-f_]/i.test(scanner.char()) && !scanner.eof()){
            acc.push(scanner.char());
            scanner.next();
        }
        if (acc.length === 1) {
            return failure();
        }
        return success(acc.join(""));
    }
    const acc1 = [];
    if (/[+-]/.test(scanner.char())) {
        acc1.push(scanner.char());
        scanner.next();
    }
    while(/[0-9_]/.test(scanner.char()) && !scanner.eof()){
        acc1.push(scanner.char());
        scanner.next();
    }
    if (acc1.length === 0 || acc1.length === 1 && /[+-]/.test(acc1[0])) {
        return failure();
    }
    const int = parseInt(acc1.filter((char)=>char !== "_").join(""));
    return success(int);
}
export function Float(scanner) {
    scanner.nextUntilChar({
        inline: true
    });
    // lookahead validation is needed for integer value is similar to float
    let position = 0;
    while(scanner.char(position) && !Patterns.END_OF_VALUE.test(scanner.char(position))){
        if (!Patterns.FLOAT.test(scanner.char(position))) {
            return failure();
        }
        position++;
    }
    const acc = [];
    if (/[+-]/.test(scanner.char())) {
        acc.push(scanner.char());
        scanner.next();
    }
    while(Patterns.FLOAT.test(scanner.char()) && !scanner.eof()){
        acc.push(scanner.char());
        scanner.next();
    }
    if (acc.length === 0) {
        return failure();
    }
    const float = parseFloat(acc.filter((char)=>char !== "_").join(""));
    if (isNaN(float)) {
        return failure();
    }
    return success(float);
}
export function DateTime(scanner) {
    scanner.nextUntilChar({
        inline: true
    });
    let dateStr = scanner.slice(0, 10);
    // example: 1979-05-27
    if (/^\d{4}-\d{2}-\d{2}/.test(dateStr)) {
        scanner.next(10);
    } else {
        return failure();
    }
    const acc = [];
    // example: 1979-05-27T00:32:00Z
    while(/[ 0-9TZ.:-]/.test(scanner.char()) && !scanner.eof()){
        acc.push(scanner.char());
        scanner.next();
    }
    dateStr += acc.join("");
    const date = new Date(dateStr.trim());
    // invalid date
    if (isNaN(date.getTime())) {
        throw new TOMLParseError(`Invalid date string "${dateStr}"`);
    }
    return success(date);
}
export function LocalTime(scanner) {
    scanner.nextUntilChar({
        inline: true
    });
    let timeStr = scanner.slice(0, 8);
    if (/^(\d{2}):(\d{2}):(\d{2})/.test(timeStr)) {
        scanner.next(8);
    } else {
        return failure();
    }
    const acc = [];
    if (scanner.char() === ".") {
        acc.push(scanner.char());
        scanner.next();
    } else {
        return success(timeStr);
    }
    while(/[0-9]/.test(scanner.char()) && !scanner.eof()){
        acc.push(scanner.char());
        scanner.next();
    }
    timeStr += acc.join("");
    return success(timeStr);
}
export function ArrayValue(scanner) {
    scanner.nextUntilChar({
        inline: true
    });
    if (scanner.char() === "[") {
        scanner.next();
    } else {
        return failure();
    }
    const array = [];
    while(!scanner.eof()){
        scanner.nextUntilChar();
        const result = Value(scanner);
        if (result.ok) {
            array.push(result.body);
        } else {
            break;
        }
        scanner.nextUntilChar({
            inline: true
        });
        // may have a next item, but trailing comma is allowed at array
        if (scanner.char() === ",") {
            scanner.next();
        } else {
            break;
        }
    }
    scanner.nextUntilChar();
    if (scanner.char() === "]") {
        scanner.next();
    } else {
        throw new TOMLParseError("Array is not closed");
    }
    return success(array);
}
export function InlineTable(scanner) {
    scanner.nextUntilChar();
    const pairs = surround("{", join(Pair, ","), "}")(scanner);
    if (!pairs.ok) {
        return failure();
    }
    let table = {};
    for (const pair of pairs.body){
        table = deepMerge(table, pair);
    }
    return success(table);
}
export const Value = or([
    MultilineBasicString,
    MultilineLiteralString,
    BasicString,
    LiteralString,
    Symbols,
    DateTime,
    LocalTime,
    Float,
    Integer,
    ArrayValue,
    InlineTable
]);
export const Pair = kv(DottedKey, "=", Value);
export function Block(scanner) {
    scanner.nextUntilChar();
    const result = merge(repeat(Pair))(scanner);
    if (result.ok) {
        return success({
            type: "Block",
            value: result.body
        });
    } else {
        return failure();
    }
}
export const TableHeader = surround("[", DottedKey, "]");
export function Table(scanner) {
    scanner.nextUntilChar();
    const header = TableHeader(scanner);
    if (!header.ok) {
        return failure();
    }
    scanner.nextUntilChar();
    const block = Block(scanner);
    return success({
        type: "Table",
        key: header.body,
        value: block.ok ? block.body.value : {}
    });
}
export const TableArrayHeader = surround("[[", DottedKey, "]]");
export function TableArray(scanner) {
    scanner.nextUntilChar();
    const header = TableArrayHeader(scanner);
    if (!header.ok) {
        return failure();
    }
    scanner.nextUntilChar();
    const block = Block(scanner);
    return success({
        type: "TableArray",
        key: header.body,
        value: block.ok ? block.body.value : {}
    });
}
export function Toml(scanner) {
    const blocks = repeat(or([
        Block,
        TableArray,
        Table
    ]))(scanner);
    if (!blocks.ok) {
        return failure();
    }
    let body = {};
    for (const block of blocks.body){
        switch(block.type){
            case "Block":
                {
                    body = deepMerge(body, block.value);
                    break;
                }
            case "Table":
                {
                    Utils.deepAssignWithTable(body, block);
                    break;
                }
            case "TableArray":
                {
                    Utils.deepAssignWithTable(body, block);
                    break;
                }
        }
    }
    return success(body);
}
export function ParserFactory(parser) {
    return function parse(tomlString) {
        const scanner = new Scanner(tomlString);
        let parsed = null;
        let err = null;
        try {
            parsed = parser(scanner);
        } catch (e) {
            err = e instanceof Error ? e : new Error("[non-error thrown]");
        }
        if (err || !parsed || !parsed.ok || !scanner.eof()) {
            const position = scanner.position();
            const subStr = tomlString.slice(0, position);
            const lines = subStr.split("\n");
            const row = lines.length;
            const column = (()=>{
                let count = subStr.length;
                for (const line of lines){
                    if (count > line.length) {
                        count -= line.length + 1;
                    } else {
                        return count;
                    }
                }
                return count;
            })();
            const message = `Parse error on line ${row}, column ${column}: ${err ? err.message : `Unexpected character: "${scanner.char()}"`}`;
            throw new TOMLParseError(message);
        }
        return parsed.body;
    };
}
/**
 * Parse parses TOML string into an object.
 * @param tomlString
 */ export const parse = ParserFactory(Toml);
//# sourceMappingURL=data:application/json;base64,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