// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Contains the enum {@linkcode Status} which enumerates standard HTTP status
 * codes and provides several type guards for handling status codes with type
 * safety.
 *
 * ### Examples
 *
 * ```ts
 * import {
 *   Status,
 *   STATUS_TEXT,
 * } from "https://deno.land/std@$STD_VERSION/http/http_status.ts";
 *
 * console.log(Status.NotFound); //=> 404
 * console.log(STATUS_TEXT[Status.NotFound]); //=> "Not Found"
 * ```
 *
 * ```ts
 * import { isErrorStatus } from "https://deno.land/std@$STD_VERSION/http/http_status.ts";
 *
 * const res = await fetch("https://example.com/");
 *
 * if (isErrorStatus(res.status)) {
 *   // error handling here...
 * }
 * ```
 *
 * @module
 */ /** Standard HTTP status codes. */ export var Status;
(function(Status) {
    Status[Status[/** RFC 7231, 6.2.1 */ "Continue"] = 100] = "Continue";
    Status[Status[/** RFC 7231, 6.2.2 */ "SwitchingProtocols"] = 101] = "SwitchingProtocols";
    Status[Status[/** RFC 2518, 10.1 */ "Processing"] = 102] = "Processing";
    Status[Status[/** RFC 8297 **/ "EarlyHints"] = 103] = "EarlyHints";
    Status[Status[/** RFC 7231, 6.3.1 */ "OK"] = 200] = "OK";
    Status[Status[/** RFC 7231, 6.3.2 */ "Created"] = 201] = "Created";
    Status[Status[/** RFC 7231, 6.3.3 */ "Accepted"] = 202] = "Accepted";
    Status[Status[/** RFC 7231, 6.3.4 */ "NonAuthoritativeInfo"] = 203] = "NonAuthoritativeInfo";
    Status[Status[/** RFC 7231, 6.3.5 */ "NoContent"] = 204] = "NoContent";
    Status[Status[/** RFC 7231, 6.3.6 */ "ResetContent"] = 205] = "ResetContent";
    Status[Status[/** RFC 7233, 4.1 */ "PartialContent"] = 206] = "PartialContent";
    Status[Status[/** RFC 4918, 11.1 */ "MultiStatus"] = 207] = "MultiStatus";
    Status[Status[/** RFC 5842, 7.1 */ "AlreadyReported"] = 208] = "AlreadyReported";
    Status[Status[/** RFC 3229, 10.4.1 */ "IMUsed"] = 226] = "IMUsed";
    Status[Status[/** RFC 7231, 6.4.1 */ "MultipleChoices"] = 300] = "MultipleChoices";
    Status[Status[/** RFC 7231, 6.4.2 */ "MovedPermanently"] = 301] = "MovedPermanently";
    Status[Status[/** RFC 7231, 6.4.3 */ "Found"] = 302] = "Found";
    Status[Status[/** RFC 7231, 6.4.4 */ "SeeOther"] = 303] = "SeeOther";
    Status[Status[/** RFC 7232, 4.1 */ "NotModified"] = 304] = "NotModified";
    Status[Status[/** RFC 7231, 6.4.5 */ "UseProxy"] = 305] = "UseProxy";
    Status[Status[/** RFC 7231, 6.4.7 */ "TemporaryRedirect"] = 307] = "TemporaryRedirect";
    Status[Status[/** RFC 7538, 3 */ "PermanentRedirect"] = 308] = "PermanentRedirect";
    Status[Status[/** RFC 7231, 6.5.1 */ "BadRequest"] = 400] = "BadRequest";
    Status[Status[/** RFC 7235, 3.1 */ "Unauthorized"] = 401] = "Unauthorized";
    Status[Status[/** RFC 7231, 6.5.2 */ "PaymentRequired"] = 402] = "PaymentRequired";
    Status[Status[/** RFC 7231, 6.5.3 */ "Forbidden"] = 403] = "Forbidden";
    Status[Status[/** RFC 7231, 6.5.4 */ "NotFound"] = 404] = "NotFound";
    Status[Status[/** RFC 7231, 6.5.5 */ "MethodNotAllowed"] = 405] = "MethodNotAllowed";
    Status[Status[/** RFC 7231, 6.5.6 */ "NotAcceptable"] = 406] = "NotAcceptable";
    Status[Status[/** RFC 7235, 3.2 */ "ProxyAuthRequired"] = 407] = "ProxyAuthRequired";
    Status[Status[/** RFC 7231, 6.5.7 */ "RequestTimeout"] = 408] = "RequestTimeout";
    Status[Status[/** RFC 7231, 6.5.8 */ "Conflict"] = 409] = "Conflict";
    Status[Status[/** RFC 7231, 6.5.9 */ "Gone"] = 410] = "Gone";
    Status[Status[/** RFC 7231, 6.5.10 */ "LengthRequired"] = 411] = "LengthRequired";
    Status[Status[/** RFC 7232, 4.2 */ "PreconditionFailed"] = 412] = "PreconditionFailed";
    Status[Status[/** RFC 7231, 6.5.11 */ "RequestEntityTooLarge"] = 413] = "RequestEntityTooLarge";
    Status[Status[/** RFC 7231, 6.5.12 */ "RequestURITooLong"] = 414] = "RequestURITooLong";
    Status[Status[/** RFC 7231, 6.5.13 */ "UnsupportedMediaType"] = 415] = "UnsupportedMediaType";
    Status[Status[/** RFC 7233, 4.4 */ "RequestedRangeNotSatisfiable"] = 416] = "RequestedRangeNotSatisfiable";
    Status[Status[/** RFC 7231, 6.5.14 */ "ExpectationFailed"] = 417] = "ExpectationFailed";
    Status[Status[/** RFC 7168, 2.3.3 */ "Teapot"] = 418] = "Teapot";
    Status[Status[/** RFC 7540, 9.1.2 */ "MisdirectedRequest"] = 421] = "MisdirectedRequest";
    Status[Status[/** RFC 4918, 11.2 */ "UnprocessableEntity"] = 422] = "UnprocessableEntity";
    Status[Status[/** RFC 4918, 11.3 */ "Locked"] = 423] = "Locked";
    Status[Status[/** RFC 4918, 11.4 */ "FailedDependency"] = 424] = "FailedDependency";
    Status[Status[/** RFC 8470, 5.2 */ "TooEarly"] = 425] = "TooEarly";
    Status[Status[/** RFC 7231, 6.5.15 */ "UpgradeRequired"] = 426] = "UpgradeRequired";
    Status[Status[/** RFC 6585, 3 */ "PreconditionRequired"] = 428] = "PreconditionRequired";
    Status[Status[/** RFC 6585, 4 */ "TooManyRequests"] = 429] = "TooManyRequests";
    Status[Status[/** RFC 6585, 5 */ "RequestHeaderFieldsTooLarge"] = 431] = "RequestHeaderFieldsTooLarge";
    Status[Status[/** RFC 7725, 3 */ "UnavailableForLegalReasons"] = 451] = "UnavailableForLegalReasons";
    Status[Status[/** RFC 7231, 6.6.1 */ "InternalServerError"] = 500] = "InternalServerError";
    Status[Status[/** RFC 7231, 6.6.2 */ "NotImplemented"] = 501] = "NotImplemented";
    Status[Status[/** RFC 7231, 6.6.3 */ "BadGateway"] = 502] = "BadGateway";
    Status[Status[/** RFC 7231, 6.6.4 */ "ServiceUnavailable"] = 503] = "ServiceUnavailable";
    Status[Status[/** RFC 7231, 6.6.5 */ "GatewayTimeout"] = 504] = "GatewayTimeout";
    Status[Status[/** RFC 7231, 6.6.6 */ "HTTPVersionNotSupported"] = 505] = "HTTPVersionNotSupported";
    Status[Status[/** RFC 2295, 8.1 */ "VariantAlsoNegotiates"] = 506] = "VariantAlsoNegotiates";
    Status[Status[/** RFC 4918, 11.5 */ "InsufficientStorage"] = 507] = "InsufficientStorage";
    Status[Status[/** RFC 5842, 7.2 */ "LoopDetected"] = 508] = "LoopDetected";
    Status[Status[/** RFC 2774, 7 */ "NotExtended"] = 510] = "NotExtended";
    Status[Status[/** RFC 6585, 6 */ "NetworkAuthenticationRequired"] = 511] = "NetworkAuthenticationRequired";
})(Status || (Status = {}));
/** A record of all the status codes text. */ export const STATUS_TEXT = {
    [Status.Accepted]: "Accepted",
    [Status.AlreadyReported]: "Already Reported",
    [Status.BadGateway]: "Bad Gateway",
    [Status.BadRequest]: "Bad Request",
    [Status.Conflict]: "Conflict",
    [Status.Continue]: "Continue",
    [Status.Created]: "Created",
    [Status.EarlyHints]: "Early Hints",
    [Status.ExpectationFailed]: "Expectation Failed",
    [Status.FailedDependency]: "Failed Dependency",
    [Status.Forbidden]: "Forbidden",
    [Status.Found]: "Found",
    [Status.GatewayTimeout]: "Gateway Timeout",
    [Status.Gone]: "Gone",
    [Status.HTTPVersionNotSupported]: "HTTP Version Not Supported",
    [Status.IMUsed]: "IM Used",
    [Status.InsufficientStorage]: "Insufficient Storage",
    [Status.InternalServerError]: "Internal Server Error",
    [Status.LengthRequired]: "Length Required",
    [Status.Locked]: "Locked",
    [Status.LoopDetected]: "Loop Detected",
    [Status.MethodNotAllowed]: "Method Not Allowed",
    [Status.MisdirectedRequest]: "Misdirected Request",
    [Status.MovedPermanently]: "Moved Permanently",
    [Status.MultiStatus]: "Multi Status",
    [Status.MultipleChoices]: "Multiple Choices",
    [Status.NetworkAuthenticationRequired]: "Network Authentication Required",
    [Status.NoContent]: "No Content",
    [Status.NonAuthoritativeInfo]: "Non Authoritative Info",
    [Status.NotAcceptable]: "Not Acceptable",
    [Status.NotExtended]: "Not Extended",
    [Status.NotFound]: "Not Found",
    [Status.NotImplemented]: "Not Implemented",
    [Status.NotModified]: "Not Modified",
    [Status.OK]: "OK",
    [Status.PartialContent]: "Partial Content",
    [Status.PaymentRequired]: "Payment Required",
    [Status.PermanentRedirect]: "Permanent Redirect",
    [Status.PreconditionFailed]: "Precondition Failed",
    [Status.PreconditionRequired]: "Precondition Required",
    [Status.Processing]: "Processing",
    [Status.ProxyAuthRequired]: "Proxy Auth Required",
    [Status.RequestEntityTooLarge]: "Request Entity Too Large",
    [Status.RequestHeaderFieldsTooLarge]: "Request Header Fields Too Large",
    [Status.RequestTimeout]: "Request Timeout",
    [Status.RequestURITooLong]: "Request URI Too Long",
    [Status.RequestedRangeNotSatisfiable]: "Requested Range Not Satisfiable",
    [Status.ResetContent]: "Reset Content",
    [Status.SeeOther]: "See Other",
    [Status.ServiceUnavailable]: "Service Unavailable",
    [Status.SwitchingProtocols]: "Switching Protocols",
    [Status.Teapot]: "I'm a teapot",
    [Status.TemporaryRedirect]: "Temporary Redirect",
    [Status.TooEarly]: "Too Early",
    [Status.TooManyRequests]: "Too Many Requests",
    [Status.Unauthorized]: "Unauthorized",
    [Status.UnavailableForLegalReasons]: "Unavailable For Legal Reasons",
    [Status.UnprocessableEntity]: "Unprocessable Entity",
    [Status.UnsupportedMediaType]: "Unsupported Media Type",
    [Status.UpgradeRequired]: "Upgrade Required",
    [Status.UseProxy]: "Use Proxy",
    [Status.VariantAlsoNegotiates]: "Variant Also Negotiates"
};
/** A type guard that determines if the status code is informational. */ export function isInformationalStatus(status) {
    return status >= 100 && status < 200;
}
/** A type guard that determines if the status code is successful. */ export function isSuccessfulStatus(status) {
    return status >= 200 && status < 300;
}
/** A type guard that determines if the status code is a redirection. */ export function isRedirectStatus(status) {
    return status >= 300 && status < 400;
}
/** A type guard that determines if the status code is a client error. */ export function isClientErrorStatus(status) {
    return status >= 400 && status < 500;
}
/** A type guard that determines if the status code is a server error. */ export function isServerErrorStatus(status) {
    return status >= 500 && status < 600;
}
/** A type guard that determines if the status code is an error. */ export function isErrorStatus(status) {
    return status >= 400 && status < 600;
}
//# sourceMappingURL=data:application/json;base64,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