// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { BytesList } from "../bytes/bytes_list.ts";
/** Transform a stream into a stream where each chunk is divided by a newline,
 * be it `\n` or `\r\n`. `\r` can be enabled via the `allowCR` option.
 *
 * ```ts
 * import { TextLineStream } from "https://deno.land/std@$STD_VERSION/streams/delimiter.ts";
 * const res = await fetch("https://example.com");
 * const lines = res.body!
 *   .pipeThrough(new TextDecoderStream())
 *   .pipeThrough(new TextLineStream());
 * ```
 */ export class TextLineStream extends TransformStream {
    #allowCR;
    #buf = "";
    constructor(options){
        super({
            transform: (chunk, controller)=>this.#handle(chunk, controller),
            flush: (controller)=>this.#handle("\r\n", controller)
        });
        this.#allowCR = options?.allowCR ?? false;
    }
    #handle(chunk, controller) {
        chunk = this.#buf + chunk;
        for(;;){
            const lfIndex = chunk.indexOf("\n");
            if (this.#allowCR) {
                const crIndex = chunk.indexOf("\r");
                if (crIndex !== -1 && crIndex !== chunk.length - 1 && (lfIndex === -1 || lfIndex - 1 > crIndex)) {
                    controller.enqueue(chunk.slice(0, crIndex));
                    chunk = chunk.slice(crIndex + 1);
                    continue;
                }
            }
            if (lfIndex !== -1) {
                let crOrLfIndex = lfIndex;
                if (chunk[lfIndex - 1] === "\r") {
                    crOrLfIndex--;
                }
                controller.enqueue(chunk.slice(0, crOrLfIndex));
                chunk = chunk.slice(lfIndex + 1);
                continue;
            }
            break;
        }
        this.#buf = chunk;
    }
}
/** Transform a stream into a stream where each chunk is divided by a given delimiter.
 *
 * ```ts
 * import { DelimiterStream } from "https://deno.land/std@$STD_VERSION/streams/delimiter.ts";
 * const res = await fetch("https://example.com");
 * const parts = res.body!
 *   .pipeThrough(new DelimiterStream(new TextEncoder().encode("foo")))
 *   .pipeThrough(new TextDecoderStream());
 * ```
 */ export class DelimiterStream extends TransformStream {
    #bufs = new BytesList();
    #delimiter;
    #inspectIndex = 0;
    #matchIndex = 0;
    #delimLen;
    #delimLPS;
    constructor(delimiter){
        super({
            transform: (chunk, controller)=>{
                this.#handle(chunk, controller);
            },
            flush: (controller)=>{
                controller.enqueue(this.#bufs.concat());
            }
        });
        this.#delimiter = delimiter;
        this.#delimLen = delimiter.length;
        this.#delimLPS = createLPS(delimiter);
    }
    #handle(chunk1, controller1) {
        this.#bufs.add(chunk1);
        let localIndex = 0;
        while(this.#inspectIndex < this.#bufs.size()){
            if (chunk1[localIndex] === this.#delimiter[this.#matchIndex]) {
                this.#inspectIndex++;
                localIndex++;
                this.#matchIndex++;
                if (this.#matchIndex === this.#delimLen) {
                    // Full match
                    const matchEnd = this.#inspectIndex - this.#delimLen;
                    const readyBytes = this.#bufs.slice(0, matchEnd);
                    controller1.enqueue(readyBytes);
                    // Reset match, different from KMP.
                    this.#bufs.shift(this.#inspectIndex);
                    this.#inspectIndex = 0;
                    this.#matchIndex = 0;
                }
            } else {
                if (this.#matchIndex === 0) {
                    this.#inspectIndex++;
                    localIndex++;
                } else {
                    this.#matchIndex = this.#delimLPS[this.#matchIndex - 1];
                }
            }
        }
    }
}
/** Transform a stream into a stream where each chunk is divided by a given delimiter.
 *
 * ```ts
 * import { TextDelimiterStream } from "https://deno.land/std@$STD_VERSION/streams/delimiter.ts";
 * const res = await fetch("https://example.com");
 * const parts = res.body!
 *   .pipeThrough(new TextDecoderStream())
 *   .pipeThrough(new TextDelimiterStream("foo"));
 * ```
 */ export class TextDelimiterStream extends TransformStream {
    #buf = "";
    #delimiter;
    #inspectIndex = 0;
    #matchIndex = 0;
    #delimLPS;
    constructor(delimiter){
        super({
            transform: (chunk, controller)=>{
                this.#handle(chunk, controller);
            },
            flush: (controller)=>{
                controller.enqueue(this.#buf);
            }
        });
        this.#delimiter = delimiter;
        this.#delimLPS = createLPS(new TextEncoder().encode(delimiter));
    }
    #handle(chunk2, controller2) {
        this.#buf += chunk2;
        let localIndex1 = 0;
        while(this.#inspectIndex < this.#buf.length){
            if (chunk2[localIndex1] === this.#delimiter[this.#matchIndex]) {
                this.#inspectIndex++;
                localIndex1++;
                this.#matchIndex++;
                if (this.#matchIndex === this.#delimiter.length) {
                    // Full match
                    const matchEnd1 = this.#inspectIndex - this.#delimiter.length;
                    const readyString = this.#buf.slice(0, matchEnd1);
                    controller2.enqueue(readyString);
                    // Reset match, different from KMP.
                    this.#buf = this.#buf.slice(this.#inspectIndex);
                    this.#inspectIndex = 0;
                    this.#matchIndex = 0;
                }
            } else {
                if (this.#matchIndex === 0) {
                    this.#inspectIndex++;
                    localIndex1++;
                } else {
                    this.#matchIndex = this.#delimLPS[this.#matchIndex - 1];
                }
            }
        }
    }
}
/** Generate longest proper prefix which is also suffix array. */ function createLPS(pat) {
    const lps = new Uint8Array(pat.length);
    lps[0] = 0;
    let prefixEnd = 0;
    let i = 1;
    while(i < lps.length){
        if (pat[i] == pat[prefixEnd]) {
            prefixEnd++;
            lps[i] = prefixEnd;
            i++;
        } else if (prefixEnd === 0) {
            lps[i] = 0;
            i++;
        } else {
            prefixEnd = lps[prefixEnd - 1];
        }
    }
    return lps;
}
//# sourceMappingURL=data:application/json;base64,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