// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
import { getLevelByName, getLevelName, LogLevels } from "./levels.ts";
export class LogRecord {
    msg;
    #args;
    #datetime;
    level;
    levelName;
    loggerName;
    constructor(options){
        this.msg = options.msg;
        this.#args = [
            ...options.args
        ];
        this.level = options.level;
        this.loggerName = options.loggerName;
        this.#datetime = new Date();
        this.levelName = getLevelName(options.level);
    }
    get args() {
        return [
            ...this.#args
        ];
    }
    get datetime() {
        return new Date(this.#datetime.getTime());
    }
}
export class Logger {
    #level;
    #handlers;
    #loggerName;
    constructor(loggerName, levelName, options = {}){
        this.#loggerName = loggerName;
        this.#level = getLevelByName(levelName);
        this.#handlers = options.handlers || [];
    }
    get level() {
        return this.#level;
    }
    set level(level) {
        this.#level = level;
    }
    get levelName() {
        return getLevelName(this.#level);
    }
    set levelName(levelName) {
        this.#level = getLevelByName(levelName);
    }
    get loggerName() {
        return this.#loggerName;
    }
    set handlers(hndls) {
        this.#handlers = hndls;
    }
    get handlers() {
        return this.#handlers;
    }
    /** If the level of the logger is greater than the level to log, then nothing
   * is logged, otherwise a log record is passed to each log handler.  `msg` data
   * passed in is returned.  If a function is passed in, it is only evaluated
   * if the msg will be logged and the return value will be the result of the
   * function, not the function itself, unless the function isn't called, in which
   * case undefined is returned.  All types are coerced to strings for logging.
   */ #_log(level, msg, ...args) {
        if (this.level > level) {
            return msg instanceof Function ? undefined : msg;
        }
        let fnResult;
        let logMessage;
        if (msg instanceof Function) {
            fnResult = msg();
            logMessage = this.asString(fnResult);
        } else {
            logMessage = this.asString(msg);
        }
        const record = new LogRecord({
            msg: logMessage,
            args: args,
            level: level,
            loggerName: this.loggerName
        });
        this.#handlers.forEach((handler)=>{
            handler.handle(record);
        });
        return msg instanceof Function ? fnResult : msg;
    }
    asString(data) {
        if (typeof data === "string") {
            return data;
        } else if (data === null || typeof data === "number" || typeof data === "bigint" || typeof data === "boolean" || typeof data === "undefined" || typeof data === "symbol") {
            return String(data);
        } else if (data instanceof Error) {
            return data.stack;
        } else if (typeof data === "object") {
            return JSON.stringify(data);
        }
        return "undefined";
    }
    debug(msg, ...args) {
        return this.#_log(LogLevels.DEBUG, msg, ...args);
    }
    info(msg, ...args) {
        return this.#_log(LogLevels.INFO, msg, ...args);
    }
    warning(msg, ...args) {
        return this.#_log(LogLevels.WARNING, msg, ...args);
    }
    error(msg, ...args) {
        return this.#_log(LogLevels.ERROR, msg, ...args);
    }
    critical(msg, ...args) {
        return this.#_log(LogLevels.CRITICAL, msg, ...args);
    }
}
//# sourceMappingURL=data:application/json;base64,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