// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/**
 * {@linkcode encode} and {@linkcode decode} for
 * [Ascii85/base85](https://en.wikipedia.org/wiki/Ascii85) encoding.
 *
 * This module is browser compatible.
 *
 * @module
 */ const rfc1924 = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz!#$%&()*+-;<=>?@^_`{|}~";
const Z85 = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ.-:+=^!/*?&<>()[]{}@%$#";
/**
 * Encodes a given Uint8Array into ascii85, supports multiple standards
 * @param uint8 input to encode
 * @param [options] encoding options
 * @param [options.standard=Adobe] encoding standard (Adobe, btoa, RFC 1924 or Z85)
 * @param [options.delimiter] whether to use a delimiter, if supported by encoding standard
 */ export function encode(uint8, options) {
    const standard = options?.standard ?? "Adobe";
    let output = [], v, n = 0, difference = 0;
    if (uint8.length % 4 !== 0) {
        const tmp = uint8;
        difference = 4 - tmp.length % 4;
        uint8 = new Uint8Array(tmp.length + difference);
        uint8.set(tmp);
    }
    const view = new DataView(uint8.buffer);
    for(let i = 0, len = uint8.length; i < len; i += 4){
        v = view.getUint32(i);
        // Adobe and btoa standards compress 4 zeroes to single "z" character
        if ((standard === "Adobe" || standard === "btoa") && v === 0 && i < len - difference - 3) {
            output[n++] = "z";
            continue;
        }
        // btoa compresses 4 spaces - that is, bytes equal to 32 - into single "y" character
        if (standard === "btoa" && v === 538976288) {
            output[n++] = "y";
            continue;
        }
        for(let j = 4; j >= 0; j--){
            output[n + j] = String.fromCharCode(v % 85 + 33);
            v = Math.trunc(v / 85);
        }
        n += 5;
    }
    switch(standard){
        case "Adobe":
            if (options?.delimiter) {
                return `<~${output.slice(0, output.length - difference).join("")}~>`;
            }
            break;
        case "btoa":
            if (options?.delimiter) {
                return `xbtoa Begin\n${output.slice(0, output.length - difference).join("")}\nxbtoa End`;
            }
            break;
        case "RFC 1924":
            output = output.map((val)=>rfc1924[val.charCodeAt(0) - 33]);
            break;
        case "Z85":
            output = output.map((val)=>Z85[val.charCodeAt(0) - 33]);
            break;
    }
    return output.slice(0, output.length - difference).join("");
}
/**
 * Decodes a given ascii85 encoded string.
 * @param ascii85 input to decode
 * @param [options] decoding options
 * @param [options.standard=Adobe] encoding standard used in the input string (Adobe, btoa, RFC 1924 or Z85)
 */ export function decode(ascii85, options) {
    const encoding = options?.standard ?? "Adobe";
    // translate all encodings to most basic adobe/btoa one and decompress some special characters ("z" and "y")
    switch(encoding){
        case "Adobe":
            ascii85 = ascii85.replaceAll(/(<~|~>)/g, "").replaceAll("z", "!!!!!");
            break;
        case "btoa":
            ascii85 = ascii85.replaceAll(/(xbtoa Begin|xbtoa End|\n)/g, "").replaceAll("z", "!!!!!").replaceAll("y", "+<VdL");
            break;
        case "RFC 1924":
            ascii85 = ascii85.replaceAll(/./g, (match)=>String.fromCharCode(rfc1924.indexOf(match) + 33));
            break;
        case "Z85":
            ascii85 = ascii85.replaceAll(/./g, (match)=>String.fromCharCode(Z85.indexOf(match) + 33));
            break;
    }
    //remove all invalid characters
    ascii85 = ascii85.replaceAll(/[^!-u]/g, "");
    const len = ascii85.length, output = new Uint8Array(len + 4 - len % 4);
    const view = new DataView(output.buffer);
    let v = 0, n = 0, max = 0;
    for(let i = 0; i < len;){
        for(max += 5; i < max; i++){
            v = v * 85 + (i < len ? ascii85.charCodeAt(i) : 117) - 33;
        }
        view.setUint32(n, v);
        v = 0;
        n += 4;
    }
    return output.slice(0, Math.trunc(len * 0.8));
}
//# sourceMappingURL=data:application/json;base64,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